/********************************************************************************
 *   Jabberoo/Judo -- C++ Jabber Library                                        *
 *                                                                              * 
 *   Copyright (C) 1999-2000 Dave Smith (dave@jabber.org)                       *
 *                                                                              *
 *   This library is free software; you can redistribute it and/or              *
 *   modify it under the terms of the GNU Lesser General Public                 *
 *   License as published by the Free Software Foundation; either               *
 *   version 2.1 of the License, or (at your option) any later version.         *
 *                                                                              *
 *   This library is distributed in the hope that it will be useful,            *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          *
 *   Lesser General Public License for more details.                            *
 *                                                                              *
 *   You should have received a copy of the GNU Lesser General Public           *
 *   License along with this library; if not, write to the Free Software        *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA  *
 ********************************************************************************
 */

#ifndef INCL_JABBEROO_H
#define INCL_JABBEROO_H

#ifdef WIN32
#pragma warning (disable:4786)
#pragma warning (disable:4503)
#include <windows.h>
#define snprintf _snprintf
#include <string.h>
#endif

#include <cstdio>
#include <cstring>
#include <map>
#include <string>
#include <set>
#include <algorithm>
#include <iostream>
using namespace std;

#include <jutil.hh>
#include <judo.hh>
using namespace judo;

#include <sigc++/signal_system.h>
#include <sigc++/object_slot.h>
using namespace SigC;

namespace jabberoo
{
     // Predeclarations
     class Message;
     class Presence;
     class Roster;
     class Session;

     const int ERR_UNAUTHORIZED = 401;

     /***********XCP CLASSES**********/
     class XCP 
	  {
	  public:
	       const string& getMessage() const { return _msg; }
	       XCP(const char* msg) : _msg(msg) {} 
	       XCP() {}
	  private:
	       string _msg;
	  };

     class XCP_NotImplemented : public XCP 
	  {
	  public:
	       XCP_NotImplemented( const char * msg ) : XCP( msg ) { }
	  };


     class Packet
	  {
	  public:
	       Packet(const string& name);
	       Packet(const Tag& t);
	       // Common ops
	       const string getFrom()    const;
	       const string getTo()      const;
	       const string toXML()      const;
	       const Tag&   getBaseTag() const;

	       // Add X tag to the packet
	       Tag&  addX();
	       // Access to the base tag
	       Tag&  getBaseTag();
	  protected:
	       Tag _base;
	  };


     class Message
	  : public Packet
	  {
	  public:
	       // Make sure numTypes is updated in jabberoo-message.cc if you add
	       // a new Message Type
	       static const unsigned int numTypes;
	       enum Type {
		    mtNormal, mtError, mtChat, mtGroupchat, mtHeadline
	       };
	       Message(const Tag& t);
	       Message(const string& jid, const string& body, Type mtype = mtNormal);
	       // Modifiers
	       void  setSubject(const string& subject);
	       // Accessors
	       const string getBody()     const;
	       const string getSubject()  const;
	       const string getError()    const;
	       const string getDateTime(const string& format = "") const;
	       const string getThread()   const;
	       Tag*         getX()        const;
	       Type         getType()     const;
	       // Factory methods
	       Message replyTo(const string& body) const;
	       // Helper funcs
	       static string translateType(Type mtype);
	       static Type   translateType(const string& mtype);
	       // Static class methods
	       static void setDateTimeFormat(const string& format);
	       static const string& getDateTimeFormat();
	  protected:
	       // Specialized reply-to constructor
	       Message(const Message& m, const string& body);
	  private:
	       Type  _type;
	       time_t _timestamp;
	       static string _dtFormat;
	  };

     class Presence
	  : public Packet
	  {
	  public:
	       enum Type {
		    ptSubRequest, ptUnsubRequest, ptSubscribed, ptUnsubscribed, ptAvailable, ptUnavailable
	       };
	       enum Show {
		    stInvalid, stOffline, stOnline, stChat, stAway, stXA, stDND
	       };

	       Presence(const Tag& t);
	       Presence(const string& jid, Type ptype, Show stype = stInvalid, const string& status = "", const string& priority = "0");
	       // Accessors
	       Type          getType()     const;
	       Show          getShow()     const;
	       const string  getStatus()   const;
	       const string  getShow_str() const;
	       int           getPriority() const;
	       // Helper funcs
	       static string translateType(Type ptype);
	       static Type   translateType(const string& ptype);
	       static string translateShow(Show stype);
	       static Show   translateShow(Type ptype, const string& stype);
	  private:
	       void setShow(Show stype);
	       void setShow(const string& show);
	       Show   _show;
	       Type   _type;
	       int    _priority;
	  };



     class PresenceDB
	  : public SigC::Object
     {
     public:
	  typedef list<Presence>::iterator             iterator;
	  typedef list<Presence>::const_iterator       const_iterator;
	  typedef pair<const_iterator, const_iterator> range;
	  class XCP_InvalidJID : public XCP{};
	  
	  PresenceDB(Session& s);
     public:
	  void           insert(const Presence& p);
	  void		 clear();
	  range          equal_range(const string& jid) const;
	  const_iterator find(const string& jid) const;
	  bool           contains(const string& jid) const;
	  bool           available(const string& jid) const;
     private:
	  typedef map<string, list<Presence>, jutil::CaseInsensitiveCmp > db;
	  db::const_iterator find_or_throw(const string& jid) const;
	  Session&   _Owner;
	  db         _DB;
     };

     class Roster
	  : public SigC::Object
	  {
	  public:
               enum Subscription {
                    rsNone, rsTo, rsFrom, rsBoth, rsRemove
               };

	       // Exceptions
	       class XCP_InvalidJID : public XCP{};

               class Item {
               public:
                    // Initializers
                    Item(Roster& r, const Tag& t);
		    Item(const string& jid, const string& nickname);
		    ~Item();
               public:
		    // Group modifiers
                    void addToGroup(const string& group);
                    void delFromGroup(const string& group);
                    void clearGroups();
		    void setNickname(const string& nickname);
                    
                    // Info ops
                    bool             isAvailable() const;
                    string           getNickname() const;
                    string           getJID() const ;
                    Subscription     getSubsType() const;

                    // Group pseudo-container ops/iterators
		    typedef set<string>::const_iterator iterator;
		    iterator begin() const { return _groups.begin(); }
		    iterator end()   const { return _groups.end(); }
		    bool     empty() const { return _groups.empty(); }

	       protected:
                    // Update handlers
                    void update(Roster& r, const string& jid, const Presence& p, Presence::Type prev_type);
                    bool update(Roster& r, const Tag& t);
               private:
		    friend class Roster;
		    int          _rescnt;
                    set<string>  _groups;
                    Subscription _type;
                    string       _nickname;
                    string       _jid;
               };
	       
	       typedef map<string, Item, jutil::CaseInsensitiveCmp> ItemMap;

	       // Non-const iterators
	       typedef jutil::ValueIterator<ItemMap::iterator, Item > iterator;
	       iterator begin() { return _items.begin(); }
	       iterator end() { return _items.end(); }

	       // Const iterators
	       typedef jutil::ValueIterator<ItemMap::const_iterator, const Item> const_iterator;
	       const_iterator begin() const { return _items.begin(); }
	       const_iterator end() const { return _items.end(); }

               // Intializers
               Roster(Session& s);

	       // Operators
	       const Roster::Item& operator[](const string& jid) const;
	       Roster& operator<<(const Item& i) { update(i); return *this; }
          public:
               void reset();

               // Information
               bool containsJID(const string& jid) const;

               // Update ops
               void update(const Tag& t);	   // Roster push
               void update(const Presence& p, Presence::Type prev_type);     // Roster presence
	       void update(const Item& i);	   // Roster item add/modify

               // Control ops
               void deleteUser(const string& jid) const;                       /* Remove the user w/ JID */
               void fetch() const;                                             /* Retrieve roster from server */             

               // Translation
               static string translateS10N(Subscription stype);
               static Subscription translateS10N(const string& stype);
               static string filterJID(const string& jid);

	       // Item/Group access -- HACK
               const map<string, set<string> >& getGroups() const { return _groups;} 
          public:
               Signal0<void, Marshal<void> >                      evtRefresh;
	       Signal3<void, const string&, bool, Presence::Type, Marshal<void> > evtPresence;
          private:
	       friend class Item;
	       void mergeItemGroups(const string& itemjid, const set<string>& oldgrp, const set<string>& newgrp);
               void removeItemFromGroup(const string& group, const string& jid);
	       void removeItemFromAllGroups(const Item& item);
	       void deleteAgent(const Tag& iq);
               ItemMap                   _items;
               map<string, set<string> > _groups;
               Session&                  _owner;
     };

     typedef Slot1<void, const Tag&> TagCallbackFunc;

     class Session : 
	  public TagStream, public SigC::Object
	  {
	  public:
	       enum AuthType
	       { 
		    atPlaintextAuth, atDigestAuth, atAutoAuth, at0kAuth
	       };

	       enum ConnectionState
	       {
		    csNotConnected, csCreateUser, csAuthReq, csAwaitingAuth, csConnected
	       };

	       // Initializers
	       Session();
	       virtual ~Session();
	       // Helper ops
	       Session& operator>>(const char* buffer) { push(buffer); return *this; } 
	       Session& operator<<(const Packet& p) { evtTransmitPacket(p); evtTransmitXML(p.toXML().c_str()); return *this;}
	       Session& operator<<(const char* buffer) { evtTransmitXML(buffer); return *this;}
	  public:
	       // Connectivity ops
	       void connect(const string& server, AuthType atype, 
			    const string& username, const string& resource, const string& password, 
			    bool createuser = false);
	       bool disconnect();

	       ConnectionState getConnState() { return _ConnState; }

	       // Misc ops
	       virtual void push(const char* data);
	       void registerIQ(const string& id, TagCallbackFunc f);
	       void queryNamespace(const string& nspace, TagCallbackFunc f, const string& to = "");

	       // Structure accessors
	       const Roster& roster() const;
	       Roster&       roster();

	       const PresenceDB& presenceDB() const;
	       PresenceDB&       presenceDB();

	       // Property accessors
	       AuthType      getAuthType() const;
	       const string& getUserName() const;

               // ID/Auth ops
	       string getNextID();
	       string getDigest();

	  public:
	       // XML transmission signals
	       Signal1<void, const char*, Marshal<void> >        evtTransmitXML;
	       Signal1<void, const char*, Marshal<void> >        evtRecvXML;
	       Signal1<void, const Packet&, Marshal<void> >      evtTransmitPacket;
	       // Connectivity signals
	       Signal1<void, const Tag&, Marshal<void> >         evtConnected;
	       Signal0<void, Marshal<void> >                     evtDisconnected;
	       // Basic signals
	       Signal1<void, const Message&, Marshal<void> >     evtMessage;
	       Signal1<void, const Tag&, Marshal<void> >         evtIQ;
	       Signal2<void, const Presence&, Presence::Type, Marshal<void> >    evtPresence;
	       Signal1<void, const Presence&, Marshal<void> >    evtPresenceRequest;
	       // Misc signals
	       Signal1<void, const Tag&, Marshal<void> >         evtUnknownPacket;
	       Signal2<void, int, const char*, Marshal<void> >   evtAuthError;	  
	       Signal3<void, string&, string&, string&, Marshal<void> > evtOnVersion;
	  protected:
	       // Authentication handler
	       void authenticate();
	       void OnAuthTypeReceived(const Tag& t);
	       void sendLogin(Session::AuthType atype, const Tag* squery);

	       // TagStream events
	       virtual void OnDocumentStart(Tag* t);
	       virtual void OnTag(Tag* t);
	       virtual void OnDocumentEnd();

	       // Basic packet handlers
	       void handleMessage(Tag& t);
	       void handlePresence(Tag& t);
	       void handleIQ(Tag& t);

	  private:
	       // Values
	       string          _ServerID;
	       string          _Username;
	       string          _Password;
	       string          _Resource;
	       string          _SessionID;
	       long            _ID;
	       // Internal roster & presence db structures
	       Roster          _Roster;
	       PresenceDB      _PDB;
	       // Enumeration values (states)
	       AuthType        _AuthType;
	       ConnectionState _ConnState;
 	       // Stream header 
	       Tag*            _StreamTag;
	       // Wether or not we have received the starting stream tag
	       bool            _StreamStart;
	       // Structures
	       multimap<string, TagCallbackFunc> _Callbacks;			 /* IQ callback funcs */
	       // Internal IQ handlers
	       void IQHandler_Auth(const Tag& t);
	       void IQHandler_CreateUser(const Tag& t);
	  };    

     class JID 
	  {
	  public:
	       static string getResource(const string& jid);
	       static string getUserHost(const string& jid);	    
	       static string getHost(const string& jid);
	       static string getUser(const string& jid);
	  };

}

#ifdef WIN32
     char* shahash(const char* str);
#else
extern "C" {
     char* shahash(const char* str);
}
#endif


#endif
