/********************************************************************************
 *   Jabberoo/Judo -- C++ Jabber Library                                        *
 *                                                                              * 
 *   Copyright (C) 1999-2000 Dave Smith (dave@jabber.org)                       *
 *                                                                              *
 *   This library is free software; you can redistribute it and/or              *
 *   modify it under the terms of the GNU Lesser General Public                 *
 *   License as published by the Free Software Foundation; either               *
 *   version 2.1 of the License, or (at your option) any later version.         *
 *                                                                              *
 *   This library is distributed in the hope that it will be useful,            *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          *
 *   Lesser General Public License for more details.                            *
 *                                                                              *
 *   You should have received a copy of the GNU Lesser General Public           *
 *   License along with this library; if not, write to the Free Software        *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA  *
 ********************************************************************************
 */

#ifndef INCL_JUDO_H
#define INCL_JUDO_H

#ifdef WIN32
#pragma warning (disable:4786)
#include <windows.h>
#endif

#include <xmlparse.h>

#include <cstdio>
#include <cstring>
#include <list>
#include <map>
#include <queue>
#include <stack>
#include <string>
#include <set>
#include <algorithm>
using namespace std;

namespace judo
{

     // Predeclarations
     class Element;
     class CDATA;
     class Attribute;
     class Tag;
     class TagStream;

     // Exceptions
     class XCP {};
     class XCP_ParserFailure : public XCP{};

     // Data structure typedefs
     typedef list<Element*> ElementList;
     typedef unary_function<Element*, bool> ElementPredicate;

     class Element
     {
     public:
	  enum Type
	  {
	       etTag, etCDATA
	  };
	  Element(const string& name, Type etype);
	  virtual ~Element() {}
     public:
	  const string&        getName()   const;
	  Element::Type        getType()   const;
	  virtual string       getXML() const = 0;
	  // Operators
	  friend bool operator==(const Element& lhs, Element::Type rhs);
	  friend bool operator==(const Element& lhs, const string& rhs);
     protected:
	  string        _name;
	  Element::Type _type;	  
     };

     // Operators
     bool operator==(const Element& lhs, Element::Type rhs);
     bool operator==(const Element& lhs, const string& rhs);


     class CDATA : public Element
     {
     public:
	  CDATA(const char* text, unsigned int textSz);
     public:
	  void          setText(const char* text, unsigned int textSz);
	  void          appendText(const char* text, unsigned int textSz);
	  const string& getText() const;
	  string getXML() const;
     protected:
	  string _text;           
     };

     class Tag : public Element
     {
     public:
	  // Initializers
	  Tag(const string& name, const char** attribs = NULL);
	  virtual ~Tag();
	  // Copy constructor & assignment operator
	  Tag(const Tag& t);
	  Tag& operator=(const Tag& t);
     public:
	  Tag&    addTag(const string& name, const char** attribs = NULL);
	  CDATA&  addCDATA(const char* pCDATA, int iCDATASz);
	  Tag&    addTaggedCDATA(const string& name, const string& value);

	  void         putAttrib(const string& name, const string& value);
	  const string getAttrib(const string& name) const;
	  void         delAttrib(const string& name);
	  bool         cmpAttrib(const string& name, const string& value) const;

	  const ElementList&          getChildren() const;
	  const map<string,string>&   getAttribs() const;
	       
	  const char* getTaggedCDATA(const char* name) const;
	  Tag*        getTag(const char* name, const char* xmlns = NULL) const;

	  bool hasTags() const;
	  bool hasCDATA() const;
	  bool hasAttribs() const;
	  bool hasChildren() const;

	  const char* getData() const;
	  string getXML() const;
     protected:
	  ElementList          _Children;
	  map<string,string>   _Attribs;
	  template <class T> inline T* addChildElement(T* t);
	  template <class T> inline Element* findChild(const T& e) const;
     };

     class TagStream 
     {
     public:
	  TagStream();
	  virtual ~TagStream();
     public:
	  virtual void push(const char* data);
     private:
	  virtual void OnDocumentStart(Tag* t) {}
	  virtual void OnTag(Tag* t) {}
	  virtual void OnDocumentEnd() {}
	  XML_Parser*  _Parser;
	  stack<Tag*>  _TagStack;
	  bool         _DocumentStarted;
	  bool	       _DocumentEnded;
	  // Expat callbacks
	  static void OnStartElement(void* pUserdata, const char* pszElementName, const char** apszAttribs);
	  static void OnEndElement(void* pUserdata, const char* pszElementName);
	  static void OnCDATA(void* pUserdata, const char* pszCDATA, int iCDATASz);
	  // Expat initializers
	  void initExpat();
	  void cleanupExpat();
     };


     // Utility routines
     template <class T> string& operator<<(string& s, T c) { s+=c; return s;}
     void   unescape(const char* src, unsigned int srcLen, string& dest, bool append = false);  
     string escape(const string& src);
}


#endif
