/********************************************************************************
 *   Jabberoo/Judo -- C++ Jabber Library                                        *
 *                                                                              * 
 *   Copyright (C) 1999-2000 Dave Smith (dave@jabber.org)                       *
 *                                                                              *
 *   This library is free software; you can redistribute it and/or              *
 *   modify it under the terms of the GNU Lesser General Public                 *
 *   License as published by the Free Software Foundation; either               *
 *   version 2.1 of the License, or (at your option) any later version.         *
 *                                                                              *
 *   This library is distributed in the hope that it will be useful,            *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          *
 *   Lesser General Public License for more details.                            *
 *                                                                              *
 *   You should have received a copy of the GNU Lesser General Public           *
 *   License along with this library; if not, write to the Free Software        *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA  *
 ********************************************************************************
 */

#include <jabberoo.hh>

using namespace jabberoo;

const unsigned int Message::numTypes = 5;

Message::Message(const Tag& t)
     : Packet(t)
{
     // Determine message type..
     _type = translateType(t.getAttrib("type"));

#ifndef WIN32
     Tag *x;
     string date;

     x = _base.getTag("x", "jabber:x:delay");
     if (x) {
	  date = x->getAttrib("stamp");
	  if (!date.empty()) {
	       struct tm ts;
	       memset(&ts, 0, sizeof(struct tm));
	       sscanf(date.c_str(), "%04d%02d%02dT%02d:%02d:%02d", &ts.tm_year, &ts.tm_mon, &ts.tm_mday, 
		      &ts.tm_hour, &ts.tm_min, &ts.tm_sec);
	       ts.tm_year -= 1900;
	       ts.tm_mon -= 1;
#ifdef MACOS
			// there is no gmtime on MacOS, but it seems the stdlib ignores timezones anyway,
			// thus we can safely use mktime here.
	       _timestamp = mktime(&ts);
#else
	       _timestamp = timegm(&ts);
#endif
	       return;
	  }
     }
     _timestamp = time(0);
#endif     
}

Message::Message(const string& jid, const string& body, Message::Type mtype)
     : Packet("message")
{
     _base.putAttrib("to", jid);
     _base.putAttrib("type", translateType(mtype));
     _base.addTaggedCDATA("body", body);
     _timestamp = time(0);
     _type      = mtype;
}

void Message::setSubject(const string& subject)
{
     _base.addTaggedCDATA("subject", subject);
}

const string Message::getBody() const
{
     return string(_base.getTaggedCDATA("body"));
}

const string Message::getSubject() const
{
     return string(_base.getTaggedCDATA("subject"));
}

const string Message::getError() const
{
     return string(_base.getTaggedCDATA("error"));
}

// added format for the future, for custom date formats in the logs
const string Message::getDateTime(const string& format) const
{
#ifndef WIN32
     char timestr[1024];
     struct tm *timestamp;

     timestamp = localtime(&_timestamp);
     if (format.empty())
	  strftime(timestr, 1024, _dtFormat.c_str(), timestamp);
     else
	  strftime(timestr, 1024, format.c_str(), timestamp);
     return string(timestr);
#else
    return "N/A";
#endif   
}

const string Message::getThread() const
{
     return _base.getTaggedCDATA("thread");
}

Tag* Message::getX() const
{
     return _base.getTag("x");
}

Message::Type Message::getType() const
{
     return _type;
}

Message::Message(const Message& m, const string& body)
     : Packet("message")
{
     // Setup basic stuff
     _base.putAttrib("to", m.getFrom());
     _base.putAttrib("type", m._base.getAttrib("type"));
     _base.addTaggedCDATA("body", body);
     // Get thread if available..
     string thread = m.getThread();
     if (thread != "")
	  _base.addTaggedCDATA("thread", thread);
}

Message Message::replyTo(const string& body) const
{
     return Message(*this, body);
}

string Message::translateType(Type mtype)
{
     switch(mtype)
     {
     case mtNormal:
	  return "normal";
     case mtError:
	  return "error";
     case mtChat:
	  return "chat";
     case mtGroupchat:
	  return "groupchat";
     case mtHeadline:
	  return "headline";
     }
}

Message::Type Message::translateType(const string& mtype)
{
     // Determine what type of message packet this is
     if (mtype == "")
	  return mtNormal;
     else if (mtype == "normal")
	  return mtNormal;
     else if (mtype == "error")
	  return mtError;
     else if (mtype == "chat")
	  return mtChat;
     else if (mtype == "groupchat")
	  return mtGroupchat;
     else if (mtype == "headline")
	  return mtHeadline;
}

string Message::_dtFormat  = "%d %b %Y %H:%M:%S";

void Message::setDateTimeFormat(const string& format)
{
     _dtFormat = format;
}

const string& Message::getDateTimeFormat()
{
     return _dtFormat;
}
