/********************************************************************************
 *   Jabberoo/Judo -- C++ Jabber Library                                        *
 *                                                                              * 
 *   Copyright (C) 1999-2000 Dave Smith (dave@jabber.org)                       *
 *                                                                              *
 *   This library is free software; you can redistribute it and/or              *
 *   modify it under the terms of the GNU Lesser General Public                 *
 *   License as published by the Free Software Foundation; either               *
 *   version 2.1 of the License, or (at your option) any later version.         *
 *                                                                              *
 *   This library is distributed in the hope that it will be useful,            *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          *
 *   Lesser General Public License for more details.                            *
 *                                                                              *
 *   You should have received a copy of the GNU Lesser General Public           *
 *   License along with this library; if not, write to the Free Software        *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA  *
 ********************************************************************************
 */

#include <jabberoo.hh>

using namespace jabberoo;

Presence::Presence(const Tag& t)
     : Packet(t)
{
     // Determine the presence type
     _type = translateType(t.getAttrib("type"));
     // Determine the show value
     _show = translateShow(_type, t.getTaggedCDATA("show"));
}

Presence::Presence(const string& jid, Presence::Type ptype, Presence::Show stype, const string& status, const string& priority)
     : Packet("presence"), _show(stype)
{
     if (!jid.empty())
	  _base.putAttrib("to", jid);
     if (ptype != Presence::ptAvailable)
	  _base.putAttrib("type", translateType(ptype));
     if (!status.empty())
	  _base.addTaggedCDATA("status", status);
     // Add a <show> element if necessary
     if (_show > stOnline)
     {
	  _base.addTaggedCDATA("show", translateShow(_show));
     }
     // Add a <priority> element if necessary
     if (priority != "0")
     {
	  _base.addTaggedCDATA("priority", priority);
     }

}


Presence::Type Presence::getType() const
{
     return _type;
}

const string Presence::getStatus() const
{
     if (_base.getTaggedCDATA("status") == "")
	  return _base.getTaggedCDATA("error");
     else
	  return _base.getTaggedCDATA("status");
}

Presence::Show Presence::getShow() const
{
     return _show;
}

const string Presence::getShow_str() const
{
     return translateShow(_show);
}

int Presence::getPriority() const
{
     if (_type == Presence::ptUnavailable)
	  return -1;
     else
	  return atoi(_base.getTaggedCDATA("priority"));
}

string Presence::translateType(Type ptype)
{
     switch(ptype)
     {
     case ptAvailable: 
	  return string("");
     case ptUnavailable: 
	  return string("unavailable");
     case ptSubRequest: 
	  return string("subscribe");
     case ptUnsubRequest: 
	  return string("unsubscribe");
     case ptSubscribed : 
	  return string("subscribed");
     case ptUnsubscribed:
	  return string("unsubscribed");
     }
     return string("");
}

string Presence::translateShow(Show stype)
{
     switch(stype)
     {
     case stInvalid:
	  return string("");
     case stOnline:
	  return string("online");
     case stOffline:
	  return string("offline");
     case stChat:
	  return string("chat");
     case stAway:
	  return string("away");
     case stXA:
	  return string("xa");
     case stDND:
	  return string("dnd");
     }
     return string("");
}

Presence::Type Presence::translateType(const string& ptype)
{
     // Determine what type of presence packet this is
     if (ptype == "")
	  return ptAvailable;
     else if (ptype == "subscribe")
	  return ptSubRequest;
     else if (ptype == "unsubscribe")
	  return ptUnsubRequest;
     else if (ptype == "subscribed")
	  return ptSubscribed;
     else if (ptype == "unsubscribed")
	  return ptUnsubscribed;
     else
	  return ptUnavailable;
}

Presence::Show Presence::translateShow(Type ptype, const string& show)
{
     // Determine what value show should return
     if ((ptype == ptAvailable) && (show == ""))
	  return stOnline;
     else if ((ptype == ptAvailable) && (show == "online"))
	  return stOnline;
     else if ((ptype == ptUnavailable) && (show == ""))
	  return stOffline;
     else if ((ptype == ptUnavailable) && (show == "offline"))
	  return stOffline;
     else if (show == "chat")
	  return stChat;
     else if (show == "away")
	  return stAway;
     else if (show == "xa")
	  return stXA;
     else if (show == "dnd")
	  return stDND;
     else
	  return stOffline;
}

void Presence::setShow(Show stype)
{
     // Ensure this is a valid packet type for <show> element
     switch (_type)
     {
     case ptAvailable:
     case ptUnavailable:
	  _show = stype;
	  if (_show > stOnline)
	       _base.addTaggedCDATA("show", translateShow(_show));
	  break;
     default:
	  _show = stInvalid;
     }
}


