/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * 
 *  Gabber
 *  Copyright (C) 1999-2001 Dave Smith & Julian Missig
 */

#include "Gabber.hh"
#include "GabberApp.hh"
#include "GabberUtility.hh"
#include "AgentInterface.hh"
#include "ContactInfoInterface.hh"
#include "AddContactDruid.hh"

using namespace jabberoo;
using namespace GabberUtil;

// ---------------------------------------------------------
//
// AgentBrowser
//
// ---------------------------------------------------------
AgentBrowser::AgentBrowser(const string& server, AgentList l)
     : BaseGabberDialog("AgentBrowser_dlg"), _agent_list(l)
{
     // Store widget pointers
     _infoBtn     = getButton("AgentBrowser_Info_btn");
     _registerBtn = getButton("AgentBrowser_Register_btn");
     _searchBtn   = getButton("AgentBrowser_Search_btn");
     _serverTxt   = getEntry("AgentBrowser_Server_txt");
     _agent_tree  = getWidget<Gtk::CTree>("AgentBrowser_ctree");

     // Setup handlers
     _infoBtn->clicked.connect(slot(this, &AgentBrowser::on_info_clicked));
     _registerBtn->clicked.connect(slot(this, &AgentBrowser::on_register_clicked));
     _searchBtn->clicked.connect(slot(this, &AgentBrowser::on_search_clicked));
     getButton("AgentBrowser_Close_btn")->clicked.connect(slot(this, &AgentBrowser::on_close_clicked));
     getButton("AgentBrowser_Browse_btn")->clicked.connect(slot(this, &AgentBrowser::on_browse_clicked));
     _thisWindow->key_press_event.connect(slot(this, &AgentBrowser::on_key_pressed));

     // Setup the tree
     _agent_tree->set_line_style(GTK_CTREE_LINES_NONE);
     _agent_tree->set_expander_style(GTK_CTREE_EXPANDER_TRIANGLE);
     _agent_tree->set_row_height(17);						    // Hack for making sure icons aren't chopped off
     _agent_tree->column_titles_hide();
     _agent_tree->set_compare_func(&GabberUtil::strcasecmp_clist_items);
     _agent_tree->set_column_auto_resize(0, true);
     _agent_tree->set_auto_sort(true);
     _agent_tree->set_indent(20);						  

     // Setup tree events
     _agent_tree->tree_expand.connect(slot(this, &AgentBrowser::on_tree_expand));
     _agent_tree->tree_select_row.connect(slot(this, &AgentBrowser::on_tree_select));
     _agent_tree->tree_unselect_row.connect(slot(this, &AgentBrowser::on_tree_unselect));

     // Grab the history and put it in the gnome entry (a cooler combo box)
     G_App->getCfg().loadEntryHistory(getWidget<Gnome::Entry>("AgentBrowser_Server_gent"));
     // Initialize server entry
     _serverTxt->set_text(fromUTF8(_serverTxt, server));

     // Walk the tree list and render
     _agent_tree->freeze();
     for(AgentList::iterator it = _agent_list.begin(); it != _agent_list.end(); it++)
	  render_agent(*it, NULL);
     _agent_tree->thaw();

     show();
}

void AgentBrowser::on_info_clicked()
{
     if (_current_agent != NULL)
     {
	  manage(new AgentInfoDlg(*_current_agent));
     }
}

void AgentBrowser::on_register_clicked()
{
     if (_current_agent != NULL)
     {
	  manage(new AgentRegisterDruid(*_current_agent));
     }
}

void AgentBrowser::on_search_clicked()
{
     if (_current_agent != NULL)
     {
          AddContactDruid::display(*_current_agent);
     }
}

void AgentBrowser::on_browse_clicked()
{
     // Send request to the server for the update
     G_App->getSession().queryNamespace("jabber:iq:agents", slot(this, &AgentBrowser::handle_agents_IQ), _serverTxt->get_text());
     // Save the gnome entry history
     G_App->getCfg().saveEntryHistory(getWidget<Gnome::Entry>("AgentBrowser_Server_gent"));
}

void AgentBrowser::on_close_clicked()
{
     close();
}

void AgentBrowser::on_tree_expand(Gtk::CTree::Row r)
{
     Agent* a = static_cast<Agent*>(r.get_data());

     // Fetch any subagents
     if ((a != NULL) && a->hasAgents())
	  a->fetch();
}

void AgentBrowser::on_tree_select(Gtk::CTree::Row r, int col)
{
     _current_agent = static_cast<Agent*>(r.get_data());

     if (_current_agent != NULL)
     {
	  // Enable and disable register and search buttons appropriately
	  _infoBtn->set_sensitive(true);
	  _registerBtn->set_sensitive(_current_agent->isRegisterable());
	  _searchBtn->set_sensitive(_current_agent->isSearchable());
     }
}

void AgentBrowser::on_tree_unselect(Gtk::CTree::Row r, int col)
{
     _current_agent = NULL;
     // Disable the buttons
     _infoBtn->set_sensitive(false);
     _registerBtn->set_sensitive(false);
     _searchBtn->set_sensitive(false);
}

void AgentBrowser::on_agent_fetch_complete(bool successful, GtkCTreeNode* agentnode, GtkCTreeNode* dummynode)
{
     _agent_tree->freeze();

     // Get the agent stored in the specified node
     void* nodedata = gtk_ctree_node_get_row_data(_agent_tree->gtkobj(), agentnode);
     Agent& a = *static_cast<Agent*>(nodedata);

     if (successful)
     {
	  // Remove dummy node
	  if (dummynode)
	       gtk_ctree_remove_node(_agent_tree->gtkobj(), dummynode);
	  
	  // Walk the agent's sublist and render individual nodes
	  for (Agent::iterator it = a.begin(); it != a.end(); it++)
	       render_agent(*it, agentnode);
     }
     else
     {
	  // Update dummy node text
	  gtk_ctree_node_set_text(_agent_tree->gtkobj(), dummynode, 0, _("Error occurred querying this agent. Please try again later."));
     }

     // Make sure this agentnode is expanded
     gtk_ctree_expand(_agent_tree->gtkobj(), agentnode);

     _agent_tree->thaw();
}

gint AgentBrowser::on_key_pressed(GdkEventKey* e)
{
     // If Enter is pressed, act like they pressed browse
     switch (e->keyval)
     {
     case GDK_Return:
	  on_browse_clicked();
	  break;
     case GDK_Escape:
	  on_close_clicked();
     }
     return FALSE;
}     

void AgentBrowser::execute()
{
     // Send request to the server for a filter
     G_App->getSession().queryNamespace("jabber:iq:agents", slot(&AgentBrowser::handle_agents_IQ_s), G_App->getCfg().getStrValue("Server/Server=jabber.com"));
}

// Static handler -- instantiates a new window
void AgentBrowser::handle_agents_IQ_s(const Tag& iq)
{
     Gnome::Dialog* d;

     // If we get a good query back, create a new AgentList
     // from the packet and display it in the browser
     if (iq.cmpAttrib("type", "result"))
     {
	  // Extract the query 
	  Tag* query = iq.getTag("query");
	  if (query != NULL)
	       // Create a new browser, passing a new AgentList
	       manage(new AgentBrowser(iq.getAttrib("from"), AgentList(*query, G_App->getSession())));
	  else
	  {
	       d = manage(Gnome::Dialogs::warning(_("Error receiving XML for Agent Browser, see standard output.")));
	       d->set_modal(true); 
	       cerr << "ERROR->Unable to extract query for AgentBrowser: " << iq.getXML() << endl;
	  }
     }
     else
     {
	  d = manage(Gnome::Dialogs::warning(_("Error, this server does not support agents.")));
	  d->set_modal(true);
     }
}

// Normal handler -- updates existing window
void AgentBrowser::handle_agents_IQ(const Tag& iq)
{
     Gnome::Dialog* d;

     // If we get a good query back, clear the existing agentlist and tree
     // and generate a new agent list
     if (iq.cmpAttrib("type", "result"))
     {
	  // Extract the query 
	  Tag* query = iq.getTag("query");
	  if (query != NULL)
	  {
	       _agent_tree->freeze();
	       // Clear the tree
	       _agent_tree->clear();	       
	       // Re load the agent list
	       _agent_list.load(*query, G_App->getSession());
	       // Render the agent list
	       for(AgentList::iterator it = _agent_list.begin(); it != _agent_list.end(); it++)
		    render_agent(*it, NULL);
	       _agent_tree->thaw();
	       // Update the server entry
	       _serverTxt->set_text(fromUTF8(_serverTxt, iq.getAttrib("from")));
	  }
	  else
	  {
	       d = manage(Gnome::Dialogs::warning(_("Error receiving XML for Agent Browser, see standard output.")));
	       d->set_modal(true); 
	       cerr << "ERROR->Unable to extract query for AgentBrowser: " << iq.getXML() << endl;
	  }
     }
     else
     {
	  d = manage(Gnome::Dialogs::warning(_("Error, this server does not support agents.")));
	  d->set_modal(true);
     }
}

void AgentBrowser::render_agent(Agent& a, GtkCTreeNode* parent)
{
     char* header[1] = { (char*)a.name().c_str()};
     // Create the node
     GtkCTreeNode* node = gtk_ctree_insert_node(_agent_tree->gtkobj(), parent, NULL, header, 0,
						NULL, NULL, NULL, NULL, !a.hasAgents(), false);
     if (a.hasAgents())
     {
	  // Insert a dummy row so the expander shows up
	  char* dummy_header[1] = { "Loading..." };
	  GtkCTreeNode* dummynode = gtk_ctree_insert_node(_agent_tree->gtkobj(), node, NULL, dummy_header, 0, 
							  NULL, NULL, NULL, NULL, true, false);
	  // Hook up to the agent's refresh event
	  a.evtFetchComplete.connect(bind(slot(this, &AgentBrowser::on_agent_fetch_complete), node, dummynode));
     }

     // Set row data to be a ref to this agent
     gtk_ctree_node_set_row_data(_agent_tree->gtkobj(), node, &a);
}

// ---------------------------------------------------------
//
// AgentRegisterDruid
//
// ---------------------------------------------------------
AgentRegisterDruid::AgentRegisterDruid(Agent& a)
     : BaseGabberWindow("RegDruid_win"), _agent(a), _init_interface(false)
{
     // Get widgets
     _druid = getWidget<Gnome::Druid>("RegDruid_druid");
     _fields_vbox = getWidget<Gtk::VBox>("RegDruid_fields_vbox");
     _instr_lbl = getLabel("RegDruid_fields_loading_lbl");
     _lblRegistered = getLabel("RegDruid_registered_explain_lbl");
     _barStatus = getWidget<Gnome::AppBar>("RegDruid_Status_bar");
     
     // Hookup event
     _druid->cancel.connect(slot(this, &AgentRegisterDruid::on_cancel));

     // Setup pages
     _RDruid_loading     = getWidget<Gnome::DruidPage>("RegDruid_loading");
     _RDruid_fields      = getWidget<Gnome::DruidPage>("RegDruid_fields");
     _RDruid_fields      ->prepare.connect(slot(this, &AgentRegisterDruid::on_Fields_prepare));
     _RDruid_fields      ->next.connect(slot(this, &AgentRegisterDruid::on_Fields_next));
     _RDruid_registering = getWidget<Gnome::DruidPage>("RegDruid_registering");
     _RDruid_registered  = getWidget<Gnome::DruidPage>("RegDruid_registered");
     _RDruid_registered  ->prepare.connect(slot(this, &AgentRegisterDruid::on_Registered_prepare));
     _RDruid_registered  ->back.connect(slot(this, &AgentRegisterDruid::on_Registered_back));
     _RDruid_registered  ->finish.connect(slot(this, &AgentRegisterDruid::on_finish));

     // Set the JID
     getLabel("RegDruid_loading_jid_lbl")    ->set_text(_agent.JID());
     getLabel("RegDruid_fields_jid_lbl")     ->set_text(_agent.JID());
     getLabel("RegDruid_registering_jid_lbl")->set_text(_agent.JID());
     getLabel("RegDruid_registered_jid_lbl") ->set_text(_agent.JID());

     // Set the agent info
     Gtk::Label* l = getLabel("RegDruid_loading_Service_lbl");
     l->set_text(fromUTF8(l, _agent.service()));
     l = getLabel("RegDruid_loading_Description_lbl");
     l->set_text(fromUTF8(l, _agent.description()));

     // Set the page to the loading page, and disallow back and next
     _druid->set_page(*_RDruid_loading);
     _druid->set_buttons_sensitive(0,0,1);

     // Display
     show();

     _barStatus->pop();
     _barStatus->push(_("Loading registration information..."));
     _barStatus->get_progress()->set_activity_mode(true);

     // Send query to the agent
     _agent.requestRegister(slot(this, &AgentRegisterDruid::handle_request_reply));

}

void AgentRegisterDruid::on_cancel()
{
     close();
}

void AgentRegisterDruid::handle_request_reply(const Tag& iq)
{
     if (iq.cmpAttrib("type", "result"))
     {
          // Extract the query 
          Tag* query = iq.getTag("query");
          if (_init_interface) {
               _key = query->getTaggedCDATA("key");
	  } else {
               ElementList::const_iterator it = query->getChildren().begin();

	       // Create fields table
	       Gtk::Table* fld_tbl = manage(new Gtk::Table(query->getChildren().size()-1, 2));
	       fld_tbl->set_row_spacings(4);
	       fld_tbl->set_col_spacings(4);
	       fld_tbl->set_border_width(0);
	       // Add the table to the vbox
	       _fields_vbox->pack_start(*fld_tbl, true, true, 8);
	       int row = 0;

               for (; it != query->getChildren().end(); it++)
               {
                    // Cast the child element into a tag
                    Tag& t = *static_cast<Tag*>(*it);

                    if (t.getName() == "instructions") {
                         _instr_lbl->set_text(fromUTF8(_instr_lbl, t.getData()));
                    } else if (t.getName() == "registered") {
                         // we are already registered with the agent
                    } else if (t.getName() == "key") {
                         _key = t.getData();
	            } else {
			 // Create a label
			 Gtk::Label* lbl = manage(new Gtk::Label());
			 lbl->set_text(fromUTF8(lbl, t.getName() + ":"));
			 lbl->set_justify(GTK_JUSTIFY_RIGHT);
			 lbl->set_alignment(1.0, 0.0);
			 fld_tbl->attach(*lbl, 0, 1, row, row+1,GTK_FILL,GTK_FILL);

			 // Create entry
			 Gtk::Entry* entry = manage(new Gtk::Entry());
			 // If it's a password, mask it.
			 if (t.getName() == "password")
			      entry->set_visibility(false);

			 fld_tbl->attach(*entry, 1, 2, row, row+1,GTK_EXPAND|GTK_FILL,0);
		         _field_names.push_back(t.getName());
		         string* fieldname = &(_field_names.back());
                         entry->set_data("fieldname", fieldname);
                         _entrys.push_back(entry);

			 row++;
	            }
	       }
	       // Show everything and indicate that we have stuff
	       _fields_vbox->show_all();
	       _init_interface = true;
          }
	  _barStatus->pop();
	  _barStatus->get_progress()->set_activity_mode(false);
	  // Send them to the fields page
          _druid->set_page(*_RDruid_fields);
     }
     else
     {
          Gnome::Dialog* d = manage(Gnome::Dialogs::warning(_("Error attempting to register with agent.")));
          d->set_modal(true);
	  close();
     }
}

void AgentRegisterDruid::on_Fields_prepare()
{
     // They can't go back
     _druid->set_buttons_sensitive(0, 1, 1);
}

bool AgentRegisterDruid::on_Fields_next() 
{
     // They can't do anything except cancel while waiting for the reply
     _druid->set_buttons_sensitive(0, 0, 1);

     EntryList::const_iterator eit = _entrys.begin();

     // Get next session ID
     string id = G_App->getSession().getNextID();

     Tag iq("iq");
     iq.putAttrib("id", id);
     iq.putAttrib("to", _agent.JID());
     iq.putAttrib("type", "set");

     Tag& query = iq.addTag("query");
     query.putAttrib("xmlns", "jabber:iq:register");
     query.addTaggedCDATA("key", _key);

     for (; eit != _entrys.end(); eit++)
     {
	     Gtk::Entry* entry = static_cast<Gtk::Entry*>(*eit);
             string& field = *static_cast<string *>(entry->get_data("fieldname"));

             if (!entry->get_text().empty()) {
                  query.addTaggedCDATA(field, toUTF8(entry, entry->get_text()));
             }
     }
     G_App->getSession() << iq.getXML().c_str();
     G_App->getSession().registerIQ(id, slot(this, &AgentRegisterDruid::on_register_reply));

     _barStatus->pop();
     _barStatus->push(_("Sending registration information..."));
     _barStatus->get_progress()->set_activity_mode(true);

     _druid->set_page(*_RDruid_registering);

     return true;
}

void AgentRegisterDruid::on_register_reply(const Tag& iq)
{
     if (!iq.cmpAttrib("type", "result"))
     {
          Tag* error = iq.getTag("error");
	  string message = _("Registration Failed");
	  _RDruid_registered->set_title(message);
	  if (error)
	  {
               message = message + ":\n" + fromUTF8(_druid, error->getData());
	  }
	  _lblRegistered->set_text(message);
     }
     _barStatus->pop();
     _barStatus->get_progress()->set_activity_mode(false);
     _druid->set_page(*_RDruid_registered);
}

void AgentRegisterDruid::on_Registered_prepare()
{
     // Show the finish button
     _druid->set_buttons_sensitive(0, 1, 1);
     _druid->set_show_finish(true);
}

bool AgentRegisterDruid::on_Registered_back()
{
     _agent.requestRegister(slot(this, &AgentRegisterDruid::handle_request_reply));

     return false;
}

void AgentRegisterDruid::on_finish()
{
     close();
}
