/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * 
 *  Gabber
 *  Copyright (C) 1999-2001 Dave Smith & Julian Missig
 */

#include "GroupsInterface.hh"
#include "Gabber.hh"
#include "GabberApp.hh"

extern "C" {
#ifdef GABBER_WINICON
#include <libgnomeui/gnome-window-icon.h>
#endif
}

using namespace jabberoo;

// ---------------------------------------------------------
//
// Edit Groups Dialog
//
// ---------------------------------------------------------

EditGroupsDlg::EditGroupsDlg(const string& jid)
     : BaseGabberDialog("EditGroups_dlg"),
       _item(G_App->getSession().roster()[jid])
{
     string nickname;
     try {
	  nickname = G_App->getSession().roster()[JID::getUserHost(jid)].getNickname();
     } catch (Roster::XCP_InvalidJID& e) {
	  // Typically we'll just use the username
	  nickname = JID::getUser(jid);
     }
     getLabel("EditGroups_User_lbl")->set_text(nickname + " (" + JID::getUserHost(jid) + ")");

     // Grab pointers
     _entName        = getEntry("EditGroups_NewGroup_txt");
     _clCurrent      = getWidget<Gtk::CList>("EditGroups_Current_clist");
     _clAvailable    = getWidget<Gtk::CList>("EditGroups_Available_clist");

     // Connect buttons
     getButton("EditGroups_Add_btn")->clicked.connect(slot(this, &EditGroupsDlg::on_add_clicked));
     getButton("EditGroups_Remove_btn")->clicked.connect(slot(this, &EditGroupsDlg::on_remove_clicked));
     getButton("EditGroups_OK_btn")->clicked.connect(slot(this, &EditGroupsDlg::on_ok_clicked));
     getButton("EditGroups_Cancel_btn")->clicked.connect(slot(this, &EditGroupsDlg::on_cancel_clicked));
     getButton("EditGroups_Help_btn")->clicked.connect(slot(this, &EditGroupsDlg::on_help_clicked));

     // Connect Available Groups list
     _clAvailable->select_row.connect(slot(this, &EditGroupsDlg::on_available_selected));
     _clCurrent->select_row.connect(slot(this, &EditGroupsDlg::on_current_selected));

     // Pixmaps
     string pix_path = ConfigManager::get_PIXPATH();
#ifdef GABBER_WINICON
     string window_icon = pix_path + "gnome-editgroups.xpm";
     gnome_window_icon_set_from_file(_thisWindow->gtkobj(),window_icon.c_str());
     gnome_window_icon_init();
#endif

     refresh();
     show();
}

void EditGroupsDlg::refresh()
{
     // Clear the lists
     _clCurrent->clear();
     _clAvailable->clear();

     // Build a list of available groups
     typedef map<string, set<string> > GMAP;
     const GMAP grps = G_App->getSession().roster().getGroups();
     for (GMAP::const_iterator it = grps.begin(); it != grps.end(); it++)
     {
	  if (it->first == "Unfiled") continue;
	  const char* data[1] = {it->first.c_str() };
	  _clAvailable->append(data);
     }

     // Load group information
     for (Roster::Item::iterator it = _item.begin(); it != _item.end(); it++)
     {
	  if (*it == "Unfiled") continue;
	  const char* data[1] = { it->c_str() };
	  _clCurrent->append(data);       
     }
}

void EditGroupsDlg::on_available_selected(int row, int col, GdkEvent* e)
{
     _entName->set_text(_clAvailable->cell(row, col).get_text());
     // If double-clicked, retrieve the row and 
     // act like they clicked "add"
     if (e->type == GDK_2BUTTON_PRESS)
     {
	  on_add_clicked();
     }
}

void EditGroupsDlg::on_current_selected(int row, int col, GdkEvent* e)
{
     // If double-clicked, retrieve the row and 
     // act like they clicked "remove"
     if (e->type == GDK_2BUTTON_PRESS)
     {
	  on_remove_clicked();
     }
}

void EditGroupsDlg::on_add_clicked()
{
     _item.addToGroup(_entName->get_text());
     refresh();
}

void EditGroupsDlg::on_remove_clicked()
{
     // If a group is selected...
     if (!_clCurrent->selection().empty())
     {
	  _item.delFromGroup(_clCurrent->selection()[0][0].get_text());
	  refresh();
     }
}

void EditGroupsDlg::on_ok_clicked()
{
     // Make sure that the "Unfiled" group is not in the item
     _item.delFromGroup("Unfiled");

     // Update the roster
     G_App->getSession().roster() << _item;

     close();
}


void EditGroupsDlg::on_cancel_clicked()
{
     close();
}


void EditGroupsDlg::on_help_clicked()
{
     // Call the manual
     GnomeHelpMenuEntry help_entry = { "gabber", "users.html#USERS-EDITUSER-GROUPS" };
     gnome_help_display (NULL, &help_entry);
}
