/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * 
 *  Gabber
 *  Copyright (C) 1999-2001 Dave Smith & Julian Missig
 */

#include "PrefsInterface.hh"
#include "GladeHelper.hh"
#include "GabberApp.hh"
#include "GabberWin.hh"
#include "RosterView.hh"
#include "TCPTransmitter.hh"
#include "WelcomeDruid.hh"
#include "GabberUtility.hh"
#include "GabberGPG.hh"

using namespace GabberUtil;

// ---------------------------------------------------------
//
// Login Dialog
//
// ---------------------------------------------------------

LoginDlg* LoginDlg::_Dialog = NULL;

LoginDlg::LoginDlg()
     : BaseGabberDialog("Login_dlg")
{
     // Connect buttons to handlers
     _btnLogin = getButton("Login_Login_btn");
     _btnLogin->clicked.connect(slot(this, &LoginDlg::on_Login_clicked));
     Gtk::Button* b1 = getButton("Login_Cancel_btn");
     b1->clicked.connect(slot(this, &LoginDlg::on_Cancel_clicked));
     b1 = getButton("Login_Help_btn");
     b1->clicked.connect(slot(this, &LoginDlg::on_Help_clicked));

     // Initialize the rest of the pointers
     _entUsername  = getEntry("Login_Username_txt");
     _entUsername  ->changed.connect(slot(this, &LoginDlg::changed));
     _entServer    = getEntry("Login_Server_txt");
     _entServer    ->changed.connect(slot(this, &LoginDlg::changed));
     _spinPort     = getWidget<Gtk::SpinButton>("Login_Port_spin");
     _spinPort     ->set_wrap(false);
     _spinPriority = getWidget<Gtk::SpinButton>("Login_Priority_spin");
     _spinPriority ->set_wrap(false);
     _entPassword  = getEntry("Login_Password_txt");
     _entPassword  ->changed.connect(slot(this, &LoginDlg::changed));
     _entResource  = getEntry("Login_Resource_txt");
     _entResource  ->changed.connect(slot(this, &LoginDlg::changed));
     _chkAutologin = getCheckButton("Login_Autologin_chk");
     _chkAutoreconnect = getCheckButton("Login_Autoreconnect_chk");
     _chkAutoreconnect->set_sensitive(true);
     _chkSavePassword = getCheckButton("Login_SavePassword_chk");
     _chkDigest    = getCheckButton("Login_Digest_chk");
     _chkSSL       = getCheckButton("Login_SSL_chk");
#ifdef WITH_SSL
     _chkSSL       ->show();
     _chkSSL       ->set_sensitive(true);
#endif
     _chkSSL       ->toggled.connect(slot(this, &LoginDlg::on_SSL_toggled));

     // Load configuration and display
     loadconfig();
     show();
}

LoginDlg::~LoginDlg()
{
}    

void LoginDlg::execute()
{
     // Create a login dialog
     _Dialog = manage(new LoginDlg());
}

void LoginDlg::on_Login_clicked()
{
     g_assert( _entUsername != NULL );
     g_assert( _entServer != NULL );
     g_assert( _entPassword != NULL );
     g_assert( _entResource != NULL );

     if (_entUsername->get_text() != "" 
	 && _entServer->get_text() != "" 
	 && _entPassword->get_text() != "" 
	 && _entResource->get_text() != "")
     {
	  // Save configuration
	  saveconfig();

	  // Hide the dialog now that we're trying to log in
	  hide();
	  
	  // Start G_Win
	  G_App->init_win();

	  // FIXME: Attempt to sign something here so it loops to get the correct passphrase.  It
	  // would be better to do this when the passphrase is first needed but there is some wierd problem
	  // with Gnome::Dialog->run() and receiving the roster at the same time causing the roster to be ignored.
	  GabberGPG& gpg = G_App->getGPG();
	  string dest;
	  if (gpg.enabled() && (gpg.sign(GPGInterface::sigClear, string(""), dest) == GPGInterface::errPass))
	  {
	       // If we couldn't get the right passphrase, disable gpg
	       gpg.disable();
	  }

	  // Start login process
	  G_App->login();
	  
	  // And now close it all up, we're done
	  close();
     }
}

void LoginDlg::on_Cancel_clicked()
{
     // Start G_Win
     G_App->init_win();

     close();
}

void LoginDlg::on_Help_clicked()
{
     // Do help stuff 
     GnomeHelpMenuEntry help_entry = { "gabber", "login.html" };
     gnome_help_display (NULL, &help_entry);
}

void LoginDlg::loadconfig()
{     
     ConfigManager& c = G_App->getCfg();

     _entUsername     ->set_text(fromUTF8(_entUsername, c.getStrValue("Server/Username=")));
     _entServer       ->set_text(fromUTF8(_entServer, c.getStrValue("Server/Server=jabber.com")));     
     _spinPort        ->set_value(c.getIntValue("Server/Port=5222"));
     _spinPriority    ->set_value(c.getIntValue("Status/Priority=0"));
     _entResource     ->set_text(fromUTF8(_entResource, c.getStrValue("Server/Resource=Gabber")));
     _chkSavePassword ->set_active(c.getBoolValue("Server/SavePassword=false"));
     if (_chkSavePassword->get_active())
	  _entPassword->set_text(fromUTF8(_entPassword, c.getPrivStrValue("Server/Password=")));
     else
	  _entPassword->set_text("");
     _chkDigest       ->set_active(c.getBoolValue("Server/Digest=true"));
     _chkSSL          ->set_active(c.getBoolValue("Server/SSL=false"));
     _chkAutologin    ->set_active(c.getBoolValue("Server/Autologin=false",false));
     _chkAutoreconnect    ->set_active(c.getBoolValue("Server/Autoreconnect=false",false));
}


void LoginDlg::saveconfig()
{
     ConfigManager& c = G_App->getCfg();
     c.putValue("Server/Username", toUTF8(_entUsername, _entUsername->get_text()));
     c.putValue("Server/Server", _entServer->get_text());
     c.putValue("Server/Port", _spinPort->get_value_as_int());
     c.putValue("Status/Priority", _spinPriority->get_value_as_int());
     c.putValue("Server/Resource", toUTF8(_entResource, _entResource->get_text()));
     c.putValue("Server/SavePassword", _chkSavePassword->get_active());
     // If they want to save the password, write it, it will be overwritten after login
     c.putPrivValue("Server/Password", toUTF8(_entPassword, _entPassword->get_text()));
     c.putValue("Server/Digest", _chkDigest->get_active());
     c.putValue("Server/SSL", _chkSSL->get_active());
     c.putValue("Server/Autoreconnect", _chkAutoreconnect->get_active());
     c.putValue("Server/Autologin", _chkAutologin->get_active(),false);
     c.sync();
}

void LoginDlg::changed()
{
     if (_entUsername->get_text() != "" && _entServer->get_text() != "" && _entPassword->get_text() != "" && _entResource->get_text() != "")
	  _btnLogin->set_sensitive(true);
     else
	  _btnLogin->set_sensitive(false);
}

void LoginDlg::on_SSL_toggled()
{
     // If they want to use SSL, set the default port to 5223
     if (_chkSSL->get_active())
	  _spinPort->set_value(5223);
     // otherwise set the default port to 5222
     else
	  _spinPort->set_value(5222);
}


// ---------------------------------------------------------
//
// Preferences Window
//
// ---------------------------------------------------------

PrefsWin* PrefsWin::_Dialog = NULL;

// Utility function for loading a color (given a config location) and
// loading it into a gnome color-picker
void loadColor(const string& cfgid, Gnome::ColorPicker& p, ConfigManager& cfgm)
{
     Gdk_Color c(cfgm.getStrValue(cfgid));
     p.set((gushort)c.get_red(), (gushort)c.get_green(), (gushort)c.get_blue(), 65535);
}

void saveColor(const string& cfgid, Gnome::ColorPicker& p, ConfigManager& cfgm)
{
     char  colorhex[8];
     guint8 r,g,b,a;
     p.get(r, g, b, a);
     g_snprintf((char*)&colorhex, 8, "#%2.2x%2.2x%2.2x", r, g, b);
     cfgm.putValue(cfgid.c_str(), (const char*)colorhex);
}

void PrefsWin::execute()
{
     if (_Dialog == NULL)
          _Dialog = manage(new PrefsWin());
}

PrefsWin::~PrefsWin()
{
     _Dialog = NULL;
}

PrefsWin::PrefsWin()
     : BaseGabberDialog("Prefs_win")
{
     // Connect the widgets
     prefs = getWidget<Gnome::PropertyBox>("Prefs_win");
     prefs->apply.connect(slot(this, &PrefsWin::on_apply));
     prefs->help.connect(slot(this, &PrefsWin::on_help));
     prefs->destroy.connect(slot(this, &PrefsWin::on_destroy));

     // Log widgets
     _chkLogsNone         = getCheckButton("Prefs_Logs_SaveLogs_chk");
     _chkLogsNone         ->toggled.connect(slot(this, &PrefsWin::changed));
     _chkLogsEncrypted    = getCheckButton("Prefs_Logs_LogEncrypted_chk");
     _chkLogsEncrypted    ->toggled.connect(slot(this, &PrefsWin::changed));
     _chkLogsSave         = getCheckButton("Prefs_Logs_Type_movemonthly_rdo");
     _chkLogsSave         ->toggled.connect(slot(this, &PrefsWin::changed));
     _chkLogsPurge        = getCheckButton("Prefs_Logs_Type_purgemonthly_rdo");
     _chkLogsPurge        ->toggled.connect(slot(this, &PrefsWin::changed));
     //_fileLogsDir         = getWidget<Gnome::FileEntry>("Prefs_Logs_location_file");
     _entLogsDir          = getEntry("Prefs_Logs_location_txt");
     _entLogsDir          ->changed.connect(slot(this, &PrefsWin::changed));
     _chkLogsHTML         = getCheckButton("Prefs_Logs_Frmt_html_rdo");
     _chkLogsHTML         ->toggled.connect(slot(this, &PrefsWin::changed));
     _chkLogsXML          = getCheckButton("Prefs_Logs_Frmt_xml_rdo");
     _chkLogsXML          ->toggled.connect(slot(this, &PrefsWin::changed));

     // Connection options
     _chkConnAutologin    = getCheckButton("Prefs_Connection_Autologin_chk");
     _chkConnAutologin    ->toggled.connect(slot(this, &PrefsWin::changed));
     _chkConnAutorecon    = getCheckButton("Prefs_Connection_Autoreconnect_chk");
     _chkConnAutorecon    ->toggled.connect(slot(this, &PrefsWin::changed));

     // Messages options
     _chkMsgsSendMsgs     = getCheckButton("Prefs_Msgs_SendMsgs_rdo");
     _chkMsgsSendMsgs     ->toggled.connect(slot(this, &PrefsWin::changed));
     _chkMsgsSendOOOChats = getCheckButton("Prefs_Msgs_SendOOOChats_rdo");
     _chkMsgsSendOOOChats ->toggled.connect(slot(this, &PrefsWin::changed));
     _chkMsgsRecvNon      = getCheckButton("Prefs_Msgs_RecvNon_rdo");
     _chkMsgsRecvNon      ->toggled.connect(slot(this, &PrefsWin::changed));
     _chkMsgsRecvMsgs     = getCheckButton("Prefs_Msgs_RecvMsgs_rdo");
     _chkMsgsRecvMsgs     ->toggled.connect(slot(this, &PrefsWin::changed));
     _chkMsgsRecvOOOChats = getCheckButton("Prefs_Msgs_RecvOOOChats_rdo");
     _chkMsgsRecvOOOChats ->toggled.connect(slot(this, &PrefsWin::changed));
     _chkMsgsOpenMsgs     = getCheckButton("Prefs_Msgs_OpenMsgs_chk");
     _chkMsgsOpenMsgs     ->toggled.connect(slot(this, &PrefsWin::changed));
     _chkMsgsOpenOOOChats = getCheckButton("Prefs_Msgs_OpenOOOChats_chk");
     _chkMsgsOpenOOOChats ->toggled.connect(slot(this, &PrefsWin::changed));

     // Colors
     _colorAvailable   = getWidget<Gnome::ColorPicker>("Prefs_Roster_available_color");
     _colorAvailable   ->color_set.connect(slot(this, &PrefsWin::colorchanged));
     _colorChat        = getWidget<Gnome::ColorPicker>("Prefs_Roster_chat_color");
     _colorChat        ->color_set.connect(slot(this, &PrefsWin::colorchanged));
     _colorAway        = getWidget<Gnome::ColorPicker>("Prefs_Roster_away_color");
     _colorAway        ->color_set.connect(slot(this, &PrefsWin::colorchanged));
     _colorXa          = getWidget<Gnome::ColorPicker>("Prefs_Roster_xa_color");
     _colorXa          ->color_set.connect(slot(this, &PrefsWin::colorchanged));
     _colorDnd         = getWidget<Gnome::ColorPicker>("Prefs_Roster_dnd_color");
     _colorDnd         ->color_set.connect(slot(this, &PrefsWin::colorchanged));
     _colorUnavailable = getWidget<Gnome::ColorPicker>("Prefs_Roster_offline_color");
     _colorUnavailable ->color_set.connect(slot(this, &PrefsWin::colorchanged));
     _colorNotInList   = getWidget<Gnome::ColorPicker>("Prefs_Roster_NotInList_color");
     _colorNotInList   ->color_set.connect(slot(this, &PrefsWin::colorchanged));
     _colorStalker     = getWidget<Gnome::ColorPicker>("Prefs_Roster_Stalker_color");
     _colorStalker     ->color_set.connect(slot(this, &PrefsWin::colorchanged));
     _colorMessage     = getWidget<Gnome::ColorPicker>("Prefs_Roster_Message_color");
     _colorMessage     ->color_set.connect(slot(this, &PrefsWin::colorchanged));
     _colorOOOChat     = getWidget<Gnome::ColorPicker>("Prefs_Roster_OOOChat_color");
     _colorOOOChat     ->color_set.connect(slot(this, &PrefsWin::colorchanged));

     // Icons
     _chkIcons       = getCheckButton("Prefs_Roster_Icons_chk");
     _chkIcons       ->toggled.connect(slot(this, &PrefsWin::changed));

     // Size and Pos
     _chkSize        = getCheckButton("Prefs_Window_SaveSize_chk");
     _chkSize        ->toggled.connect(slot(this, &PrefsWin::changed));
     _chkPos         = getCheckButton("Prefs_Window_SavePos_chk");
     _chkPos         ->toggled.connect(slot(this, &PrefsWin::changed));

     // Toolbar options
     _chkToolbarShow = getCheckButton("Prefs_Toolbar_Show_chk");
     _chkToolbarShow ->toggled.connect(slot(this, &PrefsWin::changed));
     _chkStatusShow  = getCheckButton("Prefs_Status_Show_chk");
     _chkStatusShow  ->toggled.connect(slot(this, &PrefsWin::changed));

     // Status Docklet
     _chkDockletShow = getCheckButton("Prefs_Docklet_Show_chk");
     _chkDockletShow ->toggled.connect(slot(this, &PrefsWin::changed));

     // Chats
     _chkChatsOOOTime = getCheckButton("Prefs_Chats_Time_ooochat_chk");
     _chkChatsOOOTime ->toggled.connect(slot(this, &PrefsWin::changed));
     _chkChatsGroupTime = getCheckButton("Prefs_Chats_Time_groupchat_chk");
     _chkChatsGroupTime->toggled.connect(slot(this, &PrefsWin::changed));

     // Encryption
     _chkGPGEnable = getCheckButton("Prefs_GPG_Enable_chk");
     _chkGPGEnable ->toggled.connect(slot(this, &PrefsWin::changed));
     _clistGPGKey  = getWidget<Gtk::CList>("Prefs_GPG_Key_clist");
     _clistGPGKey  ->select_row.connect(slot(this, &PrefsWin::on_GPGKey_select_row));
     // whether the secret keys have been retrieved from gpg and added to the clist yet
     _gotKeys = false;

     // Set the title, since libglade doesn't
     _thisWindow->set_title(_("Gabber: Preferences"));

     // Load configuration and display
     loadconfig();
     prefs->set_modified(false);
     show();
}

void PrefsWin::loadconfig()
{
     // Load the current configuration
     ConfigManager& c = G_App->getCfg();

     // Logs Options
     _chkLogsNone            ->set_active(!c.getBoolValue("Logs/None=false"));
     _chkLogsEncrypted       ->set_active(c.getBoolValue("Logs/Encrypted=true"));
     _chkLogsSave            ->set_active(c.getBoolValue("Logs/Save=true"));
     _chkLogsPurge           ->set_active(c.getBoolValue("Logs/Purge=false"));
     _entLogsDir             ->set_text(c.getStrValue("Logs/Dir=~/.Gabber/"));
     _chkLogsHTML            ->set_active(c.getBoolValue("Logs/HTML=true"));
     _chkLogsXML             ->set_active(c.getBoolValue("Logs/XML=false"));

     // Connection Options
     _chkConnAutologin     ->set_active(c.getBoolValue("Server/Autologin=false",false));
     _chkConnAutorecon     ->set_active(c.getBoolValue("Server/Autoreconnect=false"));

     // Message Options
     _chkMsgsSendMsgs      ->set_active(c.getBoolValue("Messages/SendMsgs=true"));
     _chkMsgsSendOOOChats  ->set_active(c.getBoolValue("Messages/SendOOOChats=false"));
     _chkMsgsRecvNon       ->set_active(c.getBoolValue("Messages/RecvNon=true"));
     _chkMsgsRecvMsgs      ->set_active(c.getBoolValue("Messages/RecvMsgs=false"));
     _chkMsgsRecvOOOChats  ->set_active(c.getBoolValue("Messages/RecvOOOChats=false"));
     _chkMsgsOpenMsgs      ->set_active(c.getBoolValue("Messages/OpenMsgs=false"));
     _chkMsgsOpenOOOChats  ->set_active(c.getBoolValue("Messages/OpenOOOChats=false"));

     // Color Options -- HACK -- Color defaults should be loaded from 
     // a central piece of code
     loadColor("Colors/available=#0066CC", *_colorAvailable, c);
     loadColor("Colors/chat=#0099CC", *_colorChat, c);
     loadColor("Colors/away=#004D80", *_colorAway, c);
     loadColor("Colors/xa=#004D4D", *_colorXa, c);
     loadColor("Colors/dnd=#666633", *_colorDnd, c);
     loadColor("Colors/unavailable=#666666", *_colorUnavailable, c);
     loadColor("Colors/NotInList=#663300", *_colorNotInList, c);
     loadColor("Colors/Stalker=#663333", *_colorStalker, c);
     loadColor("Colors/Message=#669999", *_colorMessage, c);
     loadColor("Colors/OOOChat=#996699", *_colorOOOChat, c);

     // Icons
     _chkIcons->set_active(c.getBoolValue("Colors/Icons=true"));

     // Size and Pos
     _chkSize->set_active(c.getBoolValue("Window/SaveSize=true"));
     _chkPos->set_active(c.getBoolValue("Window/SavePos=true"));

     // Toolbar options
     _chkToolbarShow->set_active(c.getBoolValue("Toolbars/toolbar=true"));
     _chkStatusShow->set_active(c.getBoolValue("Toolbars/statusbar=true"));

     // Docklet options
     _chkDockletShow->set_active(c.getBoolValue("Docklet/Show=true"));

     // Chats
     _chkChatsOOOTime->set_active(c.getBoolValue("Chats/OOOTime=true"));
     _chkChatsGroupTime->set_active(c.getBoolValue("Chats/GroupTime=false"));

     // Encryption
     GabberGPG& gpg = G_App->getGPG();
     // Changed this to use the config file instead of gpg.enabled() since gpg could be disabled while still
     // enabled in the config file
     _chkGPGEnable->set_active(c.getBoolValue("GPG/Enabled=false"));
     if (gpg.enabled() && !_gotKeys)
     {
	  for (unsigned int i = 0; i < _clistGPGKey->columns().size(); i++)
	  {
	       _clistGPGKey->set_column_resizeable(i, false);
	       _clistGPGKey->set_column_auto_resize(i, true);
	  }
	  _clistGPGKey->freeze();
	  // Add the Secret Keys to the CList
          list<GPGInterface::KeyInfo> keys;
          if (gpg.get_secret_keys(keys) == GPGInterface::errOK)
	  {
               list<GPGInterface::KeyInfo>::iterator it = keys.begin();
               string key = c.getStrValue("GPG/SecretKeyID=");
               for ( ; it != keys.end(); it++)
               {
                    const char* rowdata[2] = { it->get_userid().c_str(), it->get_keyid().c_str() };
	            gint row = _clistGPGKey->append_row(rowdata);
	            if (it->get_keyid() == key)
	                 _clistGPGKey->row(row).select();
               }
	       _gotKeys = true;
	  }
	  for (unsigned int i = 0; i < _clistGPGKey->columns().size(); i++)
	       _clistGPGKey->set_column_resizeable(i, true);
	  _clistGPGKey->thaw();
     }
     else if (!gpg.enabled())
	  getWidget<Gtk::Frame>("Prefs_GPG_Key_frm")->set_sensitive(false);
}

void PrefsWin::saveconfig(gint page_num)
{
     // Save the config
     ConfigManager& c = G_App->getCfg();
     
     // Logs config
     c.putValue("Logs/None", !_chkLogsNone->get_active());
     c.putValue("Logs/Encrypted", _chkLogsEncrypted->get_active());
     c.putValue("Logs/Save", _chkLogsSave->get_active());
     c.putValue("Logs/Purge", _chkLogsPurge->get_active());
     c.putValue("Logs/Dir", _entLogsDir->get_text());
     c.putValue("Logs/HTML", _chkLogsHTML->get_active());
     c.putValue("Logs/XML", _chkLogsXML->get_active());

     // Connection config
     c.putValue("Server/Autologin", _chkConnAutologin->get_active(), false);
     c.putValue("Server/Autoreconnect", _chkConnAutorecon->get_active());

     // Messages config
     c.putValue("Messages/SendMsgs", _chkMsgsSendMsgs->get_active());
     c.putValue("Messages/SendOOOChats", _chkMsgsSendOOOChats->get_active());
     c.putValue("Messages/RecvNon", _chkMsgsRecvNon->get_active());
     c.putValue("Messages/RecvMsgs", _chkMsgsRecvMsgs->get_active());
     c.putValue("Messages/RecvOOOChats", _chkMsgsRecvOOOChats->get_active());
     c.putValue("Messages/OpenMsgs", _chkMsgsOpenMsgs->get_active());
     c.putValue("Messages/OpenOOOChats", _chkMsgsOpenOOOChats->get_active());

     // Color config
     saveColor("Colors/available", *_colorAvailable, c);
     saveColor("Colors/chat", *_colorChat, c);
     saveColor("Colors/away", *_colorAway, c);
     saveColor("Colors/xa", *_colorXa, c);
     saveColor("Colors/dnd", *_colorDnd, c);
     saveColor("Colors/unavailable", *_colorUnavailable, c);
     saveColor("Colors/NotInList", *_colorNotInList, c);
     saveColor("Colors/Stalker", *_colorStalker, c);
     saveColor("Colors/Message", *_colorMessage, c);
     saveColor("Colors/OOOChat", *_colorOOOChat, c);

     // Icons
     c.putValue("Colors/Icons", _chkIcons->get_active());

     // Size and Pos
     c.putValue("Window/SaveSize", _chkSize->get_active());
     c.putValue("Window/SavePos", _chkPos->get_active());
  
     // Toolbar config
     c.putValue("Toolbars/toolbar", _chkToolbarShow->get_active());
     c.putValue("Toolbars/statusbar", _chkStatusShow->get_active());

     // Docklet
     c.putValue("Docklet/Show", _chkDockletShow->get_active());

     // Chats
     c.putValue("Chats/OOOTime", _chkChatsOOOTime->get_active());
     c.putValue("Chats/GroupTime", _chkChatsGroupTime->get_active());

     // Encryption
     c.putValue("GPG/Enabled", _chkGPGEnable->get_active());
     // If the clist has a selection, store the keyid
     if (_clistGPGKey->selection().begin() != _clistGPGKey->selection().end())
     {
          Gtk::CList::Row& r = *_clistGPGKey->selection().begin();
	  // keyid is second column
          c.putValue("GPG/SecretKeyID", r[1].get_text());
     }
     // Export the public key for the selected key to the keyserver to ensure it is there
     // so it can be retrieved by other jabber users
     // Only export if GPG is enabled
     if (_chkGPGEnable->get_active())
     {
          cerr << "Trying to export key" << endl;
          G_App->getGPG().send_key(c.getStrValue("GPG/SecretKeyID="));
     }

     // Sync it all and set to unmodified
     c.sync();
}

void PrefsWin::changed()
{
     if (!_entLogsDir->get_text().empty())
	  prefs->changed();
     // If they want to save logs, enable the logging options
     if (_chkLogsNone->get_active())
     {
	  _chkLogsEncrypted->set_sensitive(true);
	  getWidget<Gtk::Frame>("Prefs_Logs_Purging_frame")->set_sensitive(true);
	  getWidget<Gtk::Frame>("Prefs_Logs_Location_frame")->set_sensitive(true);
	  getWidget<Gtk::Frame>("Prefs_Logs_Format_frame")->set_sensitive(true);
     }
     else
     {
	  _chkLogsEncrypted->set_sensitive(false);
	  getWidget<Gtk::Frame>("Prefs_Logs_Purging_frame")->set_sensitive(false);
	  getWidget<Gtk::Frame>("Prefs_Logs_Location_frame")->set_sensitive(false);
	  getWidget<Gtk::Frame>("Prefs_Logs_Format_frame")->set_sensitive(false);
     }
     GabberGPG& gpg = G_App->getGPG();
     // If the user enabled gpg AND we can find gpg then add the secret keys to the CList and enable the Frame
     if (_chkGPGEnable->get_active() && gpg.find_gpg())
     {
	  for (unsigned int i = 0; i < _clistGPGKey->columns().size(); i++)
	  {
	       _clistGPGKey->set_column_resizeable(i, false);
	       _clistGPGKey->set_column_auto_resize(i, true);
	  }
	  _clistGPGKey->freeze();
          list<GPGInterface::KeyInfo> keys;
          if (gpg.get_secret_keys(keys) == GPGInterface::errOK && !_gotKeys)
	  {
               list<GPGInterface::KeyInfo>::iterator it = keys.begin();
               string key = G_App->getCfg().getStrValue("GPG/SecretKeyID=");
               for ( ; it != keys.end(); it++)
               {
                    const char* rowdata[2] = { it->get_userid().c_str(), it->get_keyid().c_str() };
                    gint row = _clistGPGKey->append_row(rowdata);
                    if (it->get_keyid() == key)
                    _clistGPGKey->row(row).select();
               }
	       _gotKeys = true;
	  }
	  for (unsigned int i = 0; i < _clistGPGKey->columns().size(); i++)
	       _clistGPGKey->set_column_resizeable(i, true);
	  _clistGPGKey->thaw();
	  getWidget<Gtk::Frame>("Prefs_GPG_Key_frm")->set_sensitive(true);
     }
     // Otherwise diable the GPG frame and disbale the GPG check
     else
     {
	  getWidget<Gtk::Frame>("Prefs_GPG_Key_frm")->set_sensitive(false);
	  _chkGPGEnable->set_active(false);
     }

}

void PrefsWin::colorchanged(unsigned int arg1, unsigned int arg2, unsigned int arg3, unsigned int arg4)
{
     if (!_entLogsDir->get_text().empty())
	  prefs->changed();
}

void PrefsWin::on_apply(gint page_num)
{
     if (!_entLogsDir->get_text().empty())
     {
	  cerr << "Trying to save stuff" << endl;
	  // Apply button has been clicked, save changes
	  saveconfig(page_num);
	  // Display/hide toolbar
	  G_Win->display_toolbars(_chkToolbarShow->get_active(), _chkStatusShow->get_active());
	  // Update the roster completely (reloads all icons and colors)
	  G_Win->refresh_roster();

	  // Update status to initiate status dock update
	  G_Win->display_status(G_App->getCfg().get_show(),
			       G_App->getCfg().get_status(),
			       G_App->getCfg().get_priority());

	  // Set the logging parameters
	  G_App->setLogHTML(_chkLogsHTML->get_active());
	  G_App->setLogDir(_entLogsDir->get_text());
	  G_App->setLogging(!_chkLogsNone->get_active());
	  // Set unmodified
	  prefs->set_modified(false);
     }
}

void PrefsWin::on_help(gint page_num)
{
     // Do help stuff
     // We can even hook this up to the proper page with # reference :)
     if (page_num == 0)
     {
	  GnomeHelpMenuEntry help_entry = { "gabber", "pref.html#PREF-MAINWINDOW" };
	  gnome_help_display (NULL, &help_entry);
     }
     else if (page_num == 1)
     {
	  GnomeHelpMenuEntry help_entry = { "gabber", "pref.html#PREF-APPEARANCE" };
	  gnome_help_display (NULL, &help_entry);
     } 
     else if (page_num == 2)
     {
	  GnomeHelpMenuEntry help_entry = { "gabber", "pref.html#PREF-MSG" };
	  gnome_help_display (NULL, &help_entry);
     }
     else if (page_num == 3)
     {
	  GnomeHelpMenuEntry help_entry = { "gabber", "pref.html#PREF-ENCRYPTION" };
	  gnome_help_display (NULL, &help_entry);
     }
     else if (page_num == 4)
     {
	  GnomeHelpMenuEntry help_entry = { "gabber", "pref.html#PREF-CONNECTION" };
	  gnome_help_display (NULL, &help_entry);
     }
     else if (page_num == 5)
     {
	  GnomeHelpMenuEntry help_entry = { "gabber", "pref.html#PREF-LOGS" };
	  gnome_help_display (NULL, &help_entry);
     }
     else
     {
	  GnomeHelpMenuEntry help_entry = { "gabber", "pref.html" };
	  gnome_help_display (NULL, &help_entry);
     }
}

void PrefsWin::on_destroy()
{
     close();
}

void PrefsWin::on_GPGKey_select_row(gint row, gint col, GdkEvent* e)
{
     // mark the preferences box as changed when the users changes what key is selected
     // in the CList
     prefs->changed();
}
