/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * 
 *  Gabber
 *  Copyright (C) 1999-2000 Dave Smith & Julian Missig
 */

#include "ConfigManager.hh"
#include "RosterView.hh"
#include "GabberWin.hh"
#include "GabberUtility.hh"

// Various dialog includes
#include "GabberApp.hh"
#include "ContactInfoInterface.hh"
#include "GroupsInterface.hh"
#include "MessageManager.hh"
#include "AddContactDruid.hh"

using Gtk::CTree;
using namespace jabberoo;
using namespace GabberUtil;

typedef map<string, GtkCTreeNode*> NodeMap;

// Temporary/scratch map for composing group nodes -- used by getGroupNode && On_Refresh
NodeMap GROUPS;
NodeMap USERS;

RosterView::RosterView(Gtk::CTree* tree, Session& session)
     : _filter(rvfAll), _session(session), _tree(tree), 
     _row(-1), _col(-1), _dnd_data(0)
{

     // Trick up tree
     _tree->set_line_style(GTK_CTREE_LINES_NONE);
     _tree->set_expander_style(GTK_CTREE_EXPANDER_TRIANGLE);
     _tree->set_column_visibility(colJID, false);
     _tree->set_row_height(17);						    // Hack for making sure icons aren't chopped off
     _tree->column_titles_hide();
     _tree->set_compare_func(&GabberUtil::strcasecmp_clist_items);
     _tree->set_column_auto_resize(colNickname, true);
     _tree->set_auto_sort(true);
     _tree->set_indent(0);						    // Set the indent to 0 so that the tree is more compact

     // Setup tree events
     _tree->tree_expand.connect(slot(this, &RosterView::on_tree_expand));
     _tree->tree_collapse.connect(slot(this, &RosterView::on_tree_collapse));
     _tree->button_press_event.connect(slot(this, &RosterView::on_button_press));
     
     // Setup session events
     _session.roster().evtRefresh.connect(slot(this, &RosterView::on_update_refresh));
     _session.roster().evtPresence.connect(slot(this, &RosterView::on_presence));

     // Connect tree-move signal
     gtk_signal_connect(GTK_OBJECT(_tree->gtkobj()), "tree-move", GTK_SIGNAL_FUNC(&RosterView::on_tree_move), (gpointer) this);
     // allow the roster to be dragged around
     _tree->set_reorderable(true);
     _tree->set_drag_compare_func(&RosterView::is_drop_ok);
     // Setup DnD targets that the roster can receive
     GtkTargetEntry roster_dest_targets[] = {
       {"gtk-clist-drag-reorder", 0, 0},
     };
     int dest_num = sizeof(roster_dest_targets) / sizeof(GtkTargetEntry);
     // Detup DnD targets that the roster can send 
     GtkTargetEntry roster_source_targets[] = {
       {"gtk-clist-drag-reorder", 0, 0},
       {"text/x-jabber-roster-item", 0, 0},
       {"text/x-vcard", 0, 0},
       {"text/x-vcard-xml", 0, 0},
       {"text/x-jabber-id", 0, 0},
     };
     int source_num = sizeof(roster_source_targets) / sizeof(GtkTargetEntry);
     // need to call set_reorderable on the tree first so that the ctree's flag gets
     // set but it sets targets which we have to remove before we can set new targets
     gtk_drag_dest_unset(GTK_WIDGET(_tree->gtkobj()));
     gtk_drag_dest_set(GTK_WIDGET(_tree->gtkobj()), (GtkDestDefaults) (GTK_DEST_DEFAULT_MOTION | GTK_DEST_DEFAULT_DROP), roster_dest_targets, dest_num, (GdkDragAction) (GDK_ACTION_COPY | GDK_ACTION_MOVE));
     gtk_drag_source_set(GTK_WIDGET(_tree->gtkobj()), (GdkModifierType) (GDK_BUTTON1_MASK | GDK_BUTTON3_MASK), roster_source_targets, source_num, (GdkDragAction) (GDK_ACTION_COPY | GDK_ACTION_MOVE));
     // Setup DnD callbacks
     _tree->drag_data_get.connect(slot(this, &RosterView::on_drag_data_get));
     _tree->drag_data_received.connect(slot(this, &RosterView::on_drag_data_received));

     Gtk::Main::timeout.connect(slot(this, &RosterView::FlashEvents), 500);
}

void RosterView::clear()
{
     _tree->clear();
}

void RosterView::refresh(bool reload_cache)
{
     if (reload_cache)
     {
          G_App->getCfg().initPresenceCache(_tree->get_window());
          G_App->getCfg().initEventCache(_tree->get_window());
     }
     cerr << "setting timer for roster connected = " << _refresh_timer.connected() << endl;
     // If ther timer is already running, stop it
     if (_refresh_timer.connected())
          _refresh_timer.disconnect();
     // start a new timer to refresh the roster
     _refresh_timer = Gtk::Main::timeout.connect(slot(this, &RosterView::on_refresh), 1000);
}

void RosterView::save_expanded_groups()
{
     // Save the list of expanded groups to the Gabber config file
     ConfigManager& cf = G_App->getCfg();

     int i = 0;
     char configpath[32];
     for(set<string>::iterator it = _expanded_groups.begin(); it != _expanded_groups.end(); it++, i++)
     {
          g_snprintf(configpath, 32, "Roster/expd_grp-%d", i);
	  cf.putValue(configpath, *it);
     }
     _expanded_groups.clear();
     cf.sync();
}

void RosterView::load_expanded_groups()
{
     // Load the list of expanded groups from the Gabber config file
     ConfigManager& cf = G_App->getCfg();

     int i;
     char configpath[64];
     for (i = 0; ; i++)
     {
          g_snprintf(configpath, 32, "Roster/expd_grp-%d=", i);
	  string groupname = cf.getStrValue(configpath);
	  if (groupname.empty())
	       return;
	  _expanded_groups.insert(groupname);
     }
}

void RosterView::set_view_filter(Filter f)
{
     _filter = f;
     refresh();
}

GtkCTreeNode* RosterView::get_group_node(const string& groupid, bool available)
{
     GtkCTreeNode* result = NULL;

     // Lookup this id
     NodeMap::iterator it = GROUPS.find(groupid);

     // If no such group is found, create one and return it
     if (it == GROUPS.end())
     {
	  // Create the node on the tree
	  char* header[2] = { (char*)fromUTF8(_tree, groupid).c_str(), (char*)groupid.c_str()};
	  result = gtk_ctree_insert_node(_tree->gtkobj(), NULL, NULL, header, 0, 
					 NULL, NULL, NULL, NULL, false, true);

	  // Ensure count of available items is set to 0
	  gtk_ctree_node_set_row_data(_tree->gtkobj(), result, NULL);

	  // Set group foreground/background colours
	  GdkColor bg = get_style_bg_color();
	  gtk_ctree_node_set_background(_tree->gtkobj(), result, &bg);
	  GdkColor fg = get_style_fg_color();
	  gtk_ctree_node_set_foreground(_tree->gtkobj(), result, &fg);

	  // Ensure the group is not selectable
	  gtk_ctree_node_set_selectable(_tree->gtkobj(), result, false);

	  // If this group name does not exist in the expanded_groups set, we need to 
	  // make sure it's collapsed
	  if (_expanded_groups.find(header[colJID]) == _expanded_groups.end())
	       gtk_ctree_collapse(_tree->gtkobj(), result);
	  else
	       gtk_ctree_expand(_tree->gtkobj(), result);

	  // Insert the node into the map for future reference
	  GROUPS.insert(make_pair(groupid, result));

     }
     // Otherwise return the node
     else
	  result = it->second;

     // Increment online cnt (stored in node data ptr) if the request
     // was made by an online item
     if (available)
     {
	  gint cnt = GPOINTER_TO_INT(gtk_ctree_node_get_row_data(_tree->gtkobj(), result)) + 1;
	  gtk_ctree_node_set_row_data(_tree->gtkobj(), result, GINT_TO_POINTER(cnt));
     }
     return result;
}

GtkCTreeNode* RosterView::create_item_node(GtkCTreeNode* groupnode, const Roster::Item& r)
{
     // Get ref to ConfigManager
     ConfigManager& cfgm = G_App->getCfg();

     // Create the node on the tree
     string nickname = fromUTF8(_tree, r.getNickname());
     char* header[2] = { (char*)nickname.c_str(), (char*)r.getJID().c_str() };
     GtkCTreeNode* n = gtk_ctree_insert_node(_tree->gtkobj(), groupnode, NULL, header, 0,
					     NULL, NULL, NULL, NULL, true, true);
     USERS.insert(make_pair(JID::getUserHost(r.getJID()), n));

     // Set color/icons appropriately
     bool showIcons = cfgm.getBoolValue("Colors/Icons=true");
  
     // If this JID is available...
     if (_session.presenceDB().available(r.getJID()))
     {
	  // Lookup presence (since it's guaranteed to be there)
	  const Presence& p = *_session.presenceDB().find(r.getJID());

          string show = p.getShow_str();
	  string status = p.getStatus();
	  // Set text color
	  gtk_ctree_node_set_foreground(_tree->gtkobj(), n, cfgm.getPresenceColor(show));
	  if (showIcons)
               // Display the appropriate icon
	       gtk_ctree_node_set_pixtext(_tree->gtkobj(), n, colNickname, nickname.c_str(), 5, cfgm.getPresencePixmap(show), cfgm.getPresenceBitmap(show));
	  else
	       gtk_ctree_node_set_text(_tree->gtkobj(), n, colNickname, nickname.c_str());	    
     }
     // Else if the user doesn't really have a subscription with this JID...
     else if (r.getSubsType() == Roster::rsFrom || r.getSubsType() == Roster::rsNone)
     {
	  gtk_ctree_node_set_foreground(_tree->gtkobj(), n, cfgm.getPresenceColor("stalker"));
	  if (showIcons)
	       gtk_ctree_node_set_pixtext(_tree->gtkobj(), n, colNickname, nickname.c_str(), 5, cfgm.getPresencePixmap("stalker"), cfgm.getPresenceBitmap("stalker"));
     }
     // Else if this JID is _not_ available...
     else
     {
	  gtk_ctree_node_set_foreground(_tree->gtkobj(), n, cfgm.getPresenceColor("offline"));
	  // Indent this node since there is no picture beside it...21 is derived from adding the width of the icon(16) + the 5 
	  // pixel text offset
	  if (showIcons)
	       gtk_ctree_node_set_shift(_tree->gtkobj(), n, colNickname, 0, 21);
     }
     return n;
}

void RosterView::render_group_nodes()
{
     for (NodeMap::iterator i = GROUPS.begin(); i != GROUPS.end(); i++)
     {
	  gint onlinecount = GPOINTER_TO_INT(gtk_ctree_node_get_row_data(_tree->gtkobj(), i->second));
	  if (onlinecount > 0)
	  {
	       gchar* grp_title = g_strdup_printf("%s (%d)", i->first.c_str(), onlinecount);
	       gtk_ctree_node_set_text(_tree->gtkobj(), i->second, colNickname, grp_title); // Set visible title appropriately
	       g_free(grp_title);
	       //  _total_online += onlinecount;
	  }
     }
}

gint RosterView::on_refresh()
{
     cerr << "refreshing" << endl;
     // Prep the tree
     _tree->freeze();
     _tree->clear();

     // Clear the groups nodemap
     GROUPS.clear();
     // CLear users
     USERS.clear();

     // Render each item in the roster
     for(Roster::iterator i = _session.roster().begin(); i != _session.roster().end(); i++)
     {
	  // Total Online
	  //_total_online = 0;

	  // Determine if this person is online
	  bool available = _session.presenceDB().available(i->getJID());

	  // If it is a transport and we're set to not display transports, skip it,
	  // otherwise display it
	  if (G_App->isTransport(i->getJID()))
	  {
	       if (_filter == rvfAllNoAgents || _filter == rvfOnlineOnlyNoAgents)
		    continue;
	  }
	  // If this JID is not online and the filter is set to display online people only,
	  // skip this item
	  else if ( !(available) && (_filter == rvfOnlineOnly || _filter == rvfOnlineOnlyNoAgents) && 
		   (G_App->getMessageManager().getEvent(i->getJID()) == G_App->getMessageManager().getEvents().end()))
	       continue;

	  // Walk the groups contained in the item, creating an entry in each
	  for (Roster::Item::iterator grp_it = i->begin(); grp_it != i->end(); grp_it++)
	  {
	       // Retrieve/create the group node
	       GtkCTreeNode* groupnode = get_group_node(*grp_it, available);
	       // Create the item node on this group
	       create_item_node(groupnode, *i);
	  }

	  // Walk the temporary group node map and render the number of people
	  // online in each group
	  render_group_nodes();
	  // Set the number in the status docklet
	  // StatusDock::num_online(_total_online);
     }

     // display users that are not in list but need to be in the roster temporarily because they have sent us a message
     for (list<Roster::Item>::iterator it = _nil_roster.begin(); it != _nil_roster.end(); it++)
     {
          for (Roster::Item::iterator grp_it = it->begin(); grp_it != it->end(); grp_it++)
          {
               GtkCTreeNode *groupnode = get_group_node(*grp_it, false);
               create_item_node(groupnode, *it);
          }
     }

     // Thaw the tree 
     _tree->thaw();

     _refresh_timer.disconnect();
     // return false to prevent the timer from being called again
     return FALSE;
}


void RosterView::on_presence(const string& jid, bool available, Presence::Type prev_type)
{
     refresh();

     // Grab the nickname
     string nickname;
     try {
	  nickname = fromUTF8(_tree, G_App->getSession().roster()[jid].getNickname());
     } catch (Roster::XCP_InvalidJID& e) {
	  nickname = fromUTF8(_tree, JID::getUser(jid));
     }

     if (available && prev_type == Presence::ptUnavailable)
     {
	  // Play UserAvailable sound
	  gnome_triggers_do(NULL, NULL, "gabber", "UserAvailable", NULL);
	  // Update the statusbar
	  G_Win->get_StatusBar()->pop();
	  G_Win->get_StatusBar()->push(nickname + _(" is now online"));
     }
     else if (available && prev_type == Presence::ptAvailable)
     {
	  // Update the statusbar
	  //G_Win->get_StatusBar()->pop();
	  //G_Win->get_StatusBar()->push(nickname + _(" changed status"));
	  ;
     }
     else if (!available && prev_type == Presence::ptAvailable)
     {
	  // Play UserOffline sound
	  gnome_triggers_do(NULL, NULL, "gabber", "UserOffline", NULL);
	  // Update the statusbar
	  G_Win->get_StatusBar()->pop();
	  G_Win->get_StatusBar()->push(nickname + _(" went offline"));
     }
}

void RosterView::on_tree_expand(Gtk::CTree::Row r)
{
     _expanded_groups.insert(r[colJID].get_text());
}

void RosterView::on_tree_collapse(Gtk::CTree::Row r)
{
     set<string>::iterator it = _expanded_groups.find(r[colJID].get_text());
     if (it != _expanded_groups.end())
	  _expanded_groups.erase(it);
}


int RosterView::on_button_press(GdkEventButton* e)
{
     if (!_tree->get_selection_info(e->x, e->y, &_row, &_col))
	  return 0;

     if (!_tree->row(_row).get_selectable())
	  return 0;

     _tree->row(_row).select();


     switch (e->type) {
     case GDK_2BUTTON_PRESS: // Double-click
     {
	  // Lookup default resource for this user
	  string defaultJID = _tree->get_text(_row, colJID);
	  try {
	       defaultJID = _session.presenceDB().find(defaultJID)->getFrom();
	  } 
	  catch (PresenceDB::XCP_InvalidJID& e) {}

          MessageManager::EventList::iterator it = G_App->getMessageManager().getEvent(defaultJID);
          if (it != G_App->getMessageManager().getEvents().end())
          {
                G_App->getMessageManager().display(defaultJID, it->second);
                refresh();
                break;
          }

 	  // Check for control key..
 	  if (e->state & GDK_CONTROL_MASK)
               G_App->getMessageManager().display(defaultJID, Message::mtChat);
	  else
	  {
	       if (G_App->getCfg().getBoolValue("Messages/SendMsgs=true"))
                    G_App->getMessageManager().display(defaultJID, Message::mtNormal);
	       else
                    G_App->getMessageManager().display(defaultJID, Message::mtChat);
	  }
	  break;
     }
     case GDK_BUTTON_PRESS: // Single-click
	  // Check for rt-click
	  if (e->button == 3) 
	  {
	       // If it's a transport
	       if (G_App->isTransport(_tree->get_text(_row, colJID)))
	       {
		    G_Win->get_menuAgent()->show_all();
		    G_Win->get_menuAgent()->popup(e->button, e->time);
	       }
	       else
	       {
		    G_Win->get_menuUser()->show_all();

                    // Hide appropriate items if the user is NIL
                    string defaultJID = _tree->get_text(_row, colJID);
                    bool on_roster = true;
                    try {
                         G_App->getSession().roster()[JID::getUserHost(defaultJID)];
                    } catch (Roster::XCP_InvalidJID& e) {
                         on_roster = false;
                    }
                    if (on_roster)
                         G_Win->toggle_roster_popup(true);
                    else
                         G_Win->toggle_roster_popup(false);

		    G_Win->get_menuUser()->popup(e->button, e->time);
	       }
	  }
	  break;
     default:
	  break;
     }

     return 0;
}

// Right-click menu for users

void RosterView::on_Message_activate()
{
     // Lookup default resource for this user
     string defaultJID = _tree->get_text(_row, colJID);
     try {
	  defaultJID = _session.presenceDB().find(defaultJID)->getFrom();
     } 
     catch (PresenceDB::XCP_InvalidJID& e) {}

     G_App->getMessageManager().display(defaultJID, Message::mtNormal);
}

void RosterView::on_OOOChat_activate()
{
     // Lookup default resource for this user
     string defaultJID = _tree->get_text(_row, colJID);
     try {
	  defaultJID = _session.presenceDB().find(defaultJID)->getFrom();
     } 
     catch (PresenceDB::XCP_InvalidJID& e) {}

     G_App->getMessageManager().display(defaultJID, Message::mtChat);
}

void RosterView::on_EditUser_activate()
{
     bool on_roster = true;
     string jid = _tree->get_text(_row, colJID);
     try {
          G_App->getSession().roster()[jid];
     } catch (Roster::XCP_InvalidJID& e) {
          on_roster = false;
     }
     if (on_roster)
          ContactInfoDlg::display(jid);
     else
          ContactInfoDlg::display(jid, Roster::rsNone);
}

void RosterView::on_EditGroups_activate()
{
     manage(new EditGroupsDlg(_tree->get_text(_row, colJID)));
}

void RosterView::on_History_activate()
{
     //HistoryDlg::display(_tree->get_text(_row, colJID));
     gnome_url_show(G_App->getLogFile(string(_tree->get_text(_row, colJID))));
}

void RosterView::on_S10nRequest_activate()
{
     G_App->getSession() << Presence(_tree->get_text(_row, colJID), Presence::ptSubRequest);  
}


void RosterView::on_DeleteUser_activate()
{
     Gnome::Dialog* d;
     string question = _("Are you sure you want to remove ");
     string jid = _tree->get_text(_row, colJID);
     if (G_App->isTransport(jid))
	  question += JID::getUserHost(jid);
     else
	  question += jid;
     question += _(" from your roster?");
     d = manage(Gnome::Dialogs::question_modal(question,
					       slot(this, &RosterView::handle_RemoveUserDlg)));
}

void RosterView::on_AddUser_activate()
{
     string jid = _tree->get_text(_row, colJID);
     AddContactDruid::display(JID::getUserHost(jid));
}

// Right-click menu for transports

void RosterView::on_LoginTransport_activate()
{
     G_App->getSession() << Presence(_tree->get_text(_row, colJID), Presence::ptAvailable);
}

void RosterView::on_LogoutTransport_activate()
{
     G_App->getSession() << Presence(_tree->get_text(_row, colJID), Presence::ptUnavailable);
}

void RosterView::on_TransInfo_activate()
{
     manage(new AgentInfoDlg(_tree->get_text(_row, colJID)));
}


void RosterView::handle_RemoveUserDlg(int code)
{
     switch (code)
     {
     case 0:
	  G_App->getSession().roster().deleteUser(_tree->get_text(_row, colJID));
	  break;
     case 1:
          // Do nothing
          ;
     }
}

GdkColor RosterView::get_style_bg_color()
{
     GtkStyle* gs = gtk_widget_get_style(GTK_WIDGET(_tree->gtkobj()));
     return gs->bg[GTK_STATE_NORMAL];
}

GdkColor RosterView::get_style_fg_color()
{
     GtkStyle* gs = gtk_widget_get_style(GTK_WIDGET(_tree->gtkobj()));
     return gs->fg[GTK_STATE_NORMAL];
}

void RosterView::addNILItem(Roster::Item& r)
{
     list<Roster::Item>::iterator it = _nil_roster.begin();
     string user = JID::getUserHost(r.getJID());
     while ((it != _nil_roster.end()) && (JID::getUserHost(it->getJID()) != user))
          it++;

     if (it == _nil_roster.end())
     {
          _nil_roster.push_back(r);
          refresh();
     }
}

void RosterView::clearNILItem(const string& jid)
{
     list<Roster::Item>::iterator it = _nil_roster.begin();
     string user = JID::getUserHost(jid);
     // Find JID in the NIL roster and remove them
     while ((it != _nil_roster.end()) && user != JID::getUserHost(it->getJID()))
	  it++;

     if (it != _nil_roster.end())
     {
	  _nil_roster.erase(it);
	  refresh();
     }
}

gint RosterView::FlashEvents()
{
     ConfigManager& cfgm = G_App->getCfg();
     bool showIcons = cfgm.getBoolValue("Colors/Icons=true");

     MessageManager::EventList& events = G_App->getMessageManager().getEvents();
     // Iterate backwards so that older events (ones closer to the front of the list) are the ones displayed
     MessageManager::EventList::reverse_iterator it = events.rbegin();
     for (; it != events.rend(); it++)
     {
          string user = JID::getUserHost(it->first);
          NodeMap::iterator uit = USERS.find(user);
          if (uit == USERS.end())
               continue;

          string nickname;
          try {
               nickname = fromUTF8(_tree, G_App->getSession().roster()[user].getNickname());
          } catch (Roster::XCP_InvalidJID& e)
          {
               list<Roster::Item>::iterator lit = _nil_roster.begin();
               while ((lit != _nil_roster.end()) && (JID::getUserHost(lit->getJID()) != user))
                    lit++;
               if (lit == _nil_roster.end())
                    continue;
               nickname = fromUTF8(_tree, lit->getNickname());
          }

          GtkCTreeNode* n = uit->second;
          GtkCTreeNode* parent = GTK_CTREE_ROW(uit->second)->parent;
          // get rid of the node shift so the text is in the right place
          gtk_ctree_node_set_shift(_tree->gtkobj(), n, colNickname, 0, 0);
          if (_event_flash)
          {
               // Change the color of the group node if it is collapsed
               if (!GTK_CTREE_ROW(parent)->expanded)
                    gtk_ctree_node_set_foreground(_tree->gtkobj(), parent, cfgm.getFlashEventColor(it->second));
               else
               {
                    // Otherwise flash the event
                    gtk_ctree_node_set_foreground(_tree->gtkobj(), n, cfgm.getFlashEventColor(it->second));
                    if (showIcons)
                         gtk_ctree_node_set_pixtext(_tree->gtkobj(), n, colNickname, nickname.c_str(), 5, cfgm.getEventPixmap(it->second), cfgm.getEventBitmap(it->second));
               }
          }
          else
          {
               GdkColor fg = get_style_fg_color();
               gtk_ctree_node_set_foreground(_tree->gtkobj(), parent, &fg);

               string show;
               try {
                    show = _session.presenceDB().find(it->first)->getShow_str();
               } catch (PresenceDB::XCP_InvalidJID& e) {
                    // JID isn't in roster so use the online color
                    show = "online";
               }
               gtk_ctree_node_set_foreground(_tree->gtkobj(), n, cfgm.getPresenceColor(show));
               if (showIcons)
                    gtk_ctree_node_set_pixtext(_tree->gtkobj(), n, colNickname, nickname.c_str(), 5, cfgm.getEventPixmap(MessageManager::evNone), cfgm.getEventBitmap(MessageManager::evNone));
          }
     }

     _event_flash = !_event_flash;
     return TRUE;
}

void RosterView::on_update_refresh()
{
     // This is called when we first receive the roster from the server (as well as a few other times)
     // We need to make sure all users not in the roster but who have spooled Message events are added
     // to the not in list group of the roster
     MessageManager::EventList& events = G_App->getMessageManager().getEvents();
     for (MessageManager::EventList::iterator it = events.begin(); it != events.end(); it++)
     {
          try {
               G_App->getSession().roster()[JID::getUserHost(it->first)];
          } catch(Roster::XCP_InvalidJID& e) {
               // User isn't in roster, add it to the NIL group
               Roster::Item t(it->first, JID::getUser(it->first));
               t.addToGroup(_("Not in Roster"));
               addNILItem(t);
          }
     }
     refresh();
}

void RosterView::on_tree_move(GtkCTree* ctree, GtkCTreeNode* node, GtkCTreeNode* new_parent, GtkCTreeNode* new_sibling, gpointer data)
{
     RosterView* roster = (RosterView *) data;
     Gtk::CTree::Row row(ctree, node);
     Gtk::CTree::Row new_parent_row(ctree, new_parent);
     Gtk::CTree::Row old_parent_row(ctree, GTK_CTREE_ROW(node)->parent);

     // Make changes to the roster item for the row being moved
     try {

          Roster::Item item = roster->_session.roster()[row[1].get_text()];
	  // remove from old group
	  item.delFromGroup(old_parent_row[1].get_text());
	  // add item to new group
	  item.addToGroup(new_parent_row[1].get_text());
	  // send off the modified roster item to the server
	  roster->_session.roster() << item;
     } catch(Roster::XCP_InvalidJID& e) {
	  cerr << "item not in roster! Oh horror of horrors, this should never happen!" << endl;
     }
}

void RosterView::on_drag_data_get(GdkDragContext* drag_ctx, GtkSelectionData* data, guint info, guint time)
{
     if (data->target == gdk_atom_intern("text/x-jabber-id", FALSE))
     {
	  // Got this out of the gtkclist drag_data_get function, should work
	  GtkCListCellInfo* info;

	  info = (GtkCListCellInfo*) g_dataset_get_data(drag_ctx, "gtk-clist-drag-source");
	  string jid = _tree->row(info->row)[1].get_text();
	  string dnddata = "jabber:" + jid + "\n";
	  gtk_selection_data_set(data, data->target, 8, (guchar *) dnddata.c_str(), dnddata.size() + 1);
     }
     else if (data->target == gdk_atom_intern("text/x-jabber-roster-item", FALSE))
     {
	  GtkCListCellInfo* info;

	  info = (GtkCListCellInfo*) g_dataset_get_data(drag_ctx, "gtk-clist-drag-source");
	  string jid = _tree->row(info->row)[1].get_text();
	  try {
		  Roster::Item item = G_App->getSession().roster()[jid];

		  // Convert roster item to a tag
		  Tag itemt("item");
		  itemt.putAttrib("jid", item.getJID());
		  if (!item.getNickname().empty())
		       itemt.putAttrib("name", item.getNickname());
		  for (Roster::Item::iterator it = item.begin(); it != item.end(); it++)
		       itemt.addTaggedCDATA("group", *it);

		  // Set the DnD Data
		  string dnddata = itemt.getXML();
		  gtk_selection_data_set(data, data->target, 8, (guchar *) dnddata.c_str(), dnddata.size() + 1);
	  } catch (Roster::XCP_InvalidJID& e) { }
     }
     else if (data->target == gdk_atom_intern("text/x-vcard-xml", FALSE))
     {
	  if (_dnd_data)
	  {
	       // already waiting for a DnD vCard request, don't start another
	       return;
	  }

	  GtkCListCellInfo* info;

	  info = (GtkCListCellInfo*) g_dataset_get_data(drag_ctx, "gtk-clist-drag-source");
	  string jid = _tree->row(info->row)[1].get_text();

	  // Get next session ID
	  string id = G_App->getSession().getNextID();

	  // create request for user's vcard
	  Tag iq("iq");
	  iq.putAttrib("id", id);
	  iq.putAttrib("to", jid);
	  iq.putAttrib("type", "get");
	  Tag& vCard = iq.addTag("vCard");
	  vCard.putAttrib("xmlns", "vcard-temp");
	  vCard.putAttrib("version", "3.0");
	  vCard.putAttrib("prodid", "-//HandGen//NONSGML vGen v1.0//EN");

	  _dnd_data = data;

	  // Send the vCard request
	  G_App->getSession() << iq.getXML().c_str();
	  G_App->getSession().registerIQ(id, slot(this, &RosterView::dnd_get_vCard));
     }
     else if (data->target == gdk_atom_intern("text/x-vcard", FALSE))
     {
	  if (_dnd_data)
	  {
	       // already waiting for a DnD vCard request, don't start another
	       return;
	  }

// FIXME: Need to convert vcard XML to normal vcard
// It supposedly shouldn't be that hard

	  GtkCListCellInfo* info;

	  info = (GtkCListCellInfo*) g_dataset_get_data(drag_ctx, "gtk-clist-drag-source");
	  string jid = _tree->row(info->row)[1].get_text();

	  // Get next session ID
	  string id = G_App->getSession().getNextID();

	  // create request for user's vcard
	  Tag iq("iq");
	  iq.putAttrib("id", id);
	  iq.putAttrib("to", jid);
	  iq.putAttrib("type", "get");
	  Tag& vCard = iq.addTag("vCard");
	  vCard.putAttrib("xmlns", "vcard-temp");
	  vCard.putAttrib("version", "3.0");
	  vCard.putAttrib("prodid", "-//HandGen//NONSGML vGen v1.0//EN");

	  _dnd_data = data;

	  // Send the vCard request
	  G_App->getSession() << iq.getXML().c_str();
	  G_App->getSession().registerIQ(id, slot(this, &RosterView::dnd_get_vCard));
     }
}

void RosterView::on_drag_data_received(GdkDragContext* drag_ctx, gint x, gint y, GtkSelectionData* data, guint info, guint time)
{
     // Check for vcard, xml vcard, etc, and do appropriate stuff
}

gboolean RosterView::is_drop_ok(GtkCTree* ctree, GtkCTreeNode* node, GtkCTreeNode* new_parent, GtkCTreeNode* new_sibling)
{
     // Prevent roster items from being moved to the toplevel of the ctree
     if (!new_parent)
	  return FALSE;
     // Prevent groups from being moved around
     if (!GTK_CTREE_ROW(node)->parent)
	  return FALSE;
     // Setup CTree rows to make getting text easier
     Gtk::CTree::Row new_parent_row(ctree, new_parent);
     Gtk::CTree::Row old_parent_row(ctree, GTK_CTREE_ROW(node)->parent);
  
     if (new_parent_row[1].get_text() == "Agents" || new_parent_row[1].get_text() == "Not in Roster" || 
	 new_parent_row[1].get_text().empty())
          return FALSE;
     if (old_parent_row[1].get_text() == "Agents" || old_parent_row[1].get_text() == "Not in Roster")
          return FALSE;

     return TRUE;
}

void RosterView::dnd_get_vCard(const Tag& iq)
{
     Tag* vCard = iq.getTag("vCard");

     if (vCard)
     {
	  string dnddata = vCard->getXML();
          gtk_selection_data_set(_dnd_data, _dnd_data->target, 8, (guchar *) dnddata.c_str(), dnddata.size() + 1);
	  _dnd_data = NULL;
     }
}
