/*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
* 
*  Gabber
*  Copyright (C) 1999-2000 Dave Smith & Julian Missig
*/

/*
* This file contains code from GNet by David A. Helder
* http://www.eecs.umich.edu/~dhelder/misc/gnet/
*/

#include "TCPTransmitter.hh"
#include "GabberApp.hh"
#include "GabberWin.hh"

#include <arpa/inet.h>
#include <errno.h>
#include <netdb.h>
#include <string.h>
#include <sys/wait.h>
#ifdef WITH_IPV6
// results in a redefinition of MIN and MAX ...
// would be better to include resolv.h before glib.h
#include <resolv.h>
#endif

using namespace std;


#ifdef HAVE_GETHOSTBYNAME_R_GLIB_MUTEX
     GStaticMutex TCPTransmitter::_gethostbyname_mutex = G_STATIC_MUTEX_INIT;
#endif


TCPTransmitter::TCPTransmitter():
     _socketfd(-1), _IOChannel(NULL), _hostResolved(false),
     _connecting(false), _reconnecting(false), _connected(false),
     _resolver_watchid(0), _poll_eventid(0), _socket_flags(0),
     _resolver_pid(-1), _socket_watchid(0)
{
#ifdef WITH_SSL
     _adapter = NULL;
#endif

#ifdef WITH_IPV6
     // get an AAAA record from gethostbyname if possible
     res_init();
     _res.options |= RES_USE_INET6;
#endif
}


TCPTransmitter::~TCPTransmitter()
{
     disconnect();
}


void
TCPTransmitter::connect(const string& host, guint port)
{
#ifdef TRANSMITTER_DEBUG
     cerr << "Connecting to " << host << ":" << port << endl;
#endif
     if (!(_connected || _connecting) || _reconnecting) {
#ifdef WITH_SSL
	  bool use_ssl = G_App->getCfg().getBoolValue("Server/SSL=false");

	  if(use_ssl && _adapter == NULL) {
	       _adapter = new SSLAdapter();
	  }
	  else if (!use_ssl && _adapter != NULL) {
	       delete _adapter;
	       _adapter = NULL;
	  }
#endif

	  _connecting = true;
	  if(!_reconnecting) {
#ifdef WITH_IPV6
	       _host_sockaddr.sin6_family = AF_INET6;
	       _host_sockaddr.sin6_port = g_htons(port);
#else
	       _host_sockaddr.sin_family = AF_INET;
	       _host_sockaddr.sin_port = g_htons(port);
#endif
	       _hostResolved = false;
	       // resolve host
	       _async_resolve(host.c_str());
	  }
	  else {
	       _hostResolved = true;
	       handleHostResolution(NULL, G_IO_NVAL, this);
	  }
     }
     else if(_connected) {
	  evtConnected();
     }
}


void
TCPTransmitter::disconnect()
{
#ifdef TRANSMITTER_DEBUG
     cerr << "disconnect() called!" << endl;
#endif
     
     if(_connected || _connecting) {

	  // remove the poll function
	  g_source_remove(_poll_eventid);
	  
#ifdef WITH_SSL
	  if(_adapter) {
	       _adapter->disconnect();
	       delete _adapter;
	       _adapter = NULL;
	  }
#endif

	  // close IOChannel
	  if(_IOChannel != NULL) 
	       g_io_channel_close(_IOChannel);
	  _connected = false;
	  _connecting = false;
          _hostResolved = false;
	  // close socket
	  if(_socketfd != -1)
	       close(_socketfd);
	  _socketfd = -1;
     }

     if(!_reconnecting) {
	  evtDisconnected();
     }
}


// Transmitter methods
void
TCPTransmitter::send(const gchar* data)
{
     guint bytes_written = 0;
     // Write the data to _Socket using glib
     GIOError result;

#ifdef WITH_SSL
     if(_adapter) {
	  result = _adapter->send(data, strlen(data), &bytes_written);
     }
     else {
	  result = g_io_channel_write(_IOChannel, const_cast< gchar* > (data),
				      strlen(data), &bytes_written);
     }
#else
     result = g_io_channel_write(_IOChannel, const_cast < gchar * >(data),
			                                  strlen(data), &bytes_written);
#endif

     // Check the error result
     switch (result)
     {
	  // TODO: SSLAdapter error reporting
     case G_IO_ERROR_NONE: break;
     case G_IO_ERROR_AGAIN: cerr << "GIOERR_AGAIN in send" << endl;
	  break;
     case G_IO_ERROR_INVAL: cerr << "GIOERR_INVAL in send" << endl;
	  //fall through
     case G_IO_ERROR_UNKNOWN: cerr << "GIOERR_UKNOWN in send" << endl;
	  //fall through
     default:
	  handleError(errSocket);
	  break;
     }
}


// Socket related callbacks
gboolean
TCPTransmitter::handleSocketEvent(GIOChannel* iochannel, GIOCondition cond, gpointer data) 
{
     TCPTransmitter& transmitter = *(static_cast<TCPTransmitter*>(data));

     if (!transmitter._connected)
     {
          // If we aren't connected don't do anything and return FALSE to get rid of the
          // Socket callback
          return FALSE;
     }

     switch(cond) {
     case G_IO_IN:
	  {
#ifdef TRANSMITTER_DEBUG
	       cout << "Transmitter: Received data from socket" << endl;
#endif
	       // Read the data from the socket and push it into the session
	       guint bytes_read = 2047;
	       char buf[2048];
	       while (bytes_read == 2047) 
		    {
			 // Ensure buffer is empty
			 memset(buf, 0, 2048);
			 // Read data from the channel
			 GIOError result;
#ifdef WITH_SSL
			 if(transmitter._adapter) {
			      result = transmitter._adapter->read(buf, 2047, &bytes_read);
			 }
			 else {
			      g_assert(iochannel != NULL);
			      result = g_io_channel_read(iochannel, buf, 2047, &bytes_read);
			 }
#else
			 g_assert(iochannel != NULL);
			 result = g_io_channel_read(iochannel, buf, 2047, &bytes_read);
#endif
			 if (((result == G_IO_ERROR_AGAIN) || (result == G_IO_ERROR_NONE)) && (bytes_read > 0)) {
#ifdef TRANSMITTER_DEBUG
			      cout << "Transmitter: Firing evtDataAvailable" << endl;
#endif
			      transmitter.evtDataAvailable(buf);
			 }
			 else
			      {
				   //TODO: Error reporting from SSLAdapter
				   transmitter.handleError(errSocket);
				   return (FALSE);
			      }
		    }
	       return (TRUE);
	  }
     case G_IO_HUP:
	  transmitter.handleError(errConnectionBroken);
	  break;
     case G_IO_ERR:
	  transmitter.handleError(errIO);
	  break;
     case G_IO_NVAL:
	  transmitter.handleError(errNoConnection);	
	  break;
     default:
	  cerr << "Unknown error on Transmitter!" << endl;
	  transmitter.handleError(errIO);
	  break;
     }

#ifdef TRANSMITTER_DEBUG
     cerr << "An error occurred on the IOChannel!" << endl;
#endif     

     transmitter._connected = false;
     return (FALSE);
	  
}


gboolean
TCPTransmitter::handleSocketConnect(GIOChannel* iochannel, GIOCondition cond, gpointer data)
{
     TCPTransmitter& transmitter = *(static_cast <TCPTransmitter*>(data));
   
     // remove watch, in case we don't return immediately to avoid being
     // called more than once
     // (since we always return FALSE, we can do it here already)
     g_source_remove(transmitter._socket_watchid);

     if (!transmitter._connecting)
     {
          // Occurs when the Transmitter is disconnected in the process of connecting.  Return to prevent
          // an error from being triggered 
          return FALSE;
     }
     
     if ((cond & G_IO_IN) || (cond & G_IO_OUT)) {
	  // was there an error?
	  gint error;
	  socklen_t len;
	  len = sizeof(error);
	  if (getsockopt(transmitter._socketfd, SOL_SOCKET, SO_ERROR,
			 static_cast <void*>(&error), &len) >= 0) {
	       if (!error) {
		    // Great everything went fine!
		    // reset the socket flags to blocking
		    if((fcntl(transmitter._socketfd, F_SETFL, transmitter._socket_flags)) == -1 ) {
#ifdef TRANSMITTER_DEBUG
			 cerr << "Couldn't reset socket flags!" << endl;
#endif
			 transmitter.handleError(errSocket);
			 return(FALSE);
		    }

#ifdef TRANSMITTER_DEBUG
		    cerr << "Async connect worked!" << endl;
#endif
		    
		    transmitter._connected = true;
		    transmitter._connecting = false;
		    transmitter._reconnecting = false;
		    
		    // register socket for notification
#ifdef WITH_SSL
		    if(transmitter._adapter) {
			 bool success =
			      transmitter._adapter->registerSocket(transmitter._socketfd);
			 //TODO: some nice error handling
			 g_assert(success);
		    }
#endif

		    transmitter._socket_watchid = g_io_add_watch(transmitter._IOChannel,
								 GIOCondition (G_IO_IN | G_IO_PRI | G_IO_ERR | G_IO_HUP | G_IO_NVAL),
								 &TCPTransmitter::handleSocketEvent, data);

		    // add poll function to event loop
		    // poll every 15sec
		    transmitter._poll_eventid = g_timeout_add(15000, &TCPTransmitter::_connection_poll, data);
#ifdef TRANSMITTER_DEBUG
		    cout << "Poll function added to event loop" << endl;
#endif

		    transmitter.evtConnected();
		    return (FALSE);
	       }
	  }
     }
     // there was some nasty error...

#ifdef TRANSMITTER_DEBUG
     cerr << "Async connect failed: " << cond << endl;
#endif

     // close the socket
     g_io_channel_close(transmitter._IOChannel);
     transmitter._socketfd = -1;
     transmitter.handleError(errSocket);

     return (FALSE);
}


gboolean
TCPTransmitter::handleHostResolution(GIOChannel* iochannel, GIOCondition cond, gpointer data)
{
     TCPTransmitter& transmitter = *(static_cast <TCPTransmitter*>(data));
     bool error = false;

     // remove the watch on the IO channel
     if(transmitter._resolver_watchid > 0)
          g_source_remove(transmitter._resolver_watchid);

     // if _hostResolved == true it means the address was already resolved
     if (!transmitter._hostResolved && transmitter._connecting) {
	  if (cond & G_IO_IN) {
	       // read from the channel

#ifdef TRANSMITTER_DEBUG
	       cerr << "reading from pipe" << endl;
#endif

#ifdef WITH_IPV6
	       guint buf_length = sizeof(guchar) + sizeof(struct in6_addr);
#else
	       guint buf_length = sizeof(guchar) + sizeof(struct in_addr);
#endif
	       gchar * buffer = new gchar[buf_length];
	       guint bytes_read = 0;
	       GIOError result = g_io_channel_read(iochannel, buffer, buf_length, &bytes_read);
	       if (result != G_IO_ERROR_NONE) {

#ifdef TRANSMITTER_DEBUG
		    cerr << "Error while reading from pipe" << endl;
#endif
		    error = true;
	       }
	       else if (buffer[0] == 0) {
		    // the hostname couldn't be resolved properly

#ifdef TRANSMITTER_DEBUG
		    cerr << "read from pipe got me a 0!" << endl;
#endif
		    error = true;
	       }
	       if (bytes_read != buf_length) {
		    // something nasty happened, bail out!
#ifdef TRANSMITTER_DEBUG
		    cerr << "Didn't receive enough data from resolver process!\n" << "Received: " << bytes_read << " Wanted: " << buf_length << endl;
#endif
		    error = true;
	       }
	       
	       if(!error) {
		    // copy the received struct in_addr into _host_sockaddr.sin_addr
#ifdef WITH_IPV6
		    memcpy(static_cast <void*>(&transmitter._host_sockaddr.sin6_addr),
			   static_cast <void*>(&buffer[sizeof(guchar)]), sizeof(struct in6_addr));
#else
		    memcpy(static_cast <void*>(&transmitter._host_sockaddr.sin_addr),
			   static_cast <void*>(&buffer[sizeof(guchar)]), sizeof(struct in_addr));
#endif
#ifdef TRANSMITTER_DEBUG
		    cerr << "wrote in_addr into _host_sockaddr" << endl;
		    cerr << "got the following from the pipe:" << endl;
#ifdef WITH_IPV6
		    cerr << "sin6_family: " << transmitter._host_sockaddr.sin6_family << endl;
		    cerr << "sin6_port:   " << transmitter._host_sockaddr.sin6_port << endl;
		    char name_buffer[INET6_ADDRSTRLEN];
		    if (inet_ntop(AF_INET6, transmitter._host_sockaddr.sin6_addr.s6_addr, name_buffer, INET6_ADDRSTRLEN))
			 cerr << "sin6_addr:   " << name_buffer << endl;
#else
		    cerr << "sin_family: " << transmitter._host_sockaddr.sin_family << endl;
		    cerr << "sin_port  : " << transmitter._host_sockaddr.sin_port << endl;
		    cerr << "sin_addr  : " << transmitter._host_sockaddr.sin_addr.s_addr << endl;
#endif
#endif
		    transmitter._hostResolved = true;
	       }
	       
	  }
	  else {
	       // should not happen...
	       error = true;
	  }
     }

     // close the pipes
     if(iochannel != NULL)
	  g_io_channel_close(iochannel);

     // wait for the resolver process to exit to avoid zombies
     if(transmitter._resolver_pid != -1) {
	  waitpid(transmitter._resolver_pid, NULL, 0);
	  transmitter._resolver_pid = -1;
     }

     if(error) {
	  transmitter.evtError(errAddressLookup);
     } else if (transmitter._connecting) {
	  transmitter._async_connect();
     }
     
     return (FALSE);
}



void
TCPTransmitter::handleError(const Error error)
{
     if(_connected || _connecting) {
	  disconnect();
     }

     // check if we should auto-reconnect: 
     if(G_App->getCfg().getBoolValue("Server/Autoreconnect=false")) {
#ifdef TRANSMITTER_DEBUG
	  cout << "Disconnected. Attempting auto-reconnect" << endl;
#endif
	  _reconnecting = true;
	  
	  G_Win->get_StatusBar()->push(_("Disconnected. Reconnecting..."));
	  connect("", 0);
	  
     } else {
	  evtError(error);
     }
}




void
TCPTransmitter::_async_resolve(const gchar* hostname)
{
     g_assert(hostname != NULL);
     // check if hostname is in dotted decimal notation
#ifdef WITH_IPV6
     if (inet_pton(AF_INET6, hostname, &_host_sockaddr.sin6_addr) != 0) {
#else
     if (inet_aton(hostname, &_host_sockaddr.sin_addr) != 0) {
#endif
	  // all done
	  _hostResolved = true;

#ifdef TRANSMITTER_DEBUG
	  cerr << "dot-dec resolved" << endl;
#endif

	  handleHostResolution(NULL, G_IO_NVAL, this);
	  
	  return;
     }

     // didn't work, we need to do a lookup
     _resolver_pid = -1;
     gint pipes[2];
     
     // create a pipe
     if (pipe(pipes) == -1)
	  {
	       handleError(errAddressLookup);
	       return;
	  }


     errno = 0;
     // try to fork as long as errno == EAGAIN
     do {
	  // fork the lookup process
	  if ((_resolver_pid = fork()) == 0) {
#ifdef WITH_IPV6
	       struct in6_addr ia;
#else
	       struct in_addr ia;
#endif
	       if(!_gethostbyname(hostname, &ia)) {
		    guchar zero = 0;
		    if (write(pipes[1], &zero, sizeof(zero)) == -1) {
			 g_warning("Problem with writing to pipe");
		    }
#ifdef TRANSMITTER_DEBUG
		    cerr << "host couldn't be resolved, wrote NULL " << endl;
#endif
	       }
	       else {
#ifdef WITH_IPV6
		    guchar size = sizeof(struct in6_addr);
#else
		    guchar size = sizeof(struct in_addr);
#endif

#ifdef TRANSMITTER_DEBUG
#ifdef WITH_IPV6
		    char name_buffer[INET6_ADDRSTRLEN];
		    if (inet_ntop(AF_INET6, ia.s6_addr, name_buffer, INET6_ADDRSTRLEN))
			 cerr << "wrote to pipe: sin6_addr:   " << name_buffer << endl;
#else
		    cerr << "wrote to pipe: sin_addr: " << ia.s_addr << endl;
#endif
#endif
		   
#ifdef WITH_IPV6
		    if ( (write(pipes[1], &size, sizeof(guchar))) == -1 ||
			 (write(pipes[1], &ia, sizeof(struct in6_addr)) == -1)) {
			 g_warning("Problem with writing to pipe");
		    }
#else 
		    if ( (write(pipes[1], &size, sizeof(guchar))) == -1 ||
			 (write(pipes[1], &ia, sizeof(struct in_addr)) == -1)) {
			 g_warning("Problem with writing to pipe");
		    }
#endif
			 
			 
#ifdef TRANSMITTER_DEBUG
		    cerr << "lookup process done!" << endl;
#endif
	  
	       }
	       // close our end of the pipe
	       close(pipes[1]);
	       
	       // exit (call _exit() to avoid atexit being called)	
	       _exit(EXIT_SUCCESS);

	  }

	  else if (_resolver_pid > 0) {
	       // parent process creates an IOChannel to read from the pipe
	       _resolver_watchid = g_io_add_watch(g_io_channel_unix_new(pipes[0]),
						  GIOCondition (G_IO_IN | G_IO_ERR | G_IO_HUP | G_IO_NVAL),
						  &TCPTransmitter::handleHostResolution,
						  this);
	       return;
	  }

	  else if (errno != EAGAIN) {
	       // nasty error
	       g_warning("Resolver fork error: %s (%d)\n", g_strerror(errno), errno);
	       handleError(errAddressLookup);
	       return;
	  }
	  
     }while (errno == EAGAIN);
	  
     return;
}



void
TCPTransmitter::_async_connect()
{
     // connect non-blocking

     // create socket
#ifdef WITH_IPV6
     _socketfd = socket(PF_INET6, SOCK_STREAM, 0);
#else
     _socketfd = socket(PF_INET, SOCK_STREAM, 0);
#endif
     if (_socketfd < 0) {
	  // something nasty happened
#ifdef TRANSMITTER_DEBUG
	  cerr << "socket() failed: " << strerror(errno) << endl;
#endif
	       
	  handleError(errSocket);
	  return;
     }
     _socket_flags = fcntl(_socketfd, F_GETFL, 0);
     if (_socket_flags == -1) {
	  // not good

#ifdef TRANSMITTER_DEBUG
	  cerr << "fcntl F_GETFL failed on socket: " << strerror(errno) << endl;
#endif

	  handleError(errSocket);
	  return;
     }
     if (fcntl(_socketfd, F_SETFL, _socket_flags | O_NONBLOCK) == -1) {
	  // damn!

#ifdef TRANSMITTER_DEBUG
	  cerr << "fcntl F_SETFL failed on socket: " << strerror(errno) << endl;
#endif

	  handleError(errSocket);
	  return;
     }

     // try to connect non-blocking
     if (::connect(_socketfd, (struct sockaddr*) (&_host_sockaddr), sizeof(_host_sockaddr)) < 0) {
	  if (errno != EINPROGRESS) {
	       // Yikes!

#ifdef TRANSMITTER_DEBUG
	       cerr << "connect failed: " << strerror(errno) << endl;
#endif	       

	       handleError(errSocket);
	       return;
	  }
     }

     _IOChannel = g_io_channel_unix_new(_socketfd);
    _socket_watchid = g_io_add_watch(_IOChannel,
		    GIOCondition(G_IO_IN | G_IO_OUT | G_IO_PRI | G_IO_ERR | G_IO_HUP | G_IO_NVAL),
		    &TCPTransmitter::handleSocketConnect, this);

#ifdef TRANSMITTER_DEBUG
     cerr << "IOChannel watch added!" << endl;
#endif
}


#ifdef WITH_IPV6
bool TCPTransmitter::_gethostbyname(const gchar* hostname, struct in6_addr* addr_result)
#else
bool TCPTransmitter::_gethostbyname(const gchar* hostname, struct in_addr* addr_result)
#endif
{
#ifdef HAVE_GETHOSTBYNAME_R_GLIBC
     {
	  struct hostent result_buf, *result;
	  size_t len;
	  char* buf;
	  int herr;
	  int res;

	  len = 1024;
	  buf = g_new(gchar, len);

	  while ((res = gethostbyname_r (hostname, &result_buf, buf, len,
					 &result, &herr)) == ERANGE) {
	       len *= 2;
	       buf = g_renew (gchar, buf, len);
	  }

	  if (res || result == NULL || result->h_addr_list[0] == NULL) {
	       g_free(buf);
	       return false;
	  }
    
	  if (addr_result)
	       {
		    memcpy(addr_result, result->h_addr_list[0], result->h_length);
	       }

	  g_free(buf);
     }

#else
#ifdef HAVE_GET_HOSTBYNAME_R_SOLARIS
     {
	  struct hostent result;
	  size_t len;
	  char* buf;
	  int herr;
	  int res;

	  len = 1024;
	  buf = g_new(gchar, len);

	  while ((res = gethostbyname_r (hostname, &result, buf, len,
					 &herr)) == ERANGE)
	       {
		    len *= 2;
		    buf = g_renew (gchar, buf, len);
	       }

	  if (res || hp == NULL || hp->h_addr_list[0] == NULL) {
	       g_free(buf);
	       return false;
	  }

	  if (addr_result)
	       {
		    memcpy(addr_result, result->h_addr_list[0], result->h_length);
	       }

	  g_free(buf);
     }
#else
#ifdef HAVE_GETHOSTBYNAME_R_HPUX
     {
	  struct hostent result;
	  struct hostent_data buf;
	  int res;

	  res = gethostbyname_r (hostname, &result, &buf);

	  if (res == 0) {
	       if (addr_result) {
		    memcpy(addr_result, result.h_addr_list[0], result.h_length);
	       }
	  }
	  else
	       return false;
     }
#else 
#ifdef HAVE_GETHOSTBYNAME_R_GLIB_MUTEX
     {
	  struct hostent* he;
	  
	  g_static_mutex_lock(&_gethostbyname_mutex);
	  he = gethostbyname(hostname);
	  g_static_mutex_unlock(&_gethostbyname_mutex);

	  if (he != NULL && he->h_addr_list[0] != NULL) {
	       if (addr_result)
		    {
			 memcpy(addr_result, he->h_addr_list[0], he->h_length);
		    }
	  }
	  else
	       return false;
     }
#else
     {
	  struct hostent* he;

	  he = gethostbyname(hostname);
	  if (he != NULL && he->h_addr_list[0] != NULL)
	       {
		    if (addr_result)
			 {
			      memcpy(addr_result, he->h_addr_list[0], he->h_length);
			 }
	       }
	  else
	       return false;
     }
#endif
#endif
#endif
#endif

     return true;
}


gboolean TCPTransmitter::_connection_poll(gpointer data) 
{
     static struct sockaddr addr;
     static socklen_t addr_length = 1;

     TCPTransmitter& transmitter = *(static_cast<TCPTransmitter*> (data));

#ifdef TRANSMITTER_DEBUG
     cout << "_connection_poll called" << endl;
#endif     
     
//      if(getpeername(transmitter._socketfd, &addr, &addr_length) != 0) {
// 	  if(transmitter._connected) {
// 	       //an error occurred
// 	       cerr << "_connection_poll: Error: " << strerror(errno) << endl;
// 	       transmitter.handleError(errSocket);
// 	  } 
// 	  return (FALSE);

//      }

     // new poll function for search of the connection-eating bug:
     if(transmitter._connected) {
       transmitter.send(" ");
     }

// #ifdef TRANSMITTER_DEBUG
//      cout << "_connection_poll: Everything fine" << endl;
// #endif
     return (TRUE);
}



