%{
/*
 * GNOME Basic Line Parser
 *
 * Author:
 *    Jody Goldberg (jgoldberg@home.com)
 */
#include <stdlib.h>
#include <string.h>

#include <gb/gb.h>
#include <gb/gb-lex.h>
#include <gb/gb-expr.h>
#include <gb/gb-eval.h>
#include <gb/gb-form.h>
#include <gb/gb-statement.h>
#include <gb/gb-type.h>
#include <gb/gb-main.h>

#define YYDEBUG         1
#define YYDEBUG         1
#define YYPARSE_PARAM	buffer
#define YYLEX_PARAM	buffer
#define YYERROR_VERBOSE 1

extern int gb_lex (void *res, void *closure);
/*
 *   The fact that gb_error is not passed the closure
 * makes this unutterably hideous.
 */
static GBParseData *yacc_hack = NULL;
extern void gb_error (const char *msg);
%}
%pure_parser

/* TODO
 * Document the specifics of the conflicts
 * Or remove them
 *
 * 4 are a result of opt_parm
 * 1 is part of the strange one_line syntax for if
 */
%expect 5

%union {
	const char      *v_str;
	double	         v_float;
	int		 v_int;
	gboolean	 v_bool;
	GSList	        *list;

	struct {
		const char *name;
		gboolean    as_new;
	} v_dim_type;

	struct {
		gboolean    is_array;
		GSList     *indicees;
	} v_array;
	GBIndex            *v_index;

	const GBExpr	   *expr;
	const GBStatement  *stmt;
	GBObjRef           *objr;
	GBArgDesc          *v_arg;
	GBSelectCase       *v_case;
	GBVar              *v_var;
	
	GBFormItem         *v_form_item;
	GBFormProperty     *v_form_prop;
}

%token <v_str> NAME STRING FRX
%token <v_float> FLOAT
%token <v_int> INTEGER
%token GB_GE GB_LE GB_MOD GB_ARG_ASSIGN GB_IS GB_EQV GB_LIKE GB_IMP
%token GB_AND GB_OR GB_XOR GB_NOT

%token GB_LEXTYPE_ASP GB_LEXTYPE_FILE GB_LEXTYPE_EXPR GB_LEXTYPE_STATEMENTS

%token GB_ACTIVATE GB_AS GB_ATTRIBUTE GB_APPEND GB_BASE GB_BEGIN GB_BEEP GB_CALL GB_CASE
%token GB_CLASS GB_CLOSE GB_COMPARE GB_CONST GB_DATE GB_DECLARE
%token GB_DEFTYPE GB_DELETESETTING GB_DIM GB_DO GB_EACH GB_ELSE GB_END
%token GB_ELSE_IF GB_ERASE GB_ERROR GB_EXIT GB_EXPLICIT GB_FALSE GB_FILECOPY GB_FOR
%token GB_FUNCTION GB_GET GB_GLOBAL GB_GOSUB GB_GOTO GB_IF GB_INPUT GB_KILL GB_LSET GB_LET GB_LEN
%token GB_LINE GB_LOAD GB_LOCK GB_LOOP GB_MID GB_MKDIR GB_NAME GB_NEXT GB_ON
%token GB_OPEN GB_OPTION GB_OUTPUT GB_PRINT GB_PRIVATE GB_PROPERTY GB_PUBLIC GB_PUT
%token GB_RSET GB_RANDOM GB_RANDOMIZE GB_REDIM GB_REM GB_RESET GB_RESUME GB_RETURN
%token GB_RMDIR GB_RND GB_SAVESETTING GB_SCALE GB_SEEK GB_SELECT GB_SENDKEYS GB_SET GB_SETATTR
%token GB_STATIC GB_STOP GB_SUB GB_THEN GB_TIME GB_TRUE GB_TYPE GB_UNLOAD GB_UNLOCK GB_VERSION
%token GB_WEND GB_WHILE GB_WIDTH GB_WITH GB_WITH_EVENTS GB_WRITE GB_NEW GB_TO
%token GB_STEP GB_PARAM_ARRAY GB_BYREF GB_BYVAL GB_OPTIONAL GB_NOTHING GB_IN
%token GB_BINARY GB_TEXT GB_DATABASE GB_MODULE GB_UNTIL, GB_PRESERVE

/* ASP specific tokens */
%token GB_ASP_START GB_ASP_START_AT GB_ASP_START_EXPR GB_ASP_END

/* Form specific tokens */
%token GB_BEGIN_PROPERTY GB_END_PROPERTY GB_OBJECT

/* To make the lexer nicer */
%token GB_KEEP_LOOKING

/* Magic GB_SPACE_DOT which flags a '.' preceded whitespace */
%token GB_SPACE_DOT

%type <v_bool> opt_new opt_events opt_static pub_private opt_parmarray
%type <v_bool> opt_optional opt_by_val_or_ref opt_preserve end_routine

%type <v_int> open_mode

%type <v_str> obj_name type as_type var_name
%type <v_case> case_stmt

%type <v_dim_type> dim_type
%type <v_arg>   arg
%type <v_array> opt_subscripts
%type <v_index> subscript
%type <v_var>   type_var_decl
 
%type <expr> object_list expr parm const_expr opt_step opt_while_until
%type <expr> opt_default sub_call opt_expr opt_open_len

%type <list> parms opt_args opt_parms statements one_line_statements
%type <list> object_refs handles case_stmts expr_csv
%type <list> else_body else_end opt_arglist arglist subscripts
%type <list> type_var_decls

%type <stmt> statement if_body on_error
%type <objr> object_ref

/* Form bits */
%type <list>        form_items form_props class_props class_item
%type <v_form_item> form_item 
%type <v_form_prop> form_prop class_prop
%type <v_str>       form_name

%left '&'
%left '<' '>' '=' GB_NE GB_GE GB_LE GB_LIKE GB_IS
%left '-' '+' 
%left '*' '/'
%left UNARY_NEG UNARY_POS
%left '^'
%left '\\'
%left GB_MOD
%left GB_NOT
%left GB_AND
%left GB_OR
%left GB_XOR
%left GB_EQV
%left GB_IMP
%left '.'
%%

Start : GlobalStatements
      | GB_LEXTYPE_FILE GlobalStatements
      | GB_LEXTYPE_ASP asp
      | GB_LEXTYPE_EXPR expr             { gb_parse_data_set_expr  (buffer, $2); }
      | GB_LEXTYPE_STATEMENTS statements { gb_parse_data_set_stmts (buffer, $2); }
      ;

GlobalStatements : GlobalStatements GlobalStatement
		 | GlobalStatements '\n'
		 |
		 ;

eol : '\n'
    | ':'
    ;

/* ASP code */

asp : GB_ASP_START_AT expr GB_ASP_END '\n' asp_stmts
    ;

asp_stmts : asp_stmts asp_stmt 
	  | asp_stmt
          ;

asp_stmt : STRING { }
	 | GB_ASP_START statements GB_ASP_END {}
	 | GB_ASP_START_EXPR expr GB_ASP_END {}
	 | '\n'

/* Form code */
form_version : GB_VERSION FLOAT eol
			{
				if ($2 != 5.0) {
					yyerror ("Unsupported version");
					YYERROR;
				}
			}
             ;

form_prop : NAME '=' const_expr '\n' { $$ = gb_form_property_new (buffer, $1, $3); }
	  ;

form_props : form_props form_prop { $$ = g_slist_append ($1, $2); }
	   | form_props GB_BEGIN_PROPERTY NAME '\n' form_props form_prop GB_END_PROPERTY '\n' { $$ = $1; }
	   | { $$ = NULL; }
	   ;

form_name  : form_name '.' NAME { $$ = g_strconcat ($1, ".", $3, NULL); }
	   | NAME               { $$ = g_strdup ($1); }
	   ;

form_item  : GB_BEGIN form_name NAME eol form_props form_prop form_items form_item GB_END eol
	{
		$$ = gb_form_item_new  ($2, $3);
		gb_form_item_add       ($$, g_slist_append ($7, $8));
		gb_form_item_props_add ($$, g_slist_append ($5, $6));
	}
	   | { $$ = NULL; }
	   ;

form_items : form_items form_item { $$ = g_slist_append ($1, $2); }
	   | form_items eol       { $$ = $1; }
	   | GB_OBJECT '=' STRING ';' STRING eol { $$ = NULL; }
	   | { $$ = NULL; }
	   ;
/* End of Form code */


/* Class Code */

class_version : GB_VERSION FLOAT GB_CLASS eol
                         { 
				 if ($2 != 1.0) {
					 yyerror ("Unsupported class version");
					 YYERROR;
				 }
			 }
              ; 

class_prop : form_prop
           ; 

class_props : class_props class_prop { $$ = g_slist_prepend ($1, $2); }
            | { $$ = NULL; }

class_item : GB_BEGIN eol class_props GB_END eol  { $$ = $3;  } 
           ;

/* End of Class Code */



/* Attribute Code */
attribute_item : GB_ATTRIBUTE object_refs '=' const_expr eol { }
               ;        
      
attribute_items : attribute_items attribute_item { }
		| attribute_item { } 
		;
/* End of Attribute Code */

GlobalStatement : '#' PreProc					{ }
		| form_version form_item 
		  { gb_parse_data_end_form (buffer, $2);          }
                | class_version class_item
                  { gb_parse_data_end_class (buffer, $2); }
		| attribute_items                               { }
		| GB_OPTION gb_options				{ }
		| pub_private GB_CONST const_statement		{ }
                | pub_private var_decls                         { }
                | type_statement                                { }
                | GB_GLOBAL GB_CONST const_statement            { }
                | dim_statement					{ }
		| pub_private opt_static declare_routine eol
		    statements
		  GB_END end_routine
		  { gb_routine_finish (buffer, $1, $7, $5); }
                ;

statements : statements statement eol
	     { $$ = gb_stmt_accumulate ($1, $2); }
	   | statements '\n'		{ $$ = $1; }
	   |				{ $$ = NULL; }
	   ;

statement : sub_call
	    { $$ = gb_stmt_new_call (buffer, $1) }
	  | object_list '=' expr                       
            { $$ = gb_stmt_new_assignment (buffer, $1, $3); }
	  | dim_statement				{ $$ = NULL; }
	  | GB_REDIM opt_preserve var_decls       	{ $$ = NULL; }

	  | GB_SET object_list '=' opt_new object_list	{ $$ = gb_stmt_new_set (buffer, $2, $4, $5); }
	  | GB_ON GB_ERROR on_error			{ $$ = $3; }
	  | GB_EXIT nesting				{ $$ = NULL; }

	  | GB_DO opt_while_until eol
		statements
	    GB_LOOP opt_while_until
	    { $$ = gb_stmt_new_do_while (buffer, $2, $6, $4); }

	  | GB_SELECT GB_CASE expr eol
		case_stmts
	    GB_END GB_SELECT
	    { $$ = gb_stmt_new_select   (buffer, $3, $5); }

	  | GB_WHILE expr eol
		statements
	    GB_WEND
	    { $$ = gb_stmt_new_do_while (buffer, $2, NULL, $4); }

	  | GB_FOR NAME '=' expr GB_TO expr opt_step eol
		statements
	    GB_NEXT opt_name /* TODO : add check for NAME == opt_name */
	    { $$ = gb_stmt_new_forloop (buffer, $2, $4, $6, $7, $9); }

	  | GB_FOR GB_EACH NAME GB_IN object_list eol
		statements
	    GB_NEXT opt_name /* TODO : add check for NAME == opt_name */
	    { $$ = gb_stmt_new_foreach (buffer, $3, $5, $7); }

	  | GB_WITH object_list eol { gb_module_with_depth_inc (buffer, $2); }
		statements
	    GB_END GB_WITH
	    { $$ = gb_stmt_new_with (buffer, $2, $5); }

	  | GB_IF expr GB_THEN if_body
	    { $$ = gb_stmt_if_set_cond ((GBStatement *)$4, $2); }

	  | GB_BEEP					{ $$ = NULL; }
	  | GB_DATE '=' expr				{ $$ = NULL; }
	  | GB_RANDOMIZE opt_expr { $$ = gb_stmt_new_randomize (buffer, $2); }

          | GB_LOAD expr   { $$ = gb_stmt_new_load (buffer, $2); }  
          | GB_UNLOAD expr { $$ = gb_stmt_new_unload (buffer, $2); }

	  /* THIS LOOKS EXTREMELY BROKEN ! */
          | object_refs '.' GB_LINE '(' expr ',' expr ')' '-' opt_line_step '(' expr ',' expr ')' ',' expr ',' expr
            { }
          
          | object_refs '.' GB_LINE '(' expr ',' expr ')' '-' opt_line_step '(' expr ',' expr ')' ',' expr 
            { }

          | object_refs '.' GB_SCALE '(' expr ',' expr ')' '-' '(' expr ',' expr ')' 
            { }
      
          /* File handling related statements */
          | GB_OPEN expr GB_FOR open_mode GB_AS expr opt_open_len          
            { $$ = gb_stmt_new_open (buffer, $2, $4, $6, $7); }
          | GB_OPEN expr GB_FOR open_mode GB_AS '#' expr opt_open_len
            { $$ = gb_stmt_new_open (buffer, $2, $4, $7, $8); }
          | GB_INPUT '#' expr ',' expr_csv                     
            { $$ = gb_stmt_new_input (buffer, $3, $5); }
          | GB_LINE GB_INPUT '#' expr ',' expr                 
            { $$ = gb_stmt_new_line_input (buffer, $4, $6); }
          | GB_CLOSE expr_csv                                  
            { $$ = gb_stmt_new_close (buffer, $2); }
          | GB_CLOSE handles                        
            { $$ = gb_stmt_new_close (buffer, $2); }
          | GB_GET '#' expr ',' opt_expr ',' expr
            { $$ = gb_stmt_new_get (buffer, $3, $5, $7); }
          | GB_PUT '#' expr ',' opt_expr ',' expr
            { $$ = gb_stmt_new_put (buffer, $3, $5, $7); }
          | GB_SEEK '#' expr ',' expr 
            { $$ = gb_stmt_new_seek (buffer, $3, $5); }
          | GB_PRINT '#' expr ',' expr_csv 
            { $$ = gb_stmt_new_print (buffer, $3, $5); }

          /* FIXME: This is a line-label (for Goto statements) */
	  | GB_GOTO NAME
	    { $$ = gb_stmt_new_goto  (buffer, $2); }
	  | NAME ':'
	    { $$ = gb_stmt_new_label (buffer, $1); }
          ;

opt_line_step : GB_STEP 
              | 
              ;

opt_open_len : GB_LEN '=' expr { $$ = $3; } 
             |                 { $$ = NULL; }
             ;

handles : handles ',' '#' expr { $$ = g_slist_append ($1, (gpointer) $4); }
        | '#' expr             { $$ = g_slist_append (NULL, (gpointer) $2); }
        |                      { $$ = NULL; }
        ;

open_mode : GB_INPUT  { $$ = GB_OPEN_INPUT; }
          | GB_OUTPUT { $$ = GB_OPEN_OUTPUT; }
          | GB_APPEND { $$ = GB_OPEN_APPEND; }
          | GB_BINARY { $$ = GB_OPEN_BINARY; }
          | GB_RANDOM { $$ = GB_OPEN_RANDOM; }
          ;

one_line_statements : one_line_statements statement ':'
		      {
			  $$ = gb_stmt_accumulate ($1, $2);
		      }
		    | { $$ = NULL; }
		    ;

type_statement : pub_private GB_TYPE NAME eol type_var_decls GB_END GB_TYPE
		{ gb_parse_data_add_type (buffer, gb_type_new ($1, $3, $5)); }
               ;

type_var_decl : var_name opt_subscripts dim_type
		{ $$ = gb_var_new ($1, $3.as_new, $2.is_array,
				   $2.indicees, $3.name); }
              ;

type_var_decls : type_var_decls type_var_decl eol { $$ = g_slist_prepend ($1, $2); }
               | type_var_decls eol               { $$ = $1; }
	       |                                  { $$ = NULL; }
               ; 


if_body : '\n' statements else_body
	  { $$ = gb_stmt_new_if (buffer, $2, $3); }


	| one_line_statements statement	{ }
	  { $$ = gb_stmt_new_if (buffer, gb_stmt_accumulate ($1, $2), NULL); }
	;

else_body : GB_ELSE_IF expr GB_THEN if_body
	    { $$ = g_slist_prepend (NULL,
				    (gpointer)gb_stmt_if_set_cond ($4, $2)); }
	  | GB_ELSE else_end 
	    { $$ = g_slist_reverse($2); }
	  | GB_END GB_IF
	    { $$ = NULL; }
	  ;

else_end : '\n' statements GB_END GB_IF
           { $$ = $2; }
         | one_line_statements statement
           { $$ = gb_stmt_accumulate ($1,$2); }
         ;

opt_while_until : GB_WHILE expr		{ $$ = $2; }
		| GB_UNTIL expr		{ $$ = gb_expr_new_unary (GB_EXPR_NOT, $2); }
	        |			{ $$ = NULL; }
		;

nesting : GB_DO
	| GB_FOR
	| GB_FUNCTION
	| GB_PROPERTY
	| GB_SUB
	; 

opt_name : NAME		{ }
         |		{ }
	 ;

dim_statement : GB_DIM var_decls 	{ }
              | GB_STATIC var_decls     { }
              ;

const_statement : const_decls const_decl	{ }
		;

const_decls : const_decls const_decl ','
	    |
	    ;

const_decl : NAME as_type opt_const_val	{ }
	   ;

opt_const_val : '=' expr  { }
              | 
              ;

on_error : GB_RESUME GB_NEXT { $$ = gb_stmt_new_on_error_next (buffer);     }
	 | GB_GOTO INTEGER   { $$ = NULL; /* FIXME: urk */                  }
	 | GB_GOTO NAME      { $$ = gb_stmt_new_on_error_goto (buffer, $2); }
         ;

gb_options : GB_EXPLICIT				{ }
           | GB_PRIVATE GB_MODULE			{ }
	   | GB_BASE INTEGER				{ }
	   | GB_COMPARE compare_types			{ }
	   ;

compare_types : GB_BINARY				{ }
 	      | GB_TEXT					{ }
	      | GB_DATABASE				{ }
	      ;

opt_preserve : GB_PRESERVE				{ $$ = TRUE; }
	     |						{ $$ = FALSE; }
	     ;

opt_step : GB_STEP expr					{ $$ = $2; }
         |						{ $$ = NULL; }
	 ;

var_decl : opt_events var_name opt_subscripts dim_type
	   { gb_parse_data_add_var (buffer, gb_var_new ($2, $4.as_new, $3.is_array,
							$3.indicees, $4.name)); }
         ;

var_name : obj_name { $$ = $1; }
         ;

var_decls : var_decls ',' var_decl
          | var_decl 
          ;

subscript : expr		{ $$ = gb_index_new (NULL, $1); }
	  | expr GB_TO expr	{ $$ = gb_index_new ($1, $3);   }
          ;

subscripts : subscripts subscript ',' { $$ = g_slist_append ($1, $2); }
	   |			      { $$ = NULL;                    }
	   ;

opt_subscripts : '(' subscripts subscript ')' { $$.is_array  = TRUE;
					        $$.indicees = g_slist_append ($2, $3);   }

	       /* A dynamic array to be set later */
	       | '(' ')'		      { $$.is_array = TRUE; $$.indicees = NULL;  }

	       /* not an array */
	       |			      { $$.is_array = FALSE; $$.indicees = NULL; }
               ;

/* NOTE : This silliness is required because MS uses keywords as
 *        types and property names ...
 */
type : GB_DATE	{ $$ = g_strdup("Date"); }
     | NAME
     {
	 extern char const * gb_is_type (char const * str, unsigned int len);
/*	 char const * typ;*/
/* Needs fixing, nice to have type checking here, but need 'As' object */
/*	 if ((typ = gb_is_type ($1,strlen($1))) == NULL)
	 {
	     yyerror ("Expecting a type\n");
	     YYERROR;
	     }*/
	 $$ = g_strdup ($1);
     }
     ;

dim_type : GB_AS opt_new type { $$.name = $3; $$.as_new = $2; }
         |	{ $$.name = g_strdup ("GBVariant"); $$.as_new = FALSE; }
	 ;

PreProc : GB_CONST NAME '=' const_expr
        | GB_IF const_expr GB_THEN
        | GB_ELSE_IF const_expr GB_THEN
        | GB_ELSE
        | GB_END GB_IF
        ;

end_routine : GB_SUB		{ $$ = TRUE;  }
	    | GB_FUNCTION	{ $$ = FALSE; }
            | GB_PROPERTY       { $$ = FALSE;  } 	    
            ;

declare_routine : GB_SUB NAME opt_arglist
		    { gb_routine_start (buffer, $2, $3, FALSE, GB_CLASS_NONE); }
		| GB_FUNCTION NAME opt_arglist as_type
		    { gb_routine_start (buffer, $2, $3, TRUE, GB_CLASS_NONE); }
                | GB_PROPERTY GB_GET NAME opt_arglist as_type
                    { gb_routine_start (buffer, $3, $4, TRUE, GB_CLASS_PROPERTY_GET); }
                | GB_PROPERTY GB_LET NAME opt_arglist
                    { gb_routine_start (buffer, $3, $4, TRUE, GB_CLASS_PROPERTY_LET); }
                ;

opt_arglist : '(' arglist ')'   { $$ = $2; }
	    | '(' ')'		{ $$ = NULL; }
            |			{ $$ = NULL; }
            ;

arglist : arglist ',' arg       { $$ = g_slist_append ($1, $3);   }
	| arg			{ $$ = g_slist_append (NULL, $1); }
        ;

arg : opt_optional opt_by_val_or_ref opt_parmarray NAME is_array as_type opt_default
    { $$ = gb_arg_desc_new ($4, $6, $7, $2, $1); /* FIXME: opt_parmarray, is_array */ }
    ;

as_type : GB_AS type { $$ = $2; }
        |            { $$ = g_strdup ("Variant"); }
        ;
 
opt_default : '=' const_expr	{ $$ = $2;   }
	    |			{ $$ = NULL; }
            ;

is_array : '(' ')'
         |
         ;

opt_by_val_or_ref : GB_BYREF	{ $$ = FALSE; }
                  | GB_BYVAL	{ $$ = TRUE;  }
                  |		{ $$ = TRUE;  }
                  ;

opt_optional : GB_OPTIONAL	{ $$ = TRUE; }
             |			{ $$ = FALSE; }
             ;

opt_parmarray : GB_PARAM_ARRAY	{ $$ = TRUE; }
              |			{ $$ = FALSE; }
              ;

pub_private : GB_PRIVATE	{ $$ = TRUE; }
            | GB_PUBLIC		{ $$ = FALSE; }
            |			{ $$ = FALSE; }
            ;

opt_static : GB_STATIC		{ $$ = TRUE; }
           |			{ $$ = FALSE; }
           ;

opt_events : GB_WITH_EVENTS	{ $$ = TRUE; }
           |			{ $$ = FALSE; }
           ;

opt_new : GB_NEW		{ $$ = TRUE; }
        |			{ $$ = FALSE; }
	;

const_expr : NAME				{ }
           | GB_FALSE           { $$ = gb_expr_new_boolean (FALSE); }
           | GB_TRUE            { $$ = gb_expr_new_boolean (TRUE); }
	   | INTEGER		{ $$ = gb_expr_new_int ($1); }
	   | FLOAT		{ $$ = gb_expr_new_float ($1); }
	   | STRING		{ $$ = gb_expr_new_string ($1); }
	   | FRX		{ $$ = gb_expr_new_frx ($1); }
	   | '-' const_expr	%prec UNARY_NEG { $$ = gb_expr_new_unary (GB_EXPR_NEGATIVE, $2);}
	   | '+' const_expr	%prec UNARY_POS { $$ = gb_expr_new_unary (GB_EXPR_POSITIVE, $2);}
	   | GB_NOT const_expr	{ $$ = gb_expr_new_unary (GB_EXPR_NOT, $2);}
	   | '(' expr ')'	{ $$ = gb_expr_new_unary (GB_EXPR_PAREN, $2); }

	   | const_expr '&' const_expr
		{ $$ = gb_expr_new_binary ($1, GB_EXPR_CONCAT, $3); }
	   | const_expr GB_AND const_expr
		{ $$ = gb_expr_new_binary ($1, GB_EXPR_AND, $3); }
	   | const_expr GB_OR const_expr
		{ $$ = gb_expr_new_binary ($1, GB_EXPR_OR, $3); }
	   | const_expr GB_XOR const_expr
		{ $$ = gb_expr_new_binary ($1, GB_EXPR_XOR, $3); }
	   | const_expr '>' const_expr
		{ $$ = gb_expr_new_binary ($1, GB_EXPR_GT, $3); }
	   | const_expr GB_GE const_expr
		{ $$ = gb_expr_new_binary ($1, GB_EXPR_GE, $3); }
	   | const_expr '=' const_expr
		{ $$ = gb_expr_new_binary ($1, GB_EXPR_EQ, $3); }
	   | const_expr GB_NE const_expr
		{ $$ = gb_expr_new_binary ($1, GB_EXPR_NE, $3); }
	   | const_expr GB_LE const_expr
		{ $$ = gb_expr_new_binary ($1, GB_EXPR_LE, $3); }
	   | const_expr '<' const_expr
		{ $$ = gb_expr_new_binary ($1, GB_EXPR_LT, $3); }
	   | const_expr '-' const_expr
		{ $$ = gb_expr_new_binary ($1, GB_EXPR_SUB, $3); }
	   | const_expr '+' const_expr
		{ $$ = gb_expr_new_binary ($1, GB_EXPR_ADD, $3); }
	   | const_expr '*' const_expr
		{ $$ = gb_expr_new_binary ($1, GB_EXPR_MULT, $3); }
	   | const_expr '/' const_expr
		{ $$ = gb_expr_new_binary ($1, GB_EXPR_DIV, $3); }
	   | const_expr '\\' const_expr
		{ $$ = gb_expr_new_binary ($1, GB_EXPR_INT_DIV, $3); }
	   | const_expr '^' const_expr
		{ $$ = gb_expr_new_binary ($1, GB_EXPR_POW, $3); }
           ;

/* VB has a nasty habit of using keywords as property/function names */
obj_name : NAME		{ $$ = $1; }
/*
 * FIXME: we need to solve this problem more elegantly
 * essentialy it is not possible to have a routine called these things,
 * only a dereferenced object method etc.
 */
     | GB_TYPE	   { $$ = "Type"; }  
     | GB_NAME	   { $$ = "Name"; }  
     | GB_SELECT   { $$ = "Select"; }
     | GB_LEN      { $$ = "Len"; }   
     | GB_SEEK     { $$ = "Seek"; } 
     | GB_PRINT    { $$ = "Print"; }
     | GB_NOTHING  { $$ = "Nothing"; }
     | GB_PROPERTY { $$ = "Property"; }
     | GB_WRITE    { $$ = "Write"; }
     ;

parm  : expr { $$ = $1 }
      ;

parms : parms parm ','		{ $$ = g_slist_prepend ($1, (gpointer) $2); }
      |				{ $$ = NULL; }
      ;

opt_parms : '(' parms parm ')'	{ $$ = g_slist_prepend ($2, (gpointer) $3); }
          | '(' ')'		{ $$ = NULL; }
          ;             

object_ref  : obj_name opt_parms { $$ = gb_obj_ref_new ($1, TRUE, $2);    }
            | obj_name           { $$ = gb_obj_ref_new ($1, FALSE, NULL); }
            ;

object_refs : object_refs '.' object_ref { $$ = g_slist_prepend ($1, $3);   }
	    | object_refs '!' obj_name   { $$ = g_slist_prepend ($1, gb_obj_ref_new (
		                                  NULL, TRUE, g_slist_prepend (NULL, (gpointer) gb_expr_new_string ($3)))); }
            | object_ref                 { $$ = g_slist_prepend (NULL, $1); }
            | object_ref  '(' expr ')'   { $$ = g_slist_prepend (NULL, $1); } /* Control arrays */
            | GB_SPACE_DOT object_refs   { $$ = g_slist_prepend (NULL, $2)  }
            ;

object_list : object_refs { $$ = gb_expr_new_obj_list ($1); }
            ;

opt_args :  parms parm	{ $$ = g_slist_prepend ($1, (gpointer)$2); }
         |		{ $$ = NULL; }
         ;             

sub_call : obj_name opt_args		 { $$ = gb_expr_new_obj_list (g_slist_prepend (NULL, gb_obj_ref_new ($1, TRUE, $2))); }
         | obj_name opt_parms            { $$ = gb_expr_new_obj_list (g_slist_prepend (NULL, gb_obj_ref_new ($1, TRUE, $2))); }
	 | GB_CALL object_refs		 { $$ = gb_expr_new_obj_list_call ($2); }
	 | object_refs '.' obj_name opt_args  { $$ = gb_expr_new_obj_list (g_slist_prepend ($1, gb_obj_ref_new ($3, TRUE, $4))); }
         | GB_SPACE_DOT NAME opt_args         { $$ = gb_expr_new_obj_list (g_slist_prepend (NULL, gb_obj_ref_new ($2, TRUE, $3))); }
         ;

case_stmt : GB_CASE GB_IS '=' expr eol statements
	    { $$ = gb_select_case_new_comparison (GB_EXPR_EQ, $4, $6); }
	  | GB_CASE GB_IS '<' expr eol statements
	    { $$ = gb_select_case_new_comparison (GB_EXPR_LT, $4, $6); }
	  | GB_CASE GB_IS '>' expr eol statements
	    { $$ = gb_select_case_new_comparison (GB_EXPR_GT, $4, $6); }
	  | GB_CASE GB_IS GB_LE expr eol statements
	    { $$ = gb_select_case_new_comparison (GB_EXPR_LE, $4, $6); }
	  | GB_CASE GB_IS GB_GE expr eol statements
	    { $$ = gb_select_case_new_comparison (GB_EXPR_GE, $4, $6); }
/* FIXME: any more comparisons ? */
	  | GB_CASE expr GB_TO expr eol statements
	    { $$ = gb_select_case_new_expr_to_expr ($2, $4, $6); }
	  | GB_CASE expr eol statements
	    { $$ = gb_select_case_new_expr ($2, $4); }
	  | GB_CASE GB_ELSE eol statements
	    { $$ = gb_select_case_new_else ($4); }
          | GB_CASE expr_csv eol statements
            { $$ = gb_select_case_new_csv ($2, $4); }
	  ;


expr_csv : expr_csv ',' expr { $$ = g_slist_append ($1, (gpointer) $3); }
         | expr              { $$ = g_slist_append (NULL, (gpointer) $1); }
         ;

/*  const_expr_csv : const_expr_csv ',' const_expr { $$ = g_slist_append ($1, (gpointer) $3); } */
/*                 | const_expr                    { $$ = g_slist_append (NULL, (gpointer) $1); } */
/*                 ; */

case_stmts : case_stmts case_stmt { $$ = g_slist_append ($1, $2); }
	   |                      { $$ = NULL;                    }
	   ;

opt_expr : expr { $$ = $1;   }
	 |      { $$ = NULL; }
	 ;

expr : GB_FALSE         { $$ = gb_expr_new_boolean (FALSE); } 
     | GB_TRUE          { $$ = gb_expr_new_boolean (TRUE); } 
     | INTEGER		{ $$ = gb_expr_new_int ($1); }
     | INTEGER'#'	{ $$ = gb_expr_new_float ($1); }
     | FLOAT		{ $$ = gb_expr_new_float ($1); }
     | STRING		{ $$ = gb_expr_new_string ($1); }
     | FRX		{ $$ = gb_expr_new_frx ($1); }
     | object_list	{ $$ = $1; }
     | '+' expr		{ $$ = gb_expr_new_unary (GB_EXPR_POSITIVE, $2);}
     | '-' expr		{ $$ = gb_expr_new_unary (GB_EXPR_NEGATIVE, $2);}
     | GB_NOT expr	{ $$ = gb_expr_new_unary (GB_EXPR_NOT, $2);}
     | '(' expr ')'	{ $$ = gb_expr_new_unary (GB_EXPR_PAREN, $2); }
     | expr '&' expr	{ $$ = gb_expr_new_binary ($1, GB_EXPR_CONCAT, $3); }
     | expr GB_AND expr	{ $$ = gb_expr_new_binary ($1, GB_EXPR_AND, $3); }
     | expr GB_OR expr	{ $$ = gb_expr_new_binary ($1, GB_EXPR_OR, $3); }
     | expr GB_XOR expr	{ $$ = gb_expr_new_binary ($1, GB_EXPR_XOR, $3); }
     | expr '>' expr	{ $$ = gb_expr_new_binary ($1, GB_EXPR_GT, $3); }
     | expr GB_GE expr	{ $$ = gb_expr_new_binary ($1, GB_EXPR_GE, $3); }
     | expr '=' expr	{ $$ = gb_expr_new_binary ($1, GB_EXPR_EQ, $3); }
     | expr GB_IS expr { $$ = gb_expr_new_binary ($1, GB_EXPR_EQ, $3); }
     | expr GB_NE expr	{ $$ = gb_expr_new_binary ($1, GB_EXPR_NE, $3); }
     | expr GB_LE expr	{ $$ = gb_expr_new_binary ($1, GB_EXPR_LE, $3); }
     | expr '<' expr	{ $$ = gb_expr_new_binary ($1, GB_EXPR_LT, $3); }
     | expr '-' expr	{ $$ = gb_expr_new_binary ($1, GB_EXPR_SUB, $3); }
     | expr '+' expr	{ $$ = gb_expr_new_binary ($1, GB_EXPR_ADD, $3); }
     | expr '*' expr	{ $$ = gb_expr_new_binary ($1, GB_EXPR_MULT, $3); }
     | expr '/' expr	{ $$ = gb_expr_new_binary ($1, GB_EXPR_DIV, $3); }
     | expr '\\' expr	{ $$ = gb_expr_new_binary ($1, GB_EXPR_INT_DIV, $3); }
     | expr '^' expr	{ $$ = gb_expr_new_binary ($1, GB_EXPR_POW, $3); }
     | expr GB_EQV expr	{ $$ = gb_expr_new_binary ($1, GB_EXPR_EQV, $3); }
     | expr GB_IMP expr	{ $$ = gb_expr_new_binary ($1, GB_EXPR_IMP, $3); }
     ;

%%

/**
 * gb_parse_gb:
 * @module: The un-parsed module.
 * 
 * parses the Module's stream data into the module.
 * 
 * Return value: whether it failed, TRUE on error.
 **/
gboolean
gb_parse_gb (GBParseData *module)
{
	gboolean ans;

	yacc_hack = module;

	ans = gb_parse (module);

	yacc_hack = NULL;

	return ans;
}

void
gb_error (const char *msg)
{
	g_return_if_fail (yacc_hack != NULL);
	g_return_if_fail (yacc_hack->ec != NULL);

	gb_eval_context_set_line (yacc_hack->ec,
				  gb_lexer_stream_line (yacc_hack->ls));
	gb_eval_exception_fire   (yacc_hack->ec, msg?msg:"no text");
}
