/*
 * GNOME Basic Enterpreter Function registration
 *
 * Authors:
 *    Michael Meeks (mmeeks@gnu.org)
 *    Thomas Meeks  (meekte95@christs-hospital.org.uk)
 *    Ariel Rios    (ariel@arcavia.com)
 *
 * Copyright 2000, Helix Code, Inc.
 */

#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>

#include <gbrun/libgbrun.h>
#include <gbrun/gbrun.h>
#include <gbrun/gbrun-value.h>
#include <gbrun/gbrun-eval.h>
#include <gbrun/gbrun-array.h>
#include <gbrun/gbrun-object.h>
#include <gbrun/gbrun-file.h>
#include <gbrun/gbrun-global.h>

#include <gb/gb-constants.h>

#include "gba-func.h"


static GBValue *
gbrun_func_vartype (GBRunEvalContext *ec,
		    GBRunObject      *object,
		    GBValue         **args)
{
	if (!args [0])
		return gbrun_exception_firev (ec, _("No argument"));

	return gb_value_new_int (gb_value_from_gtk_type (args [0]->gtk_type));
}


static GBValue *
gbrun_func_filelen (GBRunEvalContext *ec,
		  GBRunObject      *object,
		  GBValue         **args)
{
	char       *name;
	struct stat buf;
	
	GB_IS_VALUE (ec, args [0], GB_VALUE_STRING);

	name = args [0]->v.s->str;

	if (stat (name, &buf))
		return gbrun_exception_firev (ec, _("FileLen error on %s"), name);
	
	return gb_value_new_long (buf.st_size);
}


static GBValue *
gbrun_sub_print (GBRunEvalContext *ec,
	       GBRunObject      *object,
	       GSList           *expr)
{
	for (; expr; expr = expr->next) {
		GBValue *v;

		v = gbrun_eval_as (ec, expr->data, GB_VALUE_STRING);
		if (!v)
			return NULL;

		printf ("%s", v->v.s->str);

		gb_value_destroy (v);
	}
	printf ("\n");

	return gb_value_new_empty ();
}

static GBValue *
gbrun_sub_array (GBRunEvalContext *ec,
		 GBRunObject      *object,
		 GSList           *expr)
{
	GBObject *a;
	GSList   *vals = NULL;

	for (; expr; expr = expr->next) {
		GBValue *v;

		v = gb_eval_context_eval (GB_EVAL_CONTEXT (ec), expr->data);
		if (!v)
			return NULL;

		vals = g_slist_append (vals, v);
	}

	a = gbrun_array_new_vals (ec, vals);

	while (vals) {
		gb_value_destroy (vals->data);
		vals = g_slist_remove (vals, vals->data);
	}
	
	return gb_value_new_object (a);
}


static GBValue *
gbrun_func_isnumeric (GBRunEvalContext *ec,
		      GBRunObject      *object,
		      GBValue        **args)
{
	if (!args [0])
		return gb_value_new_boolean (FALSE);

	switch (gb_value_from_gtk_type (args [0]->gtk_type)) {

	case GB_VALUE_CURRENCY:
	case GB_VALUE_NUMBER:
		return gb_value_new_boolean (TRUE);

	default:
		return gb_value_new_boolean (FALSE);
	}
}

static GBValue *
gbrun_func_isnull (GBRunEvalContext *ec,
		    GBRunObject      *object,
		    GBValue        **args)
{
	if (!args [0] ||
	    args [0]->gtk_type != gb_gtk_type_from_value (GB_VALUE_NULL))
		return gb_value_new_boolean (FALSE);

	return gb_value_new_boolean (TRUE);
}


static GBValue *
gbrun_func_isobject (GBRunEvalContext *ec,
		    GBRunObject      *object,
		    GBValue        **args)
{
	if (!args [0] ||
	    args [0]->gtk_type != gb_gtk_type_from_value (GB_VALUE_OBJECT))
		return gb_value_new_boolean (FALSE);
	
	return gb_value_new_boolean (TRUE);
}

static GBValue *
gbrun_sub_call_by_name (GBRunEvalContext *ec,
			GBRunObject      *object,
			GBValue        **args)
{
	GBObjRef ref;

	GB_IS_VALUE (ec, args [0], GB_VALUE_STRING);

	ref.method = TRUE;
	ref.name   = args [0]->v.s->str;
	ref.parms  = NULL;

	return gbrun_objref_deref (ec, GB_OBJECT (object), &ref, TRUE);
}

static GBValue *
gbrun_func_eval (GBRunEvalContext *ec,
		 GBRunObject      *object,
		 GBValue         **args)
{
	GB_IS_VALUE (ec, args [0], GB_VALUE_STRING);

	return gbrun_eval_str (ec, object, args [0]->v.s->str);
}

static GBValue *
gbrun_sub_execute (GBRunEvalContext *ec,
		   GBRunObject      *object,
		   GBValue         **args)
{
	GB_IS_VALUE (ec, args [0], GB_VALUE_STRING);

	gbrun_exec_str (ec, object, args [0]->v.s->str);

	if (gbrun_eval_context_exception (ec))
		return NULL;

	return gb_value_new_empty ();
}

void
gba_func_register (GBEvalContext *ec)
{
	GBRunObject      *gba_object;
	GBRunObjectClass *gba;

	gba_object = gtk_type_new (
		gbrun_object_subclass_simple (GBRUN_TYPE_OBJECT, "vba"));
	gbrun_global_add (GB_OBJECT (gba_object), "vba");

	gba = GBRUN_OBJECT_GET_CLASS (gba_object);

	gbrun_object_add_method_arg (gba, "func;vartype;a,variant;integer;n",
				     gbrun_func_vartype);
        
	gbrun_object_add_method_arg (gba, "func;filelen;pathname,string;string;i",
				     gbrun_func_filelen);

	gbrun_object_add_method_arg (gba, "func;eof;filenumber,integer;boolean;i",
				     gbrun_func_eof);

	gbrun_object_add_method_arg (gba, "func;seek;filenumber,integer;long;i",
				     gbrun_func_seek);

	gbrun_object_add_method_arg (gba, "func;loc;filenumber,integer;long;i",
				     gbrun_func_loc);

	gbrun_object_add_method_arg (gba, "func;lof;filenumber,integer;long;i",
				     gbrun_func_lof);

	gbrun_object_add_method_arg (gba, "func;freefile;rangenumber,integer,byval,NULL;integer;i",
				     gbrun_func_freefile);

        gbrun_object_add_method_arg (gba, "func;isnumeric;expression,variant;boolean;n",
				     gbrun_func_isnumeric);

	gbrun_object_add_method_arg (gba, "func;isnull;expression,variant;boolean;n",
				     gbrun_func_isnull);

	gbrun_object_add_method_arg (gba, "func;isobject;expression,variant;boolean;n",
				     gbrun_func_isobject);

        gbrun_object_add_method_var (gba, "sub;array;...;n",
				     gbrun_sub_array);

        gbrun_object_add_method_arg (gba, "sub;callbyname;name,string;n",
				     gbrun_sub_call_by_name);

	/* Possibly only valid for vbscript etc. */
        gbrun_object_add_method_arg (gba, "func;eval;basic,string;variant;n",
				     gbrun_func_eval);

        gbrun_object_add_method_arg (gba, "sub;execute;name,string;n",
				     gbrun_sub_execute);

	gba_object = gtk_type_new (
		gbrun_object_subclass_simple (GBRUN_TYPE_OBJECT, "Debug"));
	gbrun_global_add (GB_OBJECT (gba_object), "debug");

	gba = GBRUN_OBJECT_GET_CLASS (gba_object);

	gbrun_object_add_method_var (gba, "sub;print;...;n", gbrun_sub_print);
}

void
gba_func_shutdown (void)
{
}
