/*
 * gbrun-screen.c: the 'Screen' object.
 *
 * Authors:
 *      Michael Meeks  <michael@helixcode.com>
 */
#include <gb/gb.h>
#include <gb/gb-constants.h>
#include <gbrun/gbrun-eval.h>
#include <gbrun/gbrun-global.h>
#include "gbrun-objects.h"

#define ITEM_NAME "gb-screen"

enum {
	ARG_FIRST = 0,
	MOUSE_POINTER
};

typedef struct {
	GBRunObject item;
} GBRunScreen;

static GdkCursor *
get_cursor (GBRunEvalContext *ec, int type)
{
	switch (type) {

	case GB_C_Default:
		return gdk_cursor_new (GDK_ARROW);

	case GB_C_Hourglass:
		return gdk_cursor_new (GDK_WATCH);

	default:
		gbrun_exception_firev (
			ec, "Unknown mouse pointer constant %d", type);
		return NULL;
	}
}

static gboolean
screen_setarg (GBRunEvalContext *ec,
	       GBRunObject      *object,
	       int               property,
	       GBValue          *val)
{
	switch (property) {

	case MOUSE_POINTER: {
		GdkCursor *cursor;
		GList     *roots;

		if (!(cursor = get_cursor (ec, val->v.i)))
			return FALSE;

		roots = gdk_window_get_toplevels ();
		g_list_foreach (roots, (GFunc) gdk_window_set_cursor,
				cursor);
		g_list_free (roots);

		gdk_cursor_destroy (cursor);
		return TRUE;
	}

	default:
		g_warning ("Bad property set %d", property);
		return FALSE;
	}
}

void
gbrun_screen_register (void)
{
	GBObject *gba_object;

	gba_object = gtk_type_new (
		gbrun_screen_get_type ());

	gbrun_global_add (GB_OBJECT (gba_object), "screen");
}

void
gbrun_screen_shutdown (void)
{
}

static void
gbrun_screen_class_init (GBRunObjectClass *klass)
{
	klass->set_arg = screen_setarg;
	klass->get_arg = NULL;

	gbrun_object_add_property (
		klass, "mousepointer", gb_type_int, MOUSE_POINTER);
}

GtkType
gbrun_screen_get_type (void)
{
	static GtkType object_type = 0;

	if (!object_type) {
		static const GtkTypeInfo object_info = {
			ITEM_NAME,
			sizeof (GBRunScreen),
			sizeof (GBRunObjectClass),
			(GtkClassInitFunc)  gbrun_screen_class_init,
			(GtkObjectInitFunc) NULL,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		object_type = gtk_type_unique (GBRUN_TYPE_OBJECT, &object_info);
		gtk_type_class (object_type);
	}

	return object_type;	
}
