/* GIO - GLib Input, Output and Streaming Library
 * 
 * Copyright (C) 2006-2007 Red Hat, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General
 * Public License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Alexander Larsson <alexl@redhat.com>
 */

#include <config.h>
#include <string.h>
#include <gmodule.h>
#include "gvfs.h"
#include "glocalvfs.h"
#include "giomodule.h"
#include <glib/gi18n-lib.h>

G_DEFINE_TYPE (GVfs, g_vfs, G_TYPE_OBJECT);

static void
g_vfs_class_init (GVfsClass *klass)
{
}

static void
g_vfs_init (GVfs *vfs)
{
}

/**
 * g_vfs_get_name:
 * @vfs: an input #GVfs.
 * 
 * Returns the name of @vfs.
 **/
const char *
g_vfs_get_name (GVfs *vfs)
{
  GVfsClass *class;

  g_return_val_if_fail (G_IS_VFS (vfs), NULL);

  class = G_VFS_GET_CLASS (vfs);

  return (* class->get_name) (vfs);
}

/**
 * g_vfs_get_priority:
 * @vfs: an input #GVfs.
 * 
 * Returns the priority of @vfs.
 **/
int
g_vfs_get_priority (GVfs *vfs)
{
  GVfsClass *class;
  
  g_return_val_if_fail (G_IS_VFS (vfs), G_MININT);

  class = G_VFS_GET_CLASS (vfs);

  return (* class->get_priority) (vfs);
}

/**
 * g_vfs_get_file_for_path:
 * @vfs: an input #GVfs.
 * @path: a string containing a VFS path.
 * 
 * Returns a #GFile with the given @path, or %NULL on failure.
 * 
 **/
GFile *
g_vfs_get_file_for_path (GVfs *vfs,
			 const char *path)
{
  GVfsClass *class;
  
  g_return_val_if_fail (G_IS_VFS (vfs), NULL);
  g_return_val_if_fail (path != NULL, NULL);

  class = G_VFS_GET_CLASS (vfs);

  return (* class->get_file_for_path) (vfs, path);
}

/**
 * g_vfs_get_file_for_uri:
 * @vfs: an input #GVfs.
 * @uri: an input string containing a URI path.
 * 
 * Returns a #GFile with the given @uri, or %NULL on failure.
 * 
 **/
GFile *
g_vfs_get_file_for_uri  (GVfs *vfs,
			 const char *uri)
{
  GVfsClass *class;
  
  g_return_val_if_fail (G_IS_VFS (vfs), NULL);
  g_return_val_if_fail (uri != NULL, NULL);

  class = G_VFS_GET_CLASS (vfs);

  return (* class->get_file_for_uri) (vfs, uri);
}

/**
 * g_vfs_get_supported_uri_schemes:
 * @vfs:
 * 
 * Returns:
 * 
 **/
const gchar * const *
g_vfs_get_supported_uri_schemes (GVfs *vfs)
{
  GVfsClass *class;
  
  g_return_val_if_fail (G_IS_VFS (vfs), NULL);

  class = G_VFS_GET_CLASS (vfs);

  return (* class->get_supported_uri_schemes) (vfs);
}

/**
 * g_vfs_parse_name:
 * @vfs:
 * @parse_name:
 * 
 * Returns a #GFile for the given @parse_name.
 * 
 **/
GFile *
g_vfs_parse_name (GVfs *vfs,
		  const char *parse_name)
{
  GVfsClass *class;
  
  g_return_val_if_fail (G_IS_VFS (vfs), NULL);
  g_return_val_if_fail (parse_name != NULL, NULL);

  class = G_VFS_GET_CLASS (vfs);

  return (* class->parse_name) (vfs, parse_name);
}

static gpointer
get_default_vfs (gpointer arg)
{
  volatile GType local_type;
  GType *vfs_impls;
  int i;
  guint n_vfs_impls;
  const char *use_this;
  GVfs *vfs, *max_prio_vfs;
  int max_prio;
  GType (*casted_get_type)(void);

  use_this = g_getenv ("GIO_USE_VFS");
  
  /* Ensure GLocalVfs type is availible
     the cast is required to avoid any G_GNUC_CONST optimizations */
  casted_get_type = g_local_vfs_get_type;
  local_type = casted_get_type ();
  
  /* Ensure vfs in modules loaded */
  g_io_modules_ensure_loaded (GIO_MODULE_DIR);

  vfs_impls = g_type_children (G_TYPE_VFS, &n_vfs_impls);

  max_prio = G_MININT;
  max_prio_vfs = NULL;
  for (i = 0; i < n_vfs_impls; i++)
    {
      vfs = g_object_new (vfs_impls[i], NULL);

      if (use_this && strcmp (g_vfs_get_name (vfs), use_this) == 0)
	{
	  max_prio = G_MAXINT;
	  if (max_prio_vfs)
	    g_object_unref (max_prio_vfs);
	  max_prio_vfs = g_object_ref (vfs);
	}

      if (max_prio < g_vfs_get_priority (vfs))
	{
	  max_prio = g_vfs_get_priority (vfs);
	  if (max_prio_vfs)
	    g_object_unref (max_prio_vfs);
	  max_prio_vfs = g_object_ref (vfs);
	}

      g_object_unref (vfs);
    }
  
  g_free (vfs_impls);

  /* We should at least have gotten the local implementation */
  g_assert (max_prio_vfs != NULL);

  return max_prio_vfs;
}

/**
 * g_vfs_get_default:
 * 
 * Returns the default #GVfs for the system.
 **/
GVfs *
g_vfs_get_default (void)
{
  static GOnce once_init = G_ONCE_INIT;
  
  return g_once (&once_init, get_default_vfs, NULL);
}

/**
 * g_vfs_get_local:
 * 
 * Returns the local #GVfs for the system.
 **/
GVfs *
g_vfs_get_local (void)
{
  static gsize vfs = 0;

  if (g_once_init_enter (&vfs))
    g_once_init_leave (&vfs, (gsize)g_local_vfs_new ());

  return G_VFS (vfs);
}

