/* vim: set sw=8: -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

#include <config.h>
#include "add-printer.h"

#include <cups/cups.h>
#include <cups/http.h>
#include <cups/ipp.h>
#include <unistd.h>
#include <sys/types.h>

#include <gtk/gtkcellrenderertext.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtkliststore.h>
#include <gtk/gtkradiobutton.h>
#include <gtk/gtktreeview.h>
#include <gtk/gtkcomboboxentry.h>
#include <glade/glade.h>
#include <glib/gi18n.h>
#include <libgnomeui/gnome-druid.h>
#include <libgnomeui/gnome-ui-init.h>

#include <libgnomecups/gnome-cups-ui-init.h>
#include <libgnomecups/gnome-cups-request.h>
#include <libgnomecups/gnome-cups-util.h>
#include <libgnomecups/gnome-cups-ui-util.h>
#include <libgnomecups/gnome-cups-ui-driver.h>

#include "druid-helper.h"
#include "snmpinter.h"

#include <stdarg.h>
#include <gnome-keyring.h>
#ifdef HAVE_LIBSMBCLIENT
#include <libgnomeui/gnome-password-dialog.h>
#include <libsmbclient.h>
#include <errno.h>
#endif

/* Keep in sync with the connection_types menu */
typedef enum {
	CONNECTION_TYPE_IPP,
	CONNECTION_TYPE_SMB,
	CONNECTION_TYPE_LPD,
	CONNECTION_TYPE_HP,
	CONNECTION_TYPE_LOCAL
} PrinterConnectionType;

enum {
	LOCAL_DETECTED_LABEL,
	LOCAL_DETECTED_PRINTER,
	LAST_LOCAL_DETECTED
};

typedef struct {
	char const *cannonical_name;
	char const *corporate_icon;
	GHashTable *aliases;
} Vendor;

static char *app_path = NULL;
static char *
entry_get_text_stripped (GladeXML *xml, char const *name)
{
	GtkWidget *w =  glade_xml_get_widget (xml, name);
	char const *content;

	if (GTK_IS_COMBO_BOX_ENTRY (w))
		/* cheesy hack, we should be able to do this more gracefully */
		w = gtk_bin_get_child (GTK_BIN (w));

	content = gtk_entry_get_text (GTK_ENTRY (w));
	if (content != NULL)
		return g_strstrip (g_strdup (content));
	return NULL;
}

static GCupsDriverSelector *
driver_selector (GladeXML *xml)
{
	return (GCupsDriverSelector *) glade_xml_get_widget (xml, "driver_selector");
}

static void
set_selected_uri (GladeXML *xml, char const *uri)
{
	g_object_set_data_full (G_OBJECT (xml),
		"selected_uri", g_strdup (uri), g_free);
}

static char *
get_selected_uri (GladeXML *xml)
{
	char *str = g_object_get_data (G_OBJECT (xml), "selected_uri");
	g_return_val_if_fail (str != NULL, NULL);

	return g_strdup (str);
}

/*****************************************************************************/

typedef struct {
	char *label;
	char *uri;
	char *vendor_and_model;
} LocalPrinter;

static void
local_printer_free (LocalPrinter *printer)
{
	g_free (printer->label);
	g_free (printer->uri);
	g_free (printer->vendor_and_model);
	g_free (printer);
}

static void
local_printer_list_free (GSList *list)
{
	g_slist_foreach (list, (GFunc)local_printer_free, NULL);
	g_slist_free (list);
}

/*****************************************************************************/

static char *
combine_vendor_and_model (char const *vendor, char const *model)
{
	if (strstr (model, vendor) == model)
		return g_strdup (model);
	return g_strdup_printf ("%s %s", vendor, model);
}

static char *
parse_network_detect (char const *line)
{
	char *raw_line = g_strdup (line);
	char *vendor_and_model;
	char *vendor = NULL;
	char *model = NULL;
	char **fields;
	char **p;
	char *sep;

	sep = strchr (line, '\n');
	if (sep) {
		*sep = '\0';
	}

	line = g_strstrip (raw_line);
	fields = g_strsplit (line, ";", -1);

	for (p = fields; *p != NULL; p++) {
		char **pieces = g_strsplit (*p, "=", -1);
		char *name = pieces[0];
		char *value = pieces[1];
		if (!name || !value) {
			g_strfreev (pieces);
			continue;
		}

		if (!strcmp (name, "vendor")) {
			vendor = g_strdup (value);
		} else if (!strcmp (name, "model")) {
			model = g_strdup (value);
		}

		g_strfreev (pieces);
	}
	g_strfreev (fields);

	if (!vendor || !model) {
		g_free (raw_line);
		g_free (vendor);
		g_free (model);
		return NULL;
	}

	vendor_and_model = combine_vendor_and_model (vendor, model);

	g_free (vendor);
	g_free (model);
	g_free (raw_line);

	return vendor_and_model;
}

/*************************************************************************/
/* Local Printers */

static void
local_port_init (GladeXML *xml, GSList *devices)
{
	char *label;
	GSList		*ptr;
	LocalPrinter	*desc;
	GtkTreeIter	 iter;
        GtkComboBox	*combo;
        GtkCellRenderer *renderer;
	GtkListStore	*model = gtk_list_store_new (2,
		G_TYPE_STRING, G_TYPE_POINTER);

	for (ptr = devices ; ptr != NULL; ptr = ptr->next) {
		desc = ptr->data;
		label = (desc->vendor_and_model)
			? g_strdup_printf ("%s (%s)",
					   desc->label,
					   desc->vendor_and_model)
			: g_strdup (desc->label);
		gtk_list_store_append (model, &iter);
		gtk_list_store_set (model, &iter, 0, label, 1, desc, -1);
		g_free (label);
	}
	combo = GTK_COMBO_BOX (glade_xml_get_widget (xml, "local_ports")),
	gtk_combo_box_set_model (combo, GTK_TREE_MODEL (model));
	gtk_combo_box_set_active (combo, -1);
	if (devices != NULL)
		gtk_combo_box_set_active (combo, 0);

        renderer = gtk_cell_renderer_text_new ();
        gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (combo), renderer, TRUE);
        gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (combo), renderer,
		"text", 0, NULL);
}

static void
update_local_location_sensitivities (GladeXML *xml)
{
	GtkWidget *view = glade_xml_get_widget (xml, "local_detected_view");
	GtkWidget *local_ports = glade_xml_get_widget (xml, "local_ports");

	if (toggle_button_is_active (xml, "local_use_detected_radio")) {
		gtk_widget_set_sensitive (view, TRUE);
		gtk_widget_set_sensitive (local_ports, FALSE);
	} else {
		gtk_widget_set_sensitive (view, FALSE);
		gtk_widget_set_sensitive (local_ports, TRUE);
	}
}

static GSList *
get_local_devices (void)
{
	GSList *ret = NULL;
	ipp_t *request  = gnome_cups_request_new (CUPS_GET_DEVICES);
	ipp_t *response = gnome_cups_request_execute (request, NULL, "/", NULL);

	if (response) {
		ipp_attribute_t *attr;
		LocalPrinter *desc;
		char *device_class = NULL;

		desc = g_new0 (LocalPrinter, 1);
		for (attr = response->attrs; attr != NULL; attr = attr->next) {
			if (NULL == attr->name) {
				if (device_class && strcmp (device_class, "network") &&
				    desc->label && desc->uri)
					ret = g_slist_prepend (ret, desc);
				else
					g_free (desc);

				g_free (device_class);
				device_class = NULL;

				desc = g_new0 (LocalPrinter, 1);
			} else if (!strcmp (attr->name, "device-class")) {
				g_free (device_class);
				device_class = g_strdup (attr->values[0].string.text);
			} else if (!strcmp (attr->name, "device-info")) {
				g_free (desc->label);
				desc->label = g_strdup (attr->values[0].string.text);
			} else if (!strcmp (attr->name, "device-uri")) {
				g_free (desc->uri);
				desc->uri = g_strdup (attr->values[0].string.text);
			} else if (!strcmp (attr->name, "device-make-and-model") && strcmp (attr->values[0].string.text, "Unknown")) {
				g_free (desc->vendor_and_model);
				desc->vendor_and_model = g_strdup (attr->values[0].string.text);
			}
		}

		if (device_class && strcmp (device_class, "network") &&
		    desc->label && desc->uri)
			ret = g_slist_prepend (ret, desc);
		else
			local_printer_free (desc);
		g_free (device_class);

		ippDelete (response);
	}

	return g_slist_reverse (ret);
}

/****************************************************************************/

static GHashTable *smb_servers;
static GSList *new_servers = NULL;
static GSList *new_printers = NULL;
static GStaticMutex server_mutex = G_STATIC_MUTEX_INIT;
#ifdef HAVE_LIBSMBCLIENT
static GStaticMutex smb_request_mutex = G_STATIC_MUTEX_INIT;
static GStaticMutex printer_request_mutex = G_STATIC_MUTEX_INIT;

typedef struct {
	GCond *cond;
	/* In */  char *server, *share;
		gboolean keyring;
	/* Out */ char *workgroup, *id, *passwd;
} SmbReqAuth;
typedef struct {
	char *name, *path;
} SmbNewServer;
typedef struct {
	char *server, *printer;
} SmbNewPrinter;
typedef struct {
	char *server;
} SmbGetPrinters;

static SmbReqAuth *auth_req = NULL;

/***************************************************************************/
/* On the SMB side */
static void 
smb_auth_fn (char const *server, char const *share,
	     char *workgroup, int workgroup_max,
	     char *id,	      int id_len,
	     char *passwd,    int passwd_len)
{
	SmbReqAuth *req;

	/* blah I need some extra user hooks here to store the keyring override
	 * flag safely */
	static gboolean used_keyring = FALSE;
	static char *used_keyring_for = NULL;

	req = g_new0 (SmbReqAuth, 1);
	req->cond = g_cond_new ();
	req->server = g_strdup (server);
	req->share = g_strdup (share);

	/* cheesy hack to override the use of the keyring if the previous
	 * attmempt failed */
	req->keyring == (used_keyring_for == NULL ||
			 0 != strcmp (used_keyring_for, share));

	g_static_mutex_lock (&smb_request_mutex);
	if (auth_req != NULL) { g_warning ("dropping an auth req"); }
	auth_req = req;
	g_cond_wait (req->cond, g_static_mutex_get_mutex (&smb_request_mutex));

	auth_req = NULL;
	g_static_mutex_unlock (&smb_request_mutex);

	strncpy (id,	 req->id     ? req->id     : "", id_len);
	strncpy (passwd, req->passwd ? req->passwd : "", passwd_len);

	used_keyring = req->keyring;
	g_free (used_keyring_for);
	used_keyring_for = g_strdup (server);

	g_cond_free (req->cond);
	g_free (req->server);
	g_free (req->share);
	g_free (req->workgroup);
	g_free (req->id);
	g_free (req->passwd);
	g_free (req);
}

static gpointer
cb_smb_thread (gpointer data)
{
	struct smbc_dirent const *dirent;
	int wg_handle, serve_handle;

	if ((wg_handle = smbc_opendir ("smb://")) >= 0) {
		while ((dirent = smbc_readdir (wg_handle)))
			if (dirent->smbc_type == SMBC_WORKGROUP) {
				char *path = g_strconcat ("smb://", dirent->name, NULL);
				if ((serve_handle = smbc_opendir (path)) >= 0) {
					while ((dirent = smbc_readdir (serve_handle)))
						if (dirent->smbc_type == SMBC_SERVER) {
							char *path = g_strconcat ("smb://", dirent->name, "/", NULL);
							g_static_mutex_lock (&server_mutex);
							new_servers = g_slist_append (new_servers, g_strdup (dirent->name));
							new_servers = g_slist_append (new_servers, path);
							g_static_mutex_unlock (&server_mutex);
						}
					smbc_closedir (serve_handle);
				} else {
					g_warning ("Could not list %s : %s\n", path, strerror (errno));
				}
				g_free (path);
			}

		smbc_closedir (wg_handle);
	} else {
		g_warning ("Could not list %s : %s\n", "smb://", strerror (errno));
	}

	return NULL;
}

static gpointer
cb_smb_find_printers (char const *dir_url)
{
	struct smbc_dirent const *dirent;
	int dir_handle;

reauth :
	if ((dir_handle = smbc_opendir (dir_url)) >= 0) {
		while ((dirent = smbc_readdir (dir_handle)))
			if (dirent->smbc_type == SMBC_PRINTER_SHARE) {
				g_static_mutex_lock (&printer_request_mutex);
				new_printers = g_slist_append (new_printers, g_strdup (dir_url));
				new_printers = g_slist_append (new_printers, g_strdup (dirent->name));
				g_static_mutex_unlock (&printer_request_mutex);
			}
		smbc_closedir (dir_handle);
	} else if (dir_handle == EACCES)
		goto reauth;
	else {
		g_warning ("Could not list %s : %s\n", dir_url, strerror (errno));
	}

	return NULL;
}

/***************************************************************************/
/* On the UI side */
static void
ui_auth_req_handler (GladeXML *xml)
{
	static char *default_id = NULL;
	GList *list = NULL;
	GtkWidget *w;
	GnomePasswordDialog *dialog;
	GnomeKeyringResult result = GNOME_KEYRING_RESULT_OK;
	gboolean found = FALSE;

	g_warning ("authenticating with %s for %s", auth_req->server, auth_req->share);

	if (default_id == NULL)
		default_id = g_strdup (g_getenv ("USER"));
	if (default_id == NULL)
		default_id = g_strdup (g_getenv ("LOGNAME"));

	if (default_id != NULL && auth_req->workgroup != NULL && auth_req->keyring == TRUE)
		result = gnome_keyring_find_network_password_sync (default_id,
			auth_req->workgroup, auth_req->server, auth_req->share,
			"smb", NULL, 0, &list);
	
	if (list != NULL) {
		if (result == GNOME_KEYRING_RESULT_OK) {
			GnomeKeyringNetworkPasswordData *pwd_data = list->data;
			auth_req->id = g_strdup (pwd_data->user);
			auth_req->passwd = g_strdup (pwd_data->password);
			found = TRUE;
		}
		gnome_keyring_network_password_list_free (list);
	}

	if (!found) {
		char *msg = NULL;
		if (auth_req->server != NULL) {
			if (auth_req->workgroup != NULL)
				msg = g_strdup_printf (_("Identity and Password for %s in workgroup %s"),
					auth_req->server, auth_req->workgroup);
			else
				msg = g_strdup_printf (_("Identity and Password for %s"),
					auth_req->server);
		} else if (auth_req->workgroup != NULL)
			msg = g_strdup_printf (_("Identity and Password for workgroup %s"),
				auth_req->workgroup);
		else {
			g_warning ("huh ?? what are we authenticating for ?");
			msg = g_strdup_printf (_("Identity and Password"));
		}

		dialog = GNOME_PASSWORD_DIALOG (gnome_password_dialog_new (
			_("Authentication Required"), msg, "", "", FALSE));
		if (default_id != NULL)
			gnome_password_dialog_set_username	(dialog, default_id);
		gnome_password_dialog_set_show_username		(dialog,  TRUE);
		gnome_password_dialog_set_show_domain		(dialog, FALSE);
		gnome_password_dialog_set_show_password		(dialog,  TRUE);
		gnome_password_dialog_set_show_remember		(dialog, FALSE);
#ifdef HAVE_GNOME_PASSWORD_DIALOG_SET_SHOW_USERPASS_BUTTONS
		gnome_password_dialog_set_show_userpass_buttons (dialog, FALSE);
#endif

		auth_req->keyring = FALSE;
		if (gnome_password_dialog_run_and_block (dialog)) {
			auth_req->id     = gnome_password_dialog_get_username (dialog);
			auth_req->passwd = gnome_password_dialog_get_password (dialog);
		}
		gtk_widget_destroy (GTK_WIDGET (dialog));
		g_free (msg);
	}
	if (auth_req->id != NULL) {
		w = glade_xml_get_widget (xml, "smb_username_entry");
		gtk_entry_set_text (GTK_ENTRY (w), auth_req->id);
	}
	if (auth_req->passwd != NULL) {
		w = glade_xml_get_widget (xml, "smb_password_entry");
		gtk_entry_set_text (GTK_ENTRY (w), auth_req->passwd);
	}
}

static void
ui_add_server_handler (GladeXML *xml)
{
	GtkComboBox  *combo = GTK_COMBO_BOX (glade_xml_get_widget (xml, "smb_host_entry"));
	GtkListStore *store = (GtkListStore *) gtk_combo_box_get_model (combo);
	GtkTreeIter   iter;
	char *name, *path;

	g_return_if_fail (new_servers != NULL);
	name = new_servers->data;
	new_servers = g_slist_remove (new_servers, name);

	g_return_if_fail (new_servers != NULL); /* minor leak of name */
	path = new_servers->data;
	new_servers = g_slist_remove (new_servers, path);

	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter,
		0, name,
		1, path,
		-1);
	g_free (name);
	g_free (path);
}

static void
ui_add_printer_handler (GladeXML *xml)
{
	GtkListStore *store;
	GtkTreeIter   iter;
	char *server, *printer;

	g_return_if_fail (new_printers != NULL);
	server = new_printers->data;
	new_printers = g_slist_remove (new_printers, server);

	g_return_if_fail (new_printers != NULL); /* minor leak of server */
	printer = new_printers->data;
	new_printers = g_slist_remove (new_printers, printer);

	store = g_hash_table_lookup (smb_servers, server);
	if (store != NULL) {
		gtk_list_store_append (store, &iter);
		gtk_list_store_set (store, &iter,
			0, printer,
			-1);
	} else {
		g_warning ("missing smb server model ??");
	}
	g_free (printer);
	g_free (server);
}
static gint
cb_smb_req_handler (GladeXML *xml)
{
	g_static_mutex_lock (&smb_request_mutex);
	if (auth_req != NULL) {
		ui_auth_req_handler (xml);
		g_cond_signal (auth_req->cond);
	}
	g_static_mutex_unlock (&smb_request_mutex);

	g_static_mutex_lock (&server_mutex);
	while (new_servers != NULL)
		ui_add_server_handler (xml);
	g_static_mutex_unlock (&server_mutex);

	g_static_mutex_lock (&printer_request_mutex);
	while (new_printers != NULL)
		ui_add_printer_handler (xml);
	g_static_mutex_unlock (&printer_request_mutex);
	return TRUE;
}
#endif

static void
cb_smb_host_changed (GtkComboBox *combo, GladeXML *xml)
{
	GtkTreeIter   iter;

	if (gtk_combo_box_get_active_iter (combo, &iter)) {
		gboolean make_req = FALSE;
		char *path;
		GtkTreeModel *model = gtk_combo_box_get_model (combo);
		gtk_tree_model_get (model, &iter, 1, &path, -1);
		if (smb_servers == NULL)
			smb_servers = g_hash_table_new_full (g_str_hash, g_str_equal,
					       g_free, NULL);
		if (NULL == (model = g_hash_table_lookup (smb_servers, path))) {
			model = (GtkTreeModel *)gtk_list_store_new (1, G_TYPE_STRING);
			g_hash_table_insert (smb_servers, g_strdup (path), model);
			make_req = TRUE;
		}

		gtk_combo_box_set_model (
			(GtkComboBox *)glade_xml_get_widget (xml, "smb_printer_entry"),
			GTK_TREE_MODEL (model));
#ifdef HAVE_LIBSMBCLIENT
		if (make_req)
			g_thread_create ((GThreadFunc)cb_smb_find_printers,
					 path, TRUE, NULL);
#endif
	}
}

static void
init_smb_combos (GladeXML *xml)
{
	GtkComboBox  *combo = GTK_COMBO_BOX (glade_xml_get_widget (xml, "smb_host_entry"));
	GtkListStore *store = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_STRING);
	gtk_combo_box_set_model (combo, GTK_TREE_MODEL (store));
	gtk_combo_box_entry_set_text_column (GTK_COMBO_BOX_ENTRY (combo), 0);
	gtk_combo_box_set_active (combo, -1);
	g_signal_connect (GTK_WIDGET (combo),
		"changed",
		G_CALLBACK (cb_smb_host_changed), xml);

	combo = GTK_COMBO_BOX (glade_xml_get_widget (xml, "smb_printer_entry"));
	store = gtk_list_store_new (1, G_TYPE_STRING);
	gtk_combo_box_set_model (combo, GTK_TREE_MODEL (store));
	gtk_combo_box_entry_set_text_column (GTK_COMBO_BOX_ENTRY (combo), 0);
	gtk_combo_box_set_active (combo, -1);
}

/*****************************************************************************/
/* Internal function lifted from Lifted from glib-2.5.0 */
typedef enum {
  UNSAFE_ALL        = 0x1,  /* Escape all unsafe characters   */
  UNSAFE_ALLOW_PLUS = 0x2,  /* Allows '+'  */
  UNSAFE_PATH       = 0x8,  /* Allows '/', '&', '=', ':', '@', '+', '$' and ',' */
  UNSAFE_HOST       = 0x10, /* Allows '/' and ':' and '@' */
  UNSAFE_SLASHES    = 0x20  /* Allows all characters except for '/' and '%' */
} UnsafeCharacterSet;

static const guchar acceptable[96] = {
  /* A table of the ASCII chars from space (32) to DEL (127) */
  /*      !    "    #    $    %    &    '    (    )    *    +    ,    -    .    / */ 
  0x00,0x3F,0x20,0x20,0x28,0x00,0x2C,0x3F,0x3F,0x3F,0x3F,0x2A,0x28,0x3F,0x3F,0x1C,
  /* 0    1    2    3    4    5    6    7    8    9    :    ;    <    =    >    ? */
  0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x38,0x20,0x20,0x2C,0x20,0x20,
  /* @    A    B    C    D    E    F    G    H    I    J    K    L    M    N    O */
  0x38,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,
  /* P    Q    R    S    T    U    V    W    X    Y    Z    [    \    ]    ^    _ */
  0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x20,0x20,0x20,0x20,0x3F,
  /* `    a    b    c    d    e    f    g    h    i    j    k    l    m    n    o */
  0x20,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,
  /* p    q    r    s    t    u    v    w    x    y    z    {    |    }    ~  DEL */
  0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x3F,0x20,0x20,0x20,0x3F,0x20
};

static const gchar hex[16] = "0123456789ABCDEF";

/* Note: This escape function works on file: URIs, but if you want to
 * escape something else, please read RFC-2396 */
static gchar *
g_escape_uri_string (const gchar *string, 
		     UnsafeCharacterSet mask)
{
#define ACCEPTABLE(a) ((a)>=32 && (a)<128 && (acceptable[(a)-32] & use_mask))

  const gchar *p;
  gchar *q;
  gchar *result;
  int c;
  gint unacceptable;
  UnsafeCharacterSet use_mask;
  
  g_return_val_if_fail (mask == UNSAFE_ALL
			|| mask == UNSAFE_ALLOW_PLUS
			|| mask == UNSAFE_PATH
			|| mask == UNSAFE_HOST
			|| mask == UNSAFE_SLASHES, NULL);
  
  unacceptable = 0;
  use_mask = mask;
  for (p = string; *p != '\0'; p++)
    {
      c = (guchar) *p;
      if (!ACCEPTABLE (c)) 
	unacceptable++;
    }
  
  result = g_malloc (p - string + unacceptable * 2 + 1);
  
  use_mask = mask;
  for (q = result, p = string; *p != '\0'; p++)
    {
      c = (guchar) *p;
      
      if (!ACCEPTABLE (c))
	{
	  *q++ = '%'; /* means hex coming */
	  *q++ = hex[c >> 4];
	  *q++ = hex[c & 15];
	}
      else
	*q++ = *p;
    }
  
  *q = '\0';
  
  return result;
}

/*****************************************************************************/

static char *
get_uri_ipp (GladeXML *xml)
{
	char *ret, *escaped_entry;
	char *stripped_entry = entry_get_text_stripped (xml, "ipp_uri_entry");

	if (stripped_entry == NULL || stripped_entry[0] == '\0')
		return NULL;

	escaped_entry = g_escape_uri_string (stripped_entry, UNSAFE_HOST);
	g_free (stripped_entry);
	if (escaped_entry == NULL)
		return NULL;

	if (0 != g_ascii_strncasecmp (escaped_entry, "ipp:", 4) &&
	    0 != g_ascii_strncasecmp (escaped_entry, "http:", 5))
		ret = g_strdup_printf ("ipp://%s", escaped_entry);
	else
		ret = g_strdup (escaped_entry);
	g_free (escaped_entry);

	return ret;
}

static char *
get_uri_smb (GladeXML *xml)
{
	char *ret = NULL;
	char *host = entry_get_text_stripped (xml, "smb_host_entry");
	char *printer = entry_get_text_stripped (xml, "smb_printer_entry");
	char *username = entry_get_text_stripped (xml, "smb_username_entry");
	char *password = entry_get_text_stripped (xml, "smb_password_entry");

	if (host != NULL && host[0] && printer != NULL && printer[0]) {
		if (username != NULL && username[0])
			ret = g_strdup_printf ("smb://%s:%s@%s/%s",
				username, password, host, printer);
		else
			ret = g_strdup_printf ("smb://%s/%s", host, printer);
	}

	g_free (host);
	g_free (printer);
	g_free (username);
	g_free (password);

	return ret;
}

static char *
get_uri_lpd (GladeXML *xml)
{
	char *ret = NULL;
	char *host = entry_get_text_stripped (xml, "lpd_host_entry");
	char *queue = entry_get_text_stripped (xml, "lpd_queue_entry");

	if (host[0])
		ret = g_strdup_printf ("lpd://%s/%s", host, queue);
	g_free (host);
	g_free (queue);
	return ret;
}

static char *
get_uri_hp (GladeXML *xml)
{
	char *ret = NULL;
	char *host = entry_get_text_stripped (xml, "hp_host_entry");
	char *port = entry_get_text_stripped (xml, "hp_port_entry");

	if (port == NULL || !port[0]) {
		g_free (port);
		port = g_strdup ("9100");
	}

	if (host != NULL && host[0])
		ret = g_strdup_printf ("socket://%s:%s", host, port);

	g_free (host);
	g_free (port);

	return ret;
}

static LocalPrinter const *
get_current_local (GladeXML *xml)
{
	GtkTreeIter iter;
	GtkTreeView *tree_view	    = GTK_TREE_VIEW (glade_xml_get_widget (xml, "local_detected_view"));
	GtkTreeSelection *selection = gtk_tree_view_get_selection (tree_view);
	LocalPrinter const *lp = NULL;

	if (toggle_button_is_active (xml, "local_specify_port_radio"))
		combo_selected_get (xml, "local_ports", 1, &lp, -1);
	else if (gtk_tree_selection_get_selected (selection, NULL, &iter))
		gtk_tree_model_get (gtk_tree_view_get_model (tree_view), &iter,
				    1, &lp,
				    -1);
	return lp;
}

static char *
get_uri_local (GladeXML *xml)
{
	LocalPrinter const *lp = get_current_local (xml);
	return (lp != NULL) ? g_strdup (lp->uri) : NULL;
}

static PrinterConnectionType
get_connection_type (GladeXML *xml)
{
	GtkComboBox *combo = (GtkComboBox *)glade_xml_get_widget (xml, "connection_types");
	int tmp = gtk_combo_box_get_active (combo);
	if (tmp >= 0)
		return (PrinterConnectionType)tmp;
	return CONNECTION_TYPE_IPP;
}

static char *
get_connection_uri (GladeXML *xml)
{
	switch (get_connection_type (xml)) {
	case CONNECTION_TYPE_IPP :	return get_uri_ipp (xml);
	case CONNECTION_TYPE_SMB :	return get_uri_smb (xml);
	case CONNECTION_TYPE_LPD :	return get_uri_lpd (xml);
	case CONNECTION_TYPE_HP :	return get_uri_hp (xml);
	case CONNECTION_TYPE_LOCAL :	return get_uri_local (xml);

	default :
		g_warning ("unsupported type\n");
	}

	return NULL;
}

static void
cb_connection_type_changed (GtkComboBox *combo, GladeXML *xml)
{
	static gboolean start_smb_scan = TRUE;
	GtkWidget *notebook = glade_xml_get_widget (xml, "connection_notebook");
	gtk_notebook_set_current_page (GTK_NOTEBOOK (notebook),
		gtk_combo_box_get_active (combo));

#ifdef HAVE_LIBSMBCLIENT
	if (start_smb_scan &&
	    get_connection_type (xml) == CONNECTION_TYPE_SMB) {
		start_smb_scan = FALSE;
		if (smbc_init (smb_auth_fn, 0) >= 0) {
			if (!g_thread_supported ())
				g_thread_init (NULL);
			g_thread_create (cb_smb_thread, NULL, TRUE, NULL);
			g_timeout_add (200, (GtkFunction)cb_smb_req_handler, xml);
		} else {
			g_warning ("smbc_init returned %s (%i)\nDo you have a ~/.smb/smb.conf file?\n",
				   strerror (errno), errno);
		}
	}
#endif
}

static void
setup_local_connections (GladeXML *xml)
{
	GtkTreeView	  *tree_view;
	GtkListStore	  *list_store;
	GtkTreeSelection  *selection;
	GtkCellRenderer	  *renderer;
	GtkTreeViewColumn *column;
	int num_detected;
	GSList *ptr, *devices;

	devices = get_local_devices ();
	g_object_set_data_full (G_OBJECT (xml),
		"local-devices", devices,
		(GDestroyNotify)local_printer_list_free);
	local_port_init (xml, devices);

	tree_view = GTK_TREE_VIEW (glade_xml_get_widget (xml, "local_detected_view"));

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Printer"),
							   renderer,
							   "markup",
							   LOCAL_DETECTED_LABEL,
							   NULL);
	gtk_tree_view_append_column (tree_view, column);

	list_store = gtk_list_store_new (LAST_LOCAL_DETECTED,
					 G_TYPE_STRING, G_TYPE_POINTER);
	gtk_tree_view_set_model (tree_view, GTK_TREE_MODEL (list_store));

	selection = gtk_tree_view_get_selection (tree_view);
	num_detected = 0;
	for (ptr = devices; ptr != NULL; ptr = ptr->next) {
		LocalPrinter *desc = ptr->data;
		if (desc->vendor_and_model
		    /* weed out two that are always returned
		     * regardless of what's there */
		    && strcmp (desc->vendor_and_model, "EPSON")
		    && strcmp (desc->vendor_and_model, "CANON")) {
			GtkTreeIter iter;

			gtk_list_store_append (list_store, &iter);
			gtk_list_store_set (list_store, &iter,
					    0, desc->vendor_and_model,
					    1, desc,
					    -1);
			if (num_detected == 0)
				gtk_tree_selection_select_iter (selection, &iter);
			num_detected++;
		}
	}

	if (num_detected == 0) {
		GtkTreeIter iter;
		gtk_list_store_append (list_store, &iter);
		gtk_list_store_set (list_store, &iter,
				    0, _("<i>No printers detected</i>"),
				    1, NULL,
				    -1);

		gtk_tree_selection_set_mode (selection, GTK_SELECTION_NONE);
		gtk_tree_selection_set_mode (selection, GTK_SELECTION_NONE);
		gtk_toggle_button_set_active ((GtkToggleButton *)
			glade_xml_get_widget (xml, "local_specify_port_radio"), TRUE);
	}

	update_local_location_sensitivities (xml);
	g_signal_connect_swapped (glade_xml_get_widget (xml, "local_use_detected_radio"),
		"toggled",
		G_CALLBACK (update_local_location_sensitivities), xml);
	g_signal_connect_swapped (glade_xml_get_widget (xml, "local_specify_port_radio"),
		"toggled",
		G_CALLBACK (update_local_location_sensitivities), xml);

	druid_watch_sensitivity_widget (xml, "local_use_detected_radio");
	druid_watch_sensitivity_widget (xml, "local_specify_port_radio");
}

static void
setup_network_connections (GladeXML *xml)
{
	GtkWidget *w = glade_xml_get_widget (xml, "connection_types");
	gtk_combo_box_set_active (GTK_COMBO_BOX (w), CONNECTION_TYPE_IPP);

	init_smb_combos (xml);

	druid_watch_sensitivity_widget (xml, "connection_types");
	druid_watch_sensitivity_widget (xml, "lpd_host_entry");
	druid_watch_sensitivity_widget (xml, "smb_username_entry");
	druid_watch_sensitivity_widget (xml, "smb_password_entry");
	druid_watch_sensitivity_widget (xml, "smb_printer_entry");
	druid_watch_sensitivity_widget (xml, "smb_host_entry");
	druid_watch_sensitivity_widget (xml, "ipp_uri_entry");
	druid_watch_sensitivity_widget (xml, "hp_host_entry");
	druid_watch_sensitivity_widget (xml, "hp_port_entry");
	g_signal_connect (glade_xml_get_widget (xml, "connection_types"),
		"changed",
		G_CALLBACK (cb_connection_type_changed), xml);

	w = glade_xml_get_widget (xml, "ipp_uri_entry");
	gtk_tooltips_set_tip (gtk_tooltips_new (), w,
			      _("For example :\n"
				"\tfile:/path/to/filename.prn\n"
				"\thttp://hostname:631/ipp/\n"
				"\thttp://hostname:631/ipp/port1\n"
				"\tipp://hostname/ipp/\n"
				"\tipp://hostname/ipp/port1\n"
				"\tlpd://hostname/queue\n"
				"\tsocket://hostname\n"
				"\tsocket://hostname:9100"), NULL);
}

static void
connection_page_setup (GladeXML *xml)
{
	setup_network_connections (xml);
	setup_local_connections (xml);
}

static void
connection_page_sensitivity (GladeXML *xml, gboolean *back, gboolean *next)
{
	char *uri = get_connection_uri (xml);
	*next = (uri != NULL);
	g_free (uri);

	*back = FALSE;
}

static void
cb_ipp_model (guint id, const char *path, ipp_t *response, GError **error,
	      gpointer xml)
{
	ipp_attribute_t *attr;

	if (!error && response)
		for (attr = response->attrs; attr != NULL; attr = attr->next)
			if (attr->name != NULL &&
			    !g_ascii_strcasecmp (attr->name, "printer-make-and-model")) {
				g_warning ("Found a %s", attr->values[0].string.text);
				gcups_driver_selector_set_nickname (
					driver_selector (xml), attr->values[0].string.text);
			}

	ippDelete (response);
	g_clear_error (error);
}

static GtkWidget *
connection_page_next (GladeXML *xml)
{
	char *uri = get_connection_uri (xml);

	if (uri != NULL) {
		char *nickname = NULL;

		switch (get_connection_type (xml)) {
		case CONNECTION_TYPE_IPP : {
			static char const *attrs[] = {
				"printer-make-and-model"
			};
			ipp_t *request = gnome_cups_request_new (IPP_GET_PRINTER_ATTRIBUTES);
			gnome_cups_request_add_requested_attributes (request, 
				IPP_TAG_OPERATION, G_N_ELEMENTS (attrs), (char**)attrs);
			gnome_cups_request_execute_async (request, uri, "/",
				cb_ipp_model, g_object_ref (xml), g_object_unref);
#if 0
			snprintf(uri, sizeof(uri), "ipp://%s/printers/%s", printer);
			ippAddString(request, IPP_TAG_OPERATION, IPP_TAG_URI,
				     "printer-uri", NULL, uri);
			ippAddStrings(request, IPP_TAG_OPERATION, IPP_TAG_NAME,
				      "requested-attributes",
				      G_N_ELEMENTS (attrs), NULL, attrs);
#endif

			break;
		}

		case CONNECTION_TYPE_LPD : {
			int snmp_err = 0;
			char *host = entry_get_text_stripped (xml, "lpd_host_entry");
			if (host != NULL) {
				char const *detected = get_snmp_printers (host, &snmp_err);
				if (NULL != detected && 0 == snmp_err)
					nickname = parse_network_detect (detected);
				g_free (host);
			}
			break;
		}

		case CONNECTION_TYPE_LOCAL : {
			LocalPrinter const *lp = get_current_local (xml);
			if (lp != NULL)
				nickname = g_strdup (lp->vendor_and_model);
			break;
		}

		default :
			break;
		}

		/* if we can not autorecognize the printer we may be able to
		 * use an existing instance of it to look up a driver */
		if (nickname == NULL) {
#if 0
			ipp_t *request = gnome_cups_request_new (IPP_GET_PRINTER_ATTRIBUTES);
			gnome_cups_request_add_requested_attributes (request, 
				IPP_TAG_OPERATION, G_N_ELEMENTS (attrs), (char**)attrs);
			gnome_cups_request_execute_async (request, uri, "/",
				cb_ipp_model, g_object_ref (xml), g_object_unref);
#endif
		}

		gcups_driver_selector_set_nickname (
			driver_selector (xml), nickname);
		g_free (nickname);

		set_selected_uri (xml, uri);
		g_free (uri);

		return glade_xml_get_widget (xml, "driver_page");
	} else
		set_selected_uri (xml, NULL);
	return NULL;
}

/****************************************************************************/

static void
driver_page_setup (GladeXML *xml)
{
	druid_watch_sensitivity_widget (xml, "driver_selector");
}

static void
driver_page_sensitivity (GladeXML *xml, gboolean *back, gboolean *next)
{
	GtkWidget *druid = glade_xml_get_widget (xml, "add_printer_druid");
	*back = TRUE;
	*next = NULL != gcups_driver_selector_get (driver_selector (xml));

	/* see driver_page_prepare for an explanation of the druid hack */
	gnome_druid_set_show_finish (GNOME_DRUID (druid), *next);
}

static GtkWidget *
driver_page_back (GladeXML *xml)
{
	return glade_xml_get_widget (xml, "connection_page");
}

static gboolean
add_cups_printer (GladeXML *xml, char const *device_uri, GCupsPPD  const *ppd, char const *printer_name)
{
	GError *err = NULL;
	char local_uri [HTTP_MAX_URI+1];
	ipp_t *request = gnome_cups_request_new (CUPS_ADD_PRINTER);
	g_snprintf (local_uri, sizeof local_uri - 1,
		    "ipp://localhost/printers/%s", printer_name);
	ippAddString (request, IPP_TAG_OPERATION, IPP_TAG_URI,
		      "printer-uri", NULL, local_uri);
	ippAddString (request,	IPP_TAG_PRINTER, IPP_TAG_NAME,
		      "printer-name", NULL, gnome_cups_strdup (printer_name));
	ippAddString (request,	IPP_TAG_PRINTER, IPP_TAG_TEXT,
		      "printer-location", NULL, gnome_cups_strdup (device_uri));
	ippAddString (request, IPP_TAG_PRINTER, IPP_TAG_NAME,
		      "ppd-name", NULL, gnome_cups_strdup (ppd->filename));
	ippAddString (request, IPP_TAG_PRINTER, IPP_TAG_URI,
		      "device-uri", NULL, gnome_cups_strdup (device_uri));
	ippAddBoolean (request, IPP_TAG_PRINTER, "printer-is-accepting-jobs", 1);
	ippAddInteger(request, IPP_TAG_PRINTER, IPP_TAG_ENUM, "printer-state",
		      IPP_PRINTER_IDLE);
	ippDelete (gnome_cups_request_execute (request, NULL, "/admin/", &err));

	if (err != NULL) {
		gnome_cups_error_dialog ((GtkWindow *)glade_xml_get_widget (xml, "add_printer_window"),
					 _("Couldn't add printer"), err);
		g_error_free (err);
		return FALSE;
	}

	return TRUE;
}

static void
driver_page_prepare (GladeXML *xml)
{
#if 0
	/* Blah.  GnomeDruid is stupid.  finish and next are distinct buttons internally
	 * and there is no way to desensitize finish.  We'll need to cheat */
	GtkWidget *druid = glade_xml_get_widget (xml, "add_printer_druid");
	gnome_druid_set_show_finish (GNOME_DRUID (druid), TRUE);
#endif
}

static GtkWidget *
driver_page_next (GladeXML *xml)
{
	GList *existing;
	char  *name, *uri, *ptr;
	unsigned i = 0;
	GCupsPPD const  *ppd  = gcups_driver_selector_get (driver_selector (xml));

	if (ppd == NULL)
		return NULL;

	uri  = get_selected_uri (xml);

	name = g_strdup (ppd->model);
	/* strip out the spaces */
	for (ptr = name ; *ptr ; ptr++)
		if (*ptr == ' ')
			*ptr = '-';

	existing = gnome_cups_get_printers ();
	while (NULL != g_list_find_custom (existing, name, (GCompareFunc)strcasecmp )) {
		g_free (name);
		name = g_strdup_printf ("%s-%d", ppd->model, ++i);
	}
	g_list_foreach (existing, (GFunc)g_free, NULL);
	g_list_free (existing);

	if (add_cups_printer (xml, uri, ppd, name)) {
		GError *err = NULL;
		char *argv[] = { "gnome-cups-manager", "-v", NULL, NULL };
		GtkWidget *window = glade_xml_get_widget (xml, "add_printer_window");
		gtk_widget_destroy (window);
		g_object_unref (xml);

		if (app_path != NULL)
			argv[0] = g_build_filename (app_path, "gnome-cups-manager", NULL);

		argv[2] = name;
		exit (0);
		g_spawn_async (NULL, argv, NULL, G_SPAWN_SEARCH_PATH,
			       NULL, NULL, NULL, &err);

		if (err != NULL) {
			GtkWidget *dialog = gtk_message_dialog_new_with_markup (NULL, 
				GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_MODAL, 
				GTK_MESSAGE_ERROR,
				GTK_BUTTONS_OK,
				_("Problems launching gnome-cups-manager for the new printer\n\t<b>%s</b>\n%s"), 
				name, err->message);
			gtk_dialog_run (GTK_DIALOG (dialog));
			gtk_widget_destroy (dialog); 
			g_error_free (err);
		}
		gtk_main_quit ();
	}

	g_free (uri);
	g_free (name);

	return NULL;
}

/****************************************************************************/

static DruidPageDescription pages[] = {
	{
		"connection_page",
		connection_page_setup,
		NULL,
		connection_page_sensitivity,
		NULL,
		connection_page_next,
	},
	{
		"driver_page",
		driver_page_setup,
		driver_page_prepare,
		driver_page_sensitivity,
		driver_page_back,
		driver_page_next
	},
	{ NULL }
};

static int
delete_event_cb (GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
	gtk_main_quit ();
	return FALSE;
}

static void
cancel_cb (GtkWidget *widget, gpointer user_data)
{
	gtk_main_quit ();
}

#define SU_APP	"gnomesu"

int
main (int argc, char *argv[])
{
	GladeXML  *xml;
	GtkWidget *window;

	gnome_program_init ("gnome-cups-add",
			    VERSION,
			    LIBGNOMEUI_MODULE, argc, argv,
			    GNOME_PROGRAM_STANDARD_PROPERTIES,
			    GNOME_PARAM_HUMAN_READABLE_NAME, _("Add a Printer"),
			    NULL);
	glade_init ();
	gnome_cups_ui_init ();

	bindtextdomain (GETTEXT_PACKAGE, GNOMELOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);

	if (geteuid () != 0) {
		char **args = g_new (char *, argc+2);
		unsigned i;
		GError *err = NULL;

		args[0] = SU_APP;
		for (i = 0 ; i < argc ; i++)
			args[i+1] = argv[i];
		args[i+1] = NULL;
		g_spawn_async (NULL, args, NULL, G_SPAWN_SEARCH_PATH,
			       NULL, NULL, NULL, &err);
		if (err != NULL) {
			GtkWidget *dialog = gtk_message_dialog_new_with_markup (NULL, 
				GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_MODAL, 
				GTK_MESSAGE_ERROR,
				GTK_BUTTONS_OK,
				_("Problems launching %s as root via %s\n%s"),
				argv[0], SU_APP, err->message);
			gtk_dialog_run (GTK_DIALOG (dialog));
			gtk_widget_destroy (dialog); 
			g_error_free (err);
		}
		exit (0);
	}
	/* store the path for argv[0].  If gnome-cups-add is not in someones
	 * path gnome-cups-manager probably will not be there either */
	if (argv[0] != NULL) {
		app_path = g_path_get_dirname (argv[0]);
		if (NULL != app_path && 0 == strcmp (".", app_path)) {
			g_free (app_path);
			app_path = NULL;
		}
	}

	xml = glade_xml_new (GNOME_CUPS_MANAGER_DATADIR "/gnome-cups-add.glade",
			     "add_printer_window",
			     GETTEXT_PACKAGE);

	window = glade_xml_get_widget (xml, "add_printer_window");
	set_window_icon (window, "gnome-dev-printer-new");
	g_signal_connect (window,
		"delete_event",
		G_CALLBACK (delete_event_cb), NULL);
	g_signal_connect (glade_xml_get_widget (xml, "add_printer_druid"),
		"cancel",
		G_CALLBACK (cancel_cb), NULL);

	druid_pages_setup (xml, pages);
	gtk_widget_show (window);

	gtk_main ();

	return 0;
}
