/* gnome-ppp - The GNOME PPP Dialer
 * Copyright (C) 1997 Jay Painter
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <string.h>
#include <gnome.h>
#include "account-window.h"
#include "configure-window.h"
#include "dial-window.h"
#include "global.h"
#include "misc.h"


struct
{
  int speed;
  gchar *text;
} modem_speeds[] = {
  { 115200, "115200" },
  { 57600, "57600" },
  { 38400, "38400" },
  { 19200, "19200" },
  { 9600, "9600" },
  { 2400, "2400" },
  { 0, NULL }
};


struct
{
  gchar *device;
  gchar *text;
} modem_devices[] = {
  { "/dev/modem", "/dev/modem" },
  { "/dev/ttyS0", "/dev/ttyS0 (COM1)" },
  { "/dev/ttyS1", "/dev/ttyS1 (COM2)" },
  { "/dev/ttyS2", "/dev/ttyS2 (COM3)" },
  { "/dev/ttyS3", "/dev/ttyS3 (COM4)" },
  { NULL, NULL }
};


typedef struct _AccountWindow AccountWindow;
struct _AccountWindow
{
  Account   *account;
  gboolean   free_account;

  GtkWidget *window;
  GtkWidget *name;
  GtkWidget *phone_clist;
  GtkWidget *phone_entry;
  GtkWidget *connection_speed;
  GtkWidget *modem_device;
  GtkWidget *user;
  GtkWidget *passwd;
  GtkWidget *remotename;
  GtkWidget *ip;
  GtkWidget *ip_dynamic_button;
  GtkWidget *ip_static_button;
  GtkWidget *mask;
  GtkWidget *mask_dynamic_button;
  GtkWidget *mask_static_button;
  GtkWidget *remote;
  GtkWidget *remote_dynamic_button;
  GtkWidget *remote_static_button;
  GtkWidget *dns_server_entry;
  GtkWidget *dns_server_clist;
  GtkWidget *dns_search_entry;
  GtkWidget *dns_search_clist;
  GtkWidget *script_type_option_menu;
  GtkWidget *script_entry;
  GtkWidget *script_clist;
};


/* list of all open accounts used to prevent opening 
 * the same account twice */
static GList *open_account_list = NULL;


/* text descriptions of script types */
static char *script_type_string[] =
{
  "Expect",
  "Send",
  "Send User Name",
  "Send User Password",
  NULL
};


/* data refreshing for the account window */
static void refresh_account_window (AccountWindow * aw);


/* notebook pages */
GtkWidget *create_dial_page (AccountWindow * aw);
GtkWidget *create_authentication_page (AccountWindow * aw);
GtkWidget *create_ip_page (AccountWindow * aw);
GtkWidget *create_dns_page (AccountWindow * aw);
GtkWidget *create_script_page (AccountWindow * aw);


/* account window callbacks */
static void ok_account_window (GtkWidget * widget);
static void cancel_account_window (GtkWidget * widget);


/* dial page callbacks */
static void modem_device_cb(GtkWidget * widget);
static void connection_speed_cb(GtkWidget * widget);
static void phone_number_select_cb (GtkWidget * widget,  gint row);
static void phone_number_unselect_cb (GtkWidget * widget, gint row);
static void add_phone_number_cb (GtkWidget * widget);
static void remove_phone_number_cb (GtkWidget * widget);


/* ip address callbacks */
static void ip_address_dynamic_cb (GtkWidget * widget);
static void ip_address_static_cb (GtkWidget * widget);
static void netmask_dynamic_cb (GtkWidget * widget);
static void netmask_static_cb (GtkWidget * widget);
static void remote_dynamic_cb (GtkWidget * widget);
static void remote_static_cb (GtkWidget * widget);


/* dns page callbacks */
static void dns_server_select_cb (GtkWidget * widget,  gint row);
static void dns_server_unselect_cb (GtkWidget * widget, gint row);
static void add_dns_server_cb (GtkWidget * widget);
static void remove_dns_server_cb (GtkWidget * widget);
static void dns_search_select_cb (GtkWidget * widget,  gint row);
static void dns_search_unselect_cb (GtkWidget * widget, gint row);
static void add_dns_search_cb (GtkWidget * widget);
static void remove_dns_search_cb (GtkWidget * widget);


/* script page callbacks */
static void script_type_cb (GtkWidget * widget);
static void add_script_cb (GtkWidget * widget);
static void insert_script_cb (GtkWidget * widget); 
static void remove_script_cb (GtkWidget * widget);
static void script_select_cb (GtkWidget * widget,  gint row);
static void script_unselect_cb (GtkWidget * widget, gint row);


/* macros to set and get the class data structure */
#define ACCOUNT_WINDOW_DATA_STR "account_window_data"
#define GET_ACCOUNT_WINDOW(object)\
(gtk_object_get_data(GTK_OBJECT((object)), ACCOUNT_WINDOW_DATA_STR))
#define SET_ACCOUNT_WINDOW(object, aw)\
(gtk_object_set_data(GTK_OBJECT((object)), ACCOUNT_WINDOW_DATA_STR, (gpointer)(aw)))



void 
open_account_window (Account *account)
{
  AccountWindow *aw;
  GtkWidget *bbox;
  GtkWidget *vbox;
  GtkWidget *hbox;
  GtkWidget *notebook;
  GtkWidget *button;
  GtkWidget *label;

  /* bail if the account is already open, if it isn't
   * then add the account to the open_account_list
   */
  if (account)
    {
      if (g_list_find(open_account_list, account))
	{
	  return;
	}
    }

  open_account_list = g_list_append(open_account_list, account);

  aw = g_malloc(sizeof(AccountWindow));
  if (!account)
    {
      aw->account = account_new();
      aw->free_account = TRUE;
    }
  else
    {
      aw->account = account;
      aw->free_account = FALSE;
    }

  /* main window */ 
  aw->window = gtk_dialog_new(); 
  SET_ACCOUNT_WINDOW(aw->window, aw);
  gtk_widget_set_usize(GTK_WIDGET(aw->window), ACCOUNT_WINDOW_WIDTH, ACCOUNT_WINDOW_HEIGHT);
  gtk_window_set_title(GTK_WINDOW(aw->window), "PPP Account");
  gtk_container_border_width(GTK_CONTAINER(aw->window), 0);

  gtk_signal_connect(
      GTK_OBJECT(aw->window),
      "destroy",
      (GtkSignalFunc) cancel_account_window,
      aw);
  gtk_signal_connect(
      GTK_OBJECT(aw->window),
      "delete_event",
      (GtkSignalFunc) gtk_false,
      NULL);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(aw->window)->vbox), hbox, TRUE, TRUE, 5);
  gtk_widget_show(hbox);

  /* notbook */
  notebook = gtk_notebook_new();
  gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
  gtk_box_pack_start(GTK_BOX(hbox), notebook, TRUE, TRUE, 5);
  gtk_widget_show(notebook);

  /* button box (bottom dialog) */
  bbox = gtk_hbutton_box_new();
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(aw->window)->action_area), bbox, TRUE, TRUE, 0);
  gtk_button_box_set_layout(GTK_BUTTON_BOX(bbox), GTK_BUTTONBOX_END);
  gtk_button_box_set_child_size(GTK_BUTTON_BOX(bbox), GPPP_BUTTON_WIDTH, GPPP_BUTTON_HEIGHT);
  gtk_widget_show(bbox);

  button = gnome_stock_button(GNOME_STOCK_BUTTON_OK); 
  SET_ACCOUNT_WINDOW(button, aw);
  gtk_container_add(GTK_CONTAINER(bbox), button);
  gtk_signal_connect_object(
      GTK_OBJECT(button),
      "clicked",
      (GtkSignalFunc) ok_account_window,
      GTK_OBJECT(aw->window)
      );
  gtk_widget_show(button);

  button = gnome_stock_button(GNOME_STOCK_BUTTON_CANCEL); 
  SET_ACCOUNT_WINDOW(button, aw);
  gtk_container_add(GTK_CONTAINER(bbox), button);
  gtk_signal_connect_object(
      GTK_OBJECT(button),
      "clicked",
      (GtkSignalFunc) gtk_widget_destroy, 
      GTK_OBJECT(aw->window)
      );
  gtk_widget_show(button);

  /* dial notbook page */
  label = gtk_label_new("Dial");
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), create_dial_page(aw), label);

  /* authentication notbook page */
  label = gtk_label_new("Authentication");
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), create_authentication_page(aw), label);

  /* ip notbook page */
  label = gtk_label_new("IP Address");
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), create_ip_page(aw), label);

  /* dns notbook page */
  label = gtk_label_new("DNS");
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), create_dns_page(aw), label);

  /* script notbook page */
  label = gtk_label_new("Script");
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), create_script_page(aw), label);

  /* set the account window's data */
  refresh_account_window(aw);
  gtk_widget_show(aw->window);
}


static void
refresh_account_window (AccountWindow * aw)
{
  int i;
  gchar *text[2];
  GList *list;
  GString *gstr;
  Account *account = aw->account;
  ScriptEntry *script_entry;

  g_assert(account != NULL);

  /* set information on dial window */
  gtk_entry_set_text(GTK_ENTRY(aw->name), account->name->str);

  /* modem device -- set default then override it */
  gtk_option_menu_set_history(GTK_OPTION_MENU(aw->modem_device), 0);
  gtk_object_set_user_data(GTK_OBJECT(aw->modem_device), (gpointer) modem_devices[0].text);
  for (i = 0; modem_devices[i].text != NULL; i++)
    {
      if (strcmp(modem_devices[i].device, account->modem_device->str) == 0)
	{
	  gtk_option_menu_set_history(GTK_OPTION_MENU(aw->modem_device), i);
	  gtk_object_set_user_data(GTK_OBJECT(aw->modem_device), (gpointer) modem_devices[i].text);
	  break;
	}
    }

  /* modem speed -- set default then override it */
  gtk_option_menu_set_history(GTK_OPTION_MENU(aw->connection_speed), 0);
  gtk_object_set_user_data(GTK_OBJECT(aw->connection_speed), (gpointer) modem_speeds[0].text);
  for (i = 0; modem_speeds[i].text != NULL; i++)
    {
      if (modem_speeds[i].speed == account->speed)
	{
	  gtk_option_menu_set_history(GTK_OPTION_MENU(aw->connection_speed), i);
	  gtk_object_set_user_data(GTK_OBJECT(aw->connection_speed), (gpointer) modem_speeds[i].text);
	  break;
	}
    }

  /* popluate the phone list */
  gtk_clist_clear(GTK_CLIST(aw->phone_clist));
  list = account->phone_list;
  while (list)
    {
      gstr = (GString *) list->data;
      list = list->next;
      gtk_clist_append(GTK_CLIST(aw->phone_clist), &gstr->str);
    }

  /* set information on authentication window */
  gtk_entry_set_text(GTK_ENTRY(aw->user), account->user->str);
  gtk_entry_set_text(GTK_ENTRY(aw->passwd), account->passwd->str);
  gtk_entry_set_text(GTK_ENTRY(aw->remotename), account->remotename->str);

  /* IP, Remote, Netmask */
  if (account->ip->len > 0)
    {
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(aw->ip_static_button), TRUE);
      gtk_widget_set_sensitive(GTK_WIDGET(aw->ip), TRUE);
      gtk_entry_set_text(GTK_ENTRY(aw->ip), account->ip->str);
    }
  else
    {
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(aw->ip_dynamic_button), TRUE);
      gtk_widget_set_sensitive(GTK_WIDGET(aw->ip), FALSE);
      gtk_entry_set_text(GTK_ENTRY(aw->ip), "");
    }

  if (account->mask->len > 0)
    {
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(aw->mask_static_button), TRUE);
      gtk_widget_set_sensitive(GTK_WIDGET(aw->mask), TRUE);
      gtk_entry_set_text(GTK_ENTRY(aw->mask), account->mask->str);
    }
  else
    {
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(aw->mask_dynamic_button), TRUE);
      gtk_widget_set_sensitive(GTK_WIDGET(aw->mask), FALSE);
      gtk_entry_set_text(GTK_ENTRY(aw->mask), "");
    }

  if (account->remote->len > 0)
    {
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(aw->remote_static_button), TRUE);
      gtk_widget_set_sensitive(GTK_WIDGET(aw->remote), TRUE);
      gtk_entry_set_text(GTK_ENTRY(aw->remote), account->remote->str);
    }
  else
    {
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(aw->remote_dynamic_button), TRUE);
      gtk_widget_set_sensitive(GTK_WIDGET(aw->remote), FALSE);
      gtk_entry_set_text(GTK_ENTRY(aw->remote), "");
    }

  /* dns/search domains */
  gtk_clist_clear(GTK_CLIST(aw->dns_server_clist));
  list = account->dns_list;
  while (list)
    {
      gstr = (GString *) list->data;
      list = list->next;
      gtk_clist_append(GTK_CLIST(aw->dns_server_clist), &gstr->str);
    }

  gtk_clist_clear(GTK_CLIST(aw->dns_search_clist));
  list = account->domain_list;
  while (list)
    {
      gstr = (GString *) list->data;
      list = list->next;
      gtk_clist_append(GTK_CLIST(aw->dns_search_clist), &gstr->str);
    }

  /* script */
  gtk_clist_clear(GTK_CLIST(aw->script_clist));
  list = account->script_list;
  while (list)
    {
      script_entry = (ScriptEntry *) list->data;
      list = list->next;

      text[0] = script_type_string[script_entry->type];
      text[1] = script_entry->text->str;
      gtk_clist_append(GTK_CLIST(aw->script_clist), text);
    }
}


/*
 * ok, cancel
 */
static void
ok_account_window (GtkWidget * widget)
{
  gint i;
  GList *list;
  gchar *text;
  AccountWindow *aw = GET_ACCOUNT_WINDOW(widget);
  Account *account = aw->account;

  /* save information from dial window */
  g_string_assign(account->name, gtk_entry_get_text(GTK_ENTRY(aw->name)));

  for (i = 0; modem_devices[i].text != NULL; i++)
    {
      gpointer data;

      data = gtk_object_get_user_data(GTK_OBJECT(aw->modem_device));
      if (modem_devices[i].text == (gchar *) data)
	{
	  g_string_assign(account->modem_device, modem_devices[i].device);
	  break;
	}
    }

  for (i = 0; modem_speeds[i].text != NULL; i++)
    {
      gpointer data;

      data = gtk_object_get_user_data(GTK_OBJECT(aw->connection_speed));
      if (modem_speeds[i].text == (gchar *) data)
	{
	  account->speed = modem_speeds[i].speed;
	  break;
	}
    }

  /* clear and re-poplulate the phone number list */
  account_phone_list_clear(account);
  for (i = 0; i < GTK_CLIST(aw->phone_clist)->rows; i++)
    {
      gtk_clist_get_text(GTK_CLIST(aw->phone_clist), i, 0, &text);
      account_phone_list_append(account, text);
    }

  /* save information from authentication window */
  g_string_assign(account->user, gtk_entry_get_text(GTK_ENTRY(aw->user)));
  g_string_assign(account->passwd, gtk_entry_get_text(GTK_ENTRY(aw->passwd)));
  g_string_assign(account->remotename, gtk_entry_get_text(GTK_ENTRY(aw->remotename)));

  /* ip, mask, remote */
  g_string_assign(account->ip, gtk_entry_get_text(GTK_ENTRY(aw->ip)));
  g_string_assign(account->mask, gtk_entry_get_text(GTK_ENTRY(aw->mask)));
  g_string_assign(account->remote, gtk_entry_get_text(GTK_ENTRY(aw->remote)));

  /* dns/search domains */
  account_dns_list_clear(account);
  for (i = 0; i < GTK_CLIST(aw->dns_server_clist)->rows; i++)
    {
      gtk_clist_get_text(GTK_CLIST(aw->dns_server_clist), i, 0, &text);
      account_dns_list_append(account, text);
    }

  account_domain_list_clear(account);
  for (i = 0; i < GTK_CLIST(aw->dns_search_clist)->rows; i++)
    {
      gtk_clist_get_text(GTK_CLIST(aw->dns_search_clist), i, 0, &text);
      account_domain_list_append(account, text);
    }

  /* save the script list */
  account_script_list_clear(account);
  for (i = 0; i < GTK_CLIST(aw->script_clist)->rows; i++)
    {
      ScriptEntryType script_type;
      gchar *type_string, *text_string;

      gtk_clist_get_text(GTK_CLIST(aw->script_clist), i, 0, &type_string);
      gtk_clist_get_text(GTK_CLIST(aw->script_clist), i, 1, &text_string);

      if (strcmp(script_type_string[0], type_string) == 0)
	{
	  script_type = SCRIPT_ENTRY_EXPECT;
	}
      else if (strcmp(script_type_string[1], type_string) == 0)
	{
	  script_type = SCRIPT_ENTRY_SEND;
	}
      else if (strcmp(script_type_string[2], type_string) == 0)
	{
	  script_type = SCRIPT_ENTRY_SEND_USER;
	}
      else if (strcmp(script_type_string[3], type_string) == 0)
	{
	  script_type = SCRIPT_ENTRY_SEND_PASSWORD;
	}
      else
	{
	  g_error("script type string %s unknown", type_string);
	  exit(1);
	}

      account_script_list_append(account, script_type, text_string);
    }

  /* update information in configure, dial window */
  refresh_configure_window();
  refresh_dial_window();

  /* free account window memory */
  aw->free_account = FALSE;
  gtk_widget_destroy(aw->window);
}


static void
cancel_account_window (GtkWidget * widget)
{
  AccountWindow *aw = GET_ACCOUNT_WINDOW(widget);

  if (g_list_find(open_account_list, aw->account))
    {
      open_account_list = g_list_remove (open_account_list, aw->account);
    }

  if (aw->free_account)
    {
      account_free(aw->account);
      aw->account = NULL;
    }

  g_free(aw);
}


/*
 * notebook pages
 */
GtkWidget *
create_dial_page (AccountWindow * aw)
{
  gint i;
  GtkWidget *vbox;
  GtkWidget *vbox2;
  GtkWidget *table;
  GtkWidget *table2;
  GtkWidget *label;
  GtkWidget *frame;
  GtkWidget *entry;
  GtkWidget *button;
  GtkWidget *menu;

  /* main return widget */
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_border_width(GTK_CONTAINER(vbox), 10);
  gtk_widget_show(vbox);

  table = gtk_table_new(5, 2, FALSE);
  gtk_box_pack_start(GTK_BOX(vbox), table, TRUE, TRUE, 0);
  gtk_widget_show(table);

  label = gtk_label_new("Account Name: ");
  gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 10, 10);
  gtk_widget_show(label);

  aw->name = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(table), aw->name, 1, 2, 0, 1, GTK_FILL|GTK_EXPAND, GTK_FILL, 0, 10);
  gtk_widget_show(aw->name);

  /* modem device */
  label = gtk_label_new("Modem Device:");
  gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 10, 10);
  gtk_widget_show(label);

  aw->modem_device = gtk_option_menu_new();
  menu = gtk_menu_new();
  for (i = 0; modem_devices[i].text != NULL; i++)
    {
      GtkWidget *menu_item;
      menu_item = append_menuitem_connect(
          GTK_MENU(menu), 
	  modem_devices[i].text, 
	  (GtkSignalFunc) modem_device_cb, 
	  NULL,
	  (gpointer) modem_devices[i].text
	  );
      SET_ACCOUNT_WINDOW(menu_item, aw);
    }
  gtk_option_menu_set_menu(GTK_OPTION_MENU(aw->modem_device), menu);
  gtk_widget_set_usize(aw->modem_device, 150, GPPP_BUTTON_HEIGHT);
  gtk_table_attach(GTK_TABLE(table), aw->modem_device, 1, 2, 1, 2, GTK_FILL|GTK_EXPAND, GTK_FILL, 0, 10);
  gtk_widget_show(aw->modem_device);

  /* modem speed */
  label = gtk_label_new("Connection Speed:");
  gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3, GTK_FILL, GTK_FILL, 10, 10);
  gtk_widget_show (label);

  aw->connection_speed = gtk_option_menu_new ();
  menu = gtk_menu_new();
  for (i = 0; modem_speeds[i].text != NULL; i++)
    {
      GtkWidget *menu_item;
      menu_item = append_menuitem_connect(
          GTK_MENU(menu), 
	  modem_speeds[i].text, 
	  (GtkSignalFunc) connection_speed_cb, 
	  NULL,
	  (gpointer) modem_speeds[i].text
	  );
      SET_ACCOUNT_WINDOW(menu_item, aw);
    }
  gtk_option_menu_set_menu(GTK_OPTION_MENU(aw->connection_speed), menu);
  gtk_widget_set_usize(aw->connection_speed, 150, GPPP_BUTTON_HEIGHT);
  gtk_table_attach(GTK_TABLE(table), aw->connection_speed, 1, 2, 2, 3, GTK_FILL|GTK_EXPAND, GTK_FILL, 0, 10);
  gtk_widget_show(aw->connection_speed);

  /* phone number list */
  frame = gtk_frame_new("Phone Numbers");
  gtk_table_attach(GTK_TABLE(table), frame, 0, 2, 3, 4, GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 10);
  gtk_widget_show(frame);

  table2 = gtk_table_new(3, 2, FALSE);
  gtk_container_border_width(GTK_CONTAINER(table2), 5);
  gtk_container_add(GTK_CONTAINER(frame), table2);
  gtk_widget_show(table2);

  label = gtk_label_new("#:");
  gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
  gtk_table_attach(GTK_TABLE(table2), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 10, 0);
  gtk_widget_show(label);

  /* phone number entry */
  aw->phone_entry = gtk_entry_new(); 
  SET_ACCOUNT_WINDOW(aw->phone_entry, aw);
  gtk_table_attach(GTK_TABLE(table2), aw->phone_entry, 1, 2, 0, 1, GTK_FILL|GTK_EXPAND, GTK_FILL, 0, 0);
  gtk_signal_connect(
      GTK_OBJECT(aw->phone_entry),
      "activate",
      (GtkSignalFunc) add_phone_number_cb,
      aw
      );
  gtk_widget_show(aw->phone_entry);

  /* add/remove buttons */
  button = gtk_button_new_with_label("Add"); 
  SET_ACCOUNT_WINDOW(button, aw);
  gtk_widget_set_usize(button, GPPP_BUTTON_WIDTH, GPPP_BUTTON_HEIGHT);
  gtk_table_attach(GTK_TABLE(table2), button, 0, 1, 1, 2, 0, 0, 5, 0);
  gtk_signal_connect(
      GTK_OBJECT(button),
      "clicked",
      (GtkSignalFunc) add_phone_number_cb,
      NULL
      );
  gtk_widget_show(button);

  button = gtk_button_new_with_label("Remove"); 
  SET_ACCOUNT_WINDOW(button, aw);
  gtk_widget_set_usize(button, GPPP_BUTTON_WIDTH, GPPP_BUTTON_HEIGHT);
  gtk_table_attach(GTK_TABLE(table2), button, 0, 1, 2, 3, 0, 0, 5, 0);
  gtk_signal_connect(
      GTK_OBJECT (button),
      "clicked",
      (GtkSignalFunc) remove_phone_number_cb,
      NULL
      );
  gtk_widget_show(button);

  /* phone number list */
  aw->phone_clist = gtk_clist_new(1); 
  SET_ACCOUNT_WINDOW(aw->phone_clist, aw);
  gtk_clist_set_selection_mode(GTK_CLIST(aw->phone_clist), GTK_SELECTION_SINGLE);
  gtk_clist_set_column_width(GTK_CLIST(aw->phone_clist), 0, 150);
  gtk_table_attach(GTK_TABLE(table2), aw->phone_clist, 1, 2, 1, 3, GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 5);
  gtk_signal_connect(
      GTK_OBJECT(aw->phone_clist),
      "select_row",
      (GtkSignalFunc) phone_number_select_cb,
      NULL
      );
  gtk_signal_connect(
      GTK_OBJECT(aw->phone_clist),
      "unselect_row",
      (GtkSignalFunc) phone_number_unselect_cb,
      NULL
      );
  gtk_widget_show(aw->phone_clist);

  return vbox;
}


GtkWidget *
create_authentication_page (AccountWindow * aw)
{
  gint i;
  GtkWidget *vbox;
  GtkWidget *table;
  GtkWidget *label;

  /* main return widget */
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_border_width(GTK_CONTAINER(vbox), 10);
  gtk_widget_show(vbox);

  table = gtk_table_new(2, 3, FALSE);
  gtk_box_pack_start(GTK_BOX(vbox), table, TRUE, TRUE, 0);
  gtk_widget_show(table);

  label = gtk_label_new("User ID:");
  gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 10, 10);
  gtk_widget_show(label);

  aw->user = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(table), aw->user, 1, 2, 0, 1, GTK_FILL|GTK_EXPAND, GTK_FILL, 0, 10);
  gtk_widget_show(aw->user);

  label = gtk_label_new("Password:");
  gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 10, 10);
  gtk_widget_show(label);

  aw->passwd = gtk_entry_new();
  gtk_entry_set_visibility(GTK_ENTRY(aw->passwd), FALSE);
  gtk_table_attach(GTK_TABLE(table), aw->passwd, 1, 2, 1, 2, GTK_FILL|GTK_EXPAND, GTK_FILL, 0, 10);
  gtk_widget_show(aw->passwd);

  label = gtk_label_new("Remote Name:");
  gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3, GTK_FILL, GTK_FILL, 10, 10);
  gtk_widget_show(label);

  aw->remotename = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(table), aw->remotename, 1, 2, 2, 3, GTK_FILL|GTK_EXPAND, GTK_FILL, 0, 10);
  gtk_widget_show(aw->remotename);

  return vbox;
}


GtkWidget *
create_ip_page(AccountWindow * aw)
{
  gint i;
  GtkWidget *vbox;
  GtkWidget *frame;
  GtkWidget *table;
  GtkWidget *label;
  GSList *radio_group;

  /* main return widget */
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_border_width(GTK_CONTAINER(vbox), 10);
  gtk_widget_show(vbox);

  /* local ip address */
  frame = gtk_frame_new("Local IP Address");
  gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 5);
  gtk_widget_show(frame);

  table = gtk_table_new(2, 2, FALSE);
  gtk_container_border_width(GTK_CONTAINER(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);
  gtk_widget_show(table);

  aw->ip_dynamic_button = gtk_radio_button_new_with_label(NULL, "Dynamic");
  SET_ACCOUNT_WINDOW(aw->ip_dynamic_button, aw);
  radio_group = gtk_radio_button_group(GTK_RADIO_BUTTON(aw->ip_dynamic_button));
  gtk_table_attach(GTK_TABLE(table), aw->ip_dynamic_button, 0, 1, 0, 1, GTK_FILL, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_signal_connect(
      GTK_OBJECT(aw->ip_dynamic_button),
      "clicked",
      (GtkSignalFunc) ip_address_dynamic_cb,
      NULL
      );
  gtk_widget_show(aw->ip_dynamic_button);

  aw->ip_static_button = gtk_radio_button_new_with_label(radio_group, "Static"); 
  SET_ACCOUNT_WINDOW(aw->ip_static_button, aw);
  gtk_table_attach(GTK_TABLE(table), aw->ip_static_button, 0, 1, 1, 2, GTK_FILL, GTK_FILL|GTK_EXPAND,  0, 0);
  gtk_signal_connect(
      GTK_OBJECT(aw->ip_static_button),
      "clicked",
      (GtkSignalFunc) ip_address_static_cb,
      NULL
      );
  gtk_widget_show(aw->ip_static_button);

  aw->ip = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(table), aw->ip , 1, 2, 1, 2, GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(aw->ip);

  /* netmask */
  frame = gtk_frame_new("Netmask");
  gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 5);
  gtk_widget_show(frame);

  table = gtk_table_new(2, 2, FALSE);
  gtk_container_border_width(GTK_CONTAINER(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);
  gtk_widget_show(table);

  aw->mask_dynamic_button = gtk_radio_button_new_with_label(NULL, "Dynamic"); 
  SET_ACCOUNT_WINDOW(aw->mask_dynamic_button, aw);
  radio_group = gtk_radio_button_group(GTK_RADIO_BUTTON(aw->mask_dynamic_button));
  gtk_table_attach(GTK_TABLE(table), aw->mask_dynamic_button, 0, 1, 0, 1, GTK_FILL, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_signal_connect(
      GTK_OBJECT(aw->mask_dynamic_button),
      "clicked",
      (GtkSignalFunc) netmask_dynamic_cb,
      NULL
      );
  gtk_widget_show(aw->mask_dynamic_button);

  aw->mask_static_button = gtk_radio_button_new_with_label(radio_group, "Static");
  SET_ACCOUNT_WINDOW(aw->mask_static_button, aw);
  gtk_table_attach(GTK_TABLE(table), aw->mask_static_button, 0, 1, 1, 2, GTK_FILL, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_signal_connect(
      GTK_OBJECT(aw->mask_static_button),
      "clicked",
      (GtkSignalFunc) netmask_static_cb,
      NULL
      );
  gtk_widget_show(aw->mask_static_button);

  aw->mask = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(table), aw->mask, 1, 2, 1, 2, GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(aw->mask);

  /* remote ip address */
  frame = gtk_frame_new("Remote IP Address");
  gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 5);
  gtk_widget_show(frame);

  table = gtk_table_new(2, 2, FALSE);
  gtk_container_border_width(GTK_CONTAINER(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);
  gtk_widget_show(table);

  aw->remote_dynamic_button = gtk_radio_button_new_with_label(NULL, "Dynamic");
  SET_ACCOUNT_WINDOW(aw->remote_dynamic_button, aw);
  radio_group = gtk_radio_button_group(GTK_RADIO_BUTTON(aw->remote_dynamic_button));
  gtk_table_attach(GTK_TABLE(table), aw->remote_dynamic_button, 0, 1, 0, 1, GTK_FILL, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_signal_connect(
      GTK_OBJECT(aw->remote_dynamic_button),
      "clicked",
      (GtkSignalFunc) remote_dynamic_cb,
      NULL
      );
  gtk_widget_show(aw->remote_dynamic_button);
  
  aw->remote_static_button = gtk_radio_button_new_with_label(radio_group, "Static");
  SET_ACCOUNT_WINDOW(aw->remote_static_button, aw);
  gtk_table_attach(GTK_TABLE(table), aw->remote_static_button, 0, 1, 1, 2, GTK_FILL, GTK_FILL | GTK_EXPAND, 0, 0);
  gtk_signal_connect(
      GTK_OBJECT(aw->remote_static_button),
      "clicked",
      (GtkSignalFunc) remote_static_cb,
      NULL
      );
  gtk_widget_show(aw->remote_static_button);

  aw->remote = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(table), aw->remote, 1, 2, 1, 2, GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(aw->remote);

  return vbox;
}


GtkWidget *
create_dns_page(AccountWindow *aw)
{
  gint i;
  GtkWidget *vbox;
  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *frame;
  GtkWidget *button;

  /* main return widget */
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_border_width(GTK_CONTAINER(vbox), 10);
  gtk_widget_show(vbox);

  /* dns server list */
  frame = gtk_frame_new("DNS Servers");
  gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);
  gtk_widget_show(frame);

  table = gtk_table_new(3, 2, FALSE);
  gtk_container_border_width(GTK_CONTAINER(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);
  gtk_widget_show(table);

  label = gtk_label_new("IP Address:");
  gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 10, 0);
  gtk_widget_show(label);

  /* dns server entry */
  aw->dns_server_entry = gtk_entry_new();
  SET_ACCOUNT_WINDOW(aw->dns_server_entry, aw);
  gtk_table_attach(GTK_TABLE(table), aw->dns_server_entry, 1, 2, 0, 1, GTK_FILL|GTK_EXPAND, GTK_FILL, 0, 0);
  gtk_signal_connect(
      GTK_OBJECT(aw->dns_server_entry),
      "activate",
      (GtkSignalFunc) add_dns_server_cb,
      aw
      );
  gtk_widget_show(aw->dns_server_entry);

  /* add/remove buttons */
  button = gtk_button_new_with_label("Add");
  SET_ACCOUNT_WINDOW(button, aw);
  gtk_widget_set_usize(button, GPPP_BUTTON_WIDTH, GPPP_BUTTON_HEIGHT);
  gtk_table_attach(GTK_TABLE(table), button, 0, 1, 1, 2, 0, 0, 5, 0);
  gtk_signal_connect(
      GTK_OBJECT(button),
      "clicked",
      (GtkSignalFunc) add_dns_server_cb,
      NULL
      );
  gtk_widget_show(button);

  button = gtk_button_new_with_label("Remove");
  SET_ACCOUNT_WINDOW(button, aw);
  gtk_widget_set_usize(button, GPPP_BUTTON_WIDTH, GPPP_BUTTON_HEIGHT);
  gtk_table_attach(GTK_TABLE(table), button, 0, 1, 2, 3, 0, 0, 5, 0);  
  gtk_signal_connect(
      GTK_OBJECT(button),
      "clicked",
      (GtkSignalFunc) remove_dns_server_cb,
      NULL
      );
  gtk_widget_show(button);

  /* dns server list */
  aw->dns_server_clist = gtk_clist_new(1);
  SET_ACCOUNT_WINDOW(aw->dns_server_clist, aw);
  gtk_clist_set_selection_mode(GTK_CLIST(aw->dns_server_clist), GTK_SELECTION_SINGLE);
  gtk_clist_set_column_width(GTK_CLIST(aw->dns_server_clist), 0, 150);
  gtk_table_attach(GTK_TABLE(table), aw->dns_server_clist, 1, 2, 1, 3, GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 5);
  gtk_signal_connect(
      GTK_OBJECT(aw->dns_server_clist),
      "select_row",
      (GtkSignalFunc) dns_server_select_cb,
      NULL
      );
  gtk_signal_connect(
      GTK_OBJECT(aw->dns_server_clist),
      "unselect_row",
      (GtkSignalFunc) dns_server_unselect_cb,
      NULL
      );
  gtk_widget_show(aw->dns_server_clist);

  /* search domains list */
  frame = gtk_frame_new("Search Domains");
  gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 5);
  gtk_widget_show(frame);

  table = gtk_table_new(3, 2, FALSE);
  gtk_container_border_width(GTK_CONTAINER(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);
  gtk_widget_show(table);

  label = gtk_label_new("Domain Name:");
  gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 10, 0);
  gtk_widget_show(label);

  /* dns search entry */
  aw->dns_search_entry = gtk_entry_new();
  SET_ACCOUNT_WINDOW(aw->dns_search_entry, aw);
  gtk_table_attach(GTK_TABLE(table), aw->dns_search_entry, 1, 2, 0, 1, GTK_FILL|GTK_EXPAND, GTK_FILL, 0, 0);
  gtk_signal_connect(
      GTK_OBJECT(aw->dns_search_entry),
      "activate",
      (GtkSignalFunc) add_dns_search_cb,
      aw
      );
  gtk_widget_show(aw->dns_search_entry);

  /* add/remove buttons */
  button = gtk_button_new_with_label("Add");
  SET_ACCOUNT_WINDOW(button, aw);
  gtk_widget_set_usize(button, GPPP_BUTTON_WIDTH, GPPP_BUTTON_HEIGHT);
  gtk_table_attach(GTK_TABLE(table), button, 0, 1, 1, 2, 0, 0, 5, 0);
  gtk_signal_connect(
      GTK_OBJECT(button),
      "clicked",
      (GtkSignalFunc) add_dns_search_cb,
      NULL
      );
  gtk_widget_show(button);

  button = gtk_button_new_with_label("Remove");
  SET_ACCOUNT_WINDOW(button, aw);
  gtk_widget_set_usize(button, GPPP_BUTTON_WIDTH, GPPP_BUTTON_HEIGHT);
  gtk_table_attach(GTK_TABLE(table), button, 0, 1, 2, 3, 0, 0, 5, 0);
  gtk_signal_connect(
      GTK_OBJECT(button),
      "clicked",
      (GtkSignalFunc) remove_dns_search_cb,
      NULL
      );
  gtk_widget_show(button);

  /* dns search list */
  aw->dns_search_clist = gtk_clist_new(1);
  SET_ACCOUNT_WINDOW(aw->dns_search_clist, aw);
  gtk_clist_set_selection_mode(GTK_CLIST(aw->dns_search_clist),GTK_SELECTION_SINGLE);
  gtk_clist_set_column_width(GTK_CLIST(aw->dns_search_clist), 0, 150);
  gtk_table_attach(GTK_TABLE(table), aw->dns_search_clist, 1, 2, 1, 3,GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 5);
  gtk_signal_connect(
      GTK_OBJECT(aw->dns_search_clist),
      "select_row",
      (GtkSignalFunc) dns_search_select_cb,
      NULL
      );
  gtk_signal_connect(
      GTK_OBJECT(aw->dns_search_clist),
      "unselect_row",
      (GtkSignalFunc) dns_search_unselect_cb,
      NULL
      );
  gtk_widget_show(aw->dns_search_clist);

  return vbox;
}


GtkWidget *
create_script_page(AccountWindow * aw)
{
  gint i;
  GtkWidget *vbox;
  GtkWidget *vbox2;
  GtkWidget *hbox;
  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *frame;
  GtkWidget *button;
  GtkWidget *menu;

  static char *titles[] =
  {
    "Action",
    "Text"
  };

  /* main return widget */
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_border_width(GTK_CONTAINER(vbox), 10);
  gtk_widget_show(vbox);

  /* script list */
  frame = gtk_frame_new("Login Script");
  gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);
  gtk_widget_show(frame);

  vbox2 = gtk_vbox_new(FALSE, 0);
  gtk_container_border_width(GTK_CONTAINER(vbox2), 10);
  gtk_box_set_spacing(GTK_BOX(vbox2), 5);
  gtk_container_add(GTK_CONTAINER(frame), vbox2);
  gtk_widget_show(vbox2);

  /* script type option menu */
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_set_spacing(GTK_BOX(hbox), 5);
  gtk_box_pack_start(GTK_BOX(vbox2), hbox, FALSE, TRUE, 0);
  gtk_widget_show(hbox);

  aw->script_type_option_menu = gtk_option_menu_new();
  SET_ACCOUNT_WINDOW(aw->script_type_option_menu, aw);
  menu = gtk_menu_new();
  for (i = 0; script_type_string[i]; i++)
    {
      GtkWidget *menu_item;
      menu_item = append_menuitem_connect(
          GTK_MENU(menu),
	  script_type_string[i],
	  (GtkSignalFunc) script_type_cb,
	  NULL,
	  (gpointer) script_type_string[i]
	  );
      SET_ACCOUNT_WINDOW(menu_item, aw);
    }
  /* we've got to set the inital user_data for the option menu */
  gtk_object_set_user_data(GTK_OBJECT(aw->script_type_option_menu), (gpointer) script_type_string[0]);
  gtk_option_menu_set_menu(GTK_OPTION_MENU(aw->script_type_option_menu), menu);
  gtk_widget_set_usize(aw->script_type_option_menu, 0, GPPP_BUTTON_HEIGHT);
  gtk_box_pack_start(GTK_BOX(hbox), aw->script_type_option_menu, FALSE, TRUE, 0);
  gtk_widget_show(aw->script_type_option_menu);

  /* script text entry */
  aw->script_entry = gtk_entry_new();
  SET_ACCOUNT_WINDOW(aw->script_entry, aw);
  gtk_box_pack_start(GTK_BOX(hbox), aw->script_entry, TRUE, TRUE, 0);
  gtk_signal_connect(
      GTK_OBJECT(aw->script_entry),
      "activate",
      (GtkSignalFunc) add_script_cb,
      aw
      );
  gtk_widget_show(aw->script_entry);

  /* add/insert/remove buttons */
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox2), hbox, FALSE, TRUE, 0);
  gtk_box_set_spacing(GTK_BOX(hbox), 5);
  gtk_widget_show(hbox);

  button = gtk_button_new_with_label("Add");
  SET_ACCOUNT_WINDOW(button, aw);
  gtk_widget_set_usize(button, GPPP_BUTTON_WIDTH, GPPP_BUTTON_HEIGHT);
  gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
  gtk_signal_connect(
      GTK_OBJECT(button),
      "clicked",
      (GtkSignalFunc) add_script_cb,
      NULL
      );
  gtk_widget_show(button);

  button = gtk_button_new_with_label("Insert");
  SET_ACCOUNT_WINDOW(button, aw);
  gtk_widget_set_usize(button, GPPP_BUTTON_WIDTH, GPPP_BUTTON_HEIGHT);
  gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
  gtk_signal_connect(
      GTK_OBJECT(button),
      "clicked",
      (GtkSignalFunc) insert_script_cb,
      NULL
      );
  gtk_widget_show(button);

  button = gtk_button_new_with_label("Remove");
  SET_ACCOUNT_WINDOW(button, aw);
  gtk_widget_set_usize(button, GPPP_BUTTON_WIDTH, GPPP_BUTTON_HEIGHT);
  gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
  gtk_signal_connect(
      GTK_OBJECT(button),
      "clicked",
      (GtkSignalFunc) remove_script_cb,
      NULL
      );
  gtk_widget_show(button);

  /* script list */
  aw->script_clist = gtk_clist_new_with_titles(2, titles);
  SET_ACCOUNT_WINDOW(aw->script_clist, aw);
  gtk_clist_set_selection_mode(GTK_CLIST(aw->script_clist), GTK_SELECTION_SINGLE);
  gtk_clist_set_column_width(GTK_CLIST(aw->script_clist), 0, 150);
  gtk_box_pack_start(GTK_BOX(vbox2), aw->script_clist, TRUE, TRUE, 0);
  gtk_signal_connect(
      GTK_OBJECT(aw->script_clist),
      "select_row",
      (GtkSignalFunc) script_select_cb,
      NULL
      );
  gtk_signal_connect(
      GTK_OBJECT(aw->script_clist),
      "unselect_row",
      (GtkSignalFunc) script_unselect_cb,
      NULL
      );
  gtk_widget_show(aw->script_clist);

  return vbox;
}


/*
 * dial page callbacks
 */
static void
modem_device_cb(GtkWidget * widget)
{
  gpointer data;
  AccountWindow *aw = GET_ACCOUNT_WINDOW(widget);

  data = gtk_object_get_user_data(GTK_OBJECT(widget));
  gtk_object_set_user_data(GTK_OBJECT(aw->modem_device), data);
}


static void
connection_speed_cb(GtkWidget * widget)
{
  gpointer data;
  AccountWindow *aw = GET_ACCOUNT_WINDOW(widget);
  
  data = gtk_object_get_user_data(GTK_OBJECT(widget));
  gtk_object_set_user_data(GTK_OBJECT(aw->connection_speed), data);
}


static void
phone_number_select_cb(GtkWidget * widget, gint row)
{
  gchar *text;
  AccountWindow *aw = GET_ACCOUNT_WINDOW(widget);

  g_assert(aw != NULL);

  gtk_clist_get_text(GTK_CLIST(aw->phone_clist), row, 0, &text);
  gtk_entry_set_text(GTK_ENTRY(aw->phone_entry), text);
}


static void
phone_number_unselect_cb(GtkWidget * widget, gint row)
{
  gchar *text[1];
  AccountWindow *aw = GET_ACCOUNT_WINDOW(widget);
  
  g_assert(aw != NULL);

  gtk_entry_set_text(GTK_ENTRY(aw->phone_entry), "");
}


static void 
add_phone_number_cb(GtkWidget * widget)
{
  gchar *text[1];
  gint row;
  AccountWindow *aw = GET_ACCOUNT_WINDOW(widget);

  g_assert(aw != NULL);

  /* check to see if we're editing a selection */
  if (GTK_CLIST(aw->phone_clist)->selection)
    {
      row = GPOINTER_TO_INT(GTK_CLIST(aw->phone_clist)->selection->data);
    }
  else
    {
      row = -1;
    }

  text[0] = gtk_entry_get_text(GTK_ENTRY(aw->phone_entry));
  if (strcmp(text[0], "") == 0)
    {
      return;
    }

  /* we edit the phone number in place if they user hits 
   * "enter" from the phone number entry, we add it to the list
   * if the user clicks the "add" button -- both widgets use 
   * this callback, so the choice is made here
   */
  if (row >= 0 && widget == aw->phone_entry)
    {
      gtk_clist_set_text(GTK_CLIST(aw->phone_clist), row, 0, text[0]);
    }
  else
    {
      gtk_clist_append(GTK_CLIST(aw->phone_clist), text);
    }

  if (row >= 0)
    {
      gtk_clist_unselect_row(GTK_CLIST(aw->phone_clist), row, -1);
    }

  gtk_entry_set_text(GTK_ENTRY(aw->phone_entry), "");
}


static void
remove_phone_number_cb(GtkWidget * widget)
{
  gint row;
  AccountWindow *aw = GET_ACCOUNT_WINDOW(widget);

  g_assert(aw != NULL);

  /* make sure there is somthing selected */
  if (!GTK_CLIST(aw->phone_clist)->selection)
    {
      return;
    }
  else
    {
      row = GPOINTER_TO_INT(GTK_CLIST(aw->phone_clist)->selection->data);
    }

  gtk_clist_remove(GTK_CLIST(aw->phone_clist), row);
}


/*
 * ip address page callbacks
 */
static void 
ip_address_dynamic_cb(GtkWidget * widget)
{
  AccountWindow *aw = GET_ACCOUNT_WINDOW(widget);

  g_assert(aw != NULL);

  gtk_widget_set_sensitive(GTK_WIDGET(aw->ip), FALSE);
  gtk_entry_set_text(GTK_ENTRY(aw->ip), "");
}


static void
ip_address_static_cb(GtkWidget * widget)
{
  AccountWindow *aw = GET_ACCOUNT_WINDOW(widget);

  g_assert(aw != NULL);

  gtk_widget_set_sensitive(GTK_WIDGET(aw->ip), TRUE);
  gtk_entry_set_text(GTK_ENTRY(aw->ip), "");
}


static void 
netmask_dynamic_cb(GtkWidget * widget)
{
  AccountWindow *aw = GET_ACCOUNT_WINDOW(widget);

  g_assert(aw != NULL);

  gtk_widget_set_sensitive(GTK_WIDGET(aw->mask), FALSE);
  gtk_entry_set_text(GTK_ENTRY(aw->mask), "");
}


static void
netmask_static_cb(GtkWidget * widget)
{
  AccountWindow *aw = GET_ACCOUNT_WINDOW(widget);

  g_assert(aw != NULL);

  gtk_widget_set_sensitive(GTK_WIDGET(aw->mask), TRUE);
  gtk_entry_set_text(GTK_ENTRY(aw->mask), "");
}


static void 
remote_dynamic_cb(GtkWidget * widget)
{
  AccountWindow *aw = GET_ACCOUNT_WINDOW(widget);

  g_assert(aw != NULL);

  gtk_widget_set_sensitive(GTK_WIDGET(aw->remote), FALSE);
  gtk_entry_set_text(GTK_ENTRY(aw->remote), "");
}


static void
remote_static_cb(GtkWidget * widget)
{
  AccountWindow *aw = GET_ACCOUNT_WINDOW(widget);

  g_assert(aw != NULL);

  gtk_widget_set_sensitive(GTK_WIDGET(aw->remote), TRUE);
  gtk_entry_set_text(GTK_ENTRY(aw->remote), "");
}


/* 
 * dns page callbacks 
 */
static void 
dns_server_select_cb(GtkWidget * widget,  gint row)
{
  AccountWindow *aw = GET_ACCOUNT_WINDOW(widget);
  gchar *text;

  g_assert(aw != NULL);

  gtk_clist_get_text(GTK_CLIST(aw->dns_server_clist), row, 0, &text);
  gtk_entry_set_text(GTK_ENTRY(aw->dns_server_entry), text);
}


static void
dns_server_unselect_cb(GtkWidget * widget, gint row)
{
  AccountWindow *aw = GET_ACCOUNT_WINDOW(widget);
  gchar *text[1];
  gtk_entry_set_text(GTK_ENTRY(aw->dns_server_entry), "");
}


static void 
add_dns_server_cb(GtkWidget * widget)
{
  AccountWindow *aw = GET_ACCOUNT_WINDOW(widget);
  gchar *text[1];
  gint row;

  g_assert(aw != NULL);

  /* check to see if we're editing a selection */
  if (GTK_CLIST(aw->dns_server_clist)->selection)
    {
      row = GPOINTER_TO_INT(GTK_CLIST(aw->dns_server_clist)->selection->data);
    }
  else
    {
      row = -1;
    }

  text[0] = gtk_entry_get_text(GTK_ENTRY(aw->dns_server_entry));
  if (strcmp(text[0], "") == 0)
    {
      return;
    }

  if (row >= 0 && widget == aw->dns_server_entry)
    {
      gtk_clist_set_text(GTK_CLIST(aw->dns_server_clist), row, 0, text[0]);
    }
  else
    {
      gtk_clist_append(GTK_CLIST(aw->dns_server_clist), text);
    }

  if (row >= 0)
    {
      gtk_clist_unselect_row(GTK_CLIST(aw->dns_server_clist), row, -1);
    }

  gtk_entry_set_text(GTK_ENTRY(aw->dns_server_entry), "");
}


static void
remove_dns_server_cb(GtkWidget * widget)
{
  gint row;
  AccountWindow *aw = GET_ACCOUNT_WINDOW(widget);

  g_assert(aw != NULL);

  /* make sure there is somthing selected */
  if (!GTK_CLIST(aw->dns_server_clist)->selection)
    {
      return;
    }
  else
    {
      row = GPOINTER_TO_INT(GTK_CLIST(aw->dns_server_clist)->selection->data);
    }

  gtk_clist_remove(GTK_CLIST(aw->dns_server_clist), row);
}


static void 
dns_search_select_cb(GtkWidget * widget,  gint row)
{
  gchar *text;
  AccountWindow *aw = GET_ACCOUNT_WINDOW(widget);

  g_assert(aw != NULL);

  gtk_clist_get_text(GTK_CLIST(aw->dns_search_clist), row, 0, &text);
  gtk_entry_set_text(GTK_ENTRY(aw->dns_search_entry), text);
}


static void 
dns_search_unselect_cb(GtkWidget * widget, gint row)
{
  gchar *text[1];
  AccountWindow *aw = GET_ACCOUNT_WINDOW(widget);

  g_assert(aw != NULL);

  gtk_entry_set_text(GTK_ENTRY(aw->dns_search_entry), "");
}


static void
add_dns_search_cb(GtkWidget * widget)
{
  gchar *text[1];
  gint row;
  AccountWindow *aw = GET_ACCOUNT_WINDOW(widget);

  g_assert(aw != NULL);

  /* check to see if we're editing a selection */
  if (GTK_CLIST(aw->dns_search_clist)->selection)
    {
      row = GPOINTER_TO_INT(GTK_CLIST(aw->dns_search_clist)->selection->data);
    }
  else
    {
      row = -1;
    }

  text[0] = gtk_entry_get_text(GTK_ENTRY(aw->dns_search_entry));
  if (strcmp(text[0], "") == 0)
    {
      return;
    }

  if (row >= 0 && widget == aw->dns_search_entry)
    {
      gtk_clist_set_text(GTK_CLIST(aw->dns_search_clist), row, 0, text[0]);
    }
  else
    {
      gtk_clist_append(GTK_CLIST(aw->dns_search_clist), text);
    }

  if (row >= 0)
    {
      gtk_clist_unselect_row(GTK_CLIST(aw->dns_search_clist), row, -1);
    }

  gtk_entry_set_text(GTK_ENTRY(aw->dns_search_entry), "");
}


static void
remove_dns_search_cb(GtkWidget * widget)
{
  gint row;
  AccountWindow *aw = GET_ACCOUNT_WINDOW(widget);

  g_assert(aw != NULL);

  /* make sure there is somthing selected */
  if (!GTK_CLIST(aw->dns_search_clist)->selection)
    {
      return;
    }
  else
    {
      row = GPOINTER_TO_INT(GTK_CLIST(aw->dns_search_clist)->selection->data);
    }

  gtk_clist_remove(GTK_CLIST(aw->dns_search_clist), row);
}


/*
 * script page callbacks
 */
static void
script_type_cb (GtkWidget * widget)
{
  AccountWindow *aw = GET_ACCOUNT_WINDOW(widget);
  gchar *type_string;

  g_assert(aw != NULL);

  type_string = (gchar *) gtk_object_get_user_data(GTK_OBJECT(widget));
  if (strcmp(type_string, script_type_string[0]) == 0 ||
      strcmp(type_string, script_type_string[1]) == 0)
    {
      gtk_widget_set_sensitive(GTK_WIDGET(aw->script_entry), TRUE);
      gtk_entry_set_text(GTK_ENTRY(aw->script_entry), "");
    }
  else if (strcmp(type_string, script_type_string[2]) == 0)
    {
      gtk_widget_set_sensitive(GTK_WIDGET(aw->script_entry), FALSE);
      gtk_entry_set_text(GTK_ENTRY(aw->script_entry), gtk_entry_get_text(GTK_ENTRY(aw->user)));
    }
  else if (strcmp(type_string, script_type_string[3]) == 0)
    {
      gtk_widget_set_sensitive(GTK_WIDGET(aw->script_entry), FALSE);
      gtk_entry_set_text(GTK_ENTRY(aw->script_entry), "**PASSWORD**");
    }

  /* set the user data of the option menu to the type index so it can
   * be recovered in the add_script_cb
   */
  gtk_object_set_user_data(GTK_OBJECT(aw->script_type_option_menu), (gpointer) type_string);
}


static void 
add_script_cb (GtkWidget * widget)
{
  AccountWindow *aw = GET_ACCOUNT_WINDOW(widget);
  gchar *type_string, *text_string;
  gchar *text[2];

  g_assert(aw != NULL);

  type_string = (gchar *) gtk_object_get_user_data(GTK_OBJECT(aw->script_type_option_menu));
  text_string = gtk_entry_get_text(GTK_ENTRY(aw->script_entry));

  text[0] = type_string;
  text[1] = text_string;
  gtk_clist_append(GTK_CLIST(aw->script_clist), text);
  
  if (strcmp(type_string, script_type_string[0]) == 0 ||
      strcmp(type_string, script_type_string[1]) == 0)
    {
      gtk_widget_set_sensitive(GTK_WIDGET(aw->script_entry), TRUE);
      gtk_entry_set_text(GTK_ENTRY(aw->script_entry), "");
    }
  else if (strcmp(type_string, script_type_string[2]) == 0)
    {
      gtk_widget_set_sensitive(GTK_WIDGET(aw->script_entry), FALSE);
      gtk_entry_set_text(GTK_ENTRY(aw->script_entry), gtk_entry_get_text(GTK_ENTRY(aw->user)));
    }
  else if (strcmp(type_string, script_type_string[3]) == 0)
    {
      gtk_widget_set_sensitive(GTK_WIDGET(aw->script_entry), FALSE);
      gtk_entry_set_text(GTK_ENTRY(aw->script_entry), "**PASSWORD**");
    }
}


static void
insert_script_cb (GtkWidget * widget)
{
  AccountWindow *aw = GET_ACCOUNT_WINDOW(widget);
  gint row;
  gchar *type_string, *text_string;
  gchar *text[2];

  g_assert(aw != NULL);

  /* make sure there is a row selected */
  if (!GTK_CLIST(aw->script_clist)->selection)
    {
      return;
    }

  row = GPOINTER_TO_INT(GTK_CLIST(aw->script_clist)->selection->data);
  type_string = (gchar *) gtk_object_get_user_data(GTK_OBJECT(aw->script_type_option_menu));
  text_string = gtk_entry_get_text(GTK_ENTRY(aw->script_entry));

  text[0] = type_string;
  text[1] = text_string;
  gtk_clist_insert(GTK_CLIST(aw->script_clist), row, text);
  
  if (strcmp(type_string, script_type_string[0]) == 0 ||
      strcmp(type_string, script_type_string[1]) == 0)
    {
      gtk_widget_set_sensitive(GTK_WIDGET(aw->script_entry), TRUE);
      gtk_entry_set_text(GTK_ENTRY(aw->script_entry), "");
    }
  else if (strcmp(type_string, script_type_string[2]) == 0)
    {
      gtk_widget_set_sensitive(GTK_WIDGET(aw->script_entry), FALSE);
      gtk_entry_set_text(GTK_ENTRY(aw->script_entry), gtk_entry_get_text(GTK_ENTRY(aw->user)));
    }
  else if (strcmp(type_string, script_type_string[3]) == 0)
    {
      gtk_widget_set_sensitive(GTK_WIDGET(aw->script_entry), FALSE);
      gtk_entry_set_text(GTK_ENTRY(aw->script_entry), "**PASSWORD**");
    }
}


static void
remove_script_cb (GtkWidget * widget)
{
  AccountWindow *aw = GET_ACCOUNT_WINDOW(widget);
  gint row;

  g_assert(aw != NULL);

  /* make sure there is a row selected */
  if (!GTK_CLIST(aw->script_clist)->selection)
    {
      return;
    }

  row = GPOINTER_TO_INT(GTK_CLIST(aw->script_clist)->selection->data);
  gtk_clist_remove(GTK_CLIST(aw->script_clist), row);
}


static void
script_select_cb (GtkWidget * widget,  gint row)
{
  AccountWindow *aw = GET_ACCOUNT_WINDOW(widget);
  g_assert(aw != NULL);
}


static void
script_unselect_cb (GtkWidget * widget, gint row)
{
  AccountWindow *aw = GET_ACCOUNT_WINDOW(widget);
  g_assert(aw != NULL);
}

