/* libgnome-ppp - The GNOME PPP Dialer Library
 * Copyright (C) 1997 Jay Painter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#ifndef __gnome_ppp_h__
#define __gnome_ppp_h__

#include <gnome.h>


/* enumerated types */
typedef enum
{
  SCRIPT_ENTRY_EXPECT,
  SCRIPT_ENTRY_SEND,
  SCRIPT_ENTRY_SEND_USER,
  SCRIPT_ENTRY_SEND_PASSWORD
}
ScriptEntryType;

typedef enum
{
  PPP_DISCONNECTED,
  PPP_IN_PROGRESS,
  PPP_CONNECTED
} PPPMessageType;


/* structures */
typedef struct _ScriptEntry ScriptEntry;
typedef struct _Account Account;


struct _ScriptEntry
{
  ScriptEntryType type;
  GString *text;
};

struct _Account 
{
  GString *name;             /* account name */
  GString *modem_device;     /* device path */
  gint speed;                /* connection speed */
  GList *phone_list;         /* phone number list */

  /* for scripting, PAP, CHAP and other uses */
  GString *user;
  GString *passwd;
  GString *remotename;

  /* for static IP's */
  GString *ip;
  GString *remote;
  GString *mask;
  
  /* DNS servers and search domains
   * list data type: string
   */
  GList *dns_list;
  GList *domain_list;

  /* the list of script entries forming the login script
   * list data type: ScriptEntry structures
   */
  GList *script_list;
};


/* typedefs */
typedef void (*ConnectCBFunc)(Account *account, 
			      PPPMessageType message,
			      gchar *text);


/* initalize library -- a must before using! */
void gnomeppplib_init();


/* save/restore accounts */
void          account_load();
void          account_save();

/* list of all accounts */
GList        *account_list();

/* add/free/clone account onto the account list */
Account      *account_new();
void          account_free(Account *account);
Account      *account_clone(Account *account);

/* change settings/information in a account */
void          account_phone_list_clear(Account *account);
void          account_phone_list_append(Account *account, gchar *str);
void          account_dns_list_clear(Account *account);
void          account_dns_list_append(Account *account, gchar *str);
void          account_domain_list_clear(Account *account);
void          account_domain_list_append(Account *account, gchar *str);
void          account_script_list_clear(Account *account);
void          account_script_list_append(Account *account, 
					 ScriptEntryType type, 
					 gchar *text);

/* begin a connection, messages about this connection are
 * sent back though the callback
 */
gint          connect_start(Account *account, ConnectCBFunc func);

/* this must be called into at least once every second during the entire
 * connection; I call into it once every 1/2 second; returns 0 when
 * dead
 */
gint          connect_engine_iteration(Account *account);

/* signal the end of the connection; it's not reall gone until the
 * callback signals that the connection has ended
 */
void          connect_stop(Account *account);

#endif /* __gnome_ppp_h__ */
