/* libgnome-ppp - The GNOME PPP Dialer Library
 * Copyright (C) 1997 Jay Painter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#include "gnome-ppp.h"


/* globals */
GList *__account_list = NULL;
GList *__account_list_watchers = NULL;


/* prototypes */
static GList   *duplicate_g_string_list(GList *list);
static void     free_g_string_list(GList **list);
static void     save_account(Account *account);
static void     load_account();
static gchar   *get_string_blank_default(gchar *key);

ScriptEntry  *script_entry_new(ScriptEntryType type, gchar *text);
ScriptEntry  *script_entry_clone(ScriptEntry *script_entry);
void          script_entry_free(ScriptEntry *script_entry);

static void   broadcast_watchers();


void
__init_account()
{
  static gboolean done_init = FALSE;

  if (!done_init)
    {
      done_init = TRUE;
    }
}


Account *
account_new()
{
  Account *account;

  account = g_malloc(sizeof(Account));

  account->speed                = 115200;
  account->default_route        = TRUE;
  account->lock_device          = TRUE;
  account->locked               = FALSE;

  account->name                 = g_string_new("");
  account->modem_device         = g_string_new("/dev/modem");
  account->user                 = g_string_new("");
  account->passwd               = g_string_new("");
  account->remotename           = g_string_new("");
  account->ip                   = g_string_new("");
  account->mask                 = g_string_new("");
  account->remote               = g_string_new("");
  account->modem_init           = g_string_new("ATZ");

  account->phone_list           = NULL;
  account->dns_list             = NULL;
  account->domain_list          = NULL;
  account->script_list          = NULL;

  return account;
}


void
account_add(Account *account)
{
  if (g_list_find(__account_list, account))
    {
      g_error("account_add(): re-adding existing account");
    }

  __account_list = g_list_append(__account_list, account);
  broadcast_watchers();
}


gint
account_free(Account *account)
{
  if (g_list_find(__account_list, account))
    {
      g_error("account_free(): tring to free active account");
    }
  
  g_string_free(account->name, TRUE);
  g_string_free(account->user, TRUE);
  g_string_free(account->passwd, TRUE);
  g_string_free(account->remotename, TRUE);
  g_string_free(account->ip, TRUE);
  g_string_free(account->mask, TRUE);
  g_string_free(account->remote, TRUE);
  g_string_free(account->modem_init, TRUE);

  free_g_string_list(&account->phone_list);
  free_g_string_list(&account->dns_list);
  free_g_string_list(&account->domain_list);

  account_script_list_clear(account);

  g_free(account);
  return 1;
}


gint
account_delete(Account *account)
{
  if (account->locked)
    {
      return 0;
    }

  if (!g_list_find(__account_list, account))
    {
      g_error("account_delete(): deleting non-added account");
    }

  /* remove account from the account list */
  __account_list = g_list_remove(__account_list, account);
  account_free(account);
  broadcast_watchers();
}


Account *
account_clone(Account *account)
{
  Account *new_account;
  GList *list;
  ScriptEntry *script_entry;

  new_account = account_new();

  new_account->speed               = account->speed;
  new_account->default_route       = account->default_route;
  new_account->lock_device         = account->lock_device;

  g_string_assign(new_account->name,          account->name->str);
  g_string_assign(new_account->modem_device,  account->modem_device->str);
  g_string_assign(new_account->user,          account->user->str);
  g_string_assign(new_account->passwd,        account->passwd->str);
  g_string_assign(new_account->remotename,    account->remotename->str);
  g_string_assign(new_account->ip,            account->ip->str);
  g_string_assign(new_account->mask,          account->mask->str);
  g_string_assign(new_account->remote,        account->remote->str);
  g_string_assign(new_account->modem_init,    account->modem_init->str);
  
  new_account->phone_list          = duplicate_g_string_list(account->phone_list);
  new_account->dns_list            = duplicate_g_string_list(account->dns_list);
  new_account->domain_list         = duplicate_g_string_list(account->domain_list);

  /* duplicate all the script entries */
  list = account->script_list;
  while (list)
    {
     script_entry = (ScriptEntry *) list->data;
     list = list->next;

     new_account->script_list = 
       g_list_append(new_account->script_list, script_entry_clone(script_entry));
    }

  return new_account;
}


GList *
account_list()
{
  return __account_list;
}


void
account_add_watcher(AccountWatcherCBFunc func)
{
  __account_list_watchers = g_list_append(__account_list_watchers, func);
}


void
account_remove_watcher(AccountWatcherCBFunc func)
{
  __account_list_watchers = g_list_remove(__account_list_watchers, func);
}


static void
broadcast_watchers()
{
  GList *list;
  AccountWatcherCBFunc func;

  list = __account_list_watchers;
  while (list)
    {
      func = (AccountWatcherCBFunc) list->data;
      list = list->next;
      (*func)();
    }
}


void
account_phone_list_clear(Account *account)
{
  g_assert(account != NULL);
  free_g_string_list(&account->phone_list);
}


void
account_phone_list_append(Account *account, gchar *str)
{
  g_assert(account != NULL);
  account->phone_list = g_list_append(account->phone_list, g_string_new(str));
}


void
account_dns_list_clear(Account *account)
{
  g_assert(account != NULL);
  free_g_string_list(&account->dns_list);
}


void
account_dns_list_append(Account *account, gchar *str)
{
  g_assert(account != NULL);
  account->dns_list =
    g_list_append(account->dns_list, g_string_new(str));
}


void
account_domain_list_clear(Account *account)
{
  g_assert(account != NULL);
  free_g_string_list(&account->domain_list);
}


void
account_domain_list_append(Account *account, gchar *str)
{
  g_assert(account != NULL);
  account->domain_list = 
    g_list_append(account->domain_list, g_string_new(str));
}


void
account_script_list_clear(Account *account)
{
  GList *list;
  ScriptEntry *script_entry;
  
  g_assert(account != NULL);

  list = account->script_list;
  while (list)
    {
      script_entry = (ScriptEntry *) list->data;
      list = list->next;

      script_entry_free(script_entry);
    }

  g_list_free(account->script_list);
  account->script_list = NULL;
}



void
account_script_list_append(Account *account, ScriptEntryType type, gchar *text)
{
  ScriptEntry *script_entry;

  g_assert(account != NULL);

  script_entry = script_entry_new(type, text);
  account->script_list = g_list_append(account->script_list, script_entry);
}


ScriptEntry * 
script_entry_new(ScriptEntryType type, gchar *text)
{
  ScriptEntry *script_entry;

  script_entry = g_malloc(sizeof(ScriptEntry));
  script_entry->type = type;
  script_entry->text = g_string_new(text);
  return script_entry;
}


ScriptEntry * 
script_entry_clone(ScriptEntry *script_entry)
{
  ScriptEntry *new_script_entry;

  new_script_entry = g_malloc(sizeof(ScriptEntry));
  new_script_entry->type = script_entry->type;
  new_script_entry->text = g_string_new(script_entry->text->str);
  return new_script_entry;
}


void 
script_entry_free(ScriptEntry *script_entry)
{
  g_string_free(script_entry->text, TRUE);
  g_free(script_entry);
}


void 
account_save()
{
  gint i;
  void *iter;
  gchar *some_key;
  GList *list;
  GString *key = g_string_new(NULL);
  GString *value = g_string_new(NULL);
  Account *account;

  /* drop all account sections */
  iter = gnome_config_init_iterator_sections("/gnome-ppp");
  while (gnome_config_iterator_next(iter, &some_key, NULL))
    {
      int len;

      len = strlen(some_key);
      if (len >= 9 && strcmp(some_key + len - 8, ",Account") == 0)
	{
	  g_string_sprintf(key, "/gnome-ppp/%s", some_key);
	  gnome_config_clean_section(key->str);
	}
    }

  list = account_list();
  for (i = 0; list; i++)
    {
      account = (Account *) list->data;
      list = list->next;

      /* form the section string */
      g_string_sprintf(key, "/gnome-ppp/%d,Account/", i);
      gnome_config_push_prefix(key->str);
      save_account(account);
      gnome_config_pop_prefix();
    }

  /* sync the configuration file */
  gnome_config_sync();

 cleanup:
  g_string_free(key, TRUE);
  g_string_free(value, TRUE);
}


void
account_load()
{
  gint i;
  GString *key = g_string_new(NULL);
  GString *value = g_string_new(NULL);

  for (i = 0;; i++)
    {
      /* form the section string */
      g_string_sprintf(key, "/gnome-ppp/%d,Account", i);
      if (!gnome_config_has_section(key->str))
	{
	  break;
	}

      g_string_sprintf(key, "/gnome-ppp/%d,Account/", i);

      gnome_config_push_prefix(key->str);
      load_account();
      gnome_config_pop_prefix();
    }

 cleanup:
  g_string_free(key, TRUE);
  g_string_free(value, TRUE);

}


/* utility */
static GList *
duplicate_g_string_list(GList *list)
{
  GList *new_list = NULL;
  GString *gstr;

  while (list)
    {
      gstr = (GString *) list->data;
      list = list->next;
      new_list = g_list_append(new_list, g_string_new(gstr->str));
    }

  return new_list;
}


static void
free_g_string_list(GList **list)
{
  GList *l;
  GString *gstr;

  l = *list;
  while (l)
    {
      gstr = (GString *) l->data;
      l = l->next;
      g_string_free(gstr, TRUE);
    }

  g_list_free(*list);
  *list = NULL;
}


static void
save_string_list(gchar *base_key, GList *list)
{
  int i;
  GString *key, *value;

  key = g_string_new("");

  for (i = 0; list; i++)
    {
      value = (GString *) list->data;
      list = list->next;

      g_string_sprintf(key, "%d,%s", i, base_key);
      gnome_config_set_string(key->str, value->str);
    }

  g_string_free(key, TRUE);
}


static void
save_account(Account *account)
{
  gint i;
  GList *list;
  GString *key;

  key = g_string_new("");

  gnome_config_set_string("name", account->name->str);
  gnome_config_set_string("modem_device", account->modem_device->str);
  gnome_config_set_int("speed", account->speed);

  save_string_list("phone_number", account->phone_list);

  gnome_config_set_string("user", account->user->str);
  gnome_config_set_string("passwd", account->passwd->str);
  gnome_config_set_string("remotename", account->remotename->str);

  gnome_config_set_string("ip", account->ip->str);
  gnome_config_set_string("remote", account->remote->str);
  gnome_config_set_string("mask", account->mask->str);

  save_string_list("dns_server", account->dns_list);
  save_string_list("domain", account->domain_list);

  /* save script! */
  list = account->script_list;
  for (i = 0; list; i++)
    {
      ScriptEntry *script_entry;

      script_entry = (ScriptEntry *) list->data;
      list = list->next;

      switch (script_entry->type)
	{
	case SCRIPT_ENTRY_EXPECT:
	  g_string_sprintf(key, "%d,SE_expect", i);
	  gnome_config_set_string(key->str, script_entry->text->str);
	  break;

	case SCRIPT_ENTRY_SEND:
	  g_string_sprintf(key, "%d,SE_send", i);
	  gnome_config_set_string(key->str, script_entry->text->str);
	  break;

	case SCRIPT_ENTRY_SEND_USER:
	  g_string_sprintf(key, "%d,SE_send_user", i);
	  gnome_config_set_string(key->str, "");
	  break;

	case SCRIPT_ENTRY_SEND_PASSWORD:
	  g_string_sprintf(key, "%d,SE_send_password", i);
	  gnome_config_set_string(key->str, "");
	  break;
	}
    }

  gnome_config_set_string("modem_init", account->modem_init->str);
  gnome_config_set_bool("default_route", account->default_route);
  gnome_config_set_bool("lock_device", account->lock_device);

 cleanup:
  g_string_free(key, TRUE);
}


static void
load_account()
{
  gint i;
  gboolean junk;
  GList *list;
  GString *key;
  Account *account;

  key = g_string_new("");
  account = account_new();

  g_string_assign(account->name, get_string_blank_default("name"));
  g_string_assign(account->modem_device, get_string_blank_default("modem_device"));
  account->speed = gnome_config_get_int("speed=15200");

  for (i = 0;; i++)
    {
      char *c;

      g_string_sprintf(key, "%d,phone_number", i);
      if ( (c = gnome_config_get_string(key->str)) == NULL)
	{
	  break;
	}
      account_phone_list_append(account, c);
    }

  g_string_assign(account->user, get_string_blank_default("user"));
  g_string_assign(account->passwd, get_string_blank_default("passwd"));
  g_string_assign(account->remotename, get_string_blank_default("remotename"));

  g_string_assign(account->ip, get_string_blank_default("ip"));
  g_string_assign(account->remote, get_string_blank_default("remote"));
  g_string_assign(account->mask, get_string_blank_default("mask"));

  for (i = 0;; i++)
    {
      char *c;

      g_string_sprintf(key, "%d,dns_server", i);
      if ( (c = gnome_config_get_string(key->str)) == NULL)
	{
	  break;
	}
      account_dns_list_append(account, c);
    }

  for (i = 0;; i++)
    {
      char *c;

      g_string_sprintf(key, "%d,domain", i);
      if ( (c = gnome_config_get_string(key->str)) == NULL)
	{
	  break;
	}
      account_domain_list_append(account, c);
    }

  /* load the login script */
  for (i = 0;; i++)
    {
      gchar *c;

      /* search for expect entries */
      g_string_sprintf(key, "%d,SE_expect", i);
      if ( (c = gnome_config_get_string(key->str)) != NULL)
	{
	  account_script_list_append(account, SCRIPT_ENTRY_EXPECT, c);
	  continue;
	}

      /* search for expect send */
      g_string_sprintf(key, "%d,SE_send", i);
      if ( (c = gnome_config_get_string(key->str)) != NULL)
	{
	  account_script_list_append(account, SCRIPT_ENTRY_SEND, c);
	  continue;
	}

      /* search for expect send user */
      g_string_sprintf(key, "%d,SE_send_user", i);
      if ( (c = gnome_config_get_string(key->str)) != NULL)
	{
	  account_script_list_append(account, SCRIPT_ENTRY_SEND_USER, "");
	  continue;
	}

      /* search for expect send password */
      g_string_sprintf(key, "%d,SE_send_password", i);
      if ( (c = gnome_config_get_string(key->str)) != NULL)
	{
	  account_script_list_append(account, SCRIPT_ENTRY_SEND_PASSWORD, "");
	  continue;
	}

      /* if we reach here then there's no more script */
      break;
    }

  g_string_assign(account->modem_init, get_string_blank_default("modem_init"));
  account->default_route = gnome_config_get_bool("default_route=TRUE");
  account->lock_device = gnome_config_get_bool("lock_device=TRUE");

 cleanup:
  g_string_free(key, TRUE);

  account_add(account);
}


static gchar *
get_string_blank_default(gchar *key)
{
  gchar *c;

  if ( (c = gnome_config_get_string(key)) == NULL)
    {
      return "";
    }

  return c;
}







/*** NEW SAVE STUFF W/O gnome_config ***/
/* NOT IN USE AT ALL! -JMP */
static void
append_string_var(GList **line_list, gchar *variable, gchar *value)
{
  GString *gstr;

  gstr = g_string_new("");
  g_string_sprintf(gstr, "%s = %s", variable, value);
  *line_list = g_list_append(*line_list, gstr);
}


static void
append_int_var(GList **line_list, gchar *variable, gint value)
{
  GString *gstr;

  gstr = g_string_new("");
  g_string_sprintf(gstr, "%s = %d", variable, value);
  *line_list = g_list_append(*line_list, gstr);
}


static void
append_string_list_var(GList **line_list, gchar *variable, GList *value_list)
{
  GString *gstr;

  gstr = g_string_new("");
  g_string_sprintf(gstr, "%s list", variable);
  *line_list = g_list_append(*line_list, gstr);

  while (value_list)
    {
      gstr = g_string_new(((GString *) value_list->data)->str);
      *line_list = g_list_append(*line_list, gstr);
      value_list = value_list->next;
    }

  gstr = g_string_new("");
  g_string_sprintf(gstr, "end %s list", variable);
  *line_list = g_list_append(*line_list, gstr);
}


static void
new_save_account(Account *account)
{
  gint i;
  GList *list, *line_list;
  GString *gstr;

  line_list = NULL;

  /* name, modem device */
  append_string_var(&line_list, "name", account->name->str);
  append_string_var(&line_list, "modem_device", account->modem_device->str);
  append_int_var(&line_list, "speed", account->speed);

  append_string_list_var(&line_list, "phone numbers", account->phone_list);

  append_string_var(&line_list, "user", account->user->str);
  append_string_var(&line_list, "passwd", account->passwd->str);
  append_string_var(&line_list, "remotename", account->remotename->str);

  append_string_var(&line_list, "ip", account->ip->str);
  append_string_var(&line_list, "remote", account->remote->str);
  append_string_var(&line_list, "mask", account->mask->str);

  append_string_list_var(&line_list, "dns servers", account->dns_list);
  append_string_list_var(&line_list, "search domains", account->domain_list);

  /* save script -- this is a little hackish */
  gstr = g_string_new("script list");
  line_list = g_list_append(line_list, gstr);

  list = account->script_list;
  while (list)
    {
      ScriptEntry *script_entry;

      script_entry = (ScriptEntry *) list->data;
      list = list->next;

      gstr = g_string_new("");

      switch (script_entry->type)
	{
	case SCRIPT_ENTRY_EXPECT:
	  g_string_sprintf(gstr, "expect = %s", script_entry->text->str);
	  break;

	case SCRIPT_ENTRY_SEND:
	  g_string_sprintf(gstr, "send = %s", script_entry->text->str);
	  break;

	case SCRIPT_ENTRY_SEND_USER:
	  g_string_sprintf(gstr, "send username");
	  break;

	case SCRIPT_ENTRY_SEND_PASSWORD:
	  g_string_sprintf(gstr, "send password");
	  break;
	}

      line_list = g_list_append(line_list, gstr);
    }

  gstr = g_string_new("end script list");
  line_list = g_list_append(line_list, gstr);
}

