/* gpilotd control applet ('capplet') for use with the GNOME control center */
/*                                                                          */
/* Copyright 1998 Red Hat Software                                          */
/* Author: Michael Fulbright <msf@redhat.com>                               */

#include <pwd.h>
#include <sys/types.h>
#include <signal.h>
#include <gnome.h>

#include <config.h>
#include <capplet-widget.h>
#include "pilot.h"
#include <gpilotd/gpilotd-app.h>

/* tell changes callbacks to ignore changes or not */
static gboolean ignore_changes=FALSE;

/* capplet widget */
static GtkWidget *capplet=NULL;

/* host/device/pilot configuration windows */
GtkWidget *hostCfgWindow=NULL;
GtkWidget *deviceCfgWindow=NULL;
GtkWidget *pilotCfgWindow=NULL;
GtkWidget *syncCfgWindow=NULL;
GtkWidget *dialogWindow=NULL;

/* old pilot state */
PilotState *origState=NULL;

/* current editted state */
PilotState *curState=NULL;

static void doTrySettings(GtkWidget *widget, PilotState *pilotState);
static void doRevertSettings(GtkWidget *widget, PilotState *pilotState);
static void doSaveSettings(GtkWidget *widget, PilotState *pilotState);

static void readHostCfg(GtkWidget *w, PilotState **state);
static void readDeviceCfg(GtkWidget *w, PilotState **state);
static void readPilotCfg(GtkWidget *w, PilotState **state);
static void readSyncCfg(GtkWidget *w, PilotState **state);

static void setHostCfg(GtkWidget *w, PilotState *state);
static void setDeviceCfg(GtkWidget *w, PilotState *state);
static void setPilotCfg(GtkWidget *w, PilotState *state);
static void setSyncCfg(GtkWidget *w, PilotState *state);

void run_error_dialog(gchar *,...);

void gpilotd_connect_cb(const gchar *id,const GNOME_Pilot_PilotUser *user) {
  g_message("connect from %s",id);
}

void gpilotd_disconnect_cb(const gchar *id) {
  g_message("disconnect from %s",id);
}

void gpilotd_request_completed(const gchar *id,unsigned long handle) {
  g_message("%s completed %d",id,handle);
  gnome_dialog_close(GNOME_DIALOG(dialogWindow));
}


void gpilotd_conduit_start(const gchar *id, 
			   const gchar *conduit_name, 
			   const gchar *db_name) { 
	g_message("conduit_start(%s,%s,%s)",id,conduit_name,db_name);
}

void gpilotd_conduit_end(const gchar *id) { 
	g_message("conduit_end(%s)",id);
}

void gpilotd_userinfo_requested(const gchar *device,const GNOME_Pilot_PilotUser *user) {
  g_message("device %s sent userinfo",device);
  g_message("user->userID   = %d",user->userID);
  g_message("user->username = %s",user->username);

  if(!curState->pilot) curState->pilot = g_new0(GPilotPilot,1);
  curState->pilot->pilot_id = user->userID;
  if(curState->pilot->pilot_username) g_free(curState->pilot->pilot_username);
  curState->pilot->pilot_username = g_strdup(user->username);

  setPilotCfg(pilotCfgWindow,curState);
  gnome_dialog_close(GNOME_DIALOG(dialogWindow));
  /*gtk_signal_emit_by_name(GTK_OBJECT(dialogWindow),"close"); */

  if (!ignore_changes)
    capplet_widget_state_changed(CAPPLET_WIDGET(capplet), TRUE);
}

void gpilotd_sysinfo_requested(const gchar *id, const GNOME_Pilot_SysInfo *info) {
  g_message("pilot %s",id);
  g_message("  ROMSIZE = %lu",info->romSize);
  g_message("  RAMSIZE = %lu",info->ramSize);
  g_message("  RAMFREE = %lu",info->ramFree);
}

static pid_t
getGpilotdpid(void)
{
    FILE *f;
    gchar *homedir;
    GString *str;
    gchar pidstr[100];
    pid_t pid;

    homedir = g_get_home_dir(); 
    if (!homedir)
	return -1;

    str = g_string_new(homedir);
    g_string_append(str, "/.gpilotd.pid");
    f = fopen(str->str, "r");
    g_string_free(str, TRUE);
    if (!f) {
	return -1;
    } else {
	fgets(pidstr, sizeof(pidstr), f);
	fclose(f);
	pid = strtol(pidstr, NULL, 10);
	if (pid == '\0')
	    return -1;
	else
	    return pid;
    }
}

void saveSettingsAndRestartDaemon(PilotState *state) {
  pid_t pid;

  gtk_widget_set_sensitive(gtk_object_get_data(GTK_OBJECT(pilotCfgWindow),"get_from_pilot"),TRUE);
  gtk_widget_set_sensitive(gtk_object_get_data(GTK_OBJECT(pilotCfgWindow),"send_to_pilot"),TRUE);
			   
    savePilotState(state);
    pid = getGpilotdpid();
    if (pid < 0) {
	g_message("gpilotd doesn't appear to be running");
    } else {
	kill(pid, SIGHUP);
    }
}

static void
doTrySettings(GtkWidget *widget, PilotState *pilotState)
{
    g_message("trying settings");

    readHostCfg(hostCfgWindow, &pilotState);
    readDeviceCfg(deviceCfgWindow, &pilotState);
    readPilotCfg(pilotCfgWindow, &pilotState);
    readSyncCfg(pilotCfgWindow, &pilotState);

    if(pilotState->device->device_name==NULL) {
      run_error_dialog(_("Please specify a cradle port"));
      return;
    }

    g_message("checking rw on %s",pilotState->device->device_name);
    if(access(pilotState->device->device_name,R_OK|W_OK)) {
      run_error_dialog(_("Read/Write permissions on %s failed"),pilotState->device->device_name);
      return;
    }

/* revert only back to original state, so no need to save current */
/*    copyPilotState(origState, curState); */

    /* need to HUP the current gpilotd to get it to reread settings */
    saveSettingsAndRestartDaemon(pilotState);
}

static void
doSaveSettings(GtkWidget *widget, PilotState *pilotState)
{
    g_message("Saving state");
    doTrySettings(widget, pilotState);
}


static void
doRevertSettings(GtkWidget *widget, PilotState *pilotState)
{
    g_message("reverting settings to original");

    setHostCfg(hostCfgWindow, pilotState);
    setDeviceCfg(deviceCfgWindow, pilotState);
    setPilotCfg(pilotCfgWindow, pilotState);
    setSyncCfg(syncCfgWindow, pilotState);
    saveSettingsAndRestartDaemon(pilotState);
}


static void
insert_numeric_callback (GtkEditable    *editable, const gchar    *text,
		      gint len, gint *position, void *data)
{
    gint i;

    for (i=0; i<len; i++) {
	if (!isdigit(text[i])) {
	    gtk_signal_emit_stop_by_name(GTK_OBJECT(editable), "insert_text");
	    return;
	}
    }
}

static void
statechange_cb(GtkEditable    *editable, const gchar    *text,
		      gint            length, gint           *position,
		      void *data)
{
  if (!ignore_changes) {
        capplet_widget_state_changed(CAPPLET_WIDGET (capplet), TRUE);
  }
}


static void
insert_syncpc_id(GtkEditable    *editable, const gchar    *text,
		      gint            length, gint           *position,
		      void *data)
{
  if (!ignore_changes) {
        capplet_widget_state_changed(CAPPLET_WIDGET (capplet), TRUE);
	gtk_widget_set_sensitive(gtk_object_get_data(GTK_OBJECT(pilotCfgWindow),"get_from_pilot"),FALSE);
	gtk_widget_set_sensitive(gtk_object_get_data(GTK_OBJECT(pilotCfgWindow),"send_to_pilot"),FALSE);
  }
}

static void
insert_username_callback (GtkEditable    *editable, const gchar    *text,
		      gint len, gint *position, void *data)
{
    gint i;
    gchar *curname;

    /* need to make sure that existing entry starts with a letter */
    /* since valid usernames must start with a letter             */
    curname = gtk_entry_get_text(GTK_ENTRY(editable));
    if (*curname == '\0' && len > 0) {
	if (!isalpha(text[0])) {
	    gtk_signal_emit_stop_by_name(GTK_OBJECT(editable), "insert_text");
	    return;
	}
    } else {
	/* rest of username can be alphanumeric */
	for (i=0; i<len; i++) {
	    if (!isalnum(text[i])) {
		gtk_signal_emit_stop_by_name(GTK_OBJECT(editable), 
					     "insert_text");
		return;
	    }
	}
    }
}
static void
insert_device_callback (GtkEditable    *editable, const gchar    *text,
		      gint len, gint *position, void *data)
{
    gint i;
    gchar *curname;

    curname = gtk_entry_get_text(GTK_ENTRY(editable));
    if (*curname == '\0' && len > 0) {
	if (text[0]!='/') {
	  gtk_signal_emit_stop_by_name(GTK_OBJECT(editable), "insert_text");
	  return;
	} 
    } else {
      for(i=0;i<len;i++)
	if(!(isalnum(text[i]) || text[i]=='/')) {
	  gtk_signal_emit_stop_by_name(GTK_OBJECT(editable), "insert_text");
	  return;
	}
    }
} 

/* called by the sync_type GtkOptionMenu */
static void
sync_action_selection(GtkMenuShell *widget,
		      gpointer unused) 
{
	if (!ignore_changes) {
		capplet_widget_state_changed(CAPPLET_WIDGET (capplet), TRUE);
	}
}

/* called by the sync_type GtkOptionMenu */
static void
activate_sync_type(GtkMenuItem *widget,
		   gpointer data)
{
	curState->sync_type = GPOINTER_TO_INT(data);
}

static void cancel_sync(GnomeDialog *w,gpointer data) {
  g_message("cancelling %d",GPOINTER_TO_INT(data));
  gpilotd_remove_request(GPOINTER_TO_INT(data));  
}

void run_sync_dialog(gint handle) {
  dialogWindow = gnome_message_box_new(_("Press synchronize on the cradle\n" " or cancel the operation."),GNOME_MESSAGE_BOX_GENERIC,GNOME_STOCK_BUTTON_CANCEL,NULL);
  gnome_dialog_button_connect(GNOME_DIALOG(dialogWindow),0,cancel_sync,GINT_TO_POINTER(handle));
  gnome_dialog_run_and_close(GNOME_DIALOG(dialogWindow));
}

void run_error_dialog(gchar *mesg,...) {
  char *tmp;
  va_list ap;

  va_start(ap,mesg);
  tmp = g_strdup_printf(mesg,ap);
  dialogWindow = gnome_message_box_new(tmp,GNOME_MESSAGE_BOX_ERROR,GNOME_STOCK_BUTTON_OK,NULL);
  gnome_dialog_run_and_close(GNOME_DIALOG(dialogWindow));
  va_end(ap);
  g_free(tmp);
}

static void 
get_from_pilot_cb() 
{
	int handle;
	handle = gpilotd_get_user_info(curState->device->device_name,GNOME_Pilot_PERSISTENT,0);
	/* FIXME: check handle>0 here */
	run_sync_dialog(handle);

}

static void 
send_to_pilot_cb() 
{
	int handle;
	GNOME_Pilot_PilotUser user;
	
	doTrySettings(capplet, curState);
	
	user.userID = curState->pilot->pilot_id;
	user.username = curState->pilot->pilot_username;
	handle = gpilotd_set_user_info(curState->device->device_name,user,GNOME_Pilot_PERSISTENT,0);
	/* FIXME: check handle>0 here */
	run_sync_dialog(handle);
}

#if 0
static void
entry_changed(GtkWidget *widget, gpointer data)
{
    if (!ignore_changes)
	capplet_widget_state_changed(CAPPLET_WIDGET(capplet), TRUE);
}
#endif

static void
clist_changed(GtkWidget *widget, gpointer data)
{
    if (!ignore_changes)
	capplet_widget_state_changed(CAPPLET_WIDGET(capplet), TRUE);
}
	

/* returns a hbox which has GUI for host configuration window in it */
/* Following object data exists:                                    */
/*                                                                  */
/* "syncpcid" - GtkEntry for the syncpcid                           */
/*                                                                  */
static GtkWidget
*createHostCfgWindow()
{
    GtkWidget *vbox, *hbox, *entry, *label;

    vbox = gtk_vbox_new(FALSE, GNOME_PAD);

    gtk_box_pack_start(GTK_BOX(vbox), 
		       gtk_label_new(_("SyncPCid is a integer that uniquely "
				     "identifies your PC")), 
		       FALSE, FALSE, GNOME_PAD);
 
    hbox = gtk_hbox_new(FALSE, GNOME_PAD);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, GNOME_PAD);

    label = gtk_label_new("SyncPCid");
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, GNOME_PAD);
    
    entry = gtk_entry_new_with_max_length(16);
    gtk_object_set_data(GTK_OBJECT(vbox), "syncpcid", entry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, FALSE, FALSE, GNOME_PAD);
    gtk_signal_connect(GTK_OBJECT(entry), "insert_text",
		       GTK_SIGNAL_FUNC(insert_numeric_callback),
		       NULL);
    gtk_signal_connect_after(GTK_OBJECT(entry), "insert_text",
		       GTK_SIGNAL_FUNC(insert_syncpc_id),
		       NULL);
    gtk_signal_connect(GTK_OBJECT(entry), "delete_text",
		       GTK_SIGNAL_FUNC(statechange_cb),
		       NULL);

    return vbox;
}

static void
setHostCfg(GtkWidget *hostcfg, PilotState *state)
{
    GtkWidget *entry;
    gchar id[40];

    entry = gtk_object_get_data(GTK_OBJECT(hostcfg), "syncpcid");

    g_snprintf(id, sizeof(id), "%d", state->syncPCid);

    ignore_changes = TRUE;
    gtk_entry_set_text(GTK_ENTRY(entry), id);
    ignore_changes = FALSE;
}

/* no error checking yet */
static void
readHostCfg(GtkWidget *hostcfg, PilotState **state)
{
    GtkWidget *entry;
    gchar *id;
    PilotState *p=NULL;

    entry = gtk_object_get_data(GTK_OBJECT(hostcfg), "syncpcid");
    id = gtk_entry_get_text(GTK_ENTRY(entry));
    p = *state;
    if (p)
	p->syncPCid = strtol(id, NULL, 10);

}

/* object data:                        */
/* "port_clist"  - clist for serial port */
/* "speed_clist" - clist for speed       */
static GtkWidget
*createDeviceCfgWindow(void)
{
    GtkWidget *vbox;
    GtkWidget *clist;
    GtkWidget *entry;
    GtkWidget *frame;
    gint i;

    vbox = gtk_vbox_new(FALSE, GNOME_PAD);

    frame = gtk_frame_new("Cradle Port");
    gtk_frame_set_label_align(GTK_FRAME(frame), 0.5, 0.5);
    gtk_container_border_width(GTK_CONTAINER(frame), GNOME_PAD);
    gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, GNOME_PAD);

    entry = gtk_entry_new();
    gtk_object_set_data(GTK_OBJECT(vbox),"port_entry",entry);
    gtk_container_add(GTK_CONTAINER(frame),entry);
    gtk_signal_connect(GTK_OBJECT(entry),"insert_text",
		       GTK_SIGNAL_FUNC(insert_device_callback),
		       NULL);
    gtk_signal_connect_after(GTK_OBJECT(entry), "insert_text",
		       GTK_SIGNAL_FUNC(statechange_cb),
		       NULL);
    gtk_signal_connect_after(GTK_OBJECT(entry), "delete_text",
		       GTK_SIGNAL_FUNC(statechange_cb),
		       NULL);
#if 0    
    clist = gtk_clist_new(1);
    gtk_widget_set_usize(clist, 50, 85);
/*
    gtk_clist_set_policy(GTK_CLIST(clist), GTK_POLICY_AUTOMATIC,
			 GTK_POLICY_AUTOMATIC);
*/
    gtk_clist_set_selection_mode(GTK_CLIST(clist), GTK_SELECTION_BROWSE);
    gtk_object_set_data(GTK_OBJECT(vbox), "port_clist", clist);
    gtk_container_add(GTK_CONTAINER(frame), clist);
    for (i=0; deviceList[i]; i++) {
	gchar *row[2] = {NULL, NULL};
	row[0] = (gchar *) deviceList[i];
	gtk_clist_append(GTK_CLIST(clist), row);
    }
    gtk_signal_connect(GTK_OBJECT(clist), "select_row",
		       GTK_SIGNAL_FUNC(clist_changed), NULL);
#endif

    frame = gtk_frame_new(_("Speed"));
    gtk_frame_set_label_align(GTK_FRAME(frame), 0.5, 0.5);
    gtk_container_border_width(GTK_CONTAINER(frame), GNOME_PAD);
    gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, GNOME_PAD);

    clist = gtk_clist_new(1);
    /*    gtk_widget_set_usize(clist, 50, 75); */
    
/*
    gtk_clist_set_policy(GTK_CLIST(clist), GTK_POLICY_AUTOMATIC,
			 GTK_POLICY_AUTOMATIC);
*/
    gtk_clist_set_selection_mode(GTK_CLIST(clist), GTK_SELECTION_BROWSE);
    gtk_object_set_data(GTK_OBJECT(vbox), "speed_clist", clist);
    gtk_container_add(GTK_CONTAINER(frame), clist);
    for (i=0; speedList[i]; i++) {
	gchar num[20];
	gchar *row[2] = {NULL, NULL};
	g_snprintf(num, sizeof(num), "%d", speedList[i]);
	row[0] = num;
	gtk_clist_append(GTK_CLIST(clist), row);
    }
    gtk_signal_connect(GTK_OBJECT(clist), "select_row",
		       GTK_SIGNAL_FUNC(clist_changed), NULL);
    return vbox;
}

static void
setDeviceCfg(GtkWidget *devcfg, PilotState *state)
{
    GtkWidget *port_entry, *speed_clist;
    gint      i, sel;

    port_entry  = gtk_object_get_data(GTK_OBJECT(devcfg), "port_entry");
    speed_clist = gtk_object_get_data(GTK_OBJECT(devcfg), "speed_clist");

    if (state->device) {
#if 0
	/* this piece of code is used to have the devices in a clist,
	   eg. containing COM1, COM2 ... */
	/* initialize the device setting */
	for (i=0, sel=-1; deviceList[i]; i++) {
	    if (!strncmp(deviceList[i], state->device->device_name,
			 strlen(deviceList[i])))
		sel = i;
	}
	if (sel < 0) {
	    g_message(_("Warning - unrecognized device %s, "
		        "changing to /dev/pilot"), state->device->device_name);
	    sel = 0;
	}
#endif
	ignore_changes = TRUE;
	gtk_entry_set_text(GTK_ENTRY(port_entry), state->device->device_name);
	ignore_changes = FALSE;

	/* now do the speed selection */
	for (i=0, sel=-1; speedList[i]; i++) {
	    if (speedList[i] == state->device->speed)
		sel = i;
	}
	if (sel < 0) {
	    g_message(_("Warning - unrecognized speed %d, "
		        "changing to 9600"), state->device->speed);
	    sel = 0;
	}
	ignore_changes = TRUE;
	gtk_clist_select_row(GTK_CLIST(speed_clist), sel, 0);
	ignore_changes = FALSE;
    } else {
	ignore_changes = TRUE;
#if 0
	gtk_clist_select_row(GTK_CLIST(port_clist), 0, 0);
#endif
	gtk_clist_select_row(GTK_CLIST(speed_clist), 0, 0);
        readDeviceCfg(devcfg,&state);
	ignore_changes = FALSE;
    }	

}


static void
readDeviceCfg(GtkWidget *devcfg, PilotState **state)
{
    GtkWidget *port_entry, *speed_clist;
    guint  sel;
    PilotState *p;
    GList *dlist;

    port_entry  = gtk_object_get_data(GTK_OBJECT(devcfg), "port_entry");
    speed_clist = gtk_object_get_data(GTK_OBJECT(devcfg), "speed_clist");
    p = *state;

    if (!p->device) {
	p->device = g_new0(GPilotDevice, 1);
    }

    if (p->device->device_name)
	g_free(p->device->device_name);
    p->device->device_name = g_strdup(gtk_entry_get_text(GTK_ENTRY(port_entry)));

#if 0
    dlist = GTK_CLIST(port_clist)->selection;
    sel = GPOINTER_TO_INT(dlist->data);
    if (p->device->device_name)
	g_free(p->device->device_name);
    p->device->device_name = g_strdup(deviceList[sel]);
#endif

    dlist = GTK_CLIST(speed_clist)->selection;
    sel = GPOINTER_TO_INT(dlist->data);
    p->device->speed = speedList[sel];
}

/* object data:                        */
/* "userid"   - entry box for pilot user id */
/* "username" - entry box for pilot user name */
static GtkWidget
*createPilotCfgWindow(void)
{
    GtkWidget *vbox, *table;
    GtkWidget *entry, *label;
    GtkWidget *button;

    vbox = gtk_vbox_new(FALSE, GNOME_PAD);


    table = gtk_table_new(3, 3, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 4);
    gtk_table_set_col_spacings(GTK_TABLE(table), 10);
    gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, GNOME_PAD);

    label = gtk_label_new(_("Pilot User ID"));
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
    
    entry = gtk_entry_new_with_max_length(16);
    gtk_object_set_data(GTK_OBJECT(vbox), "userid", entry);
    gtk_table_attach_defaults(GTK_TABLE(table), entry, 1, 2, 0, 1);
    gtk_signal_connect(GTK_OBJECT(entry), "insert_text",
		       GTK_SIGNAL_FUNC(insert_numeric_callback),
		       NULL);
    gtk_signal_connect_after(GTK_OBJECT(entry), "insert_text",
		       GTK_SIGNAL_FUNC(statechange_cb),
		       NULL);
    gtk_signal_connect_after(GTK_OBJECT(entry), "delete_text",
		       GTK_SIGNAL_FUNC(statechange_cb),
		       NULL);

    label = gtk_label_new(_("Pilot Username"));
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);
    
    entry = gtk_entry_new_with_max_length(128);
    gtk_object_set_data(GTK_OBJECT(vbox), "username", entry);
    gtk_table_attach_defaults(GTK_TABLE(table), entry, 1, 2, 1, 2);
    gtk_signal_connect(GTK_OBJECT(entry), "insert_text",
		       GTK_SIGNAL_FUNC(statechange_cb),
		       NULL);
    gtk_signal_connect(GTK_OBJECT(entry), "delete_text",
		       GTK_SIGNAL_FUNC(statechange_cb),
		       NULL);

    label = gtk_label_new(_("Pilot Name"));
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);
    
    entry = gtk_entry_new_with_max_length(128);
    gtk_object_set_data(GTK_OBJECT(vbox), "pilotname", entry);
    gtk_table_attach_defaults(GTK_TABLE(table), entry, 1, 2, 2, 3);
    gtk_signal_connect(GTK_OBJECT(entry), "insert_text",
		       GTK_SIGNAL_FUNC(statechange_cb),
		       NULL);
    gtk_signal_connect(GTK_OBJECT(entry), "delete_text",
		       GTK_SIGNAL_FUNC(statechange_cb),
		       NULL);

    button = gtk_button_new_with_label(_("get from pilot"));
    gtk_table_attach(GTK_TABLE(table),button,0,1,3,4,GTK_FILL,GTK_FILL,GNOME_PAD,GNOME_PAD);
    gtk_signal_connect(GTK_OBJECT(button),"clicked",GTK_SIGNAL_FUNC(get_from_pilot_cb),NULL);
    gtk_object_set_data(GTK_OBJECT(vbox),"get_from_pilot",button);
    button = gtk_button_new_with_label(_("send to pilot"));
    gtk_table_attach(GTK_TABLE(table),button,1,2,3,4,GTK_FILL,GTK_FILL,GNOME_PAD,GNOME_PAD);
    gtk_signal_connect(GTK_OBJECT(button),"clicked",GTK_SIGNAL_FUNC(send_to_pilot_cb),NULL);
    gtk_object_set_data(GTK_OBJECT(vbox),"send_to_pilot",button);
    

    return vbox;
}

static void
setPilotCfg(GtkWidget *pilotcfg, PilotState *state)
{
    GtkWidget *id, *name,*pname;
    gchar num[40];

    id  = gtk_object_get_data(GTK_OBJECT(pilotcfg), "userid");
    name = gtk_object_get_data(GTK_OBJECT(pilotcfg), "username");
    pname = gtk_object_get_data(GTK_OBJECT(pilotcfg), "pilotname");

    ignore_changes = TRUE;
    if (state->pilot) {
	    g_snprintf(num, sizeof(num), "%d", state->pilot->pilot_id);
	    gtk_entry_set_text(GTK_ENTRY(id), num);
	    gtk_entry_set_text(GTK_ENTRY(name), state->pilot->pilot_username);
	    gtk_entry_set_text(GTK_ENTRY(pname), state->pilot->name);
    } else {
	    g_snprintf(num,sizeof(num),"%d",getuid());
	    gtk_entry_set_text(GTK_ENTRY(id), num);
	    gtk_entry_set_text(GTK_ENTRY(name), g_get_real_name());
	    gtk_entry_set_text(GTK_ENTRY(pname), "My Pilot");
    }
    ignore_changes = FALSE;
}


static void
readPilotCfg(GtkWidget *pilotcfg, PilotState **state)
{
    GtkWidget *id, *name,*pname;
    guint pilotid;
    gchar *num;
    PilotState *p;

    id  = gtk_object_get_data(GTK_OBJECT(pilotcfg), "userid");
    name = gtk_object_get_data(GTK_OBJECT(pilotcfg), "username");
    pname = gtk_object_get_data(GTK_OBJECT(pilotcfg), "pilotname");
    p = *state;

    if (!p->pilot)
	p->pilot = g_new0(GPilotPilot, 1);
	
    num = gtk_entry_get_text(GTK_ENTRY(id));
    pilotid = strtol(num, NULL, 10);
    p->pilot->pilot_id = pilotid;

    if (p->pilot->pilot_username)
	g_free(p->pilot->pilot_username);
    p->pilot->pilot_username = g_strdup(gtk_entry_get_text(GTK_ENTRY(name)));

    if (p->pilot->name)
	g_free(p->pilot->name);
    p->pilot->name = g_strdup(gtk_entry_get_text(GTK_ENTRY(pname)));
}

/* object data:                                       */
/* "sync_action" - gtkoptionmenu with the sync action */
static GtkWidget
*createSyncCfgWindow(void)
{
    GtkWidget *vbox, *table;
    GtkWidget *optionMenu;
    GtkMenu *menu;
    GtkWidget *menuItem;

    vbox = gtk_vbox_new(FALSE, GNOME_PAD);

    table = gtk_table_new(1, 1, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 4);
    gtk_table_set_col_spacings(GTK_TABLE(table), 10);
    gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, GNOME_PAD);

    optionMenu = gtk_option_menu_new();
    gtk_object_set_data(GTK_OBJECT(vbox),"sync_action",optionMenu);
    menu = GTK_MENU(gtk_menu_new());

    menuItem = gtk_menu_item_new_with_label(_("Use conduit settings"));
    gtk_widget_show(menuItem);
    gtk_signal_connect(GTK_OBJECT(menuItem),"activate",
		       GTK_SIGNAL_FUNC(activate_sync_type),
		       GINT_TO_POINTER(GnomePilotConduitSyncTypeNotSet));
    gtk_menu_append(menu,menuItem);
    menuItem = gtk_menu_item_new_with_label(_("Synchronize"));
    gtk_widget_show(menuItem);
    gtk_signal_connect(GTK_OBJECT(menuItem),"activate",
		       GTK_SIGNAL_FUNC(activate_sync_type),
		       GINT_TO_POINTER(GnomePilotConduitSyncTypeSynchronize));
    gtk_menu_append(menu,menuItem);
    menuItem = gtk_menu_item_new_with_label(_("Copy from pilot"));
    gtk_widget_show(menuItem);
    gtk_signal_connect(GTK_OBJECT(menuItem),"activate",
		       GTK_SIGNAL_FUNC(activate_sync_type),
		       GINT_TO_POINTER(GnomePilotConduitSyncTypeCopyFromPilot));
    gtk_menu_append(menu,menuItem);
    menuItem = gtk_menu_item_new_with_label(_("Copy to pilot"));
    gtk_widget_show(menuItem);
    gtk_signal_connect(GTK_OBJECT(menuItem),"activate",
		       GTK_SIGNAL_FUNC(activate_sync_type),
		       GINT_TO_POINTER(GnomePilotConduitSyncTypeCopyToPilot));
    gtk_menu_append(menu,menuItem);
    menuItem = gtk_menu_item_new_with_label(_("Merge from pilot"));
    gtk_widget_show(menuItem);
    gtk_signal_connect(GTK_OBJECT(menuItem),"activate",
		       GTK_SIGNAL_FUNC(activate_sync_type),
		       GINT_TO_POINTER(GnomePilotConduitSyncTypeMergeFromPilot));
    gtk_menu_append(menu,menuItem);
    menuItem = gtk_menu_item_new_with_label(_("Merge to pilot"));
    gtk_widget_show(menuItem);
    gtk_signal_connect(GTK_OBJECT(menuItem),"activate",
		       GTK_SIGNAL_FUNC(activate_sync_type),
		       GINT_TO_POINTER(GnomePilotConduitSyncTypeMergeToPilot));
    gtk_menu_append(menu,menuItem);

    gtk_option_menu_set_menu(GTK_OPTION_MENU(optionMenu),GTK_WIDGET(menu));
    gtk_signal_connect(GTK_OBJECT(menu), "selection-done",
		       GTK_SIGNAL_FUNC(sync_action_selection),
		       NULL);
    gtk_table_attach_defaults(GTK_TABLE(table), optionMenu, 0, 1, 0, 1);

    return vbox;
}

static void
setSyncCfg(GtkWidget *synccfg, 
	   PilotState *state)
{
	GtkOptionMenu *optionMenu;
	GtkMenu *menu;

	optionMenu = gtk_object_get_data(GTK_OBJECT(synccfg), "sync_action");
	menu = GTK_MENU(gtk_option_menu_get_menu(optionMenu));

	ignore_changes = TRUE;

	/* yes, if the items in menu are the same order as in
	   GnomePilotConduitSyncType, we could loose the switch. But
	   if they're not, or the enum doesn't start at 0, boohoo */
	switch(state->sync_type) {
	case GnomePilotConduitSyncTypeCustom:
	case GnomePilotConduitSyncTypeNotSet:
		gtk_option_menu_set_history(optionMenu,0);
		break;
	case GnomePilotConduitSyncTypeSynchronize:
		gtk_option_menu_set_history(optionMenu,1);
		break;
	case GnomePilotConduitSyncTypeCopyFromPilot:
		gtk_option_menu_set_history(optionMenu,2);
		break;
	case GnomePilotConduitSyncTypeCopyToPilot:
		gtk_option_menu_set_history(optionMenu,3);
		break;
	case GnomePilotConduitSyncTypeMergeFromPilot:
		gtk_option_menu_set_history(optionMenu,4);
		break;
	case GnomePilotConduitSyncTypeMergeToPilot:
		gtk_option_menu_set_history(optionMenu,5);
		break;
	}
	ignore_changes = FALSE;
}


static void
readSyncCfg(GtkWidget *pilotcfg, PilotState **state)
{
}

static void
pilot_capplet_setup(void)
{
    GtkWidget *frame, *table;
    static GnomeHelpMenuEntry help_entry = {"gpilotd-control-applet","properties.html"};

    capplet = capplet_widget_new();

    table = gtk_table_new(2, 4, FALSE);
    gtk_container_border_width(GTK_CONTAINER(table), GNOME_PAD);
    gtk_container_add(GTK_CONTAINER(capplet), table); 

    frame = gtk_frame_new(_("Host ID"));
    gtk_container_border_width(GTK_CONTAINER(frame), GNOME_PAD_SMALL);
    gtk_table_attach_defaults(GTK_TABLE(table), frame, 0, 1, 0, 1);
    hostCfgWindow = createHostCfgWindow();
    gtk_container_add(GTK_CONTAINER(frame), hostCfgWindow);

    frame = gtk_frame_new(_("User Configuration"));
    gtk_container_border_width(GTK_CONTAINER(frame), GNOME_PAD_SMALL);
    gtk_table_attach_defaults(GTK_TABLE(table), frame, 0, 1, 1, 4);
    pilotCfgWindow = createPilotCfgWindow();
    gtk_container_add(GTK_CONTAINER(frame), pilotCfgWindow);

    frame = gtk_frame_new (_("Pilot Device"));
    gtk_container_border_width(GTK_CONTAINER(frame), GNOME_PAD_SMALL);
    gtk_table_attach_defaults(GTK_TABLE(table), frame, 1, 2, 0, 3);
    deviceCfgWindow = createDeviceCfgWindow();
    gtk_container_add(GTK_CONTAINER(frame), deviceCfgWindow);

    frame = gtk_frame_new (_("Synchronize Action"));
    gtk_container_border_width(GTK_CONTAINER(frame), GNOME_PAD_SMALL);
    gtk_table_attach_defaults(GTK_TABLE(table), frame, 1, 2, 3, 4);
    syncCfgWindow = createSyncCfgWindow();
    gtk_container_add(GTK_CONTAINER(frame), syncCfgWindow);

    gtk_signal_connect(GTK_OBJECT(capplet), "try",
		       GTK_SIGNAL_FUNC(doTrySettings), curState);
    gtk_signal_connect(GTK_OBJECT(capplet), "revert",
		       GTK_SIGNAL_FUNC(doRevertSettings), origState);
    gtk_signal_connect(GTK_OBJECT(capplet), "ok",
		       GTK_SIGNAL_FUNC(doSaveSettings), curState);
    gtk_signal_connect(GTK_OBJECT(capplet), "help",
		       GTK_SIGNAL_FUNC(gnome_help_display), 
		       &help_entry);
}

int
main( int argc, char *argv[] )
{
	/* we're a capplet */
	switch(gnome_capplet_init ("gpilot control applet", VERSION, argc, argv, 
				   NULL,
				   0, NULL)) {
	case 1: return 0; break;
	case -1: g_error(_("Error initializing gpilotd capplet")); break;
	default: break;
	}

	/* put all code to set things up in here */
	if (loadPilotState(&origState) < 0) {
		g_message(_("Error loading pilot state, aborting"));
		exit(1);
	}

	curState = dupPilotState(origState);

	if (gpilotd_init(&argc,argv)!=0) {
		run_error_dialog(_("Cannot initialze the GnomePilot Daemon"));
		g_error(_("Cannot initialze the GnomePilot Daemon"));
		return -1;
	}

	if (gpilotd_connect()!=0) {
		run_error_dialog(_("Cannot connect to the GnomePilot Daemon"));
		g_error(_("Cannot connect to the GnomePilot Daemon"));
		return -1;
	}
	if(origState->pilot!=NULL) {
		g_message("pilot = %s",origState->pilot->name);
		gpilotd_monitor_on(origState->pilot->name);
		gpilotd_notify_on(GNOME_Pilot_NOTIFY_CONNECT);
		gpilotd_notify_on(GNOME_Pilot_NOTIFY_DISCONNECT);
	}

	pilot_capplet_setup();
	setHostCfg(hostCfgWindow, origState);
	setDeviceCfg(deviceCfgWindow, origState);
	setPilotCfg(pilotCfgWindow, origState);
	setSyncCfg(syncCfgWindow, origState);
	gtk_widget_show_all(capplet);

	if(curState->device==NULL || curState->pilot==NULL) {
		gtk_widget_set_sensitive(gtk_object_get_data(GTK_OBJECT(pilotCfgWindow),"get_from_pilot"),FALSE);
		gtk_widget_set_sensitive(gtk_object_get_data(GTK_OBJECT(pilotCfgWindow),"send_to_pilot"),FALSE);
	}

	/* done setting up, now run main loop */
	capplet_gtk_main();

	return 0;
}    
