#include "config.h"
#include "gpilotd-conduit-mgmt.h"

#define USE_GMODULE
#ifndef USE_GMODULE
#include <dlfcn.h>
#endif

/* the LOG macro cannot by used without a format */
#ifdef LIBCONDUIT_MGMT_DEBUG
#define LOG(format,args...) g_log (G_LOG_DOMAIN, \
                                   G_LOG_LEVEL_MESSAGE, \
                                   "lib-gpilotd-conduit-mgmt: %s: "##format, G_GNUC_PRETTY_FUNCTION, ##args)
#else
#define LOG(format,args...)
#endif

/*********************************************************/

/**
   FIXME: Klugde, simply opens the gpilotd config file and extracts
   [Pilot0/pilotid]
 */

guint32 
gpilotd_conduit_mgmt_get_pilot_id() 
{
	return gnome_config_get_int("gnome-pilot.d/gpilotd/Pilot0/pilotid");
}

GnomePilotConduitMgmt*
gpilotd_conduit_mgmt_new(gchar *libname) 
{
	gchar *tmp;
	char *other_dirs[] = {"/usr/local/lib","/usr/lib","/lib","/opt/gnome/lib"};
	int other_dirs_num;
  
	other_dirs_num = 4;
	g_return_val_if_fail(libname!=NULL,NULL);

	LOG("v4 libname = %s",libname);
	tmp = g_strdup_printf("gnome-pilot/conduits/lib%s.so",libname);
  
	/* does it exist in gnome_lib_dir ? */
	if(!gnome_libdir_file(tmp)) {
		gchar *tmp2;
		int c;
		/* no it didn't try some other standard dirs */
		for(c=0;c<other_dirs_num;c++) {
			tmp2 = g_strdup_printf("%s/%s",other_dirs[c],tmp);
			LOG("trying %s",tmp2);
			if(access(tmp2,R_OK)==0) {
				GnomePilotConduitMgmt* retval;
				retval = gpilotd_conduit_mgmt_new_absolute(tmp2);
				g_free(tmp2);
				return retval;
			}
			g_free(tmp2);
		}
		LOG("%s","cannot find conduit file!");
		g_free(tmp);
		return NULL;
	} else {
		GnomePilotConduitMgmt* retval;
		LOG("absolute = %s",gnome_libdir_file(tmp));
		retval = gpilotd_conduit_mgmt_new_absolute(gnome_libdir_file(tmp));
		g_free(tmp);
		return retval;
	}
}

GnomePilotConduitMgmt*
gpilotd_conduit_mgmt_new_absolute(gchar *name)
{
	GnomePilotConduitMgmt *mgmt;
	GnomePilotConduit *conduit;

	g_return_val_if_fail(name!=NULL,NULL);
	g_return_val_if_fail(g_file_test(name,G_FILE_TEST_ISFILE|G_FILE_TEST_ISLINK),NULL);
	
	mgmt = g_new(GnomePilotConduitMgmt,1);  
	mgmt->library_file = g_strdup(name);
	
#ifdef USE_GMODULE
	if(g_module_supported()!=TRUE) {
		g_error(_("Module loading not supported."));
		return NULL;
	}
	if((mgmt->dlhandle = g_module_open(name,G_MODULE_BIND_LAZY))) {
		if(!g_module_symbol(mgmt->dlhandle,"conduit_get_gpilot_conduit",
				    (gpointer)&mgmt->get_gpilot_conduit)) {
			g_error(_("Malformed conduit, cannot find "
				  "conduit_get_gpilot_conduit: %s"),g_module_error());
			g_module_close(mgmt->dlhandle);
			return NULL;
		}
		if(!g_module_symbol(mgmt->dlhandle,"conduit_destroy_gpilot_conduit",
				    (gpointer)&mgmt->destroy_gpilot_conduit)) {
			g_error(_("Malformed conduit, cannot find "
				  "conduit_destroy_gpilot_conduit: %s"),g_module_error());
			g_module_close(mgmt->dlhandle);
			return NULL;
		}
#else
	if((mgmt->dlhandle = dlopen(name,RTLD_LAZY))) {
		if((mgmt->get_gpilot_conduit = dlsym(mgmt->dlhandle,"conduit_get_gpilot_conduit"))==NULL) {
			g_error(_("Malformed conduit, cannot find "
				  "conduit_get_gpilot_conduit: %s"),g_module_error());
			dlclose(mgmt->dlhandle);
			return NULL;
		}
		if((mgmt->destroy_gpilot_conduit = dlsym(mgmt->dlhandle,"conduit_destroy_gpilot_conduit"))==NULL) {
			g_error(_("Malformed conduit, cannot find "
				  "conduit_destroy_gpilot_conduit: %s"),g_module_error());
			dlclose(mgmt->dlhandle);
			return NULL;
		}
#endif
		
		conduit = mgmt->get_gpilot_conduit(0);
		if(conduit==NULL) {
			g_error(_("Could not instantiate conduit"));
			return NULL;
		}

		mgmt->name = g_strdup(gnome_pilot_conduit_get_name(conduit));
		mgmt->destroy_gpilot_conduit(conduit);
		
		/* check that the conduit has a name, if not, it cannot be used, 
		   since adding it to config lists is impossible */
		if (mgmt->name == NULL) {
			gpilotd_conduit_mgmt_free(mgmt);
			g_warning(_("Conduit %s has no name"),name);
			return NULL;
		}
		
#ifdef USE_GMODULE
		g_module_close(mgmt->dlhandle);
#else
		dlclose(mgmt->dlhandle);
#endif
	} else {
#ifdef USE_GMODULE
		g_warning("g_module_open returned %s",g_module_error());
#else
		g_warning("dlopen returned %s",dlerror());
#endif
		g_error(_("%s is not a conduit"),name);
		return NULL;
	}
	
	return mgmt;
}

void 
gpilotd_conduit_mgmt_free(GnomePilotConduitMgmt *conduit) {

  g_return_if_fail(conduit);
  g_module_close(conduit->dlhandle);
  g_free(conduit->name);
  g_free(conduit->library_file);
  g_free(conduit);
}

void 
gpilotd_conduit_mgmt_enable(GnomePilotConduitMgmt *conduit,
			    gint pilot_id,
			    GnomePilotConduitSyncType synctype) 
{
	gchar *tmp;
	gchar **conduit_name;
	char **conduit_name_copy;
	int num_conduits,cnt;
	
	g_return_if_fail(conduit);

	/* enable conduit, if it wasn't, otherwise, just write
	   the sync type in case it was changed. */
	if(gpilotd_conduit_mgmt_is_enabled(conduit,pilot_id)==FALSE) {
	
	  tmp = g_strdup_printf("/gnome-pilot.d/conduits%d/General/",pilot_id);
	  LOG("prefix = %s",tmp);
	
	  /* load the list of conduits */
	  gnome_config_push_prefix(tmp);
	  gnome_config_get_vector("conduits",&num_conduits,&conduit_name);
	  LOG("num_conduits = %d",num_conduits);
	
	  /* copy the list and attach new conduit */
	  conduit_name_copy = g_new0(gchar*,num_conduits+2);
	  for(cnt=0;cnt<num_conduits;cnt++) {
		LOG("duping %d %s",cnt,conduit_name[cnt]);
		conduit_name_copy[cnt] = g_strdup(conduit_name[cnt]);
		g_free(conduit_name[cnt]);
	  }
	  g_free(conduit_name);
	  LOG("adding %s as element %d",conduit->name,num_conduits);
	  conduit_name_copy[num_conduits] = g_strdup(conduit->name);
	
	  /* write new list */
	  LOG("writing new conduits list, size %d",num_conduits+1);
	  gnome_config_set_vector("conduits",num_conduits+1,conduit_name_copy);
	
	  /* free the copy */
	  LOG("freeing %d items",num_conduits+1);
	  for(cnt=0;cnt<num_conduits+1;cnt++) {
		LOG("should contain %s",conduit_name_copy[cnt]);
		g_free(conduit_name_copy[cnt]);
	  }
	  g_free(conduit_name_copy);
	
	  gnome_config_pop_prefix();
	
	  g_free(tmp);
	}

	/* make section for conduit and write the libfile */
        tmp = g_strdup_printf("/gnome-pilot.d/conduits%d/%s/",pilot_id,conduit->name);
	LOG("prefix = %s",tmp);
	gnome_config_push_prefix(tmp);
	gnome_config_set_string("libfile",conduit->library_file);
	gnome_config_set_string("sync_type",
				gnome_pilot_conduit_sync_type_int_to_str(synctype));
	gnome_config_pop_prefix();
	gnome_config_sync();
	g_free(tmp);
}
 
/* enable the conduit for a specific pilot, and set the conduits
   synctype and 1st sync synctype. Slow is only valid for 
   firstsynctype = GnomePilotConduitSyncTypeSynchronize */
void gpilotd_conduit_mgmt_enable_with_first_sync(GnomePilotConduitMgmt *conduit,
						 gint pilot_id,
						 GnomePilotConduitSyncType synctype,
						 GnomePilotConduitSyncType firstsynctype,
						 gboolean slow) 
{
	gchar *tmp;

	g_return_if_fail(conduit);
	/* This ensures the section is present in the conduitsPILOT file */
	gpilotd_conduit_mgmt_enable(conduit,pilot_id,synctype);
	
	/* add the first_sync info */
        tmp = g_strdup_printf("/gnome-pilot.d/conduits%d/%s/",pilot_id,conduit->name);
	gnome_config_push_prefix(tmp);
	gnome_config_set_string("first_sync_type",
				gnome_pilot_conduit_sync_type_int_to_str(firstsynctype));
	if(firstsynctype==GnomePilotConduitSyncTypeSynchronize) 
		gnome_config_set_bool("slow_sync",slow);
	else
		gnome_config_clean_key("slow_sync");
	gnome_config_pop_prefix();
	gnome_config_sync();
	g_free(tmp);
}

void gpilotd_conduit_mgmt_remove_first_sync(GnomePilotConduitMgmt *conduit,
					    gint pilot_id) 
{
	gchar *tmp;

	g_return_if_fail(conduit);
	g_return_if_fail(gpilotd_conduit_mgmt_is_enabled(conduit,pilot_id));
	/* add the first_sync info */
        tmp = g_strdup_printf("/gnome-pilot.d/conduits%d/%s/",pilot_id,conduit->name);
	if (gnome_config_has_section(tmp)) {
		gnome_config_push_prefix(tmp);
		gnome_config_clean_key("first_sync_type");
		gnome_config_clean_key("slow_sync");
		gnome_config_pop_prefix();
	}
	gnome_config_sync();
	g_free(tmp);
}


void 
gpilotd_conduit_mgmt_disable(GnomePilotConduitMgmt *conduit,gint pilot_id) 
{
  gchar *tmp;
  gchar **conduit_name;
  char **conduit_name_copy;
  int num_conduits,num_conduits_copy,cnt;

  g_return_if_fail(conduit);

  if(gpilotd_conduit_mgmt_is_enabled(conduit,pilot_id)==FALSE) return;

  tmp = g_strdup_printf("/gnome-pilot.d/conduits%d/General/",pilot_id);
  LOG("prefix = %s",tmp);
  
  /* load the list of conduits */
  gnome_config_push_prefix(tmp);
  gnome_config_get_vector("conduits",&num_conduits,&conduit_name);
 
  /* copy the list, excluding the one to delete */
  conduit_name_copy = g_new0(char*,num_conduits-1);
  num_conduits_copy = 0;
  for(cnt=1;cnt<num_conduits;cnt++) {
	  if(g_strcasecmp(conduit_name[cnt],conduit->name)) {
		  LOG("keeping %d:%s",cnt,conduit_name[cnt]);
		  conduit_name_copy[num_conduits_copy]=g_strdup(conduit_name[cnt]);
		  num_conduits_copy++;
	  } else {
		  LOG("skipping %d:%s",cnt,conduit_name[cnt]);
	  }
	  g_free(conduit_name[cnt]);
  }
  g_free(conduit_name);

  /* save new list */
  LOG("num_conduits_copy = %d",num_conduits_copy);
  if ( num_conduits_copy > 0 )
	  gnome_config_set_vector("conduits",num_conduits_copy,conduit_name_copy);
  else
	  gnome_config_clean_key("conduits");
  
  /* free copy */
  for(cnt=0;cnt<num_conduits_copy;cnt++) g_free(conduit_name_copy[cnt]);
  g_free(conduit_name_copy);
  
  gnome_config_pop_prefix();

  /* nuke the conduits config section */
  g_free(tmp);
  tmp = g_strdup_printf("/gnome-pilot.d/conduits%d/%s/",pilot_id,conduit->name);
  LOG("prefix = %s",tmp);
  gnome_config_clean_section(tmp);

  gnome_config_sync();
  g_free(tmp);
}

gboolean 
gpilotd_conduit_mgmt_is_enabled(GnomePilotConduitMgmt *conduit,gint pilot_id) 
{
  gchar *tmp;
  int num_conduits,cnt;
  gchar **conduit_name;
  gboolean retval;
  
  g_return_val_if_fail(conduit,FALSE);
  retval = FALSE;

  tmp = g_strdup_printf("/gnome-pilot.d/conduits%d/General/",pilot_id);
  LOG("prefix = %s",tmp);

  /* load the list */
  gnome_config_push_prefix(tmp);
  gnome_config_get_vector("conduits",&num_conduits,&conduit_name);
  gnome_config_pop_prefix();

  LOG("number_of_conduits = %d",num_conduits);

  /* loop, compare and free */
  for(cnt=0;cnt<num_conduits;cnt++) {
    LOG("g_str_equal(%s,%s)",conduit_name[cnt] ,conduit->name);
    if(g_str_equal(conduit_name[cnt] ,conduit->name)) {
      LOG("match on %s, it is enabled",conduit_name[cnt]);
      retval = TRUE;
    } 
    g_free(conduit_name[cnt]);
  }
  g_free(conduit_name);
  g_free(tmp);
  return retval;
}

gboolean gpilotd_conduit_mgmt_get_sync_type(GnomePilotConduitMgmt *conduit,gint pilot_id,GnomePilotConduitSyncType *synctype)
{
  gchar *tmp;
  gchar *s;
  gboolean retval=FALSE;
  g_return_val_if_fail(conduit,FALSE);

  if(gpilotd_conduit_mgmt_is_enabled(conduit,pilot_id)==FALSE) return FALSE;

  g_return_val_if_fail(synctype!=NULL,FALSE);
  
  tmp = g_strdup_printf("/gnome-pilot.d/conduits%d/%s/sync_type",pilot_id,conduit->name);
  s=gnome_config_get_string(tmp);
  if(s) {
    retval=TRUE;
    if (strcmp (s, "synchronize") == 0) {
      *synctype=GnomePilotConduitSyncTypeSynchronize;
    } else if (strcmp (s, "copy_to_pilot") == 0) {
      *synctype=GnomePilotConduitSyncTypeCopyToPilot;
    } else if (strcmp (s, "copy_from_pilot") == 0) {
      *synctype=GnomePilotConduitSyncTypeCopyFromPilot;
    } else if (strcmp (s, "merge_to_pilot") == 0) {
      *synctype=GnomePilotConduitSyncTypeMergeToPilot;
    } else if (strcmp (s, "merge_from_pilot") == 0) {
      *synctype=GnomePilotConduitSyncTypeMergeFromPilot;
    } else if (strcmp (s, "custom") == 0) {
      *synctype=GnomePilotConduitSyncTypeCustom;
    } else {
      LOG("gpilotd_conduit_mgmt_get_sync_type %s is unkown type",s);
      retval=FALSE;
    }
  }
  g_free(tmp);
  return retval;
}

void 
gpilotd_conduit_mgmt_config_push(GnomePilotConduitMgmt *conduit) 
{
	g_warning("Welcome to the unimplemented method gpilotd_conduit_mgmt_config_push");
	g_assert_not_reached();
}

void 
gpilotd_conduit_mgmt_drop_all() 
{
}
