/* gpilotd control applet ('capplet') for use with the GNOME control center */
/*                                                                          */
/* Copyright 1998 Red Hat Software                                          */
/* Author: Michael Fulbright <msf@redhat.com>                               */
/* Author: Vadim Strizhevsky <vadim@optonline.net>                          */

#include <config.h>
#include <pwd.h>
#include <sys/types.h>
#include <signal.h>
#include <ctype.h>
#include <gnome.h>
#include <sys/stat.h>

#include <config.h>
#include <capplet-widget.h>
#include <gpilotd/gnome-pilot-client.h>
#include <glade/glade.h>
#include <libgnorba/gnorba.h>
#include "pilot.h"

/* Druid declarations */
static void gpilotd_druid_show(gchar *);
static void druid_gpilotd_userinfo_requested(const gchar *,const GNOME_Pilot_PilotUser *); 
static void druid_gpilotd_request_completed(const gchar *, unsigned long );

static gboolean druid_on=FALSE;
static gboolean druid_prog=FALSE;
static gboolean druid_request_done=FALSE;
static GtkWidget *mainDruidWindow;

/* tell changes callbacks to ignore changes or not */
static gboolean ignore_changes=FALSE;

static gchar * glade_file=NULL;

/* capplet widget */
static GtkWidget *capplet=NULL;

/* device/pilot configuration windows */
static GtkWidget *deviceCfgWindow=NULL;
static GtkWidget *pilotCfgWindow=NULL;
static GtkMenu   *pilotPopupMenu=NULL;
static GtkMenu   *devicePopupMenu=NULL;

static GtkWidget *chooseDeviceDialog=NULL;

static GtkWidget *pilotCfgDialog=NULL;
static GtkWidget *deviceCfgDialog=NULL;

static GtkWidget *dialogWindow=NULL;

/* old pilot state */
static PilotState *origState=NULL;

/* current editted state */
static PilotState *curState=NULL;

static GnomePilotClient *gpc = NULL;

const guint  speedList[] = {9600, 19200, 38400, 57600, 115200, 0};
#define  DEFAULT_SPEED_INDEX  3  /* Default to 57600 */

static void doTrySettings(GtkWidget *widget, PilotState *pilotState);
static void doRevertSettings(GtkWidget *widget, PilotState *pilotState);
static void doSaveSettings(GtkWidget *widget, PilotState *pilotState);

static void readHostCfg(GtkWidget *w, PilotState **state);


static void setHostCfg(GtkWidget *w, PilotState *state);
static void setDeviceCfg(GtkWidget *w, PilotState *state);
static void setPilotCfg(GtkWidget *w, PilotState *state);

static void readDeviceDialogCfg(GtkWidget *w, GPilotDevice *device);
static void setDeviceDialogCfg(GtkWidget *w, GPilotDevice *device);

static void readPilotDialogCfg(GtkWidget *w, GPilotPilot *pilot);
static void setPilotDialogCfg(GtkWidget *w, GPilotPilot *pilot);

void run_error_dialog(gchar *,...);
gboolean run_question_dialog(gchar *,...);


static void pilots_edit_cb(GtkButton* button,GtkWidget *clist);
static void devices_edit_cb(GtkButton* button,GtkWidget *clist); 


static void 
gpilotd_request_completed(GnomePilotClient* client, const gchar *id,unsigned long handle) 
{
	g_message("%s completed %ld",id,handle);
	if(druid_on) druid_gpilotd_request_completed(id,handle);
	else gnome_dialog_close(GNOME_DIALOG(dialogWindow));
}

static void 
gpilotd_userinfo_requested(GnomePilotClient* client, const gchar *device,const GNOME_Pilot_PilotUser *user) 
{
	if(druid_on) {
		druid_gpilotd_userinfo_requested(device,user);
	} else {
	GPilotPilot *tmpPilot=g_new0(GPilotPilot,1);

	g_message("device %s sent userinfo",device);
	g_message("user->userID   = %d",user->userID);
	g_message("user->username = %s",user->username);


	readPilotDialogCfg(pilotCfgDialog,tmpPilot);
	tmpPilot->pilot_id = user->userID;
	if(tmpPilot->pilot_username) g_free(tmpPilot->pilot_username);
	tmpPilot->pilot_username = g_strdup(user->username);
	
	setPilotDialogCfg(pilotCfgDialog,tmpPilot);
	gnome_dialog_close(GNOME_DIALOG(dialogWindow));
	
	g_free(tmpPilot);
	}
}


static void saveSettingsAndRestartDaemon(PilotState *state) 
{
	savePilotState(state);
	/* FORCE the gpilotd to reread the settings */
	gnome_pilot_client_pause_daemon(gpc);
	gnome_pilot_client_unpause_daemon(gpc);
}

static void
doTrySettings(GtkWidget *widget, PilotState *pilotState)
{
	GList *tmp;
	g_message("trying settings");
	
	readHostCfg(capplet, &pilotState);

	tmp=pilotState->devices;
	while(tmp!=NULL){
		GPilotDevice *device=(GPilotDevice*)tmp->data;
		/* Not possible now? */
		if(device->port==NULL) {
			run_error_dialog(_("Please specify a cradle port"));
			return;
		}
		g_message("checking rw on %s",device->port);
		if(access(device->port,R_OK|W_OK)) {
			char *tmp;
			tmp = g_strdup_printf("%s\n%s(%s)\n%s",
					      _("Read/Write permissions failed on"),
					      device->name,device->port,
					      _("Check the permissions on the device and retry"));	  
			run_error_dialog(tmp);
			g_free(tmp);
			return ;
		}
		tmp=tmp->next;
	}

	/* need to HUP the current gpilotd to get it to reread settings */
	saveSettingsAndRestartDaemon(pilotState);
}

static void
doSaveSettings(GtkWidget *widget, PilotState *pilotState)
{
	g_message("Saving state");
	doTrySettings(widget, pilotState);
}


static void
doRevertSettings(GtkWidget *widget, PilotState *pilotState)
{
	g_message("reverting settings to original");

	setHostCfg(capplet, pilotState);
	setDeviceCfg(deviceCfgWindow, pilotState);
	setPilotCfg(pilotCfgWindow, pilotState);
	saveSettingsAndRestartDaemon(pilotState);
}


static void
insert_numeric_callback (GtkEditable    *editable, const gchar    *text,
		      gint len, gint *position, void *data)
{
	gint i;

	for (i=0; i<len; i++) {
		if (!isdigit(text[i])) {
			gtk_signal_emit_stop_by_name(GTK_OBJECT(editable), "insert_text");
			return;
	}
    }
}

static void
statechange_cb(GtkEditable    *editable, const gchar    *text,
		      gint            length, gint           *position,
		      void *data)
{
	if (!ignore_changes) {
		capplet_widget_state_changed(CAPPLET_WIDGET (capplet), TRUE);
	}
}


static void
insert_syncpc_id(GtkEditable    *editable, const gchar    *text,
		      gint            length, gint           *position,
		      void *data)
{
	if (!ignore_changes) {
		capplet_widget_state_changed(CAPPLET_WIDGET (capplet), TRUE);
	}
}

static void
insert_username_callback (GtkEditable    *editable, const gchar    *text,
		      gint len, gint *position, void *data)
{
	gint i;
	gchar *curname;

	/* need to make sure that existing entry starts with a letter */
	/* since valid usernames must start with a letter             */
	curname = gtk_entry_get_text(GTK_ENTRY(editable));
	if (*curname == '\0' && len > 0) {
		if (!isalpha(text[0])) {
			gtk_signal_emit_stop_by_name(GTK_OBJECT(editable), "insert_text");
			return;
		}
	} else {
		/* rest of username can be alphanumeric */
		for (i=0; i<len; i++) {
			if (!isalnum(text[i])) {
				gtk_signal_emit_stop_by_name(GTK_OBJECT(editable), 
							     "insert_text");
				return;
			}
		}
	}
}
static void
insert_device_callback (GtkEditable    *editable, const gchar    *text,
		      gint len, gint *position, void *data)
{
	gint i;
	gchar *curname;

	curname = gtk_entry_get_text(GTK_ENTRY(editable));
	if (*curname == '\0' && len > 0) {
		if (text[0]!='/') {
			gtk_signal_emit_stop_by_name(GTK_OBJECT(editable), "insert_text");
			return;
		} 
	} else {
		for(i=0;i<len;i++)
			if(!(isalnum(text[i]) || text[i]=='/')) {
				gtk_signal_emit_stop_by_name(GTK_OBJECT(editable), "insert_text");
				return;
			}
	}
} 

/* called by the sync_type GtkOptionMenu */
static void
activate_sync_type(GtkMenuItem *widget, gpointer data)
{
	gtk_object_set_data(GTK_OBJECT(pilotCfgDialog?pilotCfgDialog:mainDruidWindow),"sync_type",data);
}

/* called by the speed GtkOptionMenu */
static void
activate_speed(GtkMenuItem *widget, gpointer data)
{
	gtk_object_set_data(GTK_OBJECT(deviceCfgDialog?deviceCfgDialog:mainDruidWindow),"speed",data);
}

static void cancel_sync(GnomeDialog *w,gpointer data) 
{
	g_message("cancelling %d",GPOINTER_TO_INT(data));
	gnome_pilot_client_remove_request(gpc,GPOINTER_TO_INT(data));  
}

static void run_sync_dialog(GPilotDevice* device, gint handle) 
{
	gchar *txt =g_strdup_printf(_("Press synchronize on the %s (%s)\n"
				      "or cancel the operation."),device->name,device->port);
	dialogWindow = gnome_message_box_new(txt,GNOME_MESSAGE_BOX_GENERIC,GNOME_STOCK_BUTTON_CANCEL,NULL);
	g_free(txt);
	gnome_dialog_button_connect(GNOME_DIALOG(dialogWindow),0,cancel_sync,GINT_TO_POINTER(handle));
	gnome_dialog_run_and_close(GNOME_DIALOG(dialogWindow));
}

void run_error_dialog(gchar *mesg,...) 
{
	char *tmp;
	va_list ap;

	va_start(ap,mesg);
	tmp = g_strdup_vprintf(mesg,ap);
	dialogWindow = gnome_message_box_new(tmp,GNOME_MESSAGE_BOX_ERROR,GNOME_STOCK_BUTTON_OK,NULL);
	gnome_dialog_run_and_close(GNOME_DIALOG(dialogWindow));
	va_end(ap);
	g_free(tmp);
}

gint run_question_dialog(gchar *mesg,...) 
{
	char *tmp;
	gint button;
	va_list ap;
  
	va_start(ap,mesg);
	tmp = g_strdup_vprintf(mesg,ap);
	dialogWindow = gnome_message_box_new(tmp,GNOME_MESSAGE_BOX_QUESTION,
					     GNOME_STOCK_BUTTON_YES,GNOME_STOCK_BUTTON_NO,NULL);
	button=gnome_dialog_run_and_close(GNOME_DIALOG(dialogWindow));
	va_end(ap);
	g_free(tmp);
	return button==0?TRUE:FALSE;
}

static void
activate_device_menu(GtkMenuItem *widget, gpointer data)
{
	gtk_object_set_data(GTK_OBJECT(chooseDeviceDialog),"device",data);	
}

static GPilotDevice*
run_choose_dialog(PilotState *state)
{
	GList *tmp;
	GtkWidget *optionMenu,*menuItem,*menu;
	GPilotDevice *dev;
	/* first time create dialog */
	if(chooseDeviceDialog==NULL) {
		GladeXML *xml;
		xml=glade_xml_new(glade_file,"ChooseDevice");
		chooseDeviceDialog=glade_xml_get_widget(xml,"ChooseDevice");
		optionMenu=glade_xml_get_widget(xml,"device_menu");
		gtk_object_set_data(GTK_OBJECT(chooseDeviceDialog),"device_menu",optionMenu);
	} else {
		optionMenu=gtk_object_get_data(GTK_OBJECT(chooseDeviceDialog),"device_menu");
	}
	
	menu = gtk_menu_new();

	tmp=state->devices;
	while(tmp!=NULL){
		dev=(GPilotDevice*)tmp->data;
		menuItem = gtk_menu_item_new_with_label(dev->port);
		gtk_widget_show(menuItem);
		gtk_signal_connect(GTK_OBJECT(menuItem),"activate",
				   GTK_SIGNAL_FUNC(activate_device_menu),
				   dev);
		gtk_menu_append(GTK_MENU(menu),menuItem);
		tmp=tmp->next;
	}
	gtk_option_menu_set_menu(GTK_OPTION_MENU(optionMenu),menu);
	gtk_option_menu_set_history(GTK_OPTION_MENU(optionMenu),0);
	gtk_object_set_data(GTK_OBJECT(chooseDeviceDialog),"device",state->devices->data);	
	if(gnome_dialog_run_and_close(GNOME_DIALOG(chooseDeviceDialog))==0) {
		dev=(GPilotDevice*)gtk_object_get_data(GTK_OBJECT(chooseDeviceDialog),"device");
	} else {
		dev=NULL;
	}
	return dev;
}

static GPilotDevice *
pick_pilot_device(void)
{
	GPilotDevice *dev=NULL;
	if(curState->devices==NULL) {
		run_error_dialog(_("You must have at least one device setup"));
	} else if(g_list_length(curState->devices)==1) {
		dev=(GPilotDevice*)curState->devices->data;
	} else {
		dev=run_choose_dialog(curState);
	}
	return dev;
}

static void 
get_from_pilot_cb(GtkButton* button,gpointer whatever) 
{
	int handle;
	GPilotDevice *dev=pick_pilot_device();
	if(dev!=NULL){
		doTrySettings(capplet, curState);  
		if(gnome_pilot_client_get_user_info(gpc,dev->name,GNOME_Pilot_IMMEDIATE,0,&handle)==GPILOTD_OK) {
			run_sync_dialog(dev,handle);
		} else {
			run_error_dialog(_("The request to get pilot id failed"));
		}
	}
}

static void 
send_to_pilot_cb(GtkButton* button,gpointer whatever) 
{
	int handle;
	GNOME_Pilot_PilotUser user;

	GPilotPilot *tmpPilot=g_new0(GPilotPilot,1);

	GPilotDevice *dev=pick_pilot_device();
	if(dev!=NULL){
		doTrySettings(capplet, curState);  
	
		readPilotDialogCfg(pilotCfgDialog,tmpPilot);

		user.userID = tmpPilot->pilot_id;
		user.username = tmpPilot->pilot_username;

		if(gnome_pilot_client_set_user_info(gpc,
						    dev->name,
						    user,
						    FALSE,
						    GNOME_Pilot_IMMEDIATE,
						    0,
						    &handle)==GPILOTD_OK) {
			run_sync_dialog(dev,handle);
		} else {
			run_error_dialog(_("The request to set pilot id failed"));
		}
	}
	g_free(tmpPilot);
}

static void 
pilots_selection_cb(GtkWidget *clist, gint row, gint column,
		   GdkEventButton *event, gpointer data )
{
	gtk_object_set_data(GTK_OBJECT(clist),"selected",GINT_TO_POINTER(row));
	if(event && event->button==1 && event->type == GDK_2BUTTON_PRESS) {
		pilots_edit_cb(NULL,clist);
	}
}
static void 
clist_unselection_cb(GtkWidget *clist, gint row, gint column,
		   GdkEventButton *event, gpointer data )
{
	gtk_object_set_data(GTK_OBJECT(clist),"selected",GINT_TO_POINTER(-1));
}
static void 
devices_selection_cb(GtkWidget *clist, gint row, gint column,
		   GdkEventButton *event, gpointer data )
{
	gtk_object_set_data(GTK_OBJECT(clist),"selected",GINT_TO_POINTER(row));
	if(event && event->button==1 && event->type == GDK_2BUTTON_PRESS) {
		devices_edit_cb(NULL,clist);
	}
}

static void
clist_button_press(GtkCList *list, GdkEventButton *event, GtkMenu *menu)
{
	if (event && event->button==3) {
		gint  row, col;
		if (gtk_clist_get_selection_info(list,event->x,event->y,&row,&col )) {
			gtk_object_set_data(GTK_OBJECT(list),"selected",GINT_TO_POINTER(row));
			gtk_clist_select_row(list, row, 0);
			
			gtk_menu_popup (menu, NULL, NULL, NULL, NULL, 
					event->button, event->time);
		}
	}
}

static const gchar* 
sync_type_int_to_str(GnomePilotConduitSyncType e) 
{
	switch(e) {
	case GnomePilotConduitSyncTypeSynchronize:    return _("Synchronize");
	case GnomePilotConduitSyncTypeCopyFromPilot:  return _("Copy from pilot");
	case GnomePilotConduitSyncTypeCopyToPilot:    return _("Copy to pilot");
	case GnomePilotConduitSyncTypeMergeFromPilot: return _("Merge from pilot");
	case GnomePilotConduitSyncTypeMergeToPilot:   return _("Merge to pilot");
	case GnomePilotConduitSyncTypeCustom: 
	case GnomePilotConduitSyncTypeNotSet:     
	default:                                      return _("Use conduit settings");
	}
}

static gint compare_pilot_info(GPilotPilot* pilot1, GPilotPilot *pilot2)
{
	if(pilot1!=pilot2 && ( pilot1->pilot_id==pilot2->pilot_id ||
			       strcmp(pilot1->name,pilot2->name)==0)) return 0;
	else return 1;
}

static gint compare_device_info(GPilotDevice* device1, GPilotDevice *device2)
{
	if(device1!=device2 && (strcmp(device1->port, device2->port)==0 ||
				strcmp(device1->name, device2->name)==0)) return 0;
	else return 1;
}
static gboolean
check_base_directory(gchar *dir_name)
{
	gboolean ret = TRUE;
	/* check basedir validity */
	
	if(mkdir(dir_name, 0700) < 0 ) {
		struct stat buf;
		gchar *errstr;
		switch(errno) {
		case EEXIST: 
			stat(dir_name, &buf);
			if(S_ISDIR(buf.st_mode)) {  
				if(!(buf.st_mode & (S_IRUSR | S_IWUSR |S_IXUSR))) {
					run_error_dialog(_("The specified base directory exists but has the wrong permissions.\n"
							   "Please fix or choose another directory"));
					ret=FALSE;
				}
			} else {
				run_error_dialog(_("The specified base directory exists but is not a directory.\n"
						   "Please make it a directory or choose another directory"));
					ret=FALSE;
			}
			break;
			
		case EACCES:
			run_error_dialog(_("It wasn't possible to create the specified base directory.\n"
					   "Please verify the permitions on the specified path or choose another directory"));
					ret=FALSE;
			break;
		case ENOENT:
			run_error_dialog(_("The path specified for the base directory is invalid.\n"
					   "Please choose another directory"));
					ret=FALSE;
			break;
		default:
			errstr = strerror(errno);
			run_error_dialog(errstr);
			ret=FALSE;
		}
	}
	return ret;
}
	

static void 
pilots_add_cb(GtkButton* button,gpointer whatever) 
{
	gchar buf[10];
	const gchar *row[4];
	gint i;
	GPilotPilot *pilot;
	
	setPilotDialogCfg(pilotCfgDialog,NULL);
	while(gnome_dialog_run(GNOME_DIALOG(pilotCfgDialog))==0) {	
		pilot =g_new0(GPilotPilot,1);
		readPilotDialogCfg(pilotCfgDialog,pilot);

		/* check the information entered */
		if(g_list_find_custom(curState->pilots,
				      pilot,
				      (GCompareFunc)compare_pilot_info)!=NULL) {
			run_error_dialog(_("Pilot with this id or name already exists."));
			g_free(pilot);
		} else {
			if(!check_base_directory(pilot->sync_options.basedir)) {
				g_free(pilot);
				continue;
			}
			row[0]=pilot->name;
			g_snprintf(buf,sizeof(buf),"%d",pilot->pilot_id);
			row[1]=buf;
			row[2]=pilot->pilot_username;
			row[3]=sync_type_int_to_str(pilot->sync_options.default_sync_action);
			i=gtk_clist_append(GTK_CLIST(pilotCfgWindow),(gchar **)row);
			gtk_clist_set_row_data(GTK_CLIST(pilotCfgWindow),i,pilot);		
			curState->pilots=g_list_append(curState->pilots,pilot);
			capplet_widget_state_changed(CAPPLET_WIDGET (capplet), TRUE);
			break;
		}
	}
	gnome_dialog_close(GNOME_DIALOG(pilotCfgDialog));
}

static void 
pilots_edit_cb(GtkButton* button,GtkWidget *clist) 
{
	gchar buf[20];
	gint pilots_selected = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(clist),"selected"));
	if(pilots_selected>=0) {
		GPilotPilot *pilot=(GPilotPilot*)gtk_clist_get_row_data(GTK_CLIST(clist),pilots_selected);
		setPilotDialogCfg(pilotCfgDialog,pilot);
		while(gnome_dialog_run(GNOME_DIALOG(pilotCfgDialog))==0) {
			readPilotDialogCfg(pilotCfgDialog,pilot);

			/* check the information entered */
			if(g_list_find_custom(curState->pilots,
					      pilot,
					      (GCompareFunc)compare_pilot_info)!=NULL) {
				run_error_dialog(_("Pilot with this id or name already exists."));
			} else {
				if(!check_base_directory(pilot->sync_options.basedir)) {
					continue;
				}
				gtk_clist_set_text(GTK_CLIST(clist),pilots_selected,0,pilot->name);
				g_snprintf(buf,sizeof(buf),"%d",pilot->pilot_id);
				gtk_clist_set_text(GTK_CLIST(clist),pilots_selected,1,buf);
				gtk_clist_set_text(GTK_CLIST(clist),pilots_selected,2,pilot->pilot_username);
				gtk_clist_set_text(GTK_CLIST(clist),pilots_selected,3,
						   sync_type_int_to_str(pilot->sync_options.default_sync_action));
				capplet_widget_state_changed(CAPPLET_WIDGET (capplet), TRUE);
				break;
			}
		}
		gnome_dialog_close(GNOME_DIALOG(pilotCfgDialog));
	}
}

static void 
pilots_delete_cb(GtkButton* button,GtkWidget *clist) 
{
	gint pilots_selected = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(clist),"selected"));
	if(pilots_selected>=0) {
		GPilotPilot *pilot=(GPilotPilot*)gtk_clist_get_row_data(GTK_CLIST(clist),pilots_selected);
		if(run_question_dialog(_("Are you sure you want to delete %s pilot?"),
				       pilot->name)) {
			gtk_clist_remove(GTK_CLIST(clist),pilots_selected);
			curState->pilots=g_list_remove(curState->pilots,pilot);
			capplet_widget_state_changed(CAPPLET_WIDGET (capplet), TRUE);
		}
	}
}

static void 
devices_edit_cb(GtkButton* button,GtkWidget *clist) 
{
	gchar buf[20];
	gint devices_selected = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(clist),"selected"));
	if(devices_selected>=0) {
		GPilotDevice *device=(GPilotDevice*)gtk_clist_get_row_data(GTK_CLIST(clist),devices_selected);
		setDeviceDialogCfg(deviceCfgDialog,device);
		while(gnome_dialog_run(GNOME_DIALOG(deviceCfgDialog))==0) {
			readDeviceDialogCfg(deviceCfgDialog,device);
			
			/* check the information entered */
			if(g_list_find_custom(curState->devices,
					      device,
					      (GCompareFunc)compare_device_info)!=NULL) {
				
				run_error_dialog(_("Cradle with this port or name already exists."));
			} else {
				gtk_clist_set_text(GTK_CLIST(clist),devices_selected,0,device->name);
				gtk_clist_set_text(GTK_CLIST(clist),devices_selected,1,device->port);
				g_snprintf(buf,sizeof(buf),"%d",device->speed);
				gtk_clist_set_text(GTK_CLIST(clist),devices_selected,2,buf);
				capplet_widget_state_changed(CAPPLET_WIDGET (capplet), TRUE);
				break;
			}
		}
		gnome_dialog_close(GNOME_DIALOG(deviceCfgDialog));
	}
}	
static void 
devices_add_cb(GtkButton* button,gpointer whatever) 
{
	gchar buf[20];
	const gchar *row[3];
	gint i;	
	GPilotDevice *device;

	
	setDeviceDialogCfg(deviceCfgDialog,NULL);
	while(gnome_dialog_run(GNOME_DIALOG(deviceCfgDialog))==0) {
		device=g_new0(GPilotDevice,1);
		readDeviceDialogCfg(deviceCfgDialog,device);
		
		/* check the information entered */
		if(g_list_find_custom(curState->devices,
				      device,
				      (GCompareFunc)compare_device_info)!=NULL) {
			run_error_dialog(_("Cradle with this port or name already exists."));
			g_free(device);
		} else {
			row[0]=device->name;
			row[1]=device->port;
			g_snprintf(buf,sizeof(buf),"%d",device->speed);
			row[2]=buf;
			i=gtk_clist_append(GTK_CLIST(deviceCfgWindow),(gchar **)row);
			gtk_clist_set_row_data(GTK_CLIST(deviceCfgWindow),i,device);
			curState->devices=g_list_append(curState->devices,device);
			capplet_widget_state_changed(CAPPLET_WIDGET (capplet), TRUE);
			break;
		}
	}
	gnome_dialog_close(GNOME_DIALOG(deviceCfgDialog));	
}

static void 
devices_delete_cb(GtkButton* button,GtkWidget *clist) 
{
	gint devices_selected = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(clist),"selected"));
	if(devices_selected>=0) {
		GPilotDevice *device = (GPilotDevice*)gtk_clist_get_row_data(GTK_CLIST(clist),devices_selected);
		if(run_question_dialog(_("Are you sure you want to delete %s(%s) device?"),
				       device->name, device->port)) {
			gtk_clist_remove(GTK_CLIST(clist),devices_selected);
			curState->devices=g_list_remove(curState->devices,device);
			capplet_widget_state_changed(CAPPLET_WIDGET (capplet), TRUE);
		}
	}
}

/* Following object data exists:                                    */
/*                                                                  */
/* "syncpcid" - GtkEntry for the syncpcid                           */
/*                                                                  */
static void 
createHostCfgWindow(GladeXML *xml)
{
	GtkWidget *entry;
	entry = glade_xml_get_widget(xml,"syncpcid_entry");

	gtk_object_set_data(GTK_OBJECT(capplet), "syncpcid", entry);
	gtk_signal_connect(GTK_OBJECT(entry), "insert_text",
			   GTK_SIGNAL_FUNC(insert_numeric_callback),
			   NULL);
	gtk_signal_connect_after(GTK_OBJECT(entry), "insert_text",
				 GTK_SIGNAL_FUNC(insert_syncpc_id),
				 NULL);
	gtk_signal_connect(GTK_OBJECT(entry), "delete_text",
			   GTK_SIGNAL_FUNC(statechange_cb),
		       NULL);

}

static void
setHostCfg(GtkWidget *hostcfg, PilotState *state)
{
	GtkWidget *entry;
	gchar id[40];

	entry = gtk_object_get_data(GTK_OBJECT(hostcfg), "syncpcid");

	g_snprintf(id, sizeof(id), "%d", state->syncPCid);

	ignore_changes = TRUE;
	gtk_entry_set_text(GTK_ENTRY(entry), id);
	ignore_changes = FALSE;
}

/* no error checking yet */
static void
readHostCfg(GtkWidget *hostcfg, PilotState **state)
{
	GtkWidget *entry;
	gchar *id;

	if (*state){
		entry = gtk_object_get_data(GTK_OBJECT(hostcfg), "syncpcid");
		id = gtk_entry_get_text(GTK_ENTRY(entry));
		(*state)->syncPCid = strtol(id, NULL, 10);
	}
}

static void
build_speed_menu(GtkOptionMenu *optionMenu)
{
	gint i=0;
	GtkWidget *menu ,*menuItem;
	gchar buf[20];
	
	menu = gtk_menu_new();

	while(speedList[i]!=0) {
		g_snprintf(buf,sizeof(buf),"%d",speedList[i]);
		menuItem = gtk_menu_item_new_with_label(buf);
		gtk_widget_show(menuItem);
		gtk_signal_connect(GTK_OBJECT(menuItem),"activate",
				   GTK_SIGNAL_FUNC(activate_speed),
				   GINT_TO_POINTER(speedList[i]));
		gtk_menu_append(GTK_MENU(menu),menuItem);
		++i;
	}
	gtk_option_menu_set_menu(optionMenu,menu);	

}


static void
configureDeviceDialog(GtkWidget* dialog, GladeXML*xml)
{
	GtkWidget *entry,*optionMenu;
	entry = glade_xml_get_widget(xml,"device_name_entry");
	gtk_object_set_data(GTK_OBJECT(dialog),"name_entry",entry);
	
	entry= glade_xml_get_widget(xml,"device_port_entry");
	gtk_object_set_data(GTK_OBJECT(dialog),"port_entry",entry);
	gtk_signal_connect(GTK_OBJECT(entry),"insert_text",
			   GTK_SIGNAL_FUNC(insert_device_callback),
			   NULL);

	optionMenu= glade_xml_get_widget(xml,"device_speed_menu");
	gtk_object_set_data(GTK_OBJECT(dialog),"speed_menu",optionMenu);
	build_speed_menu(GTK_OPTION_MENU(optionMenu));
}

static GtkWidget
*createDeviceCfgDialog(void)
{
	GladeXML *xml;
	GtkWidget *dialog;

	xml=glade_xml_new(glade_file,"DeviceSettings");
	dialog=glade_xml_get_widget(xml,"DeviceSettings");
	
	configureDeviceDialog(dialog,xml);
	return dialog;
}

static void
setDeviceCfg(GtkWidget *devcfg, PilotState *state)
{
	GList *tmp;
	gint i=0;
	gchar *row[3];
	
	gtk_clist_clear(GTK_CLIST(devcfg));
	
	tmp=state->devices;
	while(tmp!=NULL) {
		GPilotDevice *dev=(GPilotDevice*)tmp->data;
		row[0]=dev->name;
		row[1]=dev->port;
		row[2]=g_strdup_printf("%d",dev->speed);
		gtk_clist_append(GTK_CLIST(devcfg),row);
		gtk_clist_set_row_data(GTK_CLIST(devcfg),i,dev);
		tmp=tmp->next;
		i++;
	}
}

/* find the next "Cradle#" namethat is available for use */

static gint compare_device_name(GPilotDevice *device, gchar *name)
{
	return strcmp(device->name,name);
}

static gchar *
next_cradle_name(void)
{
	int i=0;
	gchar buf[16];
	
	sprintf(buf,"Cradle");
	
	while(g_list_find_custom(curState->devices,buf,
				 (GCompareFunc)compare_device_name)!=NULL) {
		i++;
		sprintf(buf,"Cradle%d",i);
	}
	return g_strdup(buf);
}
/* find the next "MyPilot#" name that is available for use */

static gint compare_pilot_name(GPilotPilot *pilot, gchar *name)
{
	return strcmp(pilot->name,name);
}

static gchar *
next_pilot_name(void)
{
	int i=0;
	gchar buf[16];
	
	sprintf(buf,"MyPilot");
	
	while(g_list_find_custom(curState->pilots,buf,
				 (GCompareFunc)compare_pilot_name)!=NULL) {
		i++;
		sprintf(buf,"MyPilot%d",i);
	}
	return g_strdup(buf);
}

static void
setDeviceDialogCfg(GtkWidget *devcfg, GPilotDevice *device)
{	
	GtkOptionMenu *speed_menu;
	GtkWidget *port_entry, *name_entry;
	gint i;

	port_entry  = gtk_object_get_data(GTK_OBJECT(devcfg), "port_entry");
	name_entry  = gtk_object_get_data(GTK_OBJECT(devcfg), "name_entry");
	speed_menu = gtk_object_get_data(GTK_OBJECT(devcfg), "speed_menu");

	if (device) {
		gtk_entry_set_text(GTK_ENTRY(name_entry), device->name);
		gtk_entry_set_text(GTK_ENTRY(port_entry), device->port);
		i=0;
		while(speedList[i] != 0 && speedList[i] != device->speed) i++;
		if(speedList[i]==0) i=DEFAULT_SPEED_INDEX;
		gtk_option_menu_set_history(speed_menu,i);
		gtk_object_set_data(GTK_OBJECT(devcfg),"speed",GINT_TO_POINTER(speedList[i]));
	} else {
		gchar *name =next_cradle_name();
		gtk_entry_set_text(GTK_ENTRY(name_entry), name);
		g_free(name);

		gtk_entry_set_text(GTK_ENTRY(port_entry), "/dev/pilot");
		gtk_object_set_data(GTK_OBJECT(devcfg),"speed",GINT_TO_POINTER(speedList[DEFAULT_SPEED_INDEX]));
		gtk_option_menu_set_history(speed_menu,DEFAULT_SPEED_INDEX);
	}
}


static void
readDeviceDialogCfg(GtkWidget *devcfg, GPilotDevice* device)
{
	GtkWidget *port_entry, *speed_menu, *name_entry;
	g_return_if_fail(device!=NULL);

	port_entry  = gtk_object_get_data(GTK_OBJECT(devcfg), "port_entry");
	name_entry  = gtk_object_get_data(GTK_OBJECT(devcfg), "name_entry");
	speed_menu = gtk_object_get_data(GTK_OBJECT(devcfg), "speed_menu");

	if (device->port)
		g_free(device->port);
	device->port= g_strdup(gtk_entry_get_text(GTK_ENTRY(port_entry)));

	if (device->name)
		g_free(device->name);
	device->name= g_strdup(gtk_entry_get_text(GTK_ENTRY(name_entry)));
	if (device->name==NULL) device->name = g_strdup("Cradle"); 
	device->speed=GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(devcfg),"speed"));
}

/* object data:                        */
/* "userid"   - entry box for pilot user id */
/* "username" - entry box for pilot user name */

static void
configurePilotDialog(GtkWidget *dialog, GladeXML *xml)
{
	GtkMenu *menu;
	GtkWidget *entry,*optionMenu,*button,*menuItem;

	entry = glade_xml_get_widget(xml,"pilot_name_entry");
	gtk_object_set_data(GTK_OBJECT(dialog), "pilotname", entry);

	entry = glade_xml_get_widget(xml,"pilot_username_entry");
	gtk_object_set_data(GTK_OBJECT(dialog), "username", entry);
	gtk_signal_connect(GTK_OBJECT(entry), "insert_text",
			   GTK_SIGNAL_FUNC(insert_username_callback),
			   NULL);

	entry = glade_xml_get_widget(xml,"pilot_id_entry");
	gtk_object_set_data(GTK_OBJECT(dialog), "pilotid", entry);
	gtk_signal_connect(GTK_OBJECT(entry), "insert_text",
			   GTK_SIGNAL_FUNC(insert_numeric_callback),
			   NULL);

	entry = glade_xml_get_widget(xml,"pilot_basedir_entry");
	gtk_object_set_data(GTK_OBJECT(dialog), "basedir", entry);

	button = glade_xml_get_widget(xml,"get_from_pilot_button");
	if(button) {
		gtk_signal_connect(GTK_OBJECT(button),"clicked",GTK_SIGNAL_FUNC(get_from_pilot_cb),NULL);
		gtk_object_set_data(GTK_OBJECT(dialog),"get_from_pilot",button); 
	}
	button =  glade_xml_get_widget(xml,"send_to_pilot_button");
	if(button) {
		gtk_signal_connect(GTK_OBJECT(button),"clicked",GTK_SIGNAL_FUNC(send_to_pilot_cb),NULL);
		gtk_object_set_data(GTK_OBJECT(dialog),"send_to_pilot",button); 
	}
	optionMenu = glade_xml_get_widget(xml,"sync_type_menu");
	gtk_object_set_data(GTK_OBJECT(dialog), "sync_menu", optionMenu);

	menu = GTK_MENU(gtk_menu_new());

	menuItem = gtk_menu_item_new_with_label(_("Use conduit settings"));
	gtk_widget_show(menuItem);
	gtk_signal_connect(GTK_OBJECT(menuItem),"activate",
			   GTK_SIGNAL_FUNC(activate_sync_type),
			   GINT_TO_POINTER(GnomePilotConduitSyncTypeNotSet));
	gtk_menu_append(menu,menuItem);
	menuItem = gtk_menu_item_new_with_label(_("Synchronize"));
	gtk_widget_show(menuItem);
	gtk_signal_connect(GTK_OBJECT(menuItem),"activate",
			   GTK_SIGNAL_FUNC(activate_sync_type),
			   GINT_TO_POINTER(GnomePilotConduitSyncTypeSynchronize));
	gtk_menu_append(menu,menuItem);
	menuItem = gtk_menu_item_new_with_label(_("Copy from pilot"));
	gtk_widget_show(menuItem);
	gtk_signal_connect(GTK_OBJECT(menuItem),"activate",
			   GTK_SIGNAL_FUNC(activate_sync_type),
			   GINT_TO_POINTER(GnomePilotConduitSyncTypeCopyFromPilot));
	gtk_menu_append(menu,menuItem);
	menuItem = gtk_menu_item_new_with_label(_("Copy to pilot"));
	gtk_widget_show(menuItem);
	gtk_signal_connect(GTK_OBJECT(menuItem),"activate",
			   GTK_SIGNAL_FUNC(activate_sync_type),
			   GINT_TO_POINTER(GnomePilotConduitSyncTypeCopyToPilot));
	gtk_menu_append(menu,menuItem);
	menuItem = gtk_menu_item_new_with_label(_("Merge from pilot"));
	gtk_widget_show(menuItem);
	gtk_signal_connect(GTK_OBJECT(menuItem),"activate",
			   GTK_SIGNAL_FUNC(activate_sync_type),
			   GINT_TO_POINTER(GnomePilotConduitSyncTypeMergeFromPilot));
	gtk_menu_append(menu,menuItem);
	menuItem = gtk_menu_item_new_with_label(_("Merge to pilot"));
	gtk_widget_show(menuItem);
	gtk_signal_connect(GTK_OBJECT(menuItem),"activate",
			   GTK_SIGNAL_FUNC(activate_sync_type),
			   GINT_TO_POINTER(GnomePilotConduitSyncTypeMergeToPilot));
	gtk_menu_append(menu,menuItem);
	gtk_option_menu_set_menu(GTK_OPTION_MENU(optionMenu),GTK_WIDGET(menu));	
	
}

static GtkWidget
*createPilotCfgDialog(void)
{
	GladeXML *xml;
	GtkWidget *dialog;

	xml=glade_xml_new(glade_file,"PilotSettings");
	dialog=glade_xml_get_widget(xml,"PilotSettings");

	configurePilotDialog(dialog,xml);
	return dialog;
}



static void
setPilotCfg(GtkWidget *pilotcfg, PilotState *state)
{
	GList *tmp;
	gint i=0;
	const gchar *row[4];
	gchar buf[20];
	
	gtk_clist_clear(GTK_CLIST(pilotcfg));
	tmp=state->pilots;
	while(tmp!=NULL) {
		GPilotPilot *pilot=(GPilotPilot*)tmp->data;
		row[0]=pilot->name;
		g_snprintf(buf,sizeof(buf),"%d",pilot->pilot_id);
		row[1]=buf;
		row[2]=pilot->pilot_username;
		row[3]=sync_type_int_to_str(pilot->sync_options.default_sync_action);
		i=gtk_clist_append(GTK_CLIST(pilotcfg),(gchar **)row);
		gtk_clist_set_row_data(GTK_CLIST(pilotcfg),i,pilot);
		tmp=tmp->next;
		i++;
	}
}
static void
setPilotDialogCfg(GtkWidget *pilotcfg,GPilotPilot *pilot)
{
	GtkOptionMenu *optionMenu;
	GtkWidget *id, *name,*pname,*basedir;
	gchar buf[256];

	id      = gtk_object_get_data(GTK_OBJECT(pilotcfg), "pilotid");
	name    = gtk_object_get_data(GTK_OBJECT(pilotcfg), "username");
	pname   = gtk_object_get_data(GTK_OBJECT(pilotcfg), "pilotname");
	basedir = gtk_object_get_data(GTK_OBJECT(pilotcfg), "basedir");
	optionMenu = gtk_object_get_data(GTK_OBJECT(pilotcfg), "sync_menu");

	if (pilot) {
		g_snprintf(buf, sizeof(buf), "%d", pilot->pilot_id);
		gtk_entry_set_text(GTK_ENTRY(id), buf);
		gtk_entry_set_text(GTK_ENTRY(name), pilot->pilot_username);
		gtk_entry_set_text(GTK_ENTRY(pname), pilot->name);
		gtk_entry_set_text(GTK_ENTRY(basedir),pilot->sync_options.basedir);
		/* yes, if the items in menu are the same order as in
		   GnomePilotConduitSyncType, we could loose the switch. But
		   if they're not, or the ebuf doesn't start at 0, boohoo */
		switch(pilot->sync_options.default_sync_action) {
		case GnomePilotConduitSyncTypeCustom:
		case GnomePilotConduitSyncTypeNotSet:
			gtk_option_menu_set_history(optionMenu,0);
			break;
		case GnomePilotConduitSyncTypeSynchronize:
			gtk_option_menu_set_history(optionMenu,1);
			break;
		case GnomePilotConduitSyncTypeCopyFromPilot:
			gtk_option_menu_set_history(optionMenu,2);
			break;
		case GnomePilotConduitSyncTypeCopyToPilot:
			gtk_option_menu_set_history(optionMenu,3);
			break;
		case GnomePilotConduitSyncTypeMergeFromPilot:
			gtk_option_menu_set_history(optionMenu,4);
			break;
		case GnomePilotConduitSyncTypeMergeToPilot:
			gtk_option_menu_set_history(optionMenu,5);
			break;
		}
	} else {
		gchar *pilot_name=next_pilot_name();
		gtk_entry_set_text(GTK_ENTRY(pname), pilot_name);
		g_snprintf(buf, sizeof(buf), "%d", getuid());
		gtk_entry_set_text(GTK_ENTRY(id), buf);
		gtk_entry_set_text(GTK_ENTRY(name), g_get_real_name());
		g_snprintf(buf, sizeof(buf), "%s/%s", g_get_home_dir(),pilot_name);
		gtk_entry_set_text(GTK_ENTRY(basedir),buf);
		
		g_free(pilot_name);
		gtk_option_menu_set_history(optionMenu,0);
		gtk_object_set_data(GTK_OBJECT(pilotcfg),"sync_type",
				    GINT_TO_POINTER(GnomePilotConduitSyncTypeNotSet));
	}
}


static void
readPilotDialogCfg(GtkWidget *pilotcfg, GPilotPilot *pilot)
{
	GtkWidget *id, *name,*pname,*basedir,*menu;
	gchar *num;
	gint pilotid;
	g_return_if_fail(pilot!=NULL);

	id      = gtk_object_get_data(GTK_OBJECT(pilotcfg), "pilotid");
	name    = gtk_object_get_data(GTK_OBJECT(pilotcfg), "username");
	pname   = gtk_object_get_data(GTK_OBJECT(pilotcfg), "pilotname");
	basedir = gtk_object_get_data(GTK_OBJECT(pilotcfg), "basedir");
	menu    = gtk_object_get_data(GTK_OBJECT(pilotcfg), "sync_menu");
 
 	
	num = gtk_entry_get_text(GTK_ENTRY(id));
	pilotid = strtol(num, NULL, 10);
	pilot->pilot_id = pilotid;

	if (pilot->pilot_username) 
		g_free(pilot->pilot_username);
	pilot->pilot_username = g_strdup(gtk_entry_get_text(GTK_ENTRY(name)));

	if (pilot->name)
		g_free(pilot->name);
	pilot->name = g_strdup(gtk_entry_get_text(GTK_ENTRY(pname)));
	
	if (pilot->sync_options.basedir) 
		g_free(pilot->sync_options.basedir);
	pilot->sync_options.basedir = g_strdup(gtk_entry_get_text(GTK_ENTRY(basedir)));
	pilot->sync_options.default_sync_action=GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(pilotcfg),"sync_type"));
}


static void
pilot_capplet_glade_init(void)
{
	glade_gnome_init ();

	glade_file = "./gpilotd-capplet.glade";
	if (!g_file_exists (glade_file)) {
		glade_file = g_concat_dir_and_file (GLADEDATADIR, "gpilotd-capplet.glade");
	}
	if (!g_file_exists (glade_file)) {
		run_error_dialog (_("Cannot find gpilotd-capplet.glade"));
		exit(1);
	}
}

static void
pilot_capplet_setup(void)
{
	static GnomeHelpMenuEntry help_entry = {"gpilotd-control-applet","properties.html"};

	GladeXML  *xml;
	GtkWidget  *mainWindow, *button;
	GtkWidget *pilots_edit_menu, *pilots_delete_menu;
	GtkWidget *devices_edit_menu, *devices_delete_menu;

	xml = glade_xml_new (glade_file, "devices_popup_menu");
	devicePopupMenu = GTK_MENU(glade_xml_get_widget(xml,"devices_popup_menu"));
	devices_delete_menu=glade_xml_get_widget(xml,"devices_delete_menu");
	devices_edit_menu=glade_xml_get_widget(xml,"devices_edit_menu");

	xml = glade_xml_new (glade_file, "pilots_popup_menu");
	pilotPopupMenu = GTK_MENU(glade_xml_get_widget(xml,"pilots_popup_menu"));
	pilots_delete_menu=glade_xml_get_widget(xml,"pilots_delete_menu");
	pilots_edit_menu=glade_xml_get_widget(xml,"pilots_edit_menu");
	

	xml = glade_xml_new (glade_file, "main");
	mainWindow = glade_xml_get_widget (xml, "main");

	capplet = capplet_widget_new();
	gtk_container_add (GTK_CONTAINER (capplet), mainWindow);

				 
	deviceCfgWindow = glade_xml_get_widget(xml,"devices_clist");
	gtk_object_set_data(GTK_OBJECT(deviceCfgWindow),"selected",GINT_TO_POINTER(-1));
	gtk_signal_connect(GTK_OBJECT(deviceCfgWindow), "select-row",
			   GTK_SIGNAL_FUNC(devices_selection_cb), NULL);
	gtk_signal_connect(GTK_OBJECT(deviceCfgWindow), "button_press_event",
			   GTK_SIGNAL_FUNC(clist_button_press), devicePopupMenu);
	gtk_signal_connect(GTK_OBJECT(deviceCfgWindow), "unselect-row",
			   GTK_SIGNAL_FUNC(clist_unselection_cb), NULL);

	gtk_signal_connect(GTK_OBJECT(devices_edit_menu),"activate", 
			   GTK_SIGNAL_FUNC(devices_edit_cb), deviceCfgWindow);

	gtk_signal_connect(GTK_OBJECT(devices_delete_menu),"activate", 
			   GTK_SIGNAL_FUNC(devices_delete_cb), deviceCfgWindow);
	
	pilotCfgWindow = glade_xml_get_widget(xml,"pilots_clist");
	gtk_object_set_data(GTK_OBJECT(pilotCfgWindow),"selected",GINT_TO_POINTER(-1));
	gtk_signal_connect(GTK_OBJECT(pilotCfgWindow), "select-row",
			   GTK_SIGNAL_FUNC(pilots_selection_cb), NULL);
	gtk_signal_connect(GTK_OBJECT(pilotCfgWindow), "button_press_event",
			   GTK_SIGNAL_FUNC(clist_button_press), pilotPopupMenu);
	gtk_signal_connect(GTK_OBJECT(pilotCfgWindow), "unselect-row",
			   GTK_SIGNAL_FUNC(clist_unselection_cb), NULL);

	gtk_signal_connect(GTK_OBJECT(pilots_edit_menu),"activate", 
			   GTK_SIGNAL_FUNC(pilots_edit_cb), pilotCfgWindow);

	gtk_signal_connect(GTK_OBJECT(pilots_delete_menu),"activate", 
			   GTK_SIGNAL_FUNC(pilots_delete_cb), pilotCfgWindow);

	button = glade_xml_get_widget(xml,"pilots_add_button");
	gtk_signal_connect(GTK_OBJECT(button),"clicked",
			   GTK_SIGNAL_FUNC(pilots_add_cb),NULL);
	button = glade_xml_get_widget(xml,"pilots_edit_button");
	gtk_signal_connect(GTK_OBJECT(button),"clicked",
			   GTK_SIGNAL_FUNC(pilots_edit_cb),pilotCfgWindow);
	button = glade_xml_get_widget(xml,"pilots_delete_button");
	gtk_signal_connect(GTK_OBJECT(button),"clicked",
			   GTK_SIGNAL_FUNC(pilots_delete_cb),pilotCfgWindow);
	button = glade_xml_get_widget(xml,"devices_add_button");
	gtk_signal_connect(GTK_OBJECT(button),"clicked",
			   GTK_SIGNAL_FUNC(devices_add_cb),NULL);
	button = glade_xml_get_widget(xml,"devices_edit_button");
	gtk_signal_connect(GTK_OBJECT(button),"clicked",
			   GTK_SIGNAL_FUNC(devices_edit_cb),deviceCfgWindow);
	button = glade_xml_get_widget(xml,"devices_delete_button");
	gtk_signal_connect(GTK_OBJECT(button),"clicked",
			   GTK_SIGNAL_FUNC(devices_delete_cb),deviceCfgWindow);

	createHostCfgWindow(xml);
	deviceCfgDialog=createDeviceCfgDialog();
	pilotCfgDialog=createPilotCfgDialog();
	
	gtk_signal_connect(GTK_OBJECT(capplet), "try",
			   GTK_SIGNAL_FUNC(doTrySettings), curState);
	gtk_signal_connect(GTK_OBJECT(capplet), "revert",
			   GTK_SIGNAL_FUNC(doRevertSettings), origState);
	gtk_signal_connect(GTK_OBJECT(capplet), "ok",
			   GTK_SIGNAL_FUNC(doSaveSettings), curState);
	gtk_signal_connect(GTK_OBJECT(capplet), "cancel",
			   GTK_SIGNAL_FUNC(doRevertSettings), origState);
	gtk_signal_connect(GTK_OBJECT(capplet), "help",
			   GTK_SIGNAL_FUNC(gnome_help_display), 
			   &help_entry);
}

static void
monitor_pilots(PilotState * state)
{
	GList *tmp=state->pilots;
	if(tmp!=NULL){
		while(tmp!=NULL){
			GPilotPilot *pilot=(GPilotPilot*)tmp->data;
			g_message("pilot = %s",pilot->name);
			gnome_pilot_client_monitor_on(gpc,pilot->name);
			tmp=tmp->next;
		}
		gnome_pilot_client_notify_on(gpc,GNOME_Pilot_NOTIFY_CONNECT);
		gnome_pilot_client_notify_on(gpc,GNOME_Pilot_NOTIFY_DISCONNECT);
	}
}


int
main( int argc, char *argv[] )
{
	struct poptOption options[] = {{"druid",'\0',POPT_ARG_NONE,NULL,0, N_("Start druid only"),NULL },
				       {NULL,'\0',0,NULL,0,NULL,NULL  }};

	bindtextdomain (PACKAGE, GNOMELOCALEDIR);
	textdomain (PACKAGE);
	
	if(argc>1 && strcmp(argv[1],"--druid")==0) {
		/* we are a regular gnome app with CORBA support */
		CORBA_Environment ev;
		gnome_CORBA_init_with_popt_table("gpilot control applet", VERSION, &argc, argv,
						 options,0,NULL,0,&ev);
		druid_on=TRUE;
		druid_prog=TRUE;
	} else {
		/* we're a capplet (we get CORBA for free from capplet_init) */
		switch(gnome_capplet_init ("gpilot control applet", VERSION, argc, argv, 
					   options,0,NULL)) {
		case 1: return 0; break;
		case -1: g_error(_("Error initializing gpilotd capplet")); break;
		default: break;
		}
	}
	
	/* we using glade */
	pilot_capplet_glade_init();
	
	/* put all code to set things up in here */
	if (loadPilotState(&origState) < 0) {
		run_error_dialog(_("Error loading pilot state, aborting"));
		g_error(_("Error loading pilot state, aborting"));
		return -1;
	}

	curState = dupPilotState(origState);
	gpc = GNOME_PILOT_CLIENT(gnome_pilot_client_new());
	gtk_signal_connect_object(GTK_OBJECT(gpc), "completed_request",
				  GTK_SIGNAL_FUNC(gpilotd_request_completed),NULL);
	gtk_signal_connect_object(GTK_OBJECT(gpc), "user_info",
				  GTK_SIGNAL_FUNC(gpilotd_userinfo_requested),NULL);

	if(gnome_pilot_client_connect_to_daemon(gpc) != GPILOTD_OK) {
		run_error_dialog(_("Cannot connect to the GnomePilot Daemon"));
		g_error(_("Cannot connect to the GnomePilot Daemon"));
		return -1;
	}

	monitor_pilots(origState);
	if(druid_on) {
		if(curState->pilots!=NULL || curState->devices!=NULL) {
			run_error_dialog(_("Cannot run druid if pilots or cradles already configured"));
			g_error(_("Cannot run druid if pilots or cradles already configured"));
			return -1;
		}
		gpilotd_druid_show(glade_file);
		gtk_main();
	} else {
		pilot_capplet_setup();
		setHostCfg(capplet, curState);
		setDeviceCfg(deviceCfgWindow, curState);
		setPilotCfg(pilotCfgWindow, curState);

		gtk_widget_show_all(capplet);
	
		/* popup the druid anyway */
		if(curState->pilots==NULL && curState->devices==NULL) {
			druid_on=TRUE;
			gpilotd_druid_show(glade_file);
		}

		/* done setting up, now run main loop */
		capplet_gtk_main();
	}
	freePilotState(curState);
	freePilotState(origState);
	return 0;
}    

/********************  DRUID *******************/

static GtkWidget *no_button;
static int handle=-1;
static GtkWidget *sync_label;
static GnomeDruid *druid;
static GnomeDruidPage *sync_page;

static void 
druid_gpilotd_userinfo_requested(const gchar *device,const GNOME_Pilot_PilotUser *user) 
{
	GPilotPilot *pilot;
	gchar *text;
	
	g_message("device %s sent userinfo",device);
	g_message("user->userID   = %d",user->userID);
	g_message("user->username = %s",user->username);

	pilot=(GPilotPilot*)curState->pilots->data;
	

	readPilotDialogCfg(mainDruidWindow,pilot);
	pilot->pilot_id = user->userID;
	if(pilot->pilot_username) g_free(pilot->pilot_username);
	pilot->pilot_username = g_strdup(user->username);
	setPilotDialogCfg(mainDruidWindow,pilot);
	
	text=g_strdup_printf(_("Successfully retrieved UserName and ID to pilot.\n"
			       "UserName: %s\nID: %d"),
			     pilot->pilot_username,pilot->pilot_id);
	gtk_label_set_text(GTK_LABEL(sync_label),text);

	gnome_druid_set_buttons_sensitive(druid,TRUE,TRUE,TRUE);

	g_free(text);
	druid_request_done=TRUE;
	handle=-1;
}

static void
druid_gpilotd_request_completed(const gchar *id, unsigned long handle)
{
	if(druid_request_done == FALSE) {
		gtk_label_set_text(GTK_LABEL(sync_label),_("Successfully sent UserName and ID to pilot."));
		gnome_druid_set_buttons_sensitive(druid,TRUE,TRUE,TRUE);
	}
	druid_request_done=TRUE;
	handle=-1;
}

static void
no_button_toggled(GtkToggleButton *togglebutton, GtkWidget *frame)
{
	gtk_widget_set_sensitive(frame, togglebutton->active);
}


static gboolean
ask_to_cancel(void)
{
	if(run_question_dialog(_("Setup did not complete and settings will not\n"
				 "be saved. Are you sure you want to quit?"))) {
		
		if(handle>0) {
			gnome_pilot_client_remove_request(gpc,handle);
			handle=-1;
		}
		saveSettingsAndRestartDaemon(origState);
		gtk_widget_destroy(mainDruidWindow);
		if(druid_prog)	gtk_main_quit();
		return FALSE;
	}
	return TRUE;
}

static void 
druid_canceled (GnomeDruid *druid, gpointer user_data)
{
	ask_to_cancel();
}

static void
druid_finished(GnomeDruidPage *druidpage,GnomeDruid *druid,
	       gpointer user_data)
{
	GPilotPilot *pilot=(GPilotPilot*)curState->pilots->data;
	readPilotDialogCfg(mainDruidWindow,pilot);
	saveSettingsAndRestartDaemon(curState);
	gtk_widget_destroy(mainDruidWindow);
	if(druid_prog) gtk_main_quit();
	else {
		/* make sure capplet is updated with latest information */
		setHostCfg(capplet,curState);
		setDeviceCfg(deviceCfgWindow, curState);
		setPilotCfg(pilotCfgWindow, curState);
		capplet_widget_state_changed(CAPPLET_WIDGET (capplet), FALSE);
	}
}

static gboolean
druid_sync_page_back(GnomeDruidPage *druidpage,gpointer arg1,
		     gpointer user_data)
{
	if (handle>0) {
		gnome_pilot_client_remove_request(gpc,handle);  
		handle=-1;
	}
	return FALSE;
}


static gboolean
druid_presync_page_next(GnomeDruidPage *druidpage,GnomeDruid *druid,
			GtkWidget *label)
{
	gchar *text;
	GPilotDevice *device;
	GNOME_Pilot_PilotUser user;
	
	device=(GPilotDevice*)curState->devices->data;
	readDeviceDialogCfg(mainDruidWindow,device);
	if(GTK_TOGGLE_BUTTON(no_button)->active) {
		/* do send_to_pilot */
		GPilotPilot *pilot=(GPilotPilot*)curState->pilots->data;
		readPilotDialogCfg(mainDruidWindow,pilot);
		text=g_strdup_printf(_("About to send the following data to the pilot.\n"
				       "UserName: %s\nID: %d\n"
				       "Please put pilot in %s (%s) and press HotSync button."),
				     pilot->pilot_username,pilot->pilot_id,device->name,device->port);

		saveSettingsAndRestartDaemon(curState);

		user.userID = pilot->pilot_id;
		user.username = pilot->pilot_username;
		gnome_pilot_client_set_user_info(gpc,
						 device->name,
						 user,
						 FALSE,
						 GNOME_Pilot_IMMEDIATE,
						 0,
						 &handle);
	} else {
		/* do get_from_pilot */
		text=g_strdup_printf(_("About to retrieve UserName and ID from the pilot.\n"
				       "Please put pilot in %s (%s) and press HotSync button."),
				     device->name,device->port);
		saveSettingsAndRestartDaemon(curState);

		gnome_pilot_client_get_user_info(gpc,device->name,GNOME_Pilot_IMMEDIATE,0,&handle);
	}
	gtk_label_set_text(GTK_LABEL(label),text);
        g_free(text);
	if(handle<=0) {
		run_error_dialog(_("Failed sending request to gpilotd"));
		return TRUE;
	}

	/* disable NEXT until we've synced */
	gnome_druid_set_page(druid,sync_page);
	gnome_druid_set_buttons_sensitive(druid,TRUE,FALSE,TRUE);
	return TRUE;
}

static gboolean
druid_pilot2_page_next(GnomeDruidPage *druidpage,GnomeDruid *druid,
		       GtkWidget *base_dir)
{
	/* if check_base_directory fails we want to return TRUE,
	   which will tell druid not to switch pages.
	*/
	return !(check_base_directory(gtk_entry_get_text(GTK_ENTRY(base_dir))));
}

static gboolean
druid_delete_window(GtkWidget *widget,
		    GdkEvent *event,
		    gpointer user_data)
{
	return ask_to_cancel();
}

static void
gpilotd_druid_show(gchar* glade_file)
{
	GladeXML  *xml;
	GtkWidget *frame, *vbox;
	GtkWidget *druid_page, *base_dir;
	GPilotPilot *pilot;
	GPilotDevice *device;

	xml=glade_xml_new(glade_file,"DruidWindow");
	mainDruidWindow=glade_xml_get_widget(xml,"DruidWindow");

	gtk_signal_connect(GTK_OBJECT(mainDruidWindow),"delete_event",
			   GTK_SIGNAL_FUNC(druid_delete_window),NULL);

	druid=GNOME_DRUID(glade_xml_get_widget(xml,"druid"));
	gtk_signal_connect(GTK_OBJECT(druid),"cancel",
			   GTK_SIGNAL_FUNC(druid_canceled),NULL);
	
/*	sync_label=glade_xml_get_widget(xml,"sync_label"); */
	/* FIXME: is this a libglade bug or what? if sync_label 
	   is constructed in .glade it is not properly redrawn the first time. */
	vbox = glade_xml_get_widget(xml,"sync_label_vbox");
	sync_label = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(vbox),sync_label,TRUE,FALSE,GNOME_PAD_SMALL);

	druid_page=glade_xml_get_widget(xml,"druidpage_pilot1");
	gtk_signal_connect(GTK_OBJECT(druid_page),"next",
			   GTK_SIGNAL_FUNC(druid_presync_page_next),sync_label);

	base_dir = glade_xml_get_widget(xml,"pilot_basedir_entry");
	druid_page=glade_xml_get_widget(xml,"druidpage_pilot2");
	gtk_signal_connect(GTK_OBJECT(druid_page),"next",
			   GTK_SIGNAL_FUNC(druid_pilot2_page_next),base_dir);

	sync_page=GNOME_DRUID_PAGE(glade_xml_get_widget(xml,"druidpage_sync"));
	gtk_signal_connect(GTK_OBJECT(sync_page),"back",
			   GTK_SIGNAL_FUNC(druid_sync_page_back),NULL);

	druid_page=glade_xml_get_widget(xml,"druidpage_finish");
	gtk_signal_connect(GTK_OBJECT(druid_page),"finish",
			   GTK_SIGNAL_FUNC(druid_finished),sync_label);

	frame=glade_xml_get_widget(xml,"pilot_user_frame");
	
	no_button=glade_xml_get_widget(xml,"no_radio_button");
	gtk_signal_connect(GTK_OBJECT(no_button),"toggled",
			   GTK_SIGNAL_FUNC(no_button_toggled),frame);

	configureDeviceDialog(mainDruidWindow,xml);
	setDeviceDialogCfg(mainDruidWindow,NULL);

	configurePilotDialog(mainDruidWindow,xml);
	setPilotDialogCfg(mainDruidWindow,NULL);

	pilot=g_new0(GPilotPilot,1);
	curState->pilots=g_list_append(curState->pilots,pilot);
	readPilotDialogCfg(mainDruidWindow,pilot);
	
	device=g_new0(GPilotDevice,1);
	curState->devices=g_list_append(curState->devices,device);
	readDeviceDialogCfg(mainDruidWindow,device);

	gtk_widget_show_all(mainDruidWindow);
}

