/* handles pilot issues */

#include <config.h>
#include <gnome.h>

#include "pilot.h"

/* create new pilotstate structure, initialize to sane state */
PilotState
*newPilotState(void)
{
    PilotState *p;
    p = g_new0(PilotState, 1);
    return p;
}

static void
copy_device(GPilotDevice *device, PilotState *dest)
{
    GPilotDevice *device2=gpilot_device_new();
    device2->name=g_strdup(device->name);
    device2->port=g_strdup(device->port);
    device2->speed=device->speed;
    dest->devices=g_list_append(dest->devices,device2);
}

static void
copy_pilot(GPilotPilot *pilot, PilotState *dest)
{
    GPilotPilot *pilot2=gpilot_pilot_new();
    pilot2->name=g_strdup(pilot->name);
    pilot2->passwd=g_strdup(pilot->passwd);
    pilot2->pilot_username=g_strdup(pilot->pilot_username);
    pilot2->pilot_id=pilot->pilot_id;
    pilot2->sync_options.default_sync_action=pilot->sync_options.default_sync_action;
    pilot2->sync_options.basedir=g_strdup(pilot->sync_options.basedir);
    dest->pilots=g_list_append(dest->pilots,pilot2);
}

void
copyPilotState(PilotState *dest, PilotState *src)
{
    dest->syncPCid=src->syncPCid;
    g_list_foreach(src->pilots,(GFunc)copy_pilot,dest);
    g_list_foreach(src->devices,(GFunc)copy_device,dest);
}

PilotState*
dupPilotState(PilotState *src)
{
    PilotState *p;
    p = g_new0(PilotState, 1);
    copyPilotState(p, src);
    return p;
}

void
freePilotState(PilotState *state)
{
    g_list_foreach(state->pilots,(GFunc)gpilot_pilot_free,NULL);
    g_list_free(state->pilots);
    g_list_foreach(state->devices,(GFunc)gpilot_device_free,NULL);
    g_list_free(state->devices);
    g_free(state);
}

static gint
loadHostID(PilotState *p)
{
    guint32  id;
    gboolean notfound;

    gnome_config_push_prefix("/gnome-pilot.d/gpilotd/General/");
    id = gnome_config_get_int_with_default("sync_PC_Id=0", &notfound);
    gnome_config_pop_prefix();
  
    if (notfound) {
	p->syncPCid = random();
	return -1;
    } else {
	p->syncPCid = id;
	return 0;
    }
}    

static gint
loadDeviceCfg(PilotState *p)
{
  GPilotDevice *device;
  gchar *prefix;
  gchar buf[25];
  int i,num;

  gnome_config_push_prefix("/gnome-pilot.d/gpilotd/General/");
  num = gnome_config_get_int("num_devices=0");
  gnome_config_pop_prefix();

  if (num == 0) {
	  g_message("No pilot craddle information located");
	  p->devices = NULL;
	  return -1;
  } else {
	  for(i=0;i<num;i++){
		  device = g_new0(GPilotDevice, 1);
		  prefix=g_strdup_printf("/gnome-pilot.d/gpilotd/Device%d/",i);
		  
		  gnome_config_push_prefix(prefix);
		  device->port = gnome_config_get_string("device");
		  g_message("cradle device name -> %s",device->port);
		  sprintf(buf,"name=Cradle%d",i);
		  device->name = gnome_config_get_string(buf);
		  g_message("cradle device name -> %s",device->name);
		  device->speed = gnome_config_get_int("speed=9600");
		  g_message("Pilot Speed  -> %d",device->speed);  
		  gnome_config_pop_prefix();
		  g_free(prefix);
		  p->devices = g_list_append(p->devices,device);
	  }
  }
  return 0;
}


static gint
loadPilotPilot(PilotState *p)
{
  GPilotPilot *pilot;
  gboolean notfound;
  gchar *prefix;
  int i,num;

  gnome_config_push_prefix("/gnome-pilot.d/gpilotd/General/");
  num = gnome_config_get_int("num_pilots=0");
  gnome_config_pop_prefix();

  if (num == 0) {
      g_message("No pilot userid/username information located");
      p->pilots = NULL;
      return -1;
  } else {
	  for(i=0;i<num;i++){
		  gchar *tmp;
		  pilot = g_new0(GPilotPilot, 1);
		  prefix=g_strdup_printf("/gnome-pilot.d/gpilotd/Pilot%d/",i);
		  gnome_config_push_prefix(prefix);
		  pilot->name = gnome_config_get_string("name=MyPilot");
		  g_message("Pilot name -> %s",pilot->name);
		  pilot->pilot_id = gnome_config_get_int_with_default("pilotid",&notfound);
		  if (notfound)
			  pilot->pilot_id = getuid();
		  g_message("Pilot id   -> %d",pilot->pilot_id);
		  pilot->pilot_username=gnome_config_get_string("pilotusername");
		  g_message("Pilot username -> %s",pilot->pilot_username);
	  
		  tmp = gnome_config_get_string("default_sync_action=custom");
		  g_message("Pilot sync_action -> %s",tmp);
		  pilot->sync_options.default_sync_action = gnome_pilot_conduit_sync_type_str_to_int(tmp);
		  g_free(tmp);


		  pilot->sync_options.basedir = gnome_config_get_string("basedir");
		  if(pilot->sync_options.basedir==NULL) {
			  pilot->sync_options.basedir=g_strdup_printf("%s/%s",g_get_home_dir(),pilot->name);
		  }
	  
		  /* set these to something sane since we arent using them */
		  pilot->trusted_users = NULL;
		  pilot->number = i;
	  
		  g_free(prefix);
		  gnome_config_pop_prefix();
		  
		  p->pilots = g_list_append(p->pilots,pilot);
	  }
  }
  return 0;
}

/* allocates a pilotstate, load pilot state, return 0 if ok, -1 otherwise*/
gint
loadPilotState(PilotState **pilotState)
{
    PilotState *p;

    p = newPilotState();

    /* load host information */
    if (loadHostID(p) < 0) {
	g_message("Unable to load host id information, assuming unset");
    }

    if (loadPilotPilot(p) < 0) {
	g_message("Unable to load pilot id/username, assuming unset");	
    }

    if (loadDeviceCfg(p) < 0) {
	g_message("Unable to load pilot craddle info, assuming unset");
    }

    *pilotState = p;
    return 0;
}


gint
savePilotState(PilotState *state)
{
  int i;
  GList *tmp;
  gchar *prefix;

  gnome_config_clean_section("/gnome-pilot.d/gpilotd/General/");

  gnome_config_set_int("/gnome-pilot.d/gpilotd/General/sync_PC_Id",state->syncPCid);

  tmp=state->devices;
  i=0;
  while(tmp!=NULL)
  {
	  GPilotDevice *device=(GPilotDevice*)tmp->data;
	  prefix=g_strdup_printf("/gnome-pilot.d/gpilotd/Device%d/",i);

	  gnome_config_clean_section(prefix);
	  gnome_config_push_prefix(prefix);
	  gnome_config_set_string("device", device->port);
	  gnome_config_set_string("name", device->name);
	  gnome_config_set_int("speed", device->speed);
	  gnome_config_pop_prefix();
	  g_free(prefix);
	  tmp=tmp->next;
	  i++;
  }  
  if (i>0) {
      gnome_config_push_prefix("/gnome-pilot.d/gpilotd/General/");
      gnome_config_set_int("num_devices", i);
      gnome_config_pop_prefix();
  }

  tmp=state->pilots;
  i=0;
  while(tmp!=NULL)
  {
	  GPilotPilot *pilot=(GPilotPilot*)tmp->data;
	  prefix=g_strdup_printf("/gnome-pilot.d/gpilotd/Pilot%d/",i);
	  gnome_config_clean_section(prefix);

	  gnome_config_push_prefix(prefix);
	  gnome_config_set_string("name", pilot->name);
	  gnome_config_set_int("pilotid", pilot->pilot_id);
	  gnome_config_set_string("pilotusername", pilot->pilot_username);
	  gnome_config_set_string("default_sync_action", 
				  gnome_pilot_conduit_sync_type_int_to_str(pilot->sync_options.default_sync_action));
	  gnome_config_set_string("basedir", pilot->sync_options.basedir);
	  gnome_config_pop_prefix();
	  g_free(prefix);
	  tmp=tmp->next;
	  i++;
  }
  if (i>0) {
      gnome_config_push_prefix("/gnome-pilot.d/gpilotd/General/");
      gnome_config_set_int("num_pilots", i);
      gnome_config_pop_prefix();
  }

  gnome_config_sync();
  return 0;
}



