
/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- *//* 
 * Copyright (C) 1998-2000 Free Software Foundation
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors: Eskil Heyn Olsen
 *          Dave Camp
 *
 */

#include <config.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include "gnome-pilot-structures.h"
#include "gpilot-gui.h"
#include <libgnome/libgnome.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <fcntl.h>

/* From pi-csd */
#include <signal.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <net/if.h>
#include <arpa/inet.h>
#include <netdb.h>

#define LOCK_DIR "/var/lock"
#define LOCK_BINARY 0

#ifdef HAVE_SA_LEN
#ifndef max
#define max(a,b) ((a) > (b) ? (a) : (b))
#endif
#define ifreq_size(i) max(sizeof(struct ifreq),\
     sizeof((i).ifr_name)+(i).ifr_addr.sa_len)
#else
#define ifreq_size(i) sizeof(struct ifreq)
#endif /* HAVE_SA_LEN */

/* What, me worry? */
#ifndef IFF_POINTOPOINT
# ifdef IFF_POINTTOPOINT
#  define IFF_POINTOPOINT IFF_POINTTOPOINT
# endif
#endif
/* end of from pi-csd */

static GList *get_devices (void);
static GList *get_pilots (void);


/* context stuff first */
GPilotContext *gpilot_context_new (void)
{
	gchar *str;
	gint id;
	
	GPilotContext *retval;
	
	retval = (GPilotContext *)g_malloc (sizeof (GPilotContext));
	retval->paused = FALSE;
	retval->devices = NULL;
	retval->pilots = NULL;
	retval->user = NULL;
#ifdef WITH_USB_VISOR
	retval->visor_fd = -1;
	retval->visor_io = NULL;
	retval->visor_in_handle = -1;
	retval->visor_err_handle = -1;
#endif

	/* get the id.  Does anyone know the valid range for this? */
	srandom (time (NULL));
	id = 1 + ((guint) 1000000.0*random ());
	gnome_config_push_prefix ("/gnome-pilot.d/gpilotd/General/");
	str = g_strdup_printf ("sync_PC_Id=%d", id);
	retval->sync_PC_Id=gnome_config_get_int (str);
	g_free (str);

	/* if retval is the default value, store it now */
	if (retval->sync_PC_Id == id) {
		gnome_config_set_int ("sync_PC_Id",
				     retval->sync_PC_Id);
	}

	/* get progress stepping, default is -1, if default is returned,
	   default to one and set it */
	retval->progress_stepping = gnome_config_get_int ("progress_stepping=-1");
	if (retval->progress_stepping ==  -1) {
		retval->progress_stepping = 1;
		gnome_config_set_int ("progress_stepping", retval->progress_stepping);
	}

	gnome_config_pop_prefix ();
	gnome_config_sync ();
	return retval;
}

/* this will initialize the user context from their config
 * files.  If it has already been initialized, it will reread
 * the files, and free the old data. */
void
gpilot_context_init_user (GPilotContext *context)
{
	gchar *str;

	if (!context->user) {
		context->user = gpilot_user_new ();
	}

	str=getenv ("USER");
	if (str) {
		g_free (context->user->username);
		context->user->username = g_strdup (str);
	}
 
	context->devices = get_devices ();
	context->pilots = get_pilots ();

	context->sync_PC_Id=gnome_config_get_int ("/gnome-pilot.d/gpilotd/General/sync_PC_Id");
}

void
gpilot_context_free (GPilotContext *context)
{
	g_free (context->user->username);
	context->user->username = NULL;

	g_free (context->user);
	context->user = NULL;

	g_list_foreach (context->pilots, (GFunc)gpilot_pilot_free, NULL);
	g_list_free (context->pilots);
	context->pilots = NULL;
	
	g_list_foreach (context->devices, (GFunc)gpilot_device_free, NULL);
	g_list_free (context->devices);
	context->devices = NULL;
}

#define LOCK_DIR "/var/lock"

static int
gpilot_hdb_uucp_lock (GPilotDevice *device) 
{
#ifndef LOCK_BINARY
	char lock_buffer[12];
#endif
	int fd, pid, n;
	char *p = NULL;
	char *dev = device->port;

	if (geteuid () != 0) {
		/* g_message ("Not root, won't do UUCP lock on %s", device->port); */
		return 1;
	}
	
	if ((p = strrchr(dev, '/')) != NULL) {
		dev = p + 1;
	}
	
	device->lock_file = g_new0 (gchar, 128);

	snprintf (device->lock_file, 127, "%s/LCK..%s", LOCK_DIR, dev);

	while ((fd = open (device->lock_file, O_EXCL | O_CREAT | O_RDWR, 0644)) < 0) {
		if (errno != EEXIST) {
			g_warning ("Can't create lock file %s: %m", device->lock_file);
			break;
		}
		
		/* Read the lock file to find out who has the device locked. */
		fd = open (device->lock_file, O_RDONLY, 0);
		if (fd < 0) {
			if (errno == ENOENT) /* This is just a timing problem. */
				continue;
			g_warning ("Can't open existing lock file %s: %m", device->lock_file);
			break;
		}

#ifndef LOCK_BINARY
		n = read (fd, lock_buffer, 11);
#else
		n = read (fd, &pid, sizeof(pid));
#endif /* LOCK_BINARY */
		close (fd);
		fd = -1;
		if (n <= 0) {
			g_warning ("Can't read pid from lock file %s", device->lock_file);
			break;
		}
		
		/* See if the process still exists. */
#ifndef LOCK_BINARY
		lock_buffer[n] = 0;
		pid = atoi (lock_buffer);
#endif /* LOCK_BINARY */
		if (pid == getpid ()) {
			g_warning ("Port %s is already locked", device->port);
			return 0;           /* somebody else locked it for us */
		}
		if (pid == 0
		    || (kill (pid, 0) == -1 && errno == ESRCH)) {
			if (unlink (device->lock_file) == 0) {
				g_message ("Removed stale lock on %s (pid %d)", device->port, pid);
				continue;
			}
			g_warning ("Couldn't remove stale lock on %s", device->port);
		} else {
			g_message ("Device %s is locked by pid %d", device->port, pid);
		}
		break;
	}

	if (fd < 0) {
		g_free (device->lock_file);
		device->lock_file = NULL;
		return 0;
	}
	
	pid = getpid ();
#ifndef LOCK_BINARY
	snprintf(lock_buffer, 11, "%10d\n", pid);
	write (fd, lock_buffer, 11);
#else
	write (fd, &pid, sizeof (pid));
#endif
	close (fd);
	return 1;
}


static void
gpilot_hdb_uucp_unlock (GPilotDevice *device) 
{
	if (geteuid () != 0) {
		/* g_message ("Not root, won't do UUCP unlock on %s", device->port); */
	}

	if (device->lock_file) {
		unlink(device->lock_file);	
		g_free (device->lock_file);
		device->lock_file = NULL;
	}
}

/* device stuff next */
GPilotDevice *
gpilot_device_new (void)
{
	GPilotDevice *device;
	device = g_new0(GPilotDevice, 1);
	return device;
}

static gint 
gpilot_serial_device_init (GPilotDevice *device)
{
	if (!gpilot_hdb_uucp_lock (device)) {
		return -1;
	}

	device->fd=open (device->port, O_RDWR|O_NOCTTY|O_NONBLOCK);
	if (device->fd < 0) {
		g_warning (_("Could not open device %s (%s): reason: \"%s\"."),
			  device->name, device->port,
			  g_strerror (errno));
#ifdef GUI
		gpilot_gui_warning_dialog (_("GnomePilot could not open device %s (%s).\n"
					    "Reason: \"%s\"."),
					  device->name, device->port,
					  g_strerror (errno));
#endif
		gpilot_hdb_uucp_unlock (device);
	
		g_free (device->name);
		device->name = NULL;

		g_free (device->port);
		device->port = NULL;

		close (device->fd);
		device->fd = 0;

		device->io = NULL;

		return -1;
	}
	device->io = g_io_channel_unix_new (device->fd);
	g_io_channel_ref (device->io);

	return 0;
}

/* This free is used by USB and IrDA devices as well */
static void
gpilot_serial_device_free (GPilotDevice *device)
{
	if (device->fd) {
		close (device->fd);
	}
	g_free (device->name);
	g_free (device->port);
	gpilot_hdb_uucp_unlock (device);
}

static void
gpilot_serial_device_deinit (GPilotDevice *device)
{
	if (device->io) {
		g_source_remove (device->in_handle);
		g_source_remove (device->err_handle);
		g_io_channel_close (device->io);
		g_io_channel_unref (device->io);
	}
}

#ifdef WITH_USB_VISOR
static gint 
gpilot_usb_device_init (GPilotDevice *device)
{
	gpilot_hdb_uucp_lock (device);
	device->fd = -1;
	device->io = NULL;
	device->device_exists = FALSE;
	return 0;
}

static void
gpilot_usb_device_free (GPilotDevice *device)
{
	gpilot_hdb_uucp_unlock (device);
}
static void
gpilot_usb_device_deinit (GPilotDevice *device)
{
}
#endif /* WITH_USB_VISOR */

#ifdef WITH_IRDA /* WITH_IRDA */
static gint 
gpilot_irda_device_init (GPilotDevice *device)
{
	return gpilot_serial_device_init (device);
}

static void
gpilot_irda_device_free (GPilotDevice *device)
{
	gpilot_serial_device_free (device);
}

static void
gpilot_irda_device_deinit (GPilotDevice *device)
{
	gpilot_serial_device_deinit (device);
}
#endif /* WITH_IRDA */

#ifdef WITH_NETWORK

/* 
 * pi-csd.c: Connection Service Daemon, required for accepting 
 *           logons via NetSync (tm)
 * Copyright (c) 1997, Kenneth Albanowski
 *
 * While this function is useful in pi-csd, it is intended also to
 * be a demonstration of the proper (or improper, if I'm unlucky) techniques
 * to retrieve networking information.
 */
static void
fetch_host (char *hostname, int hostlen, struct in_addr *address,
	   struct in_addr *mask)
{
	int s, n, i;
	struct ifconf ifc;
	struct ifreq *ifr, ifreqaddr, ifreqmask;
	struct hostent *hent;

#ifdef HAVE_GETHOSTNAME
	/* Get host name the easy way */

	gethostname (hostname, hostlen);
#else
# ifdef HAVE_UNAME
	struct utsname uts;

	if (uname (&uts) == 0) {
		strncpy (hostname, uts.nodename, hostlen - 1);
		hostname[hostlen - 1] = '\0';
	}
# endif	/*def HAVE_UNAME */
#endif /*def HAVE_GETHOSTNAME */

	/* Get host address through DNS */
	hent = gethostbyname (hostname);

	if (hent) {
		while (*hent->h_addr_list) {
			struct in_addr haddr;

			memcpy (&haddr, *(hent->h_addr_list++), sizeof (haddr));
			if (haddr.s_addr != inet_addr ("127.0.0.1"))
				memcpy (address, &haddr, sizeof (haddr));
		}
	}
#if defined (SIOCGIFCONF) && defined (SIOCGIFFLAGS)
	s = socket (AF_INET, SOCK_DGRAM, 0);

	if (s < 0)
		return;

	ifc.ifc_buf = calloc (1024, 1);
	ifc.ifc_len = 1024;

	if (ioctl (s, SIOCGIFCONF, (char *) &ifc) < 0)
		goto done;

	n = ifc.ifc_len;
	for (i = 0; i < n; i += ifreq_size (*ifr)) {
		struct sockaddr_in *a;
		struct sockaddr_in *b;

		ifr = (struct ifreq *) ((caddr_t) ifc.ifc_buf + i);
		a = (struct sockaddr_in *) &ifr->ifr_addr;
		strncpy (ifreqaddr.ifr_name, ifr->ifr_name, sizeof (ifreqaddr.ifr_name));
		strncpy (ifreqmask.ifr_name, ifr->ifr_name, sizeof (ifreqmask.ifr_name));

		if (ioctl (s, SIOCGIFFLAGS, (char *) &ifreqaddr) < 0)
			continue;

		/* Reject loopback device */
#ifdef IFF_LOOPBACK
		if (ifreqaddr.ifr_flags & IFF_LOOPBACK)
			continue;
#endif /*def IFF_LOOPBACK */

#ifdef IFF_UP
		/* Reject down devices */
		if (!(ifreqaddr.ifr_flags & IFF_UP))
			continue;
#endif /*def IFF_UP */

		if (ifr->ifr_addr.sa_family != AF_INET)
			continue;

		/* If it is a point-to-point device, use the dest address */
#if defined (IFF_POINTOPOINT) && defined (SIOCGIFDSTADDR)
		if (ifreqaddr.ifr_flags & IFF_POINTOPOINT) {
			if (ioctl (s, SIOCGIFDSTADDR, (char *) &ifreqaddr) < 0)
				break;

			a = (struct sockaddr_in *) &ifreqaddr.ifr_dstaddr;

			if (address->s_addr == 0) {
				memcpy (address, &a->sin_addr, sizeof (struct in_addr));
			}
		}
		else
#endif /*defined (IFF_POINTOPOINT) && defined (SIOCGIFDSTADDR) */
			/* If it isn't a point-to-point device, use the address */
#ifdef SIOCGIFADDR
		{
			if (ioctl (s, SIOCGIFADDR, (char *) &ifreqaddr) < 0)
				break;

			a = (struct sockaddr_in *) &ifreqaddr.ifr_addr;

			if (address->s_addr == 0) {
				memcpy (address, &a->sin_addr, sizeof (struct in_addr));
			}
		}
#endif /*def SIOCGIFADDR */
		/* OK, we've got an address */

		/* Compare netmask against the current address and see if it seems to match. */
#ifdef SIOCGIFNETMASK
		if (ioctl (s, SIOCGIFNETMASK, (char *) &ifreqmask) < 0)
			break;

/* Is there any system where we need to use ifr_netmask?  */
#if 1
		b = (struct sockaddr_in *) &ifreqmask.ifr_addr;
#else
		b = (struct sockaddr_in *) &ifreqmask.ifr_netmask;
#endif

		if ((mask->s_addr == 0) && (address->s_addr != 0)) {
			if ((b->sin_addr.s_addr & a->sin_addr.s_addr) ==
			    (b->sin_addr.s_addr & address->s_addr)) {
				memcpy (mask, &b->sin_addr, sizeof (struct in_addr));

				/* OK, we've got a netmask */

				break;
			}
		}
#endif /*def SIOCGIFNETMASK */

	}

 done:
	free (ifc.ifc_buf);
	close (s);
#endif /*defined (SIOCGIFCONF) && defined (SIOCGIFFLAGS) */
}

/* gpilot_network_device_init (GPilotDevice *device)
 * pi-csd.c: Connection Service Daemon, required for accepting 
 *           logons via NetSync (tm)
 * Copyright (c) 1997, Kenneth Albanowski
 *
 */
static gint 
gpilot_network_device_init (GPilotDevice *device)
{
/*
 * Open a file descriptor. device->fd
 * Check fd. return -1 if fail.
 * device->io = g_io_channel_unix_new (device->fd);
 * g_io_channel_ref (device->io);
 * return 0;
 *
 */

/* char hostname[130];	this is device->host */
	struct in_addr address, netmask;


/*   int sockfd;	this is device->fd */
	struct sockaddr_in serv_addr, cli_addr;
	int clilen;
	char mesg[1026];
	fd_set rset;
	int n;
	struct in_addr raddress;

	extern char *optarg;
	extern int optind;

	struct hostent *hent;
/*   int quiet = 0;		It's Always quiet here */

	memset (&address, 0, sizeof (address));
	memset (&netmask, 0, sizeof (netmask));

/*   hostname[0] = 0; */
/* device->host = g_new (gchar, 130);
   g_free (device->host);
*/

/* This is used if the hostname is not correct in the conf file
   I'll move it to the conf file saving later */
	fetch_host (device->host, 128, &address, &netmask); 
	
/* device->host I already have from the config file */

	if (inet_aton (device->ip, &address) == 0) /* Invalid */
	{
		if ((hent = gethostbyname (device->ip))) {
			memcpy (&address.s_addr, hent->h_addr, sizeof (address));
		} else {
			g_message ("Invalid ip address '%s'", device->ip);
			return -1;
		}
	}

	if (inet_aton (device->netmask, &netmask) == 0) {
		g_message ("Invalid netmask '%s'", device->netmask);
		return -1;
	}

	/* cannot execute without address and hostname */
	if ((address.s_addr == 0) || (strlen (device->host) == 0))
	{
		g_message ("Cannot execute without ip address and hostname.");
		return -1;
	}

	device->fd = socket (AF_INET, SOCK_DGRAM, 0);
	if (device->fd < 0) {
		g_message ("Unable to get socket");
		return -1;
	}

	memset (&serv_addr, 0, sizeof (serv_addr));
	serv_addr.sin_family = AF_INET;
	serv_addr.sin_addr.s_addr = htonl (INADDR_ANY);
	serv_addr.sin_port = htons (14237);

	if (bind (device->fd, (struct sockaddr *) &serv_addr, sizeof (serv_addr)) < 0) {
		g_message ("Unable to bind socket");
		return -1;
	}
	/*
	g_message ("Connection Service Daemon for Palm Computing (tm) device active.");
	g_message ("Accepting connection requests for '%s' at %s with mask %s", 
		   device->host, inet_ntoa (address), inet_ntoa (netmask));
	*/

	/* We don't want to infinite loops here */
	/* Lets try something else instead */
	device->io = g_io_channel_unix_new (device->fd);
	g_io_channel_ref (device->io);

	/* All went well.. */
	return 0;
}

static void
gpilot_network_device_free (GPilotDevice *device)
{
	if (device->fd) {
		close (device->fd);
	}

	g_free(device->ip);
	g_free(device->host);
	g_free(device->netmask);
}

static void
gpilot_network_device_deinit (GPilotDevice *device)
{
}
#endif /* WITH_NETWORK */

gint
gpilot_device_init (GPilotDevice *device)
{
	gint result;

	g_return_val_if_fail (device != NULL,-1);
	
	if (device->type == PILOT_DEVICE_SERIAL) {
		result = gpilot_serial_device_init (device);
#ifdef WITH_IRDA
	} else if (device->type == PILOT_DEVICE_IRDA) {
		result = gpilot_irda_device_init (device);
#endif /* WITH_IRDA */
#ifdef WITH_USB_VISOR
	} else if (device->type == PILOT_DEVICE_USB_VISOR) {
		result = gpilot_usb_device_init (device);
#endif /* WITH_USB_VISOR */
#ifdef WITH_NETWORK
	} else if (device->type == PILOT_DEVICE_NETWORK) {
		result = gpilot_network_device_init (device);
#endif /* WITH_NETWORK */
	} else {
		g_warning (_("Unknown device type"));
	}

	return result;
}

static void
gpilot_serial_device_load (GPilotDevice *device)
{
	device->port = gnome_config_get_string ("device");
	device->speed = (guint)gnome_config_get_int ("speed=57600");
}

static void
gpilot_network_device_load (GPilotDevice *device)
{
	device->ip = gnome_config_get_string ("ip");
	device->host = gnome_config_get_string ("host");
	device->netmask = gnome_config_get_string ("netmask");
}

gint
gpilot_device_load (GPilotDevice *device, gint i)
{
	gchar prefix[40];
	gchar tmp[40];
	gint result = 0;

	g_return_val_if_fail (device != NULL,-1);
	g_return_val_if_fail (prefix != NULL,-1);
	
	g_snprintf (prefix, 39,"/gnome-pilot.d/gpilotd/Device%d/", i);
	gnome_config_push_prefix (prefix);

	g_snprintf (tmp, 39,"name=Cradle%d", i);
	device->type = gnome_config_get_int ("type=0");
	device->name = gnome_config_get_string (tmp);
	device->timeout = gnome_config_get_int ("timeout=3");

	switch (device->type) {
	case PILOT_DEVICE_SERIAL:
	case PILOT_DEVICE_USB_VISOR:
	case PILOT_DEVICE_IRDA:
		/* These devices share the serial loader */
		gpilot_serial_device_load (device);
		break;
	case PILOT_DEVICE_NETWORK:
		gpilot_network_device_load (device);
		break;
	default:
		g_warning (_("Unknown device type"));
	}

	gnome_config_pop_prefix ();

	return result;
}

void
gpilot_device_deinit (GPilotDevice *device)
{
	g_assert (device != NULL);

	if (device->type == PILOT_DEVICE_SERIAL) {
		gpilot_serial_device_deinit (device);
#ifdef WITH_IRDA
	} else if (device->type == PILOT_DEVICE_IRDA) {
		gpilot_irda_device_deinit (device);
#endif /* WITH_IRDA */
#ifdef WITH_USB_VISOR
	} else if (device->type == PILOT_DEVICE_USB_VISOR) {
		gpilot_usb_device_deinit (device);
#endif /* WITH_USB_VISOR */
#ifdef WITH_NETWORK
	} else if (device->type == PILOT_DEVICE_NETWORK) {
		gpilot_network_device_deinit (device);
#endif /* WITH_NETWORK */
	} else {
		g_warning (_("Unknown device type"));
	}
}

void
gpilot_device_free (GPilotDevice *device)
{
	g_assert (device != NULL);

	gpilot_device_deinit (device);

	if (device->type == PILOT_DEVICE_SERIAL) {
		gpilot_serial_device_free (device);
#ifdef WITH_IRDA
	} else if (device->type == PILOT_DEVICE_IRDA) {
		gpilot_irda_device_free (device);
#endif /* WITH_IRDA */
#ifdef WITH_USB_VISOR
	} else if (device->type == PILOT_DEVICE_USB_VISOR) {
		gpilot_usb_device_free (device);
#endif /* WITH_USB_VISOR */
#ifdef WITH_NETWORK
	} else if (device->type == PILOT_DEVICE_NETWORK) {
		gpilot_network_device_free (device);
#endif /* WITH_NETWORK */
	} else {
		g_warning (_("Unknown device type"));
	}

	g_free (device);
}

static GList *
get_devices (void)
{
	GList * retval = NULL;
	int n, i, final;
  
	final = n = gnome_config_get_int ("/gnome-pilot.d/gpilotd/General/num_devices=0");

	if (n==0) {
		g_warning (_("Number of devices is configured to 0"));
#ifdef GUI
		gpilot_gui_warning_dialog (_("No cradles configure.\n"
					    "Please run gpilotd-control-applet (use gnomecc)\n" 
					    "to configure gnome-pilot."));    
#endif
	}

	for (i=0;i<n;i++) {
		GPilotDevice *device;

		device = gpilot_device_new ();

		if (gpilot_device_load (device, i) == 0) {
			if (gpilot_device_init (device) == 0) {
				retval = g_list_append (retval, device);
			}
		} else {
			final --;
		}
	}

	if (final == 0) {
		g_warning (_("No accessible devices available"));
	}

	return retval;
}

static GList *
get_pilots (void)
{
	GList * retval = NULL;
	int n, i;
  
	n = gnome_config_get_int ("/gnome-pilot.d/gpilotd/General/num_pilots=0");

	if (n==0) {
		g_warning (_("Number of pilots is configured to 0"));
#ifdef GUI
		gpilot_gui_warning_dialog (_("No pilots configured.\n"
					    "Please run gpilotd-control-applet (use gnomecc)\n" 
					    "to configure gnome-pilot."));    
#endif
	}


	for (i=0; i<n; i++) {
		GPilotPilot *pilot;
		pilot = gpilot_pilot_new ();
		gpilot_pilot_init (pilot, i);
		retval = g_list_append (retval, pilot);
	}
	return retval;
}



GPilotPilot *
gpilot_pilot_new (void)
{
	GPilotPilot *retval;
	retval = g_new0(GPilotPilot,1);
	return retval;
}

void
gpilot_pilot_init (GPilotPilot *pilot, 
		   gint i)
{
	gchar *tmp_str;
	gchar *prefix;

	/* set up stuff  */
	g_free (pilot->name);
	g_free (pilot->passwd);
	g_free (pilot->pilot_username);
	g_list_foreach (pilot->trusted_users, (GFunc) g_free, NULL);
	g_list_free (pilot->trusted_users);
	prefix = g_strdup_printf ("/gnome-pilot.d/gpilotd/Pilot%d/", i);
	gnome_config_push_prefix (prefix);

	/* start filling in fields */
	pilot->name = gnome_config_get_string("name");
	pilot->pilot_id = gnome_config_get_int("pilotid");
	pilot->pilot_username = gnome_config_get_string("pilotusername");
	pilot->passwd = gnome_config_get_string("password");
	pilot->creation = gnome_config_get_int ("creation");
	pilot->romversion = gnome_config_get_int ("romversion");
	pilot->number=i;
	pilot->trusted_users=NULL;
	tmp_str = gnome_config_get_string("default_sync_action=synchronize");
	pilot->sync_options.default_sync_action = gnome_pilot_conduit_sync_type_str_to_int(tmp_str);
	g_free (tmp_str);
	pilot->sync_options.basedir = gnome_config_get_string("basedir");

	gnome_config_pop_prefix ();
	g_free(prefix);
}

void
gpilot_pilot_free (GPilotPilot *pilot)
{
	g_free (pilot->name);
	g_free (pilot->passwd);
	g_free (pilot->pilot_username);
	g_free (pilot->sync_options.basedir);
	g_list_foreach (pilot->trusted_users, (GFunc) g_free, NULL);
	g_list_free (pilot->trusted_users);
	g_free (pilot);
}

GPilotUser *
gpilot_user_new (void) {
	GPilotUser *retval;

	retval = (GPilotUser *)g_malloc(sizeof(GPilotUser));
	retval->username = NULL;

	return retval;
}

void
gpilot_user_free (GPilotUser *user)
{
	g_free (user->username);
	g_free (user);
}
 
GPilotPilot*
gpilot_find_pilot_by_name(gchar *id,GList *inlist) 
{
	GList *iterator;
	iterator = inlist;
	while(iterator!=NULL) {
		GPilotPilot *retval;
		retval = GPILOT_PILOT(iterator->data);
		if (strcmp(retval->name,id)==0) return retval;
		iterator = iterator->next;
	}
	return NULL;
}

GPilotPilot*
gpilot_find_pilot_by_id(guint32 id,GList *inlist) 
{
	GList *iterator;
	iterator = inlist;
	while(iterator!=NULL) {
		GPilotPilot *retval;
		retval = GPILOT_PILOT(iterator->data);
		if (retval->pilot_id ==id) return retval;
		iterator = iterator->next;
	}
	return NULL;
}

gint
gnome_pilot_conduit_sync_type_str_to_int(const gchar *s) 
{
	g_return_val_if_fail(s!=NULL,GnomePilotConduitSyncTypeNotSet);

	if (strcmp (s, "synchronize") == 0) {
		return GnomePilotConduitSyncTypeSynchronize;
	} else if (strcmp (s, "copy_to_pilot") == 0) {
		return GnomePilotConduitSyncTypeCopyToPilot;
	} else if (strcmp (s, "copy_from_pilot") == 0) {
		return GnomePilotConduitSyncTypeCopyFromPilot;
	} else if (strcmp (s, "merge_to_pilot") == 0) {
		return GnomePilotConduitSyncTypeMergeToPilot;
	} else if (strcmp (s, "merge_from_pilot") == 0) {
		return GnomePilotConduitSyncTypeMergeFromPilot;
	} else if (strcmp (s, "custom") == 0) {
		return GnomePilotConduitSyncTypeCustom;
	} else if (strcmp (s, "not_set") == 0) {
		return GnomePilotConduitSyncTypeNotSet;
	}
	return GnomePilotConduitSyncTypeNotSet;
}

const gchar* 
gnome_pilot_conduit_sync_type_int_to_str(GnomePilotConduitSyncType e) 
{
	switch(e) {
	case GnomePilotConduitSyncTypeCustom:         return "custom";
	case GnomePilotConduitSyncTypeSynchronize:    return "synchronize";
	case GnomePilotConduitSyncTypeCopyFromPilot:  return "copy_from_pilot";
	case GnomePilotConduitSyncTypeCopyToPilot:    return "copy_to_pilot";
	case GnomePilotConduitSyncTypeMergeFromPilot: return "merge_from_pilot";
	case GnomePilotConduitSyncTypeMergeToPilot:   return "merge_to_pilot";
	case GnomePilotConduitSyncTypeNotSet:         return "not_set";
	default: 
	  g_message ("gnome_pilot_conduit_sync_type_int_to_str: invalid sync_type %d",e);
	  return "not_set";
	}
}
