#include "parseAFM.h"
#include <gnome.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <dirent.h>
#include <string.h>
#include <ctype.h>
#include "gnome-font.h"

#define noVERBOSE

typedef struct _GnomeFontType1EncTab GnomeFontType1EncTab;

struct _GnomeFontType1EncTab {
  int unicode;
  char *type1_name;
};

static void gnome_font_class_init (GnomeFontClass *klass);

static void
gnome_font_init (GnomeFont *font);

static void
gnome_font_finalize (GtkObject *object);

static GtkObjectClass *parent_class = NULL;

/* These are just the glyphs in the standard encoding. I need to add
   the rest of the glyphs, accessible through nonstandard
   encodings. */

/* This table is adapted very straightforwardly from
   ftp.unicode.org/Public/MAPPINGS/VENDORS/ADOBE/STDENC.TXT.

   I have reviewed them and compared them against an independently
   constructed encoding, and believe they are correct.
  */

const GnomeFontType1EncTab gnome_font_type1_std_enc[] = {
  { 0x0020, "space" },
  { 0x0021, "exclam" },
  { 0x0022, "quotedbl" },
  { 0x0023, "numbersign" },
  { 0x0024, "dollar" },
  { 0x0025, "percent" },
  { 0x0026, "ampersand" },
  { 0x0027, "quotesingle" },
  { 0x0028, "parenleft" },
  { 0x0029, "parenright" },
  { 0x002A, "asterisk" },
  { 0x002B, "plus" },
  { 0x002C, "comma" },
  { 0x002D, "hyphen" },
  { 0x002E, "period" },
  { 0x002F, "slash" },
  { 0x0030, "zero" },
  { 0x0031, "one" },
  { 0x0032, "two" },
  { 0x0033, "three" },
  { 0x0034, "four" },
  { 0x0035, "five" },
  { 0x0036, "six" },
  { 0x0037, "seven" },
  { 0x0038, "eight" },
  { 0x0039, "nine" },
  { 0x003A, "colon" },
  { 0x003B, "semicolon" },
  { 0x003C, "less" },
  { 0x003D, "equal" },
  { 0x003E, "greater" },
  { 0x003F, "question" },
  { 0x0040, "at" },
  { 0x0041, "A" },
  { 0x0042, "B" },
  { 0x0043, "C" },
  { 0x0044, "D" },
  { 0x0045, "E" },
  { 0x0046, "F" },
  { 0x0047, "G" },
  { 0x0048, "H" },
  { 0x0049, "I" },
  { 0x004A, "J" },
  { 0x004B, "K" },
  { 0x004C, "L" },
  { 0x004D, "M" },
  { 0x004E, "N" },
  { 0x004F, "O" },
  { 0x0050, "P" },
  { 0x0051, "Q" },
  { 0x0052, "R" },
  { 0x0053, "S" },
  { 0x0054, "T" },
  { 0x0055, "U" },
  { 0x0056, "V" },
  { 0x0057, "W" },
  { 0x0058, "X" },
  { 0x0059, "Y" },
  { 0x005A, "Z" },
  { 0x005B, "bracketleft" },
  { 0x005C, "backslash" },
  { 0x005D, "bracketright" },
  { 0x005E, "asciicircum" },
  { 0x005F, "underscore" },
  { 0x0060, "grave" },
  { 0x0061, "a" },
  { 0x0062, "b" },
  { 0x0063, "c" },
  { 0x0064, "d" },
  { 0x0065, "e" },
  { 0x0066, "f" },
  { 0x0067, "g" },
  { 0x0068, "h" },
  { 0x0069, "i" },
  { 0x006A, "j" },
  { 0x006B, "k" },
  { 0x006C, "l" },
  { 0x006D, "m" },
  { 0x006E, "n" },
  { 0x006F, "o" },
  { 0x0070, "p" },
  { 0x0071, "q" },
  { 0x0072, "r" },
  { 0x0073, "s" },
  { 0x0074, "t" },
  { 0x0075, "u" },
  { 0x0076, "v" },
  { 0x0077, "w" },
  { 0x0078, "x" },
  { 0x0079, "y" },
  { 0x007A, "z" },
  { 0x007B, "braceleft" },
  { 0x007C, "bar" },
  { 0x007D, "braceright" },
  { 0x007E, "asciitilde" },
  { 0x00A1, "exclamdown" },
  { 0x00A2, "cent" },
  { 0x00A3, "sterling" },
  { 0x00A4, "currency" },
  { 0x00A5, "yen" },
  { 0x00A7, "section" },
  { 0x00A8, "dieresis" },
  { 0x00AA, "ordfeminine" },
  { 0x00AB, "guillemotleft" },
  { 0x00AF, "macron" },
  { 0x00B4, "acute" },
  { 0x00B6, "paragraph" },
  { 0x00B7, "periodcentered" },
  { 0x00B8, "cedilla" },
  { 0x00BA, "ordmasculine" },
  { 0x00BB, "guillemotright" },
  { 0x00BF, "questiondown" },
  { 0x00C6, "AE" },
  { 0x00D8, "Oslash" },
  { 0x00DF, "germandbls" },
  { 0x00E6, "ae" },
  { 0x00F8, "oslash" },
  { 0x0131, "dotlessi" },
  { 0x0141, "Lslash" },
  { 0x0142, "lslash" },
  { 0x0152, "OE" },
  { 0x0153, "oe" },
  { 0x0192, "florin" },
  { 0x02C6, "circumflex" },
  { 0x02C7, "caron" },
  { 0x02D8, "breve" },
  { 0x02D9, "dotaccent" },
  { 0x02DA, "ring" },
  { 0x02DB, "ogonek" },
  { 0x02DC, "tilde" },
  { 0x02DD, "hungarumlaut" },
  { 0x2013, "endash" },
  { 0x2014, "emdash" },
  { 0x2018, "quoteleft" },
  { 0x2019, "quoteright" },
  { 0x201A, "quotesinglbase" },
  { 0x201C, "quotedblleft" },
  { 0x201D, "quotedblright" },
  { 0x201E, "quotedblbase" },
  { 0x2020, "dagger" },
  { 0x2021, "daggerdbl" },
  { 0x2022, "bullet" },
  { 0x2026, "ellipsis" },
  { 0x2030, "perthousand" },
  { 0x2039, "guilsinglleft" },
  { 0x203A, "guilsinglright" },
  { 0x2044, "fraction" },
  { 0xFB01, "fi" },
  { 0xFB02, "fl" }
};

GtkType
gnome_font_get_type (void)
{
  static GtkType font_type = 0;

  if (!font_type)
    {
      GtkTypeInfo font_info =
      {
	"GnomeFont",
	sizeof (GnomeFont),
	sizeof (GnomeFontClass),
	(GtkClassInitFunc) gnome_font_class_init,
	(GtkObjectInitFunc) gnome_font_init,
	/* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      font_type = gtk_type_unique (gtk_object_get_type (), &font_info);
    }

  return font_type;
}

static gboolean
gnome_font_is_afm_file (const char *fn)
{
  int len;

  len = strlen (fn);
  return (len > 4 &&
	  fn[len - 4] == '.' &&
	  tolower (fn[len - 3]) == 'a' &&
	  tolower (fn[len - 2]) == 'f' &&
	  tolower (fn[len - 1]) == 'm');
}

/* Try to find the filename for the font */
static int
gnome_font_find (GnomeFontClass *class, char *fontname)
{
  int i;

  for (i = 0; i < class->n_fonts; i++)
    if (!strcmp (class->fontmap[i].font_name, fontname))
      return i;
  return -1;
}

static char *
gnome_font_find_filename (GnomeFontClass *class, char *fontname)
{
  int i;

  i = gnome_font_find (class, fontname);
  if (i >= 0)
    return class->fontmap[i].afm_fn;
  else
    return NULL;
}

typedef struct _GnomeFontWeightTab GnomeFontWeightTab;

struct _GnomeFontWeightTab {
  const char *name;
  int code;
};

static void
gnome_font_add_mapping (GnomeFontClass *class, char *fontname,
			char *afm_fn, char *pfb_fn, char *fullname,
			char *familyname, char *weight)
{
  int n_fonts;
  const GnomeFontWeightTab weight_tab[] = {
    { "Extra Light", GNOME_FONT_EXTRA_LIGHT },
    { "Extralight", GNOME_FONT_EXTRA_LIGHT },

    { "Thin", GNOME_FONT_THIN },

    { "Light", GNOME_FONT_LIGHT },

    { "Book", GNOME_FONT_BOOK },
    { "Roman", GNOME_FONT_BOOK },
    { "Regular", GNOME_FONT_BOOK },

    { "Medium", GNOME_FONT_MEDIUM },

    { "Semi", GNOME_FONT_SEMI },
    { "Semibold", GNOME_FONT_SEMI },
    { "Demi", GNOME_FONT_SEMI },
    { "Demibold", GNOME_FONT_SEMI },

    { "Bold", GNOME_FONT_BOLD },

    { "Heavy", GNOME_FONT_HEAVY },
 
    { "Extra", GNOME_FONT_EXTRABOLD },
    { "Extra Bold", GNOME_FONT_EXTRABOLD },

    { "Black", GNOME_FONT_BLACK },

    { "Extra Black", GNOME_FONT_EXTRABLACK },
    { "Extrablack", GNOME_FONT_EXTRABLACK },
    { "Ultra Bold", GNOME_FONT_EXTRABLACK }
  };
  int i;
  int len_full;
  GnomeFontWeight weight_code;


#ifdef VERBOSE
  g_print ("Adding %s %s\n", fontname, afm_fn);
#endif
  if (gnome_font_find (class, fontname) >= 0)
    {
      /* already present in fontmap */
#ifdef VERBOSE
      g_print ("Font %s already present\n", fontname);
#endif
      return;
    }

  n_fonts = (class->n_fonts)++;
  if (n_fonts == 0)
    class->fontmap = g_new (GnomeFontMap, 1);
  else if ((n_fonts & (n_fonts - 1)) == 0)
    {
#ifdef VERBOSE
      g_print ("reallocating to %d\n", n_fonts * 2);
#endif
      class->fontmap = g_realloc (class->fontmap,
				  sizeof(GnomeFontMap) * n_fonts * 2);
    }
  class->fontmap[n_fonts].font_name = g_strdup (fontname);
  class->fontmap[n_fonts].afm_fn = g_strdup (afm_fn);
  class->fontmap[n_fonts].pfb_fn = g_strdup (pfb_fn);
  class->fontmap[n_fonts].fullname = g_strdup (fullname);
  class->fontmap[n_fonts].familyname = g_strdup (familyname);
  class->fontmap[n_fonts].weight = g_strdup (weight);
  class->fontmap[n_fonts].widths = NULL;
  class->fontmap[n_fonts].subst_glyph = NULL;
  weight_code = GNOME_FONT_BOOK;
  for (i = 0; i < sizeof(weight_tab) / sizeof(weight_tab[0]); i++)
    if (!g_strcasecmp (weight, weight_tab[i].name))
      {
	weight_code = weight_tab[i].code;
	break;
      }
  class->fontmap[n_fonts].weight_code = weight_code;
  len_full = strlen (fullname);
  class->fontmap[n_fonts].italic =
    (len_full >= 7 && !g_strcasecmp (fullname + len_full - 7, "Oblique")) ||
    (len_full >= 6 && !g_strcasecmp (fullname + len_full - 6, "Italic"));
}

static void
gnome_font_add_glyph_alias (GnomeFontClass *class, char *fontname,
			    char *subst_font_name)
{
  int n_glyph_alias;
  int i;
  int font_idx, subst_idx;

#ifdef VERBOSE
  g_print ("Adding glyph-alias %s %s\n", fontname, subst_font_name);
#endif
  n_glyph_alias = class->n_glyph_alias;

  for (i = 0; i < n_glyph_alias; i++)
    if (!strcmp (class->glyph_alias[i].font_name, fontname))
      {
	/* already present in glyph alias table */
#ifdef VERBOSE
	g_print ("Glyph alias %s already present\n", fontname);
#endif
	return;
      }

  class->n_glyph_alias++;
  if (n_glyph_alias == 0)
    class->glyph_alias = g_new (GnomeFontGlyphAlias, 1);
  else if ((n_glyph_alias & (n_glyph_alias - 1)) == 0)
    {
#ifdef VERBOSE
      g_print ("reallocating to %d\n", n_glyph_alias * 2);
#endif
      class->glyph_alias = g_realloc (class->glyph_alias,
				      sizeof(GnomeFontGlyphAlias) *
				      n_glyph_alias * 2);
    }
  class->glyph_alias[n_glyph_alias].font_name = g_strdup (fontname);
  class->glyph_alias[n_glyph_alias].subst_font_name =
    g_strdup (subst_font_name);

  font_idx = gnome_font_find (class, fontname);
  if (font_idx < 0)
    {
#ifdef VERBOSE
      g_print ("Aliased font %s not found\n", fontname);
#endif
    }
  subst_idx = gnome_font_find (class, subst_font_name);
  if (subst_idx < 0)
    {
#ifdef VERBOSE
      g_print ("Alias %s not found\n", subst_font_name);
#endif
    }
  class->fontmap[font_idx].subst_glyph = &class->fontmap[subst_idx];
}

#if 0
/* We're not doing it this way, we're using fontmap files */
static void
gnome_font_refresh_font_dir (GnomeFontClass *class, char *dirname)
{
  DIR *dir;
  struct dirent *dir_ent;
  char *fn;
  FILE *afm_f;
  FontInfo *fi;
  int status;

  g_print (_("scanning dir %s\n"), dirname);
  dir = opendir (dirname);
  if (dir != NULL)
    {
      while (1)
	{
	  dir_ent = readdir (dir);
	  if (dir_ent == NULL)
	    break;
	  if (gnome_font_is_afm_file (dir_ent->d_name))
	    {
	      g_print (_("checking file %s\n"), dir_ent->d_name);
	      fn = g_malloc (strlen (dirname) + strlen (dir_ent->d_name) + 2);
	      sprintf (fn, _("%s/%s"), dirname, dir_ent->d_name);
	      afm_f = fopen (fn, "rb");
	      if (afm_f != NULL)
		{
		  status = parseFile (afm_f, &fi, P_G);
		  g_print ("status = %d\n", status);
		  if (status == 0)
		    {
		      gnome_font_add_mapping (class, fi->gfi->fontName, fn);
		      /* todo: free fi */
		    }

		  fclose (afm_f);
		}
	    }
	}
    }
}
#endif

static char *
gnome_font_get_field (char **pp) {
  char *p;
  int i, j, n;
  char *field;

  p = *pp;
  for (i = 0; p[i] && p[i] != ' ' && p[i] != '\n'; i++)
    if (p[i] == '\\' && p[i + 1])
      i++;
  n = i;
  field = g_malloc (n + 1);
  j = 0;
  for (i = 0; i < n; i++)
    if (p[i] == '\\' && p[i + 1])
      field[j++] = p[++i];
    else
      field[j++] = p[i];
  field[j++] = 0;

  /* update pointer to point to next field */
  if (p[n]) n++;
  *pp += n;

  return field;
}

static void
gnome_font_load_fontmap (GnomeFontClass *class, const char *fn)
{
  FILE *f;
  char linebuf[1024];
  char *p;
  char *type;
  char *fontname, *afmfile, *pfbfile, *fullname, *familyname, *weight;
  char *subst_font_name;

#ifdef VERBOSE
  g_print ("filename %s\n", fn);
#endif
  f = fopen (fn, "r");
  if (f == NULL)
    return;
  while (fgets (linebuf, sizeof(linebuf), f))
    {
      /* parse the fontmap line */
      p = linebuf;
      type = gnome_font_get_field (&p);
      if (!strcmp (type, "type1"))
	{
	  fontname = gnome_font_get_field (&p);
	  afmfile = gnome_font_get_field (&p);
	  pfbfile = gnome_font_get_field (&p);
	  fullname = gnome_font_get_field (&p);
	  familyname = gnome_font_get_field (&p);
	  weight = gnome_font_get_field (&p);
	  gnome_font_add_mapping (class, fontname, afmfile, pfbfile,
				  fullname, familyname, weight);
#ifdef VERBOSE
	  printf ("%s %s %s %s %s %s\n",
		  fontname, afmfile, pfbfile, fullname, familyname, weight);
#endif
	  g_free (weight);
	  g_free (familyname);
	  g_free (fullname);
	  g_free (pfbfile);
	  g_free (afmfile);
	  g_free (fontname);
	}
      else if (!strcmp (type, "glyph-alias"))
	{
	  fontname = gnome_font_get_field (&p);
	  subst_font_name = gnome_font_get_field (&p);
	  gnome_font_add_glyph_alias (class, fontname, subst_font_name);
#ifdef VERBOSE
	  printf ("%s %s\n",
		  fontname, subst_font_name);
#endif
	  g_free (subst_font_name);
	  g_free (fontname);
	}
      g_free (type);
    }
  fclose (f);
}

static void
gnome_font_refresh_fontmap (GnomeFontClass *class)
{
  char *fontmap_fn;
  char *env_home;

  /* build fontmap from scanning the font path */
  env_home = getenv ("HOME");
  if (env_home != NULL)
    {
      fontmap_fn = g_malloc (strlen (env_home) + 32);
      sprintf (fontmap_fn, "%s/.gnome/fonts/fontmap", env_home);
      gnome_font_load_fontmap (class, fontmap_fn);
      g_free (fontmap_fn);
      fontmap_fn = gnome_datadir_file ("fonts/fontmap");
      if (fontmap_fn)
	{
	  gnome_font_load_fontmap (class, fontmap_fn);
	  g_free (fontmap_fn);
	}
    }
}

/* Looks up the name, returning the glyph number in the font's
   standard encoding. Return -1 if the glyph is not found in the
   font's standard encoding.

   There are two problems with this function: first, the implementation
   is quite slow (the analogous operation in libhnj uses a _much_
   faster hash table implementation. Second, we eventually want to
   break free of the font's standard encoding. My guess is that
   values < 256 will be glyphnums in the standard encoding, while
   values >= 256 will be in the "extension" encoding.

   Anyway, the point of this function is merely that it works. */
static int
gnome_font_find_glyphnum (Font_Info *fi, const char *name)
{
  int i;

  for (i = 0; i < fi->numOfChars; i++)
    {
      if (!strcmp (name, fi->cmi[i].name))
	return fi->cmi[i].code;
    }
  return -1;
}

static int
gnome_font_find_unicode_encoding (const GnomeFontType1EncTab *enc, int n_enc,
				  const char *glyphname)
{
  int i;

  for (i = 0; i < n_enc; i++)
    {
      if (!strcmp (glyphname, enc[i].type1_name))
	return enc[i].unicode;
    }
  return -1;
}

/* Add a mapping to the font's coverage table. */
static void
gnome_font_add_unimapping (GnomeFontMap *fontmap, int unicode, int glyphnum)
{
  int page;
  int **cov_pages;
  int *cov;
  int i;

  page = unicode >> 8;

  if (fontmap->cov_pages == NULL)
    {
      cov_pages = g_new (int *, 1);
      fontmap->cov_pages = cov_pages;
      fontmap->first_cov_page = page;
      fontmap->num_cov_pages = 1;
      cov = g_new (int, 256);
      for (i = 0; i < 256; i++)
	cov[i] = -1;
      cov_pages[0] = cov;
    }
  else if (page < fontmap->first_cov_page ||
	   page >= fontmap->first_cov_page + fontmap->num_cov_pages)
    {
      int first_cov_page, num_cov_pages;

      first_cov_page = MIN (page, fontmap->first_cov_page);
      num_cov_pages = MAX (page + 1 - first_cov_page, fontmap->num_cov_pages);
      cov_pages = g_new (int *, num_cov_pages);

      for (i = 0; i < first_cov_page - fontmap->first_cov_page; i++)
	cov_pages[i] = NULL;
      memcpy (cov_pages + fontmap->first_cov_page - first_cov_page,
	      fontmap->cov_pages,
	      fontmap->num_cov_pages * sizeof (int *));
      for (i = fontmap->first_cov_page + fontmap->num_cov_pages -
	     first_cov_page; i < num_cov_pages; i++)
	cov_pages[i] = NULL;

      fontmap->first_cov_page = first_cov_page;
      fontmap->num_cov_pages = num_cov_pages;
      g_free (fontmap->cov_pages);
      fontmap->cov_pages = cov_pages;

      cov = g_new (int, 256);
      for (i = 0; i < 256; i++)
	cov[i] = -1;
      fontmap->cov_pages[page - fontmap->first_cov_page] = cov;
      cov_pages[page - first_cov_page] = cov;
    }
  else
    {
      cov = fontmap->cov_pages[page - fontmap->first_cov_page];
      if (cov == NULL)
	{
	  cov = g_new (int, 256);
	  for (i = 0; i < 256; i++)
	    cov[i] = -1;
	  fontmap->cov_pages[page - fontmap->first_cov_page] = cov;
	}
    }
  cov[unicode & 0xff] = glyphnum;
}

#define KERN_PAIR_HASH(g1, g2) ((g1) * 367 + (g2) * 31)
static void
gnome_font_load_afm (GnomeFontMap *fontmap_entry)
{
  Font_Info *fi;
  FILE *afm_f;
  int status;

  afm_f = fopen (fontmap_entry->afm_fn, "rb");
  if (afm_f != NULL)
    {
      status = parseFile (afm_f, &fi, P_M | P_P);
#ifdef VERBOSE
      g_print (_("status loading %s = %d\n"), fontmap_entry->afm_fn, status);
#endif
      if (status == 0)
	{
	  int *widths;
	  GnomeFontLigList **ligtab;
	  Ligature *ligs;
	  int i;
	  GnomeFontKernPair *ktab;
	  int ktabsize;

	  widths = g_malloc (256 * sizeof (int));
	  fontmap_entry->widths = widths;

	  ligtab = g_malloc (256 * sizeof (GnomeFontLigList *));
	  fontmap_entry->ligs = ligtab;

#if 0
	  for (i = 0; i < 256; i++)
	    widths[i] = fi->cwi[i];
#else
	  for (i = 0; i < 256; i++)
	    widths[i] = 0;

	  for (i = 0; i < fi->numOfChars; i++)
	    {
	      int code;
	      int unicode;

	      /* Get the width */
	      code = fi->cmi[i].code;
	      if (code >= 0 && code < 256)
		{
		  widths[code] = fi->cmi[i].wx;

		  /* Get the ligature info */
		  for (ligs = fi->cmi[i].ligs; ligs != NULL; ligs = ligs->next)
		    {
		      int succ, lig;
		      GnomeFontLigList *ll;

		      succ = gnome_font_find_glyphnum (fi, ligs->succ);
		      lig = gnome_font_find_glyphnum (fi, ligs->lig);
		      if ((succ | lig) >= 0)
			{
			  ll = g_new (GnomeFontLigList, 1);
			  ll->succ = succ;
			  ll->lig = lig;
			  ll->next = ligtab[code];
			  ligtab[code] = ll;
			}
		    }

		  /* add in the mapping from Unicode */
		  unicode = gnome_font_find_unicode_encoding
		    (gnome_font_type1_std_enc,
		     sizeof(gnome_font_type1_std_enc) /
		     sizeof(gnome_font_type1_std_enc[0]),
		     fi->cmi[i].name);

		  if (unicode > 0)
		    gnome_font_add_unimapping (fontmap_entry, unicode, code);
		}
	    }
#endif

	  /* process the kern pairs */
	  for (ktabsize = 1; ktabsize < fi->numOfPairs << 1; ktabsize <<= 1);
	  ktab = g_new (GnomeFontKernPair, ktabsize);
	  fontmap_entry->kerns = ktab;
	  fontmap_entry->num_kerns = ktabsize;
	  for (i = 0; i < ktabsize; i++)
	    {
	      ktab[i].glyph1 = -1;
	      ktab[i].glyph2 = -1;
	      ktab[i].x_amt = 0;
	    }

	  for (i = 0; i < fi->numOfPairs; i++)
	    {
	      int glyph1, glyph2;
	      int j;

	      glyph1 = gnome_font_find_glyphnum (fi, fi->pkd[i].name1);
	      glyph2 = gnome_font_find_glyphnum (fi, fi->pkd[i].name2);
	      for (j = KERN_PAIR_HASH (glyph1, glyph2) & (ktabsize - 1);
		   ktab[j].glyph1 != -1;
		   j = (j + 1) & (ktabsize - 1));
	      ktab[j].glyph1 = glyph1;
	      ktab[j].glyph2 = glyph2;
	      ktab[j].x_amt = fi->pkd[i].xamt;
#ifdef VERBOSE
	      g_print ("kern pair %s(%d) %s(%d) %d\n",
		       fi->pkd[i].name1, glyph1,
		       fi->pkd[i].name2, glyph2,
		       fi->pkd[i].xamt);
#endif
	    }

#ifdef VERBOSE
	  for (i = 0; i < ktabsize; i++)
	    g_print ("%d %d %d\n",
		     ktab[i].glyph1, ktab[i].glyph2, ktab[i].x_amt);
#endif

	  /* Free the Font_Info structure */
	  free (fi->pkd);
	  for (i = 0; i < fi->numOfChars; i++)
	    {
	      Ligature *ligs, *next;

	      for (ligs = fi->cmi[i].ligs; ligs != NULL; ligs = next)
		{
		  next = ligs->next;
		  free (ligs->succ);
		  free (ligs->lig);
		  free (ligs);
		}
	      free (fi->cmi[i].name);
	    }
	  free (fi->cmi);
	  free (fi);
	}

      fclose (afm_f);
    }
}

/* Return the amount of kerning for the two glyphs. */
double
gnome_font_kern (GnomeFont *font, int glyph1, int glyph2)
{
  int ktabsize;
  GnomeFontKernPair *ktab;
  int j;
    
  ktabsize = font->fontmap_entry->num_kerns;
  ktab = font->fontmap_entry->kerns;
  for (j = KERN_PAIR_HASH (glyph1, glyph2) & (ktabsize - 1);
       ktab[j].glyph1 != -1;
       j = (j + 1) & (ktabsize - 1))
    if (ktab[j].glyph1 == glyph1 && ktab[j].glyph2 == glyph2)
      return ktab[j].x_amt * font->scale;
  return 0;
}

static void
gnome_font_class_init (GnomeFontClass *class)
{
  GtkObjectClass *object_class;

  object_class = (GtkObjectClass*) class;

  parent_class = gtk_type_class (gtk_object_get_type ());

  object_class->finalize = gnome_font_finalize;

  class->n_fonts = 0;
  class->fontmap = NULL;

  class->n_glyph_alias = 0;
  class->glyph_alias = NULL;

  gnome_font_refresh_fontmap (class);
}

static void
gnome_font_init (GnomeFont *font)
{
  font->num = -1;
  font->size = 0;
}

GnomeFont *
gnome_font_new (const char *name, double size)
{
  GnomeFont *font;
  GnomeFontClass *klass = gtk_type_class (gnome_font_get_type ());
  GnomeFontMap *fontmap = klass->fontmap;
  int n_fonts = klass->n_fonts;
  int i;

  for (i = 0; i < n_fonts; i++)
    if (!strcmp (name, fontmap[i].font_name))
      break;

  if (i == n_fonts)
    {
      g_warning (_("gnome_font_new: font %s not found\n"), name);
      return NULL;
    }

  if (fontmap[i].widths == NULL)
      gnome_font_load_afm (&fontmap[i]);

  font = gtk_type_new (gnome_font_get_type ());
  font->num = i;
  font->size = size;
  font->fontmap_entry = &fontmap[i];
  font->scale = 0.001 * size;

  return font;
}

/* Find the closest weight matching the family name, weight, and italic
   specs. Return the unsized font. */
GnomeFontUnsized *
gnome_font_name_closest (const char *family_name,
			 GnomeFontWeight weight,
			 gboolean italic,
			 double size)
{
  GnomeFontClass *klass = gtk_type_class (gnome_font_get_type ());
  GnomeFontMap *fontmap = klass->fontmap;
  int n_fonts = klass->n_fonts;
  int i;
  int best, best_dist, dist;

  best = -1;
  best_dist = 1000000;
  for (i = 0; i < n_fonts; i++)
    if (!strcmp (family_name, fontmap[i].familyname))
      {
	dist = abs (weight - fontmap[i].weight_code) +
	  100 * (italic != fontmap[i].italic);
	if (dist < best_dist)
	  {
	    best_dist = dist;
	    best = i;
	  }
      }

  if (best == -1)
    {
      g_warning (_("gnome_font_name_closest: font family %s not found\n"),
		 family_name);
      return NULL;
    }

  if (fontmap[best].widths == NULL)
      gnome_font_load_afm (&fontmap[best]);

  return &fontmap[best];
}

/* Find the closest weight matching the family name, weight, and italic
   specs. */
GnomeFont *
gnome_font_new_closest (const char *family_name,
			GnomeFontWeight weight,
			gboolean italic,
			double size)
{
  GnomeFont *font;
  GnomeFontClass *klass = gtk_type_class (gnome_font_get_type ());
  GnomeFontMap *fontmap = klass->fontmap;
  int n_fonts = klass->n_fonts;
  int i;
  int best, best_dist, dist;

  best = -1;
  best_dist = 1000000;
  for (i = 0; i < n_fonts; i++)
    if (!strcmp (family_name, fontmap[i].familyname))
      {
	dist = abs (weight - fontmap[i].weight_code) +
	  100 * (italic != fontmap[i].italic);
	if (dist < best_dist)
	  {
	    best_dist = dist;
	    best = i;
	  }
      }

  if (best == -1)
    {
      g_warning (_("gnome_font_new_closest: font family %s not found\n"),
		 family_name);
      return NULL;
    }

  if (fontmap[best].widths == NULL)
      gnome_font_load_afm (&fontmap[best]);

  font = gtk_type_new (gnome_font_get_type ());
  font->num = i;
  font->size = size;
  font->fontmap_entry = &fontmap[best];
  font->scale = 0.001 * size;

  return font;
}


/* return a pointer to the (PostScript) name of the font */
char *
gnome_font_get_name (GnomeFont *font)
{
  g_return_val_if_fail (font != NULL, NULL);

  return font->fontmap_entry->font_name;
}

/* return a pointer to the (PostScript) name of the font */
char *
gnome_font_get_glyph_name (GnomeFont *font)
{
  GnomeFontMap *fontmap_entry;
  GnomeFontMap *subst;

  g_return_val_if_fail (font != NULL, NULL);

  fontmap_entry = font->fontmap_entry;

  if (strcmp ("-", fontmap_entry->pfb_fn))
    return fontmap_entry->font_name;

  subst = fontmap_entry->subst_glyph;
  g_return_val_if_fail (subst != NULL, NULL);

  return subst->font_name;
}

static int
read_int32_lsb (const char *p)
{
  const unsigned char *q = (unsigned char *)p;

  return q[0] + (q[1] << 8) + (q[2] << 16) + (q[3] << 24);
}

/* this is actually the same as a pfb to pfa converter

 Reference: Adobe technical note 5040, "Supporting Downloadable PostScript
 Language Fonts", page 9 */
static char *
pfb_to_flat (const char *input, int input_size)
{
  const unsigned char *in = (unsigned char *)input;
  char *flat;
  int flat_size, flat_size_max;
  int in_idx;
  int length;
  int i;
  const char hextab[16] = "0123456789abcdef";

  flat_size = 0;
  flat_size_max = 32768;
  flat = g_new (char, flat_size_max);

  for (in_idx = 0; in_idx < input_size;)
    {
      if (in[in_idx] != 128)
	{
	  g_free (flat);
	  return NULL;
	}
      switch (in[in_idx + 1])
	{
	case 1:
	  length = read_int32_lsb (input + in_idx + 2);
	  if (flat_size + length > flat_size_max)
	    {
	      do
		flat_size_max <<= 1;
	      while (flat_size + length > flat_size_max);
	      flat = g_realloc (flat, flat_size_max);
	    }
	  in_idx += 6;
	  memcpy (flat + flat_size, in + in_idx, length);
	  flat_size += length;
	  in_idx += length;
	  break;
	case 2:
	  length = read_int32_lsb (input + in_idx + 2);
	  if (flat_size + length * 3 > flat_size_max)
	    {
	      do
		flat_size_max <<= 1;
	      while (flat_size + length * 3 > flat_size_max);
	      flat = g_realloc (flat, flat_size_max);
	    }
	  in_idx += 6;
	  for (i = 0; i < length; i++)
	    {
	      flat[flat_size++] = hextab[in[in_idx] >> 4];
	      flat[flat_size++] = hextab[in[in_idx] & 15];
	      in_idx++;
	      if ((i & 31) == 31 || i == length - 1)
		flat[flat_size++] = '\n';
	    }
	  break;
	case 3:
	  /* zero terminate the returned string */
	  if (flat_size == flat_size_max)
	    flat = g_realloc (flat, flat_size_max <<= 1);
	  flat[flat_size] = 0;
	  return flat;
	default:
	    g_free (flat);
	    return NULL;
	}
    }
  return flat;
}

char *
gnome_font_get_pfa (GnomeFont *font)
{
  char *pfb_fn;
  FILE *f;
  GnomeFontMap *subst;

  char *pfb;
  int pfb_size, pfb_size_max;
  int bytes_read;

  char *flat;

  if (font == NULL)
    return NULL;

  pfb_fn = font->fontmap_entry->pfb_fn;
  if (!strcmp (pfb_fn, "-"))
    {
      subst = font->fontmap_entry->subst_glyph;
      g_return_val_if_fail (subst != NULL, NULL);

      pfb_fn = subst->pfb_fn;
    }
  f = fopen (pfb_fn, "rb");
  if (f == NULL)
    {
      g_warning (_("Couldn't open font file %s\n"), pfb_fn);
      return NULL;
    }
  
  pfb_size = 0;
  pfb_size_max = 32768;
  pfb = g_new (char, pfb_size_max);
  while (1)
    {
      bytes_read = fread (pfb + pfb_size, 1, pfb_size_max - pfb_size, f);
      if (bytes_read == 0) break;
      pfb_size += bytes_read;
      pfb = g_realloc (pfb, pfb_size_max <<= 1);
    }

  if (pfb_size)
    {
      if (((unsigned char *)pfb)[0] == 128)
	flat = pfb_to_flat (pfb, pfb_size);
      else
	{
	  flat = g_new (char, pfb_size + 1);
	  memcpy (flat, pfb, pfb_size);
	  flat[pfb_size] = 0;
	}
    }
  else
    flat = NULL;

  g_free (pfb);
  return flat;
}

double
gnome_font_get_width (GnomeFont *font, int ch)
{
  if (ch < 0 || ch >= 256)
    return 0.0;
  return font->fontmap_entry->widths[ch] * font->scale;
}

/* This works with the standard Adobe encoding and without kerning or
   ligatures. When the text libs get written, this function will be
   deprecated. */
double
gnome_font_get_width_string (GnomeFont *font, const char *s)
{
  unsigned char *us = (unsigned char *)s;
  int *widths = font->fontmap_entry->widths;
  double width;
  unsigned char c;

  width = 0;
  while ((c = *(us++)) != 0)
    width += widths[c];
  return width * font->scale;
}

/* Get the glyph number corresponding to a given unicode, or -1 if it
   is not mapped. */
int
gnome_font_get_glyph (GnomeFont *font, int unicode)
{
  int page;
  GnomeFontMap *fontmap;
  int *cov_page;

  fontmap = font->fontmap_entry;

  page = unicode >> 8;
  page -= fontmap->first_cov_page;
  if (page < 0 || page >= fontmap->num_cov_pages)
    return -1;
  cov_page = fontmap->cov_pages[page];
  if (cov_page == NULL)
    return -1;
  else
    return cov_page[unicode & 0xff];
}

static void
gnome_font_finalize (GtkObject *object)
{
  GnomeFont *font;

  g_return_if_fail (object != NULL);
  g_return_if_fail (GNOME_IS_FONT (object));

  font = GNOME_FONT (object);

  (* GTK_OBJECT_CLASS (parent_class)->finalize) (object);
}
