#include "parseAFM.h"
#include <gnome.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <dirent.h>
#include <string.h>
#include <ctype.h>
#include <libgnomeprint/gnome-font.h>
#include <gnome-xml/tree.h>
#include <gnome-xml/parser.h>

#define noVERBOSE

typedef struct _GnomeFontType1EncTab GnomeFontType1EncTab;

struct _GnomeFontType1EncTab {
  int unicode;
  char *type1_name;
};

static void gnome_font_class_init (GnomeFontClass *klass);

static void
gnome_font_init (GnomeFont *font);

static void
gnome_font_finalize (GtkObject *object);

static GtkObjectClass *parent_class = NULL;

/* These are just the glyphs in the standard encoding. I need to add
   the rest of the glyphs, accessible through nonstandard
   encodings. */

/* This table is adapted very straightforwardly from
   ftp.unicode.org/Public/MAPPINGS/VENDORS/ADOBE/STDENC.TXT.

   I have reviewed them and compared them against an independently
   constructed encoding, and believe they are correct.
  */

const GnomeFontType1EncTab gnome_font_type1_std_enc[] = {
  { 0x0020, "space" },
  { 0x0021, "exclam" },
  { 0x0022, "quotedbl" },
  { 0x0023, "numbersign" },
  { 0x0024, "dollar" },
  { 0x0025, "percent" },
  { 0x0026, "ampersand" },
  { 0x0027, "quotesingle" },
  { 0x0028, "parenleft" },
  { 0x0029, "parenright" },
  { 0x002A, "asterisk" },
  { 0x002B, "plus" },
  { 0x002C, "comma" },
  { 0x002D, "hyphen" },
  { 0x002E, "period" },
  { 0x002F, "slash" },
  { 0x0030, "zero" },
  { 0x0031, "one" },
  { 0x0032, "two" },
  { 0x0033, "three" },
  { 0x0034, "four" },
  { 0x0035, "five" },
  { 0x0036, "six" },
  { 0x0037, "seven" },
  { 0x0038, "eight" },
  { 0x0039, "nine" },
  { 0x003A, "colon" },
  { 0x003B, "semicolon" },
  { 0x003C, "less" },
  { 0x003D, "equal" },
  { 0x003E, "greater" },
  { 0x003F, "question" },
  { 0x0040, "at" },
  { 0x0041, "A" },
  { 0x0042, "B" },
  { 0x0043, "C" },
  { 0x0044, "D" },
  { 0x0045, "E" },
  { 0x0046, "F" },
  { 0x0047, "G" },
  { 0x0048, "H" },
  { 0x0049, "I" },
  { 0x004A, "J" },
  { 0x004B, "K" },
  { 0x004C, "L" },
  { 0x004D, "M" },
  { 0x004E, "N" },
  { 0x004F, "O" },
  { 0x0050, "P" },
  { 0x0051, "Q" },
  { 0x0052, "R" },
  { 0x0053, "S" },
  { 0x0054, "T" },
  { 0x0055, "U" },
  { 0x0056, "V" },
  { 0x0057, "W" },
  { 0x0058, "X" },
  { 0x0059, "Y" },
  { 0x005A, "Z" },
  { 0x005B, "bracketleft" },
  { 0x005C, "backslash" },
  { 0x005D, "bracketright" },
  { 0x005E, "asciicircum" },
  { 0x005F, "underscore" },
  { 0x0060, "grave" },
  { 0x0061, "a" },
  { 0x0062, "b" },
  { 0x0063, "c" },
  { 0x0064, "d" },
  { 0x0065, "e" },
  { 0x0066, "f" },
  { 0x0067, "g" },
  { 0x0068, "h" },
  { 0x0069, "i" },
  { 0x006A, "j" },
  { 0x006B, "k" },
  { 0x006C, "l" },
  { 0x006D, "m" },
  { 0x006E, "n" },
  { 0x006F, "o" },
  { 0x0070, "p" },
  { 0x0071, "q" },
  { 0x0072, "r" },
  { 0x0073, "s" },
  { 0x0074, "t" },
  { 0x0075, "u" },
  { 0x0076, "v" },
  { 0x0077, "w" },
  { 0x0078, "x" },
  { 0x0079, "y" },
  { 0x007A, "z" },
  { 0x007B, "braceleft" },
  { 0x007C, "bar" },
  { 0x007D, "braceright" },
  { 0x007E, "asciitilde" },
  { 0x00A1, "exclamdown" },
  { 0x00A2, "cent" },
  { 0x00A3, "sterling" },
  { 0x00A4, "currency" },
  { 0x00A5, "yen" },
  { 0x00A7, "section" },
  { 0x00A8, "dieresis" },
  { 0x00AA, "ordfeminine" },
  { 0x00AB, "guillemotleft" },
  { 0x00AF, "macron" },
  { 0x00B4, "acute" },
  { 0x00B6, "paragraph" },
  { 0x00B7, "periodcentered" },
  { 0x00B8, "cedilla" },
  { 0x00BA, "ordmasculine" },
  { 0x00BB, "guillemotright" },
  { 0x00BF, "questiondown" },
  { 0x00C6, "AE" },
  { 0x00D8, "Oslash" },
  { 0x00DF, "germandbls" },
  { 0x00E6, "ae" },
  { 0x00F8, "oslash" },
  { 0x0131, "dotlessi" },
  { 0x0141, "Lslash" },
  { 0x0142, "lslash" },
  { 0x0152, "OE" },
  { 0x0153, "oe" },
  { 0x0192, "florin" },
  { 0x02C6, "circumflex" },
  { 0x02C7, "caron" },
  { 0x02D8, "breve" },
  { 0x02D9, "dotaccent" },
  { 0x02DA, "ring" },
  { 0x02DB, "ogonek" },
  { 0x02DC, "tilde" },
  { 0x02DD, "hungarumlaut" },
  { 0x2013, "endash" },
  { 0x2014, "emdash" },
  { 0x2018, "quoteleft" },
  { 0x2019, "quoteright" },
  { 0x201A, "quotesinglbase" },
  { 0x201C, "quotedblleft" },
  { 0x201D, "quotedblright" },
  { 0x201E, "quotedblbase" },
  { 0x2020, "dagger" },
  { 0x2021, "daggerdbl" },
  { 0x2022, "bullet" },
  { 0x2026, "ellipsis" },
  { 0x2030, "perthousand" },
  { 0x2039, "guilsinglleft" },
  { 0x203A, "guilsinglright" },
  { 0x2044, "fraction" },
  { 0xFB01, "fi" },
  { 0xFB02, "fl" }
};

GtkType
gnome_font_get_type (void)
{
  static GtkType font_type = 0;

  if (!font_type)
    {
      GtkTypeInfo font_info =
      {
	"GnomeFont",
	sizeof (GnomeFont),
	sizeof (GnomeFontClass),
	(GtkClassInitFunc) gnome_font_class_init,
	(GtkObjectInitFunc) gnome_font_init,
	/* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      font_type = gtk_type_unique (gtk_object_get_type (), &font_info);
    }

  return font_type;
}

static gboolean
gnome_font_is_afm_file (const char *fn)
{
  int len;

  len = strlen (fn);
  return (len > 4 &&
	  fn[len - 4] == '.' &&
	  tolower (fn[len - 3]) == 'a' &&
	  tolower (fn[len - 2]) == 'f' &&
	  tolower (fn[len - 1]) == 'm');
}

/* Try to find the filename for the font */
static int
gnome_font_find (GnomeFontClass *class, char *fontname)
{
  int i;

  for (i = 0; i < class->n_fonts; i++)
    if (!strcmp (class->fontmap[i].font_name, fontname))
      return i;
  return -1;
}

static char *
gnome_font_find_filename (GnomeFontClass *class, char *fontname)
{
  int i;

  i = gnome_font_find (class, fontname);
  if (i >= 0)
    return class->fontmap[i].afm_fn;
  else
    return NULL;
}

typedef struct _GnomeFontWeightTab GnomeFontWeightTab;

struct _GnomeFontWeightTab {
  const char *name;
  int code;
};

static void
gnome_font_add_mapping (GnomeFontClass *class, char *fontname,
			char *afm_fn, char *pfb_fn, char *fullname,
			char *familyname, char *weight, char *alias)
{
  int n_fonts;
  const GnomeFontWeightTab weight_tab[] = {
    { "Extra Light", GNOME_FONT_EXTRA_LIGHT },
    { "Extralight", GNOME_FONT_EXTRA_LIGHT },

    { "Thin", GNOME_FONT_THIN },

    { "Light", GNOME_FONT_LIGHT },

    { "Book", GNOME_FONT_BOOK },
    { "Roman", GNOME_FONT_BOOK },
    { "Regular", GNOME_FONT_BOOK },

    { "Medium", GNOME_FONT_MEDIUM },

    { "Semi", GNOME_FONT_SEMI },
    { "Semibold", GNOME_FONT_SEMI },
    { "Demi", GNOME_FONT_SEMI },
    { "Demibold", GNOME_FONT_SEMI },

    { "Bold", GNOME_FONT_BOLD },

    { "Heavy", GNOME_FONT_HEAVY },
 
    { "Extra", GNOME_FONT_EXTRABOLD },
    { "Extra Bold", GNOME_FONT_EXTRABOLD },

    { "Black", GNOME_FONT_BLACK },

    { "Extra Black", GNOME_FONT_EXTRABLACK },
    { "Extrablack", GNOME_FONT_EXTRABLACK },
    { "Ultra Bold", GNOME_FONT_EXTRABLACK }
  };
  int i;
  int len_full;
  GnomeFontWeight weight_code;


#ifdef VERBOSE
  g_print ("Adding %s %s\n", fontname, afm_fn);
#endif
  if (gnome_font_find (class, fontname) >= 0)
    {
      /* already present in fontmap */
#ifdef VERBOSE
      g_print ("Font %s already present\n", fontname);
#endif
      return;
    }

  n_fonts = (class->n_fonts)++;
  if (n_fonts == 0)
    class->fontmap = g_new (GnomeFontMap, 1);
  else if ((n_fonts & (n_fonts - 1)) == 0)
    {
#ifdef VERBOSE
      g_print ("reallocating to %d\n", n_fonts * 2);
#endif
      class->fontmap = g_realloc (class->fontmap,
				  sizeof(GnomeFontMap) * n_fonts * 2);
    }
  class->fontmap[n_fonts].font_name = g_strdup (fontname);
  class->fontmap[n_fonts].afm_fn = g_strdup (afm_fn);
  class->fontmap[n_fonts].pfb_fn = g_strdup (pfb_fn);
  class->fontmap[n_fonts].fullname = g_strdup (fullname);
  class->fontmap[n_fonts].familyname = g_strdup (familyname);
  class->fontmap[n_fonts].weight = g_strdup (weight);
  class->fontmap[n_fonts].widths = NULL;
  if ( alias )
    class->fontmap[n_fonts].alias = g_strdup( alias );
  else
    class->fontmap[n_fonts].alias = NULL;
  class->fontmap[n_fonts].cov_pages = NULL;
  weight_code = GNOME_FONT_BOOK;
  for (i = 0; i < sizeof(weight_tab) / sizeof(weight_tab[0]); i++)
    if (!g_strcasecmp (weight, weight_tab[i].name))
      {
	weight_code = weight_tab[i].code;
	break;
      }
  class->fontmap[n_fonts].weight_code = weight_code;
  len_full = strlen (fullname);
  class->fontmap[n_fonts].italic =
    (len_full >= 7 && !g_strcasecmp (fullname + len_full - 7, "Oblique")) ||
    (len_full >= 6 && !g_strcasecmp (fullname + len_full - 6, "Italic"));
  class->fontmap[n_fonts].kerns = NULL;
  class->fontmap[n_fonts].num_kerns = 0;
  class->fontmap[n_fonts].ligs = NULL; /* one liglist for each glyph */

  class->fontmap[n_fonts].first_cov_page = 0;
  class->fontmap[n_fonts].num_cov_pages = 0;
}

#if 0
/* We're not doing it this way, we're using fontmap files */
static void
gnome_font_refresh_font_dir (GnomeFontClass *class, char *dirname)
{
  DIR *dir;
  struct dirent *dir_ent;
  char *fn;
  FILE *afm_f;
  FontInfo *fi;
  int status;

  g_print (_("scanning dir %s\n"), dirname);
  dir = opendir (dirname);
  if (dir != NULL)
    {
      while (1)
	{
	  dir_ent = readdir (dir);
	  if (dir_ent == NULL)
	    break;
	  if (gnome_font_is_afm_file (dir_ent->d_name))
	    {
	      g_print (_("checking file %s\n"), dir_ent->d_name);
	      fn = g_malloc (strlen (dirname) + strlen (dir_ent->d_name) + 2);
	      sprintf (fn, _("%s/%s"), dirname, dir_ent->d_name);
	      afm_f = fopen (fn, "rb");
	      if (afm_f != NULL)
		{
		  status = parseFile (afm_f, &fi, P_G);
		  g_print ("status = %d\n", status);
		  if (status == 0)
		    {
		      gnome_font_add_mapping (class, fi->gfi->fontName, fn);
		      /* todo: free fi */
		    }

		  fclose (afm_f);
		}
	    }
	}
    }
}
#endif

static char *
gnome_font_get_field (char **pp) {
  char *p;
  int i, j, n;
  char *field;

  p = *pp;
  for (i = 0; p[i] && p[i] != ' ' && p[i] != '\n'; i++)
    if (p[i] == '\\' && p[i + 1])
      i++;
  n = i;
  field = g_malloc (n + 1);
  j = 0;
  for (i = 0; i < n; i++)
    if (p[i] == '\\' && p[i + 1])
      field[j++] = p[++i];
    else
      field[j++] = p[i];
  field[j++] = 0;

  /* update pointer to point to next field */
  if (p[n]) n++;
  *pp += n;

  return field;
}

/*
 * Get a value for a node either carried as an attibute or as
 * the content of a child.
 */
static char *
xmlGetValue (xmlNodePtr node, const char *name)
{
	char *ret;
	xmlNodePtr child;

	ret = (char *) xmlGetProp (node, name);
	if (ret != NULL)
		return g_strdup (ret);
	child = node->childs;

	while (child != NULL) {
		if (!strcmp (child->name, name)) {
		        /*
			 * !!! Inefficient, but ...
			 */
			ret = xmlNodeGetContent(child);
			if (ret != NULL)
			  {
			    char *ret2 = g_strdup(ret);
			    free(ret);
			    return (ret2);
			  }
		}
		child = child->next;
	}

	return NULL;
}

static void
gnome_font_load_fontmap (GnomeFontClass *class, const char *fn)
{
  FILE *f;
  char linebuf[1024];
  char *p;
  char *type;
  char *fontname, *afmfile, *pfbfile, *fullname, *familyname, *weight, *alias;
  char *subst_font_name;
  xmlDoc *doc;

#ifdef VERBOSE
  g_print ("filename %s\n", fn);
#endif

  doc = xmlParseFile( fn );
  if ( doc && doc->root && doc->root->name && ( ! strcmp( doc->root->name, "fontmap" ) ) ) /* && doc->root->ns && doc->root->ns->href && ( ! strcmp( doc->root->ns->href, "http://www.gnome.org/gnome-font/0.0" ) ) ) */
    {
      xmlNode *font = doc->root->childs;
      while(font)
	{
	  /* parse the fontmap line */
	  type = xmlGetValue( font, "format" );
	  if (type && !strcmp (type, "type1"))
	    {
              gboolean have_all_info = FALSE;

	      fontname = xmlGetValue( font, "name" );
	      afmfile = xmlGetValue( font, "metrics" );
	      pfbfile = xmlGetValue( font, "glyphs" );
	      fullname = xmlGetValue( font, "fullname" );
	      familyname = xmlGetValue( font, "familyname" );
	      weight = xmlGetValue( font, "weight" );
	      alias = xmlGetValue( font, "alias" );

              /* alias is the only optional one, AFAICT */
              have_all_info = fontname && afmfile && pfbfile && 
                fullname && familyname && weight;

              if (have_all_info)
                {
                  gnome_font_add_mapping (class, fontname, afmfile, pfbfile,
                                          fullname, familyname, weight, alias);
                }
              else 
                {
                  g_warning("Missing data in font map `%s':\n"
                            "  Font name: %s\n"
                            "  Metrics:   %s\n"
                            "  Glyphs:    %s\n"
                            "  Full name: %s\n"
                            "  Family:    %s\n"
                            "  Weight:    %s\n",
                            fn,
                            fontname ? fontname : "**missing**",
                            afmfile ? afmfile : "**missing**",
                            pfbfile ? pfbfile : "**missing**",
                            fullname ? fullname : "**missing**",
                            familyname ? familyname : "**missing**",
                            weight ? weight : "**missing**");
                }
#ifdef VERBOSE
	      printf ("%s %s %s %s %s %s\n",
		      fontname, afmfile, pfbfile, fullname, familyname, weight);
#endif
	      g_free (weight);
	      g_free (familyname);
	      g_free (fullname);
	      g_free (pfbfile);
	      g_free (afmfile);
	      g_free (fontname);
	    }
	  g_free (type);
	  font = font->next;
	}
    }
}

static void
gnome_font_refresh_fontmap (GnomeFontClass *class)
{
  char *fontmap_fn;
  char *env_home;

  /* build fontmap from scanning the font path */
  env_home = getenv ("HOME");
  if (env_home != NULL)
    {
      fontmap_fn = g_malloc (strlen (env_home) + 32);
      sprintf (fontmap_fn, "%s/.gnome/fonts/fontmap", env_home);
      gnome_font_load_fontmap (class, fontmap_fn);
      g_free (fontmap_fn);
      fontmap_fn = gnome_datadir_file ("fonts/fontmap");
      if (fontmap_fn)
	{
	  gnome_font_load_fontmap (class, fontmap_fn);
	  g_free (fontmap_fn);
	}
      else
        {
          /* Check gnome-print installation prefix */
          fontmap_fn = g_strconcat(DATADIR, "/fonts/fontmap", NULL);
          gnome_font_load_fontmap (class, fontmap_fn);
          g_free (fontmap_fn);
        }
    }
}

/* Looks up the name, returning the glyph number in the font's
   standard encoding. Return -1 if the glyph is not found in the
   font's standard encoding.

   There are two problems with this function: first, the implementation
   is quite slow (the analogous operation in libhnj uses a _much_
   faster hash table implementation. Second, we eventually want to
   break free of the font's standard encoding. My guess is that
   values < 256 will be glyphnums in the standard encoding, while
   values >= 256 will be in the "extension" encoding.

   Anyway, the point of this function is merely that it works. */
static int
gnome_font_find_glyphnum (Font_Info *fi, const char *name)
{
  int i;

  for (i = 0; i < fi->numOfChars; i++)
    {
      if (!strcmp (name, fi->cmi[i].name))
	return fi->cmi[i].code;
    }
  return -1;
}

static int
gnome_font_find_unicode_encoding (const GnomeFontType1EncTab *enc, int n_enc,
				  const char *glyphname)
{
  int i;

  for (i = 0; i < n_enc; i++)
    {
      if (!strcmp (glyphname, enc[i].type1_name))
	return enc[i].unicode;
    }
  return -1;
}

/* Add a mapping to the font's coverage table. */
static void
gnome_font_add_unimapping (GnomeFontMap *fontmap, int unicode, int glyphnum)
{
  int page;
  int **cov_pages;
  int *cov;
  int i;

  page = unicode >> 8;

  if (fontmap->cov_pages == NULL)
    {
      cov_pages = g_new (int *, 1);
      fontmap->cov_pages = cov_pages;
      fontmap->first_cov_page = page;
      fontmap->num_cov_pages = 1;
      cov = g_new (int, 256);
      for (i = 0; i < 256; i++)
	cov[i] = -1;
      cov_pages[0] = cov;
    }
  else if (page < fontmap->first_cov_page ||
	   page >= fontmap->first_cov_page + fontmap->num_cov_pages)
    {
      int first_cov_page, num_cov_pages;

      first_cov_page = MIN (page, fontmap->first_cov_page);
      num_cov_pages = MAX (page + 1 - first_cov_page, fontmap->num_cov_pages);
      cov_pages = g_new (int *, num_cov_pages);

      for (i = 0; i < first_cov_page - fontmap->first_cov_page; i++)
	cov_pages[i] = NULL;
      memcpy (cov_pages + fontmap->first_cov_page - first_cov_page,
	      fontmap->cov_pages,
	      fontmap->num_cov_pages * sizeof (int *));
      for (i = fontmap->first_cov_page + fontmap->num_cov_pages -
	     first_cov_page; i < num_cov_pages; i++)
	cov_pages[i] = NULL;

      fontmap->first_cov_page = first_cov_page;
      fontmap->num_cov_pages = num_cov_pages;
      g_free (fontmap->cov_pages);
      fontmap->cov_pages = cov_pages;

      cov = g_new (int, 256);
      for (i = 0; i < 256; i++)
	cov[i] = -1;
      fontmap->cov_pages[page - fontmap->first_cov_page] = cov;
      cov_pages[page - first_cov_page] = cov;
    }
  else
    {
      cov = fontmap->cov_pages[page - fontmap->first_cov_page];
      if (cov == NULL)
	{
	  cov = g_new (int, 256);
	  for (i = 0; i < 256; i++)
	    cov[i] = -1;
	  fontmap->cov_pages[page - fontmap->first_cov_page] = cov;
	}
    }
  cov[unicode & 0xff] = glyphnum;
}

#define KERN_PAIR_HASH(g1, g2) ((g1) * 367 + (g2) * 31)
static void
gnome_font_load_afm (GnomeFontMap *fontmap_entry)
{
  Font_Info *fi;
  FILE *afm_f;
  int status;

  afm_f = fopen (fontmap_entry->afm_fn, "rb");
  if (afm_f != NULL)
    {
      status = parseFile (afm_f, &fi, P_M | P_P);
#ifdef VERBOSE
      g_print (_("status loading %s = %d\n"), fontmap_entry->afm_fn, status);
#endif
      if (status == 0)
	{
	  int *widths;
	  GnomeFontLigList **ligtab;
	  Ligature *ligs;
	  int i;
	  GnomeFontKernPair *ktab;
	  int ktabsize;

	  widths = g_malloc (256 * sizeof (int));
	  fontmap_entry->widths = widths;

	  ligtab = g_malloc (256 * sizeof (GnomeFontLigList *));
	  fontmap_entry->ligs = ligtab;

#if 0
	  for (i = 0; i < 256; i++)
	    widths[i] = fi->cwi[i];
#else
	  for (i = 0; i < 256; i++)
	    {
	      widths[i] = 0;
	      ligtab[i] = 0;
	    }

	  for (i = 0; i < fi->numOfChars; i++)
	    {
	      int code;
	      int unicode;

	      /* Get the width */
	      code = fi->cmi[i].code;
	      if (code >= 0 && code < 256)
		{
		  widths[code] = fi->cmi[i].wx;

		  /* Get the ligature info */
		  for (ligs = fi->cmi[i].ligs; ligs != NULL; ligs = ligs->next)
		    {
		      int succ, lig;
		      GnomeFontLigList *ll;

		      succ = gnome_font_find_glyphnum (fi, ligs->succ);
		      lig = gnome_font_find_glyphnum (fi, ligs->lig);
		      if ((succ | lig) >= 0)
			{
			  ll = g_new (GnomeFontLigList, 1);
			  ll->succ = succ;
			  ll->lig = lig;
			  ll->next = ligtab[code];
			  ligtab[code] = ll;
			}
		    }

		  /* add in the mapping from Unicode */
		  unicode = gnome_font_find_unicode_encoding
		    (gnome_font_type1_std_enc,
		     sizeof(gnome_font_type1_std_enc) /
		     sizeof(gnome_font_type1_std_enc[0]),
		     fi->cmi[i].name);

		  if (unicode > 0)
		    gnome_font_add_unimapping (fontmap_entry, unicode, code);
		}
	    }
#endif

	  /* process the kern pairs */
	  for (ktabsize = 1; ktabsize < fi->numOfPairs << 1; ktabsize <<= 1);
	  ktab = g_new (GnomeFontKernPair, ktabsize);
	  fontmap_entry->kerns = ktab;
	  fontmap_entry->num_kerns = ktabsize;
	  for (i = 0; i < ktabsize; i++)
	    {
	      ktab[i].glyph1 = -1;
	      ktab[i].glyph2 = -1;
	      ktab[i].x_amt = 0;
	    }

	  for (i = 0; i < fi->numOfPairs; i++)
	    {
	      int glyph1, glyph2;
	      int j;

	      glyph1 = gnome_font_find_glyphnum (fi, fi->pkd[i].name1);
	      glyph2 = gnome_font_find_glyphnum (fi, fi->pkd[i].name2);
	      for (j = KERN_PAIR_HASH (glyph1, glyph2) & (ktabsize - 1);
		   ktab[j].glyph1 != -1;
		   j = (j + 1) & (ktabsize - 1));
	      ktab[j].glyph1 = glyph1;
	      ktab[j].glyph2 = glyph2;
	      ktab[j].x_amt = fi->pkd[i].xamt;
#ifdef VERBOSE
	      g_print ("kern pair %s(%d) %s(%d) %d\n",
		       fi->pkd[i].name1, glyph1,
		       fi->pkd[i].name2, glyph2,
		       fi->pkd[i].xamt);
#endif
	    }

#ifdef VERBOSE
	  for (i = 0; i < ktabsize; i++)
	    g_print ("%d %d %d\n",
		     ktab[i].glyph1, ktab[i].glyph2, ktab[i].x_amt);
#endif

	  /* Free the Font_Info structure */
	  free (fi->pkd);
	  for (i = 0; i < fi->numOfChars; i++)
	    {
	      Ligature *ligs, *next;

	      for (ligs = fi->cmi[i].ligs; ligs != NULL; ligs = next)
		{
		  next = ligs->next;
		  free (ligs->succ);
		  free (ligs->lig);
		  free (ligs);
		}
	      free (fi->cmi[i].name);
	    }
	  free (fi->cmi);
	  free (fi);
	}

      fclose (afm_f);
    }
}

/* Return the amount of kerning for the two glyphs. */
double
gnome_font_kern (GnomeFont *font, int glyph1, int glyph2)
{
  int ktabsize;
  GnomeFontKernPair *ktab;
  int j;
    
  ktabsize = font->fontmap_entry->num_kerns;
  ktab = font->fontmap_entry->kerns;
  for (j = KERN_PAIR_HASH (glyph1, glyph2) & (ktabsize - 1);
       ktab[j].glyph1 != -1;
       j = (j + 1) & (ktabsize - 1))
    if (ktab[j].glyph1 == glyph1 && ktab[j].glyph2 == glyph2)
      return ktab[j].x_amt * font->scale;
  return 0;
}

/* Return the amount of kerning for the two glyphs. */
int
gnome_font_unsized_kern (GnomeFontUnsized *font, int glyph1, int glyph2)
{
  int ktabsize;
  GnomeFontKernPair *ktab;
  int j;
    
  ktabsize = font->num_kerns;
  ktab = font->kerns;
  for (j = KERN_PAIR_HASH (glyph1, glyph2) & (ktabsize - 1);
       ktab[j].glyph1 != -1;
       j = (j + 1) & (ktabsize - 1))
    if (ktab[j].glyph1 == glyph1 && ktab[j].glyph2 == glyph2)
      return ktab[j].x_amt;
  return 0;
}

static void
gnome_font_class_init (GnomeFontClass *class)
{
  GtkObjectClass *object_class;

  object_class = (GtkObjectClass*) class;

  parent_class = gtk_type_class (gtk_object_get_type ());

  object_class->finalize = gnome_font_finalize;

  class->n_fonts = 0;
  class->fontmap = NULL;

  gnome_font_refresh_fontmap (class);
}

static void
gnome_font_init (GnomeFont *font)
{
  font->num = -1;
  font->size = 0;
}

GnomeFont *
gnome_font_new (const char *name, double size)
{
  GnomeFont *font;
  GnomeFontClass *klass = gtk_type_class (gnome_font_get_type ());
  GnomeFontMap *fontmap = klass->fontmap;
  int n_fonts = klass->n_fonts;
  int i;

  for (i = 0; i < n_fonts; i++)
    if (!strcmp (name, fontmap[i].font_name))
      break;

  if (i == n_fonts)
    {
      g_warning (_("gnome_font_new: font %s not found\n"), name);
      return NULL;
    }

  if (fontmap[i].widths == NULL)
      gnome_font_load_afm (&fontmap[i]);

  font = gtk_type_new (gnome_font_get_type ());
  font->num = i;
  font->size = size;
  font->fontmap_entry = &fontmap[i];
  font->scale = 0.001 * size;

  return font;
}

/* Find the closest weight matching the family name, weight, and italic
   specs. Return the unsized font. */
GnomeFontUnsized *
gnome_font_unsized_closest (const char *family_name,
			    GnomeFontWeight weight,
			    gboolean italic)
{
  GnomeFontClass *klass = gtk_type_class (gnome_font_get_type ());
  GnomeFontMap *fontmap = klass->fontmap;
  int n_fonts = klass->n_fonts;
  int i;
  int best, best_dist, dist;

  /* This should be reimplemented to use the gnome_font_family_hash. */
  best = -1;
  best_dist = 1000000;
  for (i = 0; i < n_fonts; i++)
    if (!strcmp (family_name, fontmap[i].familyname))
      {
	dist = abs (weight - fontmap[i].weight_code) +
	  100 * (italic != fontmap[i].italic);
	if (dist < best_dist)
	  {
	    best_dist = dist;
	    best = i;
	  }
      }

  if (best == -1)
    {
      g_warning (_("gnome_font_new_closest: font family %s not found\n"),
		 family_name);
      return NULL;
    }

  if (fontmap[best].widths == NULL)
      gnome_font_load_afm (&fontmap[best]);

  return &fontmap[best];
}

/* Find the closest weight matching the family name, weight, and italic
   specs. */
GnomeFont *
gnome_font_new_closest (const char *family_name,
			GnomeFontWeight weight,
			gboolean italic,
			double size)
{
  GnomeFont *font;
  GnomeFontClass *klass = gtk_type_class (gnome_font_get_type ());
  GnomeFontMap *fontmap = klass->fontmap;
  int n_fonts = klass->n_fonts;
  int i;
  int best, best_dist, dist;

  best = -1;
  best_dist = 1000000;
  for (i = 0; i < n_fonts; i++)
    if (!strcmp (family_name, fontmap[i].familyname))
      {
	dist = abs (weight - fontmap[i].weight_code) +
	  100 * (italic != fontmap[i].italic);
	if (dist < best_dist)
	  {
	    best_dist = dist;
	    best = i;
	  }
      }

  if (best == -1)
    {
      g_warning (_("gnome_font_new_closest: font family %s not found\n"),
		 family_name);
      return NULL;
    }

  if (fontmap[best].widths == NULL)
      gnome_font_load_afm (&fontmap[best]);

  font = gtk_type_new (gnome_font_get_type ());
  font->num = i;
  font->size = size;
  font->fontmap_entry = &fontmap[best];
  font->scale = 0.001 * size;

  return font;
}


/* return a pointer to the (PostScript) name of the font */
char *
gnome_font_unsized_get_name (GnomeFontUnsized *font)
{
  g_return_val_if_fail (font != NULL, NULL);

  return font->font_name;
}

/* return a pointer to the (PostScript) name of the font */
char *
gnome_font_unsized_get_glyph_name (GnomeFontUnsized *font)
{
  GnomeFontMap *subst;

  g_return_val_if_fail (font != NULL, NULL);

  if (font->alias)
    return font->alias;
  else
    return font->font_name;
}

/* return a pointer to the (PostScript) name of the font */
char *
gnome_font_get_name (GnomeFont *font)
{
  g_return_val_if_fail (font != NULL, NULL);

  return font->fontmap_entry->font_name;
}

/* return a pointer to the (PostScript) name of the font */
char *
gnome_font_get_glyph_name (GnomeFont *font)
{
  GnomeFontMap *fontmap_entry;

  g_return_val_if_fail (font != NULL, NULL);

  fontmap_entry = font->fontmap_entry;

  if (fontmap_entry->alias)
    return fontmap_entry->alias;
  else
    return fontmap_entry->font_name;
}

static int
read_int32_lsb (const char *p)
{
  const unsigned char *q = (unsigned char *)p;

  return q[0] + (q[1] << 8) + (q[2] << 16) + (q[3] << 24);
}

/* this is actually the same as a pfb to pfa converter

 Reference: Adobe technical note 5040, "Supporting Downloadable PostScript
 Language Fonts", page 9 */
static char *
pfb_to_flat (const char *input, int input_size)
{
  const unsigned char *in = (unsigned char *)input;
  char *flat;
  int flat_size, flat_size_max;
  int in_idx;
  int length;
  int i;
  const char hextab[16] = "0123456789abcdef";

  flat_size = 0;
  flat_size_max = 32768;
  flat = g_new (char, flat_size_max);

  for (in_idx = 0; in_idx < input_size;)
    {
      if (in[in_idx] != 128)
	{
	  g_free (flat);
	  return NULL;
	}
      switch (in[in_idx + 1])
	{
	case 1:
	  length = read_int32_lsb (input + in_idx + 2);
	  if (flat_size + length > flat_size_max)
	    {
	      do
		flat_size_max <<= 1;
	      while (flat_size + length > flat_size_max);
	      flat = g_realloc (flat, flat_size_max);
	    }
	  in_idx += 6;
	  memcpy (flat + flat_size, in + in_idx, length);
	  flat_size += length;
	  in_idx += length;
	  break;
	case 2:
	  length = read_int32_lsb (input + in_idx + 2);
	  if (flat_size + length * 3 > flat_size_max)
	    {
	      do
		flat_size_max <<= 1;
	      while (flat_size + length * 3 > flat_size_max);
	      flat = g_realloc (flat, flat_size_max);
	    }
	  in_idx += 6;
	  for (i = 0; i < length; i++)
	    {
	      flat[flat_size++] = hextab[in[in_idx] >> 4];
	      flat[flat_size++] = hextab[in[in_idx] & 15];
	      in_idx++;
	      if ((i & 31) == 31 || i == length - 1)
		flat[flat_size++] = '\n';
	    }
	  break;
	case 3:
	  /* zero terminate the returned string */
	  if (flat_size == flat_size_max)
	    flat = g_realloc (flat, flat_size_max <<= 1);
	  flat[flat_size] = 0;
	  return flat;
	default:
	    g_free (flat);
	    return NULL;
	}
    }
  return flat;
}

char *
gnome_font_unsized_get_pfa (GnomeFontUnsized *font)
{
  char *pfb_fn;
  FILE *f;
  GnomeFontMap *subst;

  char *pfb;
  int pfb_size, pfb_size_max;
  int bytes_read;

  char *flat;

  if (font == NULL)
    return NULL;

  pfb_fn = font->pfb_fn;
#if 0
  if (!strcmp (pfb_fn, "-"))
    {
      subst = font->subst_glyph;
      g_return_val_if_fail (subst != NULL, NULL);

      pfb_fn = subst->pfb_fn;
    }
#endif
  f = fopen (pfb_fn, "rb");
  if (f == NULL)
    {
      g_warning (_("Couldn't open font file %s\n"), pfb_fn);
      return NULL;
    }
  
  pfb_size = 0;
  pfb_size_max = 32768;
  pfb = g_new (char, pfb_size_max);
  while (1)
    {
      bytes_read = fread (pfb + pfb_size, 1, pfb_size_max - pfb_size, f);
      if (bytes_read == 0) break;
      pfb_size += bytes_read;
      pfb = g_realloc (pfb, pfb_size_max <<= 1);
    }

  if (pfb_size)
    {
      if (((unsigned char *)pfb)[0] == 128)
	flat = pfb_to_flat (pfb, pfb_size);
      else
	{
	  flat = g_new (char, pfb_size + 1);
	  memcpy (flat, pfb, pfb_size);
	  flat[pfb_size] = 0;
	}
    }
  else
    flat = NULL;

  g_free (pfb);
  return flat;
}

char *
gnome_font_get_pfa (GnomeFont *font)
{
  return gnome_font_unsized_get_pfa (font->fontmap_entry);
}

double
gnome_font_get_width (GnomeFont *font, int ch)
{
  if (ch < 0 || ch >= 256)
    return 0.0;
  return font->fontmap_entry->widths[ch] * font->scale;
}

/* Returns the glyph width in 0.001 unit */
int
gnome_font_unsized_get_width (GnomeFontUnsized *font, int ch)
{
  if (ch < 0 || ch >= 256)
    return 0;
  return font->widths[ch];
}

/* This works with the standard Adobe encoding and without kerning or
   ligatures. When the text libs get written, this function will be
   deprecated. */
double
gnome_font_get_width_string (GnomeFont *font, const char *s)
{
  unsigned char *us = (unsigned char *)s;
  int *widths = font->fontmap_entry->widths;
  double width;
  unsigned char c;

  width = 0;
  while ((c = *(us++)) != 0)
    width += widths[c];
  return width * font->scale;
}

/* Get the glyph number corresponding to a given unicode, or -1 if it
   is not mapped. */
int
gnome_font_unsized_get_glyph (GnomeFontUnsized *font, int unicode)
{
  int page;
  int *cov_page;

  page = unicode >> 8;
  page -= font->first_cov_page;
  if (page < 0 || page >= font->num_cov_pages)
    return -1;
  cov_page = font->cov_pages[page];
  if (cov_page == NULL)
    return -1;
  else
    return cov_page[unicode & 0xff];
}

/* Get the glyph number corresponding to a given unicode, or -1 if it
   is not mapped. */
int
gnome_font_get_glyph (GnomeFont *font, int unicode)
{
  return gnome_font_unsized_get_glyph (font->fontmap_entry, unicode);
}

static void
gnome_font_finalize (GtkObject *object)
{
  GnomeFont *font;

  g_return_if_fail (object != NULL);
  g_return_if_fail (GNOME_IS_FONT (object));

  font = GNOME_FONT (object);

  (* GTK_OBJECT_CLASS (parent_class)->finalize) (object);
}

/* Return a list of fonts, as a g_list of strings */
GList *
gnome_font_list (GnomeFontClass *class)
{
  GList *list;
  int i;
  GnomeFontMap *fontmap = class->fontmap;

  list = NULL;

  for (i = 0; i < class->n_fonts; i++)
   list = g_list_append (list, fontmap[i].font_name);

  return list;
}

void
gnome_font_list_free (GList *fontlist)
{
  g_list_free (fontlist);
}

/* These two should probably go into the class */
GHashTable *gnome_font_family_hash = NULL;
GList *gnome_font_family_the_list = NULL;

/* Return a list of font families, as a g_list of newly allocated strings */
GList *
gnome_font_family_list (GnomeFontClass *class)
{
  GList *list;
  GHashTable *hash;
  int i;
  GList *the_family;
  GnomeFontMap *fontmap = class->fontmap;

  if (gnome_font_family_the_list != NULL)
    return gnome_font_family_the_list;

  list = NULL;
  hash = g_hash_table_new (g_str_hash, g_str_equal);

  for (i = 0; i < class->n_fonts; i++)
    {
      the_family = g_hash_table_lookup (hash, fontmap[i].familyname);
      if (the_family == NULL)
	{
	  the_family = g_list_prepend (NULL, &fontmap[i]);
	  g_hash_table_insert (hash, fontmap[i].familyname, the_family);
	  list = g_list_append (list, fontmap[i].familyname);

	}
      else
	g_list_append (the_family, &fontmap[i]);
    }

  gnome_font_family_the_list = list;
  gnome_font_family_hash = hash;

  return list;
}

void
gnome_font_family_list_free (GList *fontlist)
{
}

/*
 * Copyright  1998-1999 The Hungry Programmers
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gnome.h>

#ifndef GNOME_FONT_EXTRA_BLACK
#define GNOME_FONT_EXTRA_BLACK GNOME_FONT_EXTRABLACK
#endif

static char *getFontComponent (char *font_name, unsigned int pos);
static char *setFontComponent (char *font_name, unsigned int pos, char *s);

static char *getFontFoundry (char *font_name);
static char *getFontFamily (char *font_name);
static char *getFontWeight (char *font_name);
static GnomeFontWeight getGnomeFontWeight (char *font_name);
static char *getFontSlant (char *font_name);
static gboolean fontIsItalic (char *font_name);
static char *getFontSWidth (char *font_name);
static char *getFontAdStyle (char *font_name);
static int getFontPixelSize (char *font_name);
static int getFontPointSize (char *font_name);
static int getFontResolutionX (char *font_name);
static int getFontResolutionY (char *font_name);
static char *getFontSpace (char *font_name);
static int getFontAverageWidth (char *font_name);
static char *getFontRegistry (char *font_name);
static char *getFontEncoding (char *font_name);


/* these malloc, so free what they return */

static char *setFontFoundry (char *font_name, char *s);
static char *setFontFamily (char *font_name, char *s);
static char *setFontWeight (char *font_name, char *s);
static char *setFontSlant (char *font_name, char *s);
static char *setFontSWidth (char *font_name, char *s);
static char *setFontAdStyle (char *font_name, char *s);
static char *setFontPixelSize (char *font_name, int i);
static char *setFontPointSize (char *font_name, int i);
static char *setFontResolutionX (char *font_name, int i);
static char *setFontResolutionY (char *font_name, int i);
static char *setFontSpace (char *font_name, char *s);
static char *setFontAverageWidth (char *font_name, int i);
static char *setFontRegistry (char *font_name, char *s);
static char *setFontEncoding (char *font_name, char *s);

static void setComponentReplace (char **io,
				 char *(*func) (char *font_name, char *s),
				 char *part);


GHashTable *scaled_display_fonts = NULL;
GHashTable *gnome_font_family_to_x_hash = NULL;
GHashTable *gnome_font_x_to_family_hash = NULL;
GHashTable *gnome_font_weight_to_string_hash = NULL;
GHashTable *gnome_font_string_to_weight_hash = NULL;

typedef struct
{
  char *print_name;
  char *x_name;
} gnome_font_family_to_x_map;

gnome_font_family_to_x_map gnome_font_family_to_x_maps[] =
{
  {"Bitstream Charter", "charter"},
  {"Century Schoolbook L", "new century schoolbook"},
  {"Courier", "courier"},
  {"Dingbats", "dingbats"},
  {"Helvetica", "helvetica"},
  {"ITC Avant Garde Gothic", "gothic"},
  {"ITC Bookman", "bookman"},
  {"ITC Zapf Chancery", "zapf chancery"},
  {"ITC Zapf Dingbats", "zapf dingbats"},
  {"New Century Schoolbook", "new century schoolbook"},
  {"Nimbus Mono L", "nimbus"}, /* times roman? */
  {"Nimbus Roman No9 L", "nimbus"}, /* times roman? */
  {"Nimbus Sans L", "nimbus"}, /* times roman? */
  {"Palatino", "palatino"},
  {"Standard Symbols L", "symbol"},
  {"Symbol", "symbol"},
#if 1
  {"Times", "times"},
#else
  {"Times", "times new roman"},
#endif
  {"URW Bookman L", "bookman"},
  {"URW Chancery L", "chancery"},
  {"URW Gothic L", "gothic"},
  {"URW Palladio L", "palladio"},
  {0, 0}
};


typedef struct
{
  GnomeFontWeight weight;
  char *name;
} gnome_font_weight_to_string_map;

gnome_font_weight_to_string_map gnome_font_weight_to_string_maps [] =
{
  {GNOME_FONT_EXTRA_LIGHT, "extralight"},
  {GNOME_FONT_THIN, "thin"},
  {GNOME_FONT_LIGHT, "light"},
  {GNOME_FONT_BOOK, "regular"},
  {GNOME_FONT_MEDIUM, "medium"},
  {GNOME_FONT_SEMI, "demibold"},
  {GNOME_FONT_BOLD, "bold"},
  {GNOME_FONT_HEAVY, "heavy"},
  {GNOME_FONT_EXTRABOLD, "extrabold"},
  {GNOME_FONT_BLACK, "black"},
  {GNOME_FONT_EXTRA_BLACK, "extrablack"},
  {(GnomeFontWeight)0, 0}
};


static void initialize_hashes ()
{
  gnome_font_family_to_x_map *ftx = gnome_font_family_to_x_maps;
  gnome_font_weight_to_string_map *wtx = gnome_font_weight_to_string_maps;

  scaled_display_fonts = g_hash_table_new (g_str_hash, g_str_equal);
  gnome_font_family_to_x_hash = g_hash_table_new (g_str_hash, g_str_equal);
  gnome_font_x_to_family_hash = g_hash_table_new (g_str_hash, g_str_equal);
  gnome_font_weight_to_string_hash =
    g_hash_table_new (g_int_hash, g_int_equal);
  gnome_font_string_to_weight_hash =
    g_hash_table_new (g_str_hash, g_str_equal);

  /* build 2 way hashes for x vs font family names */
  while (ftx->print_name)
    {
      printf ("inserting family-->x mapping: '%s' --> '%s'\n",
	      ftx->print_name, ftx->x_name);
      g_hash_table_insert (gnome_font_family_to_x_hash,
			   ftx->print_name, ftx->x_name);
      g_hash_table_insert (gnome_font_x_to_family_hash,
			   ftx->x_name, ftx->print_name);
      ftx ++;
    }


  /* build 2 way hashes for gnome font weight to x names */
  while (wtx->name)
    {
      printf ("inserting weight-->string mapping: '%d' --> '%s'\n",
	      wtx->weight, wtx->name);
      g_hash_table_insert (gnome_font_weight_to_string_hash,
			   & (wtx->weight), wtx->name);
      g_hash_table_insert (gnome_font_string_to_weight_hash,
			   wtx->name, & (wtx->weight));
      wtx ++;
    }
}


char *gnome_font_family_to_x_name (char *family)
{
  char *x_name;
  if (! scaled_display_fonts) initialize_hashes ();

  x_name = (char *) g_hash_table_lookup (gnome_font_family_to_x_hash, family);
  if (! x_name)
    return "helvetica";

  return x_name;
}


char *gnome_font_x_name_to_family (char *x_name)
{
  char *family;
  if (! scaled_display_fonts) initialize_hashes ();

  family = (char *) g_hash_table_lookup (gnome_font_x_to_family_hash, x_name);
  if (! family)
    return "Helvetica";

  return x_name;
}


char *gnome_font_weight_to_string (GnomeFontWeight gfw)
{
  char *x_weight;
  if (! scaled_display_fonts) initialize_hashes ();

  x_weight =
    (char *) g_hash_table_lookup (gnome_font_weight_to_string_hash, &gfw);

  if (! x_weight)
    x_weight = "medium";

  return x_weight;
}


GnomeFontWeight string_to_gnome_font_weight (char *x_weight)
{
  GnomeFontWeight *gfw;

  if (! scaled_display_fonts) initialize_hashes ();

  gfw =
    (GnomeFontWeight *)
    g_hash_table_lookup (gnome_font_string_to_weight_hash, x_weight);

  return (*gfw);
}


static int iabs (int a)
{
  if (a < 0) return -a;
  return a;
}

static GdkFont *
find_best_x_weight (char *x_font_name_ret, GnomeFontWeight start_weight)
{
  gchar *current_x_name = g_strdup( x_font_name_ret );
  GdkFont *current_font = gdk_font_load (current_x_name);
  GnomeFontWeight weighta, weightb;
  char *x_weight;

  if (! scaled_display_fonts) initialize_hashes ();

  weighta = weightb = start_weight;
  if ( current_font )
    {
      g_free( current_x_name );
      return current_font;
    }
  while( weighta >= GNOME_FONT_LIGHTEST || weightb <= GNOME_FONT_HEAVIEST )
    {
      weighta --;
      weightb ++;

      x_weight =
	(char *) g_hash_table_lookup (gnome_font_weight_to_string_hash, &weighta);

      if ( x_weight )
	{
	  setComponentReplace (&current_x_name, setFontWeight, x_weight);
	  current_font = gdk_font_load (current_x_name);
	  if ( current_font )
	    {
	      g_free( current_x_name );
	      return current_font;
	    }
	}

      x_weight =
	(char *) g_hash_table_lookup (gnome_font_weight_to_string_hash, &weightb);

      if ( x_weight )
	{
	  setComponentReplace (&current_x_name, setFontWeight, x_weight);
	  current_font = gdk_font_load (current_x_name);
	  if ( current_font )
	    {
	      g_free( current_x_name );
	      return current_font;
	    }
	}
  
    }
  
  g_free( current_x_name );
  return NULL;
}


/* here is a horrible little routine that tries to find an X font
   that is about the same size as a printer font */
static GdkFont *find_best_x_font (char *unsized_name,
				  GnomeFont *gnome_font,
				  char **x_font_name_ret)
{
  /* sample string used during the search for a good size */
  /*char *test_string = "abcdefghijklmnopqrstuvwxyz";*/
  /*char *test_string = "snider";*/
  char *test_string = "oooowwww";
  int length = strlen (test_string);

  double perfect_width; /* the printer-font width of the sample string */
  double max_width; /* we must end up below this size */

  /* information about the best match found so far */
  int current_size; /* point size */
  int current_width; /* width of rendered sample string */
  int current_diff; /* difference between current_width and perfect_width */
  char *current_x_name; /* the x-style name of the current font */
  GdkFont *current_font; /* the current font */

  int direction; /* either +1 or -1 -- are we looking for a bigger font
		    or a smaller one? */
#if 0
  g_return_val_if_fail (unsized_name, NULL);
  g_return_val_if_fail (gnome_font, NULL);
  g_return_val_if_fail (x_font_name_ret, NULL);

  perfect_width = gnome_font_get_width_string (gnome_font, test_string);
  /*space_width = gnome_font_get_width_string (gnome_font, " ");*/
  /*max_width = perfect_width + space_width / 2;*/
  max_width = perfect_width /*- 1*/;
  g_print( "perfect_width = %f\n", perfect_width );
#endif
  current_size = (int) (gnome_font->size);
  current_x_name = setFontPixelSize (unsized_name, current_size);
  current_font = find_best_x_weight (current_x_name, gnome_font->fontmap_entry->weight_code);

  /* start by getting some x font loaded */
  if (current_font == NULL)
    {
      printf ("couldn't load first font '%s'\n", current_x_name);
      /* is it italic? */
      if (gnome_font->fontmap_entry->italic)
	{
	  /* try oblique */
	  printf ("trying oblique...\n");
	  setComponentReplace (&current_x_name, setFontSlant, "o");
	  current_font = find_best_x_weight (current_x_name, gnome_font->fontmap_entry->weight_code);

	  if (current_font == NULL)
	    {
	      /* failed.  put italic back */
	      printf ("oblique failed: '%s'\n", current_x_name);
	      setComponentReplace (&current_x_name, setFontSlant, "i");
	    }
	}

      if (current_font == NULL)
	{
	  /* try helvetica */
	  printf ("trying hevletica...\n");
	  setComponentReplace (&current_x_name, setFontFamily, "helvetica");
	  current_font = find_best_x_weight (current_x_name, gnome_font->fontmap_entry->weight_code);

	  if (current_font == NULL)
	    {
	      printf ("helvetica failed. giving up...\n");
	      return NULL;
	    }
	}
    }

  printf ("initial name loaded: '%s'\n", current_x_name);

#if 0
  current_width = gdk_text_width (current_font, test_string, length);
  current_diff = (int) (perfect_width - current_width);

  printf ("size=%d, starting diff=%d\n", current_size, current_diff);

  /* are we looking for a bigger x font or a smaller one? */
  /*
  if (current_diff > 0) direction = 1;
  else direction = -1;
  */
  if (current_diff > 0)
    {
      /*direction = (int) gnome_font->size / 6.0;*/
      direction = 10;
      /*if (direction < 1) direction = 1;*/
      printf ("size=%f, direction=%d\n", gnome_font->size, direction);
    }
  else
    {
      /*direction = (int) gnome_font->size / -6.0;*/
      direction = -10;
      /*if (direction > -1) direction = -1;*/
      printf ("size=%f, direction=%d\n", gnome_font->size, direction);
    }


  while (current_diff != 0)
    {
      int next_size = current_size + direction;
      char *next_x_name = setFontPixelSize (current_x_name, next_size);
      GdkFont *next_font = find_best_x_weight (next_x_name, gnome_font->fontmap_entry->weight_code);
      int next_width;
      int next_diff;

      if (next_font == NULL)
	{
	  printf ("couldn't load '%s'\n", next_x_name);
	  continue; /* ??? */
	}
      next_width = gdk_text_width (next_font, test_string, length);
      next_diff = (int) (perfect_width - next_width);

      /* if we've gotten bigger than max_width, stop */
      if (next_width >= max_width && direction > 0)
	{
	  printf (" bailing because next_width[%d] >= max_width[%d]\n",
		  next_width, (int) max_width);
	  break;
	}

      /* if the previous font was closer than this one, stop */
      if (iabs (next_diff) > iabs (current_diff))
	{
	  printf (" bailing because next_diff[%d] > current_diff[%d]\n",
		  next_diff, current_diff);
	  break;
	}

      g_free (current_x_name);
      gdk_font_unref (current_font);

      current_size = next_size;
      current_x_name = next_x_name;
      current_font = next_font;
      current_diff = next_diff;

      printf (" new size=%d, new diff=%d\n", current_size, current_diff);
    }
#endif
  (*x_font_name_ret) = current_x_name;
  return current_font;
}



/* construct an x font name and find the best point size */

static GnomeDisplayFont *create_display_font (char *family,
					      GnomeFontWeight weight,
					      gboolean italic,
					      double points,
					      double scale)
{
  char *x_family;
  char *xfn = g_strdup ("-*-helvetica-*-r-*-*-12-*-*-*-*-*-*-*");
  GnomeFontUnsized *gfus;
  GnomeFont *gnome_font, *closest;
  GnomeDisplayFont *wpdf;

  g_return_val_if_fail (family, NULL);

  if (! scaled_display_fonts) initialize_hashes ();

  if ((closest = gnome_font_new_closest (family, weight, italic, 0)) == NULL)
	  return NULL;
	
  gfus = closest->fontmap_entry;
  gnome_font = gnome_font_new_closest (family, weight, italic, points * scale);
  wpdf = (GnomeDisplayFont *) g_malloc (sizeof (GnomeDisplayFont));

  wpdf->unsized_font = gfus;
  wpdf->gnome_font = gnome_font;
  wpdf->scale = scale;

  x_family = gnome_font_family_to_x_name (gfus->familyname);
  setComponentReplace (&xfn, setFontFamily, x_family);
  setComponentReplace (&xfn, setFontWeight,
		       gnome_font_weight_to_string (gfus->weight_code));
  if (gfus->italic)
    setComponentReplace (&xfn, setFontSlant, "i");

  wpdf->gdk_font =
    find_best_x_font (xfn, wpdf->gnome_font, &(wpdf->x_font_name));

  g_free (xfn);

  g_return_val_if_fail (wpdf->gdk_font, NULL);
  return wpdf;
}


/* cache for create_display_font */

GnomeDisplayFont *gnome_get_display_font (char *family,
					  GnomeFontWeight weight,
					  gboolean italic,
					  double points,
					  double scale)
{
  char key[ 1024 ];
  GnomeDisplayFont *wpdf;

  g_snprintf (key, sizeof (key), "%s.%s.%s.%d",
	      family,
	      gnome_font_weight_to_string (weight),
	      italic?"t":"f",
	      (int) (points * scale));

  if (! scaled_display_fonts) initialize_hashes ();

  /*
  printf ("lookup_display_font -- hash:%p key='%s'\n",
	  scaled_display_fonts,
	  key);
  */
  wpdf =
    (GnomeDisplayFont *) g_hash_table_lookup (scaled_display_fonts, key);

  if (wpdf == NULL)
    {
      wpdf = create_display_font (family, weight, italic, points, scale);
      g_return_val_if_fail (wpdf, NULL);
      g_hash_table_insert (scaled_display_fonts, g_strdup (key), wpdf);
    }

  return wpdf;
}

GnomeDisplayFont *gnome_font_get_display_font( GnomeFont *font )
{
  return gnome_get_display_font( font->fontmap_entry->familyname,
				 font->fontmap_entry->weight_code,
				 font->fontmap_entry->italic,
				 font->size,
				 font->scale );
}



/*** X font name munging ***/

/*

 0  char *foundry
 1  char *family
 2  char *weight
 3  char *slant
 4  char *sWidth
 5  char *adStyle
 6   int *pixelSize
 7   int *pointSize
 8   int resolutionX
 9   int *resolutionY
10  char *space
11  char *averageWidth
12  char *registry
13  char *encoding

*/


char *getFontComponent (char *font_name, unsigned int pos)
{
  char *p, *parts[ 14 ];
  char *scratch;
  char *ret;
  int i;

  if (pos > 13)
    {
      fprintf (stderr, "replaceFontComponent -- pos out of ");
      fprintf (stderr, "range: %d\n", pos);
      return NULL;
    }

  if (font_name == NULL || strcmp (font_name, "") == 0)
    return g_strdup ("*");

  scratch = g_strdup (font_name);

  for (i=0, p=scratch+1; i<14 && *p != '\0'; i++)
    {
      parts[ i ] = p;
      for (; *p != '-' && *p != '\0'; p ++)
	;
      *p = '\0';
      p ++;
    }

  ret = g_strdup (parts[ pos ]);
  g_free (scratch);

  return ret;
}


char *setFontComponent (char *font_name, unsigned int pos, char *s)
{
  char *p, *parts[ 14 ];
  char *scratch;
  char *new_name;
  int i, len;

  if (pos > 13)
    {
      fprintf (stderr, "replaceFontComponent -- pos out of ");
      fprintf (stderr, "range: %d\n", pos);
      return g_strdup (font_name);
    }

  if (font_name != NULL && *font_name != '\0')
    scratch = g_strdup (font_name);
  else
    {
      char *d = "-*-*-*-*-*-*-*-*-*-*-*-*-*-*";
      scratch = g_strdup (d);
    }

  for (i=0, p=scratch+1; i<14 && (*p) != '\0'; i++)
    {
      parts[ i ] = p;
      for (; *p != '-' && *p != '\0'; p ++)
	;
      *p = '\0';
      p ++;
    }

  if (s == NULL)
    parts[ pos ] = "*";
  else
    parts[ pos ] = s;

  for (i=0, len=0; i<14; i++)
    len += strlen (parts[ i ]);

  new_name = (char *) g_malloc (len + 17);

  sprintf(new_name, "-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s",
	  parts[ 0 ], parts[ 1 ], parts[ 2 ], parts[ 3 ], 
	  parts[ 4 ], parts[ 5 ], parts[ 6 ], parts[ 7 ], 
	  parts[ 8 ], parts[ 9 ], parts[ 10 ], parts[ 11 ],
	  parts[ 12 ], parts[ 13 ]);

  g_free (scratch);
  return new_name;
}


char *getFontFoundry (char *font_name)
{
  return getFontComponent (font_name, 0);
}

char *getFontFamily (char *font_name)
{
  return getFontComponent (font_name, 1);
}

char *getFontWeight (char *font_name)
{
  return getFontComponent (font_name, 2);
}

GnomeFontWeight getGnomeFontWeight (char *font_name)
{
  char *weight = getFontComponent (font_name, 2);
  GnomeFontWeight fw = string_to_gnome_font_weight (weight);
  g_free (weight);
  return fw;
}

char *getFontSlant (char *font_name)
{
  return getFontComponent (font_name, 3);
}

gboolean fontIsItalic (char *font_name)
{
  char *slant = getFontComponent (font_name, 3);
  gboolean s = FALSE;

  if (strcasecmp (slant, "i") == 0) s = TRUE;
  else if (strcasecmp (slant, "o") == 0) s = TRUE;
  else if (strcasecmp (slant, "r") == 0) s = FALSE;

  g_free (slant);
  return s;
}

char *getFontSWidth (char *font_name)
{
  return getFontComponent (font_name, 4);
}

char *getFontAdStyle (char *font_name)
{
  return getFontComponent (font_name, 5);
}

int getFontPixelSize (char *font_name)
{
  int ret;
  char *p = getFontComponent (font_name, 6);
  ret = atoi (p);
  g_free (p);
  return ret;
}

int getFontPointSize (char *font_name)
{
  int ret;
  char *p = getFontComponent (font_name, 7);
  ret = atoi (p);
  g_free (p);
  return ret;
}

int getFontResolutionX (char *font_name)
{
  int ret;
  char *p = getFontComponent (font_name, 8);
  ret = atoi (p);
  g_free (p);
  return ret;
}

int getFontResolutionY (char *font_name)
{
  int ret;
  char *p = getFontComponent (font_name, 9);
  ret = atoi (p);
  g_free (p);
  return ret;
}

char *getFontSpace (char *font_name)
{
  return getFontComponent (font_name, 10);
}

int getFontAverageWidth (char *font_name)
{
  int ret;
  char *p = getFontComponent (font_name, 11);
  ret = atoi (p);
  g_free (p);
  return ret;
}

char *getFontRegistry (char *font_name)
{
  return getFontComponent (font_name, 12);
}

char *getFontEncoding (char *font_name)
{
  return getFontComponent (font_name, 13);
}




char *setFontFoundry (char *font_name, char *s)
{
  return setFontComponent (font_name, 0, s);
}

char *setFontFamily (char *font_name, char *s)
{
  return setFontComponent (font_name, 1, s);
}

char *setFontWeight (char *font_name, char *s)
{
  return setFontComponent (font_name, 2, s);
}

char *setFontSlant (char *font_name, char *s)
{
  return setFontComponent (font_name, 3, s);
}

char *setFontSWidth (char *font_name, char *s)
{
  return setFontComponent (font_name, 4, s);
}

char *setFontAdStyle (char *font_name, char *s)
{
  return setFontComponent (font_name, 5, s);
}

char *setFontPixelSize (char *font_name, int i)
{
  gchar *temp;
  gchar *returnval;
  temp = g_strdup_printf( "%d", i );
  returnval = setFontComponent (font_name, 6, temp);
  g_free( temp );
  return returnval;
}

char *setFontPointSize (char *font_name, int i)
{
  gchar *temp;
  gchar *returnval;
  temp = g_strdup_printf( "%d", i );
  returnval = setFontComponent (font_name, 7, temp);
  g_free( temp );
  return returnval;
}

char *setFontResolutionX (char *font_name, int i)
{
  gchar *temp;
  gchar *returnval;
  temp = g_strdup_printf( "%d", i );
  returnval = setFontComponent (font_name, 8, temp);
  g_free( temp );
  return returnval;
}

char *setFontResolutionY (char *font_name, int i)
{
  gchar *temp;
  gchar *returnval;
  temp = g_strdup_printf( "%d", i );
  returnval = setFontComponent (font_name, 9, temp);
  g_free( temp );
  return returnval;
}

char *setFontSpace (char *font_name, char *s)
{
  return setFontComponent (font_name, 10, s);
}

char *setFontAverageWidth (char *font_name, int i)
{
  gchar *temp;
  gchar *returnval;
  temp = g_strdup_printf( "%d", i );
  returnval = setFontComponent (font_name, 11, temp);
  g_free( temp );
  return returnval;
}

char *setFontRegistry (char *font_name, char *s)
{
  return setFontComponent (font_name, 12, s);
}

char *setFontEncoding (char *font_name, char *s)
{
  return setFontComponent (font_name, 13, s);
}

static void setComponentReplace (char **io,
				 char *(*func) (char *font_name, char *s),
				 char *part)
{
  char *new_name = (*func) (*io, part);
  g_free (*io);
  (*io) = new_name;
}
