/* srmain.c
 *
 * Copyright 2001, 2002 Sun Microsystems, Inc.,
 * Copyright 2001, 2002 BAUM Retec, A.G.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/* #define DEBUG_SRCORE  */
#include "config.h"
#include <stdio.h>
#include <gnome.h>
#include "srbrl.h"
#include "srspc.h"
#include "srmag.h"
#include "libsrconf.h"
#include "brlinp.h"
#include "libke.h"
#include "SREvent.h"
#include "srctrl.h"
#include "SRObject.h"
#include "SRLow.h"
#include "srmain.h"
#include "srpres.h"
#include "srintl.h"

#define MIN_BRAILLE_PORT 1
#define MAX_BRAILLE_PORT 4

static SRLClientHandle src_srl_client;
gboolean src_use_braille = FALSE;
gboolean src_use_magnifier = FALSE;
gboolean src_use_speech	= FALSE;
gboolean src_use_braille_monitor = FALSE;

extern gchar *src_braille_device;
extern gint src_braille_port;
extern gint src_braille_position_sensor;
extern gint src_braille_optical_sensor;
extern gint src_speech_mode;
gboolean src_mouse_take;
gboolean src_mouse_click;
gboolean src_enable_format_string = FALSE; 

extern gboolean panning;

extern gint src_braille_offset;

void
src_present_null_sro ()
{
    if (src_use_braille) 
	src_braille_show (_("No focus"));
	
    if (src_use_braille_monitor) 
	src_brlmon_show (_("No focus")); 

    if (src_use_speech)
    	src_speech_say_message (_("No accessible object is focused"));
}


static gchar *src_last_key_echo		= NULL;
static gchar *src_last_punct_echo 	= NULL;
static gchar *src_last_space_echo 	= NULL;
static gchar *src_last_modifier_echo 	= NULL;
static gchar *src_last_cursor_echo 	= NULL;

static gboolean src_process_key_  	(gchar *key);
static gboolean src_process_punct 	(gchar *punct);
static gboolean src_process_space 	(gchar *space);
static gboolean src_process_modifier 	(gchar *modifier);
static gboolean src_process_cursor 	(gchar *cursor);



void
src_present_real_sro (SRObject *obj)
{
    static SRObject *old = NULL;
    gchar *reason;
    if (!obj)
	return;
    
    if (old != obj)
	src_braille_offset = 0;
    old = obj;

    if (!src_use_braille && !src_use_speech && !src_use_magnifier && !src_use_braille_monitor)
	return;
    reason = NULL;
    if (!sro_get_reason (obj, &reason))
	return;
    
    if (src_use_braille || src_use_braille_monitor)
    {
	gchar *txt;
	
	src_enable_format_string = FALSE; 
    	txt = src_presentation_present_sro_for_device (obj, reason, "braille");
	if (txt)
	{
	    gchar *tmp;
	    gchar *cursor = NULL;
	    SRObjectRoles role;
	    gchar *status = NULL;
	    gchar offset[5];

	    sro_get_role (obj, &role, SR_INDEX_CONTAINER);
	    if (role == SR_ROLE_TEXT || role == SR_ROLE_TERMINAL)
	    {
		SRLong offset;
		SRTextAttribute *attr;
		gchar status_[5] = "    ";

		if (sro_text_get_caret_offset (obj, &offset, SR_INDEX_CONTAINER))
		    cursor = g_strdup_printf (" cursor=\"%ld\" cstyle=\"underline\"", offset + 4);
	    
		if (sro_text_get_text_attr_from_caret (obj, SR_TEXT_BOUNDARY_CHAR, 
				    &attr, SR_INDEX_CONTAINER))
		{
		    gchar *tmp;
		    if (sra_get_attribute_value (attr[0], "bold", &tmp))
		    {
			if (strcmp (tmp, "true") == 0)
			    status_[0] = 'B';
			SR_freeString (tmp);
		    }
	    	    if (sra_get_attribute_value (attr[0], "italic", &tmp))
		    {
			if (strcmp (tmp, "true") == 0)
			    status_[1] = 'I';
			SR_freeString (tmp);
		    }
	    	    if (sra_get_attribute_value (attr[0], "underline", &tmp))
		    {
			if (strcmp (tmp, "single") == 0 || strcmp (tmp, "double") == 0)
			    status_[2] = 'U';
			SR_freeString (tmp);
		    }
	    	    if (sra_get_attribute_value (attr[0], "selected", &tmp))
		    {
			if (strcmp (tmp, "true") == 0)
			    status_[3] = 'S';
			SR_freeString (tmp);
		    }
		    sra_free (attr);
		    status = g_strconcat("<BRLDISP role=\"status\" start=\"0\" clear=\"true\">"
					 "<TEXT>",
					 status_,
					 "</TEXT>"
					 "</BRLDISP>", 
					 NULL);
		}
	    }
	    sprintf (offset, "%d", src_braille_offset);
	    tmp = g_strconcat ("<BRLOUT language=\"us\" bstyle=\"8\" clear=\"true\">",
				"<BRLDISP role=\"main\" offset=\"", offset, "\" clear=\"true\"",
				cursor ? cursor : "",
				">",
				txt,
				"</BRLDISP>",
				status ? status : "",
				"</BRLOUT>",
				NULL);
	    /*fprintf (stderr, "\n%s", tmp);*/
	    if (src_use_braille)
		src_braille_send (tmp);
	    if (src_use_braille_monitor)
		src_brlmon_send (tmp);
	    g_free (tmp);
	    g_free (cursor);
	}
	g_free (txt);
    }
    
    if (src_use_speech)
    {
	src_enable_format_string = TRUE; 
	{
	    SRObjectRoles role;
	    sro_get_role (obj, &role, SR_INDEX_CONTAINER);
	    if (strcmp (reason, "object:text-changed:insert") == 0
		    && (role == SR_ROLE_TEXT || role == SR_ROLE_TERMINAL))
	    {
		gchar *diff;
		if (sro_text_get_difference (obj, &diff, SR_INDEX_CONTAINER))
		{
		    if (diff[1] != '\0')
		    {
			if (src_last_key_echo)
			{
			    g_free (src_last_key_echo);
			    src_last_key_echo = NULL;
			}
		    }
		    else
		    {
			if (g_ascii_ispunct (diff[0]))
	    		    src_process_punct (diff);
			else if (g_ascii_isspace (diff[0]))
	    		    src_process_space (diff);
			else
	    		    src_process_key_ (diff);
		    }
		    SR_freeString (diff);
		}
	    }
	    else
	    {
		gchar *txt;

	    	if (src_last_key_echo)
		{
		    g_free (src_last_key_echo);
		    src_last_key_echo = NULL;
		}
		txt = src_presentation_present_sro_for_device (obj, reason, "speech");
    		if (txt && txt[0])
    		{
		    src_speech_send_chunk (txt, SRC_SPEECH_PRIORITY_IDLE);
		}
		g_free (txt);
	    }
	}
    }
        

    if (src_use_magnifier)
    {
	gchar *txt;
	txt = src_presentation_present_sro_for_device (obj, reason, "magnifier");
	if (txt)
	{
	    gchar *tmp;
	    tmp = g_strconcat ("<MAGOUT><ZOOMER ID=\"generic_zoomer\" tracking=\"focus\" ",
				txt,
				"></ZOOMER></MAGOUT>",
				NULL);
	    if (tmp)
		src_magnifier_send (tmp);
	    g_free (tmp);
	    g_free (txt);
	}
    }
    SR_freeString (reason);
}


SRObject *src_focused_sro;
SRObject *src_crt_sro;
SRObject *src_crt_tooltip;
SRObject *src_crt_window;


gboolean
src_present_crt_sro ()
{
    if (!src_crt_sro)
	src_present_null_sro ();
    else
	src_present_real_sro (src_crt_sro);

    return TRUE;
}



gboolean
src_present_crt_window ()
{
    gchar *reason;
    sru_assert (src_crt_window);
    if (!sro_get_reason (src_crt_window, &reason))
	return FALSE;
    
    if (src_use_speech)
    {
    	gchar *txt;

	src_enable_format_string = FALSE; 
	txt = src_presentation_present_sro_for_device (src_crt_window, reason, "speech");
    	if (txt && txt[0])
    	{
	    src_speech_send_chunk (txt, SRC_SPEECH_PRIORITY_SYSTEM);
	}
	g_free (txt);
    }
    SR_freeString (reason);
    return TRUE;
}


gboolean
src_present_crt_tooltip ()
{
    gchar *reason;
    sru_assert (src_crt_tooltip);

    if (!sro_get_reason (src_crt_tooltip, &reason))
	return FALSE;
    
    if (src_use_speech)
    {
    	gchar *txt;

	src_enable_format_string = FALSE; 
	txt = src_presentation_present_sro_for_device (src_crt_tooltip, reason, "speech");
    	if (txt && txt[0])
    	{
	    src_speech_send_chunk (txt, SRC_SPEECH_PRIORITY_SYSTEM);
	}
	g_free (txt);
    }
    SR_freeString (reason);
    return TRUE;
}




gboolean
src_present_details ()
{
    gchar *name, *role, *description, *location, *message, *tmp;
    SRRectangle rect;
    
    sru_assert (src_crt_sro);
    
    role = NULL;
    if (sro_get_role_name (src_crt_sro, &tmp, SR_INDEX_CONTAINER))
    {
	role = src_xml_make_part ("TEXT", src_speech_get_voice ("role"), tmp);
	SR_freeString (tmp);
    }
    
    location = g_strdup (_("unknown location"));
    if (sro_get_location (src_crt_sro, SR_COORD_TYPE_SCREEN, 
				&rect, SR_INDEX_CONTAINER))
    {
	location = g_strdup_printf ("x %d y %d width %d height %d", rect.x, rect.y,
			    rect.width, rect.height);
    }

    tmp = location;
    location = src_xml_make_part ("TEXT", src_speech_get_voice ("location"), location);
    g_free (tmp);
    
    name = NULL;
    if (sro_get_name (src_crt_sro, &tmp, SR_INDEX_CONTAINER))
    {
	name = src_xml_make_part ("TEXT", src_speech_get_voice ("name"), tmp);
        SR_freeString (tmp);
    }
    
    description = NULL;
    if (sro_get_description (src_crt_sro, &tmp, SR_INDEX_CONTAINER))
    {
	description = src_xml_make_part ("TEXT", src_speech_get_voice ("name"), tmp);
        SR_freeString (tmp);
    }
    
    message = NULL;
    if (name || description || location || role)
    {
	message = g_strconcat (	name ? name : "",
				role ? role : "",
				description ? description : "",
				location ? location : "",
				NULL); 
    }
    else
    {
	message = src_xml_make_part ("TEXT", src_speech_get_voice ("system"), _("no details")); 
    }    
    
    if (message && message[0])
	src_speech_send_chunk (message, SRC_SPEECH_PRIORITY_IDLE);
    
    g_free (name);
    g_free (role);
    g_free (description);
    g_free (location);
    g_free (message);
    
    return TRUE;
}


static void
src_process_layer_event (SREvent *event, 
			 unsigned long flags)
{
    gchar *key;
    
    sru_assert (event);

    sre_get_event_data (event, (void **)&key);
    if (key)
	src_ctrl_process_key (key);
}

static gchar *src_layer = NULL;

gboolean
src_present_layer_changed ()
{
    sru_assert (src_layer);
    
    if (src_use_speech)
    {    
	gchar *message, *tmp;
	
	tmp = g_strdup_printf ("%s %s", _("layer"), src_layer + (src_layer[1] == '0' ? 2 : 1));
		
	message = src_xml_make_part ("TEXT", src_speech_get_voice ("system"), tmp);
	if (message && message[0])
	    src_speech_send_chunk (message, SRC_SPEECH_PRIORITY_IDLE);

	g_free (tmp);
	g_free (message);
    }
    
    return TRUE;
}

gboolean
src_present_layer_timeout ()
{
    sru_assert (src_layer);

    if (src_use_speech)
    {
	gchar *message, *tmp;
	tmp = g_strdup_printf ("%s %s", _("back to layer"), src_layer + (src_layer[1] == '0' ? 2 : 1));
	
	message = src_xml_make_part ("TEXT", src_speech_get_voice ("system"), tmp);
	if (message && message[0])
	    src_speech_send_chunk (message, SRC_SPEECH_PRIORITY_IDLE);

	g_free (tmp);
	g_free (message);
    }
    return TRUE;
}


static void
src_process_layer_changed_event (SREvent *event, 
				 unsigned long flags)
{
    gchar *layer, *cmd;
    
    sru_assert (event);
    
    sre_get_event_data (event, (void **)&layer);
    src_layer = g_strdup (layer);
    
    cmd = NULL;
    if (flags == KE_LAYER_TIME_OUT)
	cmd = "present layer timeout";
    else if (flags == KE_LAYER_CHANGED)
	cmd = "present layer changed";
    else
	sru_assert_not_reached ();    
    if (cmd)
    {
	src_cmd_queue_add (cmd);
	src_cmd_queue_process ();
    }
}


static void
src_process_sro (SRObject *obj)
{
    if (src_focused_sro)
    {
	sro_release_reference (src_focused_sro);
	src_focused_sro = NULL;
    }
    if (src_crt_sro)
    {
	sro_release_reference (src_crt_sro);
	src_crt_sro = NULL;
    }

    if (obj)
    {
        src_focused_sro = src_crt_sro = obj;
	sro_add_reference (src_focused_sro);
	sro_add_reference (src_crt_sro);
    }
    
    src_cmd_queue_add ("present current object");
    src_cmd_queue_process ();
}



static void
src_process_window (SRObject *obj)
{
    if (src_crt_window)
    	sro_release_reference (src_crt_window);
    src_crt_window = obj;

    if (obj)
	sro_add_reference (src_crt_window);
/*    {
	gchar *r = NULL, *n = NULL;
	sro_get_reason (obj, &r);
	sro_get_name (obj, &n, SR_INDEX_CONTAINER);
	fprintf (stderr, "\n%s for %s", r, n);
	SR_freeString (r);
	SR_freeString (n);
    }
*/        
    src_cmd_queue_add ("present current window");
    src_cmd_queue_process ();
}



static void
src_process_tooltip (SRObject *obj)
{

    if (src_crt_tooltip)
    	sro_release_reference (src_crt_tooltip);
    src_crt_tooltip = obj;

    if (obj)
	sro_add_reference (src_crt_tooltip);
        
    src_cmd_queue_add ("present current tooltip");
    src_cmd_queue_process ();
}


/*************************************************************/
/* GCONF CALLBACKS */

static void
src_process_config_changed_for_braille (SRConfigStructure *config)
{
    gboolean restart = FALSE;
    
    if (!config)
	return;
	
    if (config->key && 
	config->type == CFGT_STRING && 
	strcmp (config->key, BRAILLE_DEVICE) == 0)
    {
	if (config->newvalue)
	{
	    if (strcmp (config->newvalue,src_braille_device))
	    {
		if (src_braille_device) 
		    g_free (src_braille_device);
		
		src_braille_device = g_strdup (config->newvalue);			
	    
		restart = TRUE;
	    }
	    else
	    {
		fprintf (stderr, "SR: process config changed for braille: brl_device did not change!!!\n");
	    }
	}
	else
	{
	    fprintf (stderr, "SR: process config changed for braille: brl_device NULL!!!\n");
	}
    }
    else
    if (config->key && 
	config->type == CFGT_INT && 
	strcmp (config->key, BRAILLE_PORT_NO) == 0)
    {
    	if (config->newvalue)
	{
	    if (src_braille_port != *((gint*)config->newvalue))
	    {
		src_braille_port = *((gint*)config->newvalue);
		restart = TRUE;
	    }
	}
    }
    else
/*    if (config->key && strcmp (config->key, BRAILLE_ATTRIBUTES) == 0)
    {
    }
    else
    if (config->key && strcmp (config->key, BRAILLE_STYLE) == 0)
    {
    }
    else
    if (config->key && strcmp (config->key, BRAILLE_CURSOR) == 0)
    {
    }
    else
    if (config->key && strcmp (config->key, BRAILLE_TRANSLATION) == 0)
    {
    }
    else
    if (config->key && strcmp (config->key, BRAILLE_FILL_CHAR) == 0)
    {
    }
    else
    if (config->key && strcmp (config->key, BRAILLE_STATUS_CELL) == 0)
    {
    }
    else*/
    if (config->key && 
	config->type == CFGT_INT && 
	strcmp (config->key, BRAILLE_POSITION_SENSOR) == 0)
    {
	if (config->newvalue)
	{
	    if (src_braille_position_sensor != *((gint*)config->newvalue))
	    {
		src_braille_position_sensor = *((gint*)config->newvalue);
	    }
	}
    }
    else
    if (config->key && 
	config->type == CFGT_INT && 
	strcmp (config->key, BRAILLE_OPTICAL_SENSOR) == 0)
    {
	if (config->newvalue)
	{
	    if (src_braille_optical_sensor != *((gint*)config->newvalue))
	    {
		src_braille_optical_sensor = *((gint*)config->newvalue);
	    }
	}
    }
    
    if (restart)
    {
	src_use_braille = src_braille_restart ();
    }	
}



static void
src_process_config_changed_for_srcore (SRConfigStructure *config)
{    
    if (!config)
	return;
	
    if (config->key && 
	config->type == CFGT_BOOL && 
	strcmp (config->key, SRCORE_EXIT_KEY) == 0)
    {
	if (config->newvalue)
	{
	    if (*(gboolean *)(config->newvalue) == TRUE)
	    {
		sru_exit_loop();
	    }
	}
    }
/*    else	
    if (config->key && strcmp (config->key, SRCORE_FIND_TYPE) == 0)
    {
	sru_message ("Find Type event occured");
    }
    else
    if (config->key && strcmp (config->key, SRCORE_FIND_TEXT) == 0)
    {
	sru_message ("Find Text event occured");
    }
*/    else
    if (config->key && 
	config->type == CFGT_BOOL && 
	strcmp (config->key, SRCORE_BRAILLE_ACTIVE) == 0)
    {
	if (config->newvalue)
	{
	    if (src_use_braille != *((gboolean*)config->newvalue))
	    {
		src_use_braille = *((gboolean*)config->newvalue);
	    
		if (src_use_braille)
		{
		    src_use_braille = src_braille_init ();
		}
		else
		{
		    src_braille_terminate ();
		}
	    }
	}
    }
    else
    if (config->key &&
	config->type == CFGT_BOOL &&
	strcmp (config->key, SRCORE_BRAILLE_MONITOR_ACTIVE) == 0)
    {
	if (config->newvalue)
	{
	    if (src_use_braille_monitor != *((gboolean*)config->newvalue))
	    {
		src_use_braille_monitor = *((gboolean*)config->newvalue);
	    
		if (src_use_braille_monitor)
		{
		    src_use_braille_monitor = src_braille_monitor_init ();
		}
		else
		{
		    src_braille_monitor_terminate ();
		}
	    }
	}
    }
    else
    if (config->key && 
	config->type == CFGT_BOOL && 
	strcmp (config->key, SRCORE_SPEECH_ACTIVE) == 0)
    {
	if (config->newvalue)
	{
	    if (src_use_speech != *((gboolean*)config->newvalue))
	    {
		src_use_speech = *((gboolean*)config->newvalue);
	    
		if (src_use_speech)
		{
		    src_use_speech = src_speech_init();
		}
		else
		{
		    src_speech_terminate ();
		}
	    }
	}
    }
    else	
    if (config->key && 
	config->type == CFGT_BOOL && 
	strcmp (config->key, SRCORE_MAGNIF_ACTIVE) == 0)
    {
	if (config->newvalue)
	{
	    if (src_use_magnifier != *((gboolean*)config->newvalue))
	    {
		src_use_magnifier = *((gboolean*)config->newvalue);
	    
		if (src_use_magnifier)
		{
		    src_use_magnifier = src_magnifier_init();
		    if (src_use_magnifier) 
			src_magnifier_create ();
		}
		else
		{
		    /*sleep (1); ADI: TBR */
		    src_magnifier_terminate(); 
		}
	    }
	}
    }
    else
    if (config->key && 
	config->type == CFGT_BOOL &&
	strcmp (config->key, KEYBOARD_TAKE_MOUSE) == 0)
    {
	if (config->newvalue)
	{
	    if (src_mouse_take != *((gboolean*)config->newvalue))
	    {
		src_mouse_take = *((gboolean*)config->newvalue);
	    }
	}
    }
    else
    if (config->key && 
	config->type == CFGT_BOOL &&
	strcmp (config->key, KEYBOARD_SIMULATE_CLICK) == 0)
    {
	if (config->newvalue)
	{
	    if (src_mouse_click != *((gboolean*)config->newvalue))
	    {
		src_mouse_click = *((gboolean*)config->newvalue);
	    }
	}
    }
    else
    if (config->key && 
	config->type == CFGT_INT &&
	strcmp (config->key, SRCORE_SCREEN_REVIEW) == 0)
    {
	if (config->newvalue)
	{
	    /* TO DO */
	}
    }
}


static void
src_process_config_changed_for_speech (SRConfigStructure *config)
{
    if (!src_use_speech)
	return;

    src_speech_proces_config_changed (config);
}

static void
src_process_config_changed_for_magnifier (SRConfigStructure *config)
{
static gboolean src_zoom_factor_lock 	= FALSE;
static gboolean src_cursor_mag      	= FALSE;
static gboolean src_cursor	    	= FALSE;
static gboolean src_crosswire	    	= FALSE;

    if (!src_use_magnifier)
	return;


    if (config->key && 
	strcmp (config->key, MAGNIFIER_ACTIVE_SCHEMA MAGNIFIER_CURSOR) == 0)
    {
	if (src_use_magnifier && 
	    config->type == CFGT_BOOL)
	{
	    if (config->newvalue)
	    {
		src_cursor = *((gboolean*)config->newvalue);
	    
		if (src_use_speech)
		{
		    gchar *tmp;
		    tmp = g_strdup_printf (src_cursor ? 
				        _("cursor on") : 
				        _("cursor off"));
		    src_say_message (tmp);
		    g_free (tmp);
		}
 
		if (src_cursor) 
		{
		    src_magnifier_send ("<MAGOUT cursor=\"true\"></MAGOUT>");
		}	    
		else 
		{
	    	    src_magnifier_send ("<MAGOUT cursor=\"false\"></MAGOUT>");
		}
	    }
	}
    }
    else
    if (config->key && 
	strcmp (config->key, 
		MAGNIFIER_ACTIVE_SCHEMA MAGNIFIER_CURSOR_NAME) == 0)
    {
	if (config->newvalue && 
	    config->type == CFGT_STRING)
	{
	    gchar *cursors = (gchar*)config->newvalue;
	
	    if (src_use_magnifier) 
	    {
		gchar *mag_out = NULL;
		if (src_use_speech)
		{
		    gchar *tmp;
		    tmp = g_strdup_printf ("%s %s", 
					   _("curent cursor is"), 
					   cursors);
		    src_say_message (tmp);
		    g_free (tmp);
		}
 
		mag_out = g_strconcat("<MAGOUT CursorName=\"", 
				      cursors,
				      "\"></MAGOUT>",
				      NULL);
		if (mag_out)
		{
		    src_magnifier_send (mag_out);
		    g_free (mag_out);
		    mag_out = NULL;
		}
	    }
	}
    }
    else
    if (config->key && 
	strcmp (config->key, MAGNIFIER_ACTIVE_SCHEMA MAGNIFIER_CURSOR_SIZE) == 0)
    {
	if (src_use_magnifier && 
	    config->newvalue &&
	    /*src_cursor &&*/
	     config->type == CFGT_INT) 
	{
	    gchar *mag_out = NULL;
	    gint cursor_size;
	    
	    cursor_size = *((gint*)config->newvalue);
	    
	    cursor_size = (cursor_size > MAX_CURSOR_SIZE) ? 
			    MAX_CURSOR_SIZE : 
			    ( cursor_size < MIN_CURSOR_SIZE) ?
			    MIN_CURSOR_SIZE :
			    cursor_size;
	
	    if (src_use_speech)
	    {
		gchar *tmp;
		tmp = g_strdup_printf ("%s %d", 
					_("cursor size"), 
					cursor_size);
		src_say_message (tmp);
		g_free (tmp);
	    }

	    mag_out = g_strdup_printf ("<MAGOUT CursorSize=\"%d\"></MAGOUT>",
					cursor_size);

	    if (mag_out)
    	    {		
		src_magnifier_send (mag_out);
		g_free (mag_out);
		mag_out = NULL;
	    }
	}
    }
    else
    if (config->key && 
	strcmp (config->key, MAGNIFIER_ACTIVE_SCHEMA MAGNIFIER_CURSOR_MAG) == 0)
    {
	if (src_use_magnifier && 
	    config->newvalue &&
	    config->type == CFGT_BOOL)
	{
	    gchar *mag_out = NULL;
	    
	    src_cursor_mag = *((gboolean*)config->newvalue);
	    
	    if (src_use_speech)
	    {
		gchar *tmp;
		tmp = g_strdup_printf (src_cursor_mag ? 
				       _("cursor magnification on") :
				       _("cursor magnification off"));
		src_say_message (tmp);
		g_free (tmp);
	    }

	    mag_out = g_strdup_printf ("<MAGOUT CursorScale=\"%d\"></MAGOUT>",
					src_cursor_mag);
	    if (mag_out)
    	    {
		src_magnifier_send (mag_out);
		g_free (mag_out);
		mag_out = NULL;
 	    }
	}
    }
    else
    if (config->key && 
	strcmp (config->key, MAGNIFIER_ACTIVE_SCHEMA MAGNIFIER_CURSOR_COLOR) == 0)
    {
    	if (src_use_magnifier &&
	    config->newvalue &&
	    /*src_cursor &&
	    src_crosswire  && */
	    config->type == CFGT_INT)
	{

	    gint32 cursor_color;
	    gchar *mag_out = NULL;
	    cursor_color = *((gint32*)config->newvalue);
	    
	    if (src_use_speech)
	    {
		gchar *tmp;
		tmp = g_strdup_printf ("%s %u",
				       _("cursor color"),
				       cursor_color);
		src_say_message (tmp);
		g_free (tmp);
	    }

	    mag_out = g_strdup_printf ("<MAGOUT CursorColor=\"%u\"></MAGOUT>",
					cursor_color);

	    if (mag_out)
    	    {
		src_magnifier_send (mag_out);
		g_free (mag_out);
		mag_out = NULL;
    	    }
	}
    }

    else
   if (config->key && 
	strcmp (config->key, MAGNIFIER_ACTIVE_SCHEMA MAGNIFIER_CROSSWIRE) == 0)
    {
	if (src_use_magnifier && 
	    config->newvalue &&
	    /*src_cursor && */
	    config->type == CFGT_BOOL)
	{
    	    gchar *mag_out = NULL;
	    
	    src_crosswire = *((gboolean*)config->newvalue);
	

	    if (src_use_speech)
	    {
		gchar *tmp;
		tmp = g_strdup_printf (src_crosswire ? 
				       _("crosswire on") :
				       _("crosswire off"));
		src_say_message (tmp);
		g_free (tmp);
	    }

    	    mag_out = g_strdup_printf ("<MAGOUT crosswire=\"%d\"></MAGOUT>",
					src_crosswire);
    	    if (mag_out)
    	    {
    		src_magnifier_send (mag_out);
		g_free (mag_out);
		mag_out = NULL;	    
    	    }
	}
    }
    else
    if (config->key && 
	strcmp (config->key, MAGNIFIER_ACTIVE_SCHEMA MAGNIFIER_CROSSWIRE_CLIP) == 0)
    {
	if (src_use_magnifier && 
	    config->newvalue &&
	    /*src_cursor && 
	    src_crosswire && */
	    config->type == CFGT_BOOL)
	{
	    gboolean crosswire_clip;
	    
	    crosswire_clip = *((gboolean*)config->newvalue);
	    
	    if (src_use_speech)
	    {
		gchar *tmp;
		tmp = g_strdup_printf (crosswire_clip ? 
				       _("crosswire clip on") :
				       _("crosswire clipp off"));
		src_say_message (tmp);
		g_free (tmp);
	    }
	    
	    if (crosswire_clip)
	    {
		src_magnifier_send ("<MAGOUT CrosswireClip =\"true\"></MAGOUT>");
	    }
	    else
	    {
		src_magnifier_send ("<MAGOUT CrosswireClip=\"false\"></MAGOUT>");
	    }
	}
    }
    else
    if (config->key && 
	strcmp (config->key, MAGNIFIER_ACTIVE_SCHEMA MAGNIFIER_CROSSWIRE_SIZE) == 0)
    {
	if (src_use_magnifier && 
	    config->newvalue &&
	    /*src_cursor && 
	    src_crosswire  && */
	    config->type == CFGT_INT)
	{
	    gint crosswire_size;
	    gchar *mag_out = NULL;
	    
	    crosswire_size = *((gint*)config->newvalue);
	    
	    crosswire_size = 	(crosswire_size < MIN_CROSSWIRE_SIZE) ? 
				MIN_CROSSWIRE_SIZE : 
				(crosswire_size > MAX_CROSSWIRE_SIZE) ? 
				MAX_CROSSWIRE_SIZE :
				crosswire_size;
	    if (src_use_speech)
	    {
		gchar *tmp;
		tmp = g_strdup_printf ("%s %d", 
				       _("crosswire size"), 
				       crosswire_size);
		src_say_message (tmp);
		g_free (tmp);
	    }

	    mag_out = g_strdup_printf ("<MAGOUT CrosswireSize=\"%d\"></MAGOUT>",
					crosswire_size);

	    if (mag_out)
    	    {
		src_magnifier_send (mag_out);
		g_free (mag_out);
		mag_out = NULL;
    	    }
	}
    }
    else
    if (config->key && 
	strcmp (config->key, MAGNIFIER_ACTIVE_SCHEMA MAGNIFIER_CROSSWIRE_COLOR) == 0)
    {
    	if (src_use_magnifier &&
	    config->newvalue &&
	    /*src_cursor &&
	    src_crosswire  && */
	    config->type == CFGT_INT)
	{

	    gint32 crosswire_color;
	    gchar *mag_out = NULL;
	    crosswire_color = *((gint32*)config->newvalue);

	    if (src_use_speech)
	    {
		gchar *tmp;
		tmp = g_strdup_printf ("%s %u",
				       _("crosswire color"),
				       crosswire_color);
		src_say_message (tmp);
		g_free (tmp);
	    }

	    mag_out = g_strdup_printf ("<MAGOUT CrosswireColor=\"%u\"></MAGOUT>",
					crosswire_color);

	    if (mag_out)
    	    {
		src_magnifier_send (mag_out);
		g_free (mag_out);
		mag_out = NULL;
    	    }
	}
    }
    else
    if (config->key && 
	strcmp (config->key, MAGNIFIER_ACTIVE_SCHEMA MAGNIFIER_ZP_LEFT) == 0)
    {
    	if (src_use_magnifier &&
	    config->newvalue &&
	    config->type == CFGT_INT)
	{
	    gint zp_size = *((gint*)config->newvalue);
	    gchar *mag_out = NULL;
	    
	    if (src_use_speech)
	    {
		gchar *tmp;
		tmp = g_strdup_printf ("%s %u",
				       _("zoomer placement left"),
				       zp_size);
		src_say_message (tmp);
		g_free (tmp);
	    }
    	    mag_out = g_strdup_printf ("<MAGOUT><ZOOMER ID=\"generic_zoomer\" ZPLeft=\"%u\"></ZOOMER></MAGOUT>",
				    zp_size);

	    if (mag_out)
    	    {
		src_magnifier_send (mag_out);
		g_free (mag_out);
		mag_out = NULL;
    	    }
	}
    }
    else
    if (config->key && 
	strcmp (config->key, MAGNIFIER_ACTIVE_SCHEMA MAGNIFIER_ZP_TOP) == 0)
    {
    	if (src_use_magnifier &&
	    config->newvalue &&
	    config->type == CFGT_INT)
	{
	    gint zp_size = *((gint*)config->newvalue);
	    gchar *mag_out = NULL;

	    if (src_use_speech)
	    {
		gchar *tmp;
		tmp = g_strdup_printf ("%s %u",
				       _("zoomer placement top"),
				       zp_size);
		src_say_message (tmp);
		g_free (tmp);
	    }
    	    mag_out = g_strdup_printf ("<MAGOUT><ZOOMER ID=\"generic_zoomer\" ZPTop=\"%u\"></ZOOMER></MAGOUT>",
				    zp_size);

	    if (mag_out)
    	    {
		src_magnifier_send (mag_out);
		g_free (mag_out);
		mag_out = NULL;
    	    }
	}
    }
    else
    if (config->key && 
	strcmp (config->key, MAGNIFIER_ACTIVE_SCHEMA MAGNIFIER_ZP_WIDTH) == 0)
    {
    	if (src_use_magnifier &&
	    config->newvalue &&
	    config->type == CFGT_INT)
	{
	    gint zp_size = *((gint*)config->newvalue);
	    gchar *mag_out = NULL;

	    if (src_use_speech)
	    {
		gchar *tmp;
		tmp = g_strdup_printf ("%s %u",
				       _("zoomer placement width"),
				       zp_size);
		src_say_message (tmp);
		g_free (tmp);
	    }
    	    mag_out = g_strdup_printf ("<MAGOUT><ZOOMER ID=\"generic_zoomer\" ZPWidth=\"%u\"></ZOOMER></MAGOUT>",
				    zp_size);

	    if (mag_out)
    	    {
		src_magnifier_send (mag_out);
		g_free (mag_out);
		mag_out = NULL;
    	    }

	}

    }
    else
    if (config->key && 
	strcmp (config->key, MAGNIFIER_ACTIVE_SCHEMA MAGNIFIER_ZP_HEIGHT) == 0)
    {
    	if (src_use_magnifier &&
	    config->newvalue &&
	    config->type == CFGT_INT)
	{
	    gint zp_size = *((gint*)config->newvalue);
	    gchar *mag_out = NULL;

	    if (src_use_speech)
	    {
		gchar *tmp;
		tmp = g_strdup_printf ("%s %u",
				       _("zoomer placement height"),
				       zp_size);
		src_say_message (tmp);
		g_free (tmp);
	    }
    	    mag_out = g_strdup_printf ("<MAGOUT><ZOOMER ID=\"generic_zoomer\" ZPheight=\"%u\"></ZOOMER></MAGOUT>",
				    zp_size);

	    if (mag_out)
    	    {
		src_magnifier_send (mag_out);
		g_free (mag_out);
		mag_out = NULL;
    	    }

	}
    }
    
    else
/*    if (config->key && 
	strcmp (config->key, MAGNIFIER_ACTIVE_SCHEMA MAGNIFIER_BORDER_WIDTH) == 0)
    {
    }
    else
    if (config->key && 
	strcmp (config->key, MAGNIFIER_ACTIVE_SCHEMA MAGNIFIER_BORDER_COLOR) == 0)
    {
    }

    else
    if (config->key && 
	strcmp (config->key, MAGNIFIER_ACTIVE_SCHEMA MAGNIFIER_TARGET) == 0)
    {
    }
    else*/
      if (config->key &&
	strcmp (config->key, MAGNIFIER_ACTIVE_SCHEMA MAGNIFIER_SOURCE) == 0)
    {
	if (config->newvalue &&
	    config->type == CFGT_STRING)
	{
	    gchar *source = (gchar*)config->newvalue;

	    if (src_use_magnifier)
	    {
		gchar *mag_out;

		if (src_use_speech)
		{
		    gchar *tmp;
		    tmp = g_strdup_printf ("%s %s",
					   _("magnifier source"),
					   source);
		    src_say_message (tmp);
		    g_free (tmp);
		}

		mag_out = g_strconcat ("<MAGOUT><ZOOMER ID=\"generic_zoomer\" source=\"",
			    		source,
					" \"></ZOOMER></MAGOUT>",
					NULL);
		if (mag_out)
		{
		 /*   fprintf (stderr, "\n%s %d : source %s",
					__FILE__,
					__LINE__,
					source);
		 */ src_magnifier_send (mag_out);
		    g_free (mag_out);
		    mag_out = NULL;
		}
	    }
	}

    }

    else
    if (config->key && 
	strcmp (config->key, MAGNIFIER_ACTIVE_SCHEMA MAGNIFIER_ZOOM_FACTOR_X) == 0)
    {
	if (src_use_magnifier && 
	    config->newvalue &&
	    config->type == CFGT_INT)
	{
	    gint zoom_factor; 
	    	    
	    gchar *mag_out = NULL;
	    
	    zoom_factor = *((gint*)config->newvalue);

	    if (src_use_speech)
	    {
		gchar *tmp;
		tmp = g_strdup_printf ("%s %d", 
				       _("zoom factor x"), 
				       zoom_factor);
		src_say_message (tmp);
		g_free (tmp);
	    }
	    	    	    
	    mag_out = g_strdup_printf ("<MAGOUT><ZOOMER ID=\"generic_zoomer\" ZoomFactorX=\"%d\"></ZOOMER></MAGOUT>", 
						zoom_factor);
	    if (mag_out)
	    {			
		src_magnifier_send (mag_out);
		g_free (mag_out);
		mag_out = NULL;
	    }
	}
    }
    else
    if (config->key && 
	strcmp (config->key, MAGNIFIER_ACTIVE_SCHEMA MAGNIFIER_ZOOM_FACTOR_Y) == 0)
    {
	if (src_use_magnifier && 
	    config->newvalue &&
	    config->type == CFGT_INT)
	{
	    gint zoom_factor = *((gint*)config->newvalue);
	    gchar *mag_out = NULL;

	    if (src_use_speech)
	    {
		gchar *tmp;
		tmp = g_strdup_printf ("%s %d", 
					_("zoom factor y"), 
					zoom_factor);
		src_say_message (tmp);
		g_free (tmp);
	    }
		    	    
	    mag_out = g_strdup_printf ("<MAGOUT><ZOOMER ID=\"generic_zoomer\" ZoomFactorY=\"%d\"></ZOOMER></MAGOUT>", 
						zoom_factor);
	    if (mag_out)
	    {			
		src_magnifier_send (mag_out);
		g_free (mag_out);
		mag_out = NULL;
	    }
	}
    }
    else
    if (config->key && 
	strcmp (config->key, MAGNIFIER_ACTIVE_SCHEMA MAGNIFIER_ZOOM_FACTOR_LOCK) == 0)
    {
	if (src_use_magnifier && 
	    config->newvalue &&
	    config->type == CFGT_BOOL)
	{
	    src_zoom_factor_lock = *((gboolean*)config->newvalue);
	    
	    if (src_use_speech)
	    {
		gchar *tmp;
		tmp = g_strdup_printf (src_zoom_factor_lock ? 
				       _("zoom factor locked") :
				       _("zoom factor unlocked"));
		src_say_message (tmp);
		g_free (tmp);
	    }
	}
    }
    else
    if (config->key && 
	strcmp (config->key, MAGNIFIER_ACTIVE_SCHEMA MAGNIFIER_INVERT) == 0)
    {
	if (src_use_magnifier && 
	    config->newvalue &&
	    config->type == CFGT_BOOL);
	{
	    gboolean invert = *((gboolean*)config->newvalue);

	    
	    if (src_use_speech)
	    {
		gchar *tmp;
		tmp = g_strdup_printf (invert ? 
				       _("invert on") : 
				       _("invert off"));
		src_say_message (tmp);
		g_free (tmp);
	    }

	    if (invert) 
	    {
		src_magnifier_send ("<MAGOUT><ZOOMER ID=\"generic_zoomer\" invert=\"true\"></ZOOMER></MAGOUT>");
	    }	    
	    else 
	    {
		src_magnifier_send ("<MAGOUT><ZOOMER ID=\"generic_zoomer\" invert=\"false\"></ZOOMER></MAGOUT>");
	    }
	}
    }
    else
    if (config->key && 
	strcmp (config->key, MAGNIFIER_ACTIVE_SCHEMA MAGNIFIER_SMOOTHING) == 0)
    {
	if (config->newvalue && 
	    config->type == CFGT_STRING)
	{
	    gchar *smoothing = (gchar*)config->newvalue;
	
	    if (src_use_magnifier) 
	    {
		gchar *mag_out;
		
		if (src_use_speech)
		{
		    gchar *tmp;
		    tmp = g_strdup_printf ("%s %s", 
					   _("magnifier smoothing type"), 
					   smoothing);;
		    src_say_message (tmp);
		    g_free (tmp);
		}

		mag_out = g_strconcat ("<MAGOUT><ZOOMER ID=\"generic_zoomer\" smoothing=\"",
			    		smoothing,
					" \"></ZOOMER></MAGOUT>",
					NULL);
		if (mag_out)
		{
		    src_magnifier_send (mag_out);	    
		    g_free (mag_out);
		    mag_out = NULL;
		}
	    }	
	}
    }
/*    else
    if (config->key && 
	strcmp (config->key, MAGNIFIER_ACTIVE_SCHEMA MAGNIFIER_TRACKING) == 0)
    {
    }*/
    else
    if (config->key && 
	strcmp (config->key, MAGNIFIER_ACTIVE_SCHEMA MAGNIFIER_MOUSE_TRACKING) == 0)
    {
	if (config->newvalue && 
	    config->type == CFGT_STRING)
	{
	    gchar *mouse_tracking = (gchar*)config->newvalue;
	
	    if (src_use_magnifier) 
	    {
		gchar *mag_out = NULL;
		
		if (src_use_speech)
		{
		    gchar *tmp;
		    tmp = g_strdup_printf ("%s %s", 
					   _("mouse tracking mode"), 
					   mouse_tracking);
		    src_say_message (tmp);
		    g_free (tmp);
		}

		mag_out = g_strconcat("<MAGOUT><ZOOMER ID=\"generic_zoomer\"  MouseTracking=\"", 
				    mouse_tracking, 
				    "\"></ZOOMER></MAGOUT>",
				    NULL);
		if (mag_out)
		{
		    src_magnifier_send (mag_out);
		    g_free (mag_out);
		    mag_out = NULL;
		}
	    }
	}
    }
/*    else
    if (config->key && 
	strcmp (config->key, MAGNIFIER_ACTIVE_SCHEMA MAGNIFIER_VISIBLE) == 0)
    {
    }*/
    else
    if (config->key && 
	strcmp (config->key, MAGNIFIER_ACTIVE_SCHEMA MAGNIFIER_PANNING) == 0)
    {
    	if (src_use_magnifier && 
	    config->newvalue &&
	    config->type == CFGT_BOOL);
	{
	    panning = *((gboolean*)config->newvalue);
		
	    if (src_use_speech)
	    {
		gchar *tmp;
		tmp = g_strdup ( panning ? 
				_("panning on") : 
				_("panning off"));
		src_say_message (tmp);
		g_free (tmp);
	    }

	    if (!panning) 
		src_magnifier_stop_panning (0);
	}
    }

}

static void
src_process_config_changed_for_control (SRConfigStructure *config)
{
    if (config)
    {
	src_ctrl_terminate 	();
	src_ctrl_init 		();
	ke_config_changed	();
    }
    /*src_update_key ();*/
}

static void
src_process_config_changed_for_presentation (SRConfigStructure *config)
{
    if (config)
    {
	src_presentation_terminate ();
	src_presentation_init ();
    }
}

static void
src_process_config_changed_for_keyboard (SRConfigStructure *config)
{
    if (!config) 
	return;
    if (config->newvalue)
    {
/*	ke_config_changed (config);*/
    }
}



static void
src_process_config_changed (SREvent *event,
			    unsigned long flags)
{
    SRConfigStructure *config;
    
    if (!sre_get_event_data (event, (void**)&config))
	return;
    if (!config)
	return;
    
/* for debuggung purpose */
#ifdef DEBUG_SRCORE
    if (config != NULL)
    {
	printf ("\n*********************************************************************\n");
	printf ("SRCORE: CONFIG_CHANGED/process_config_changed(): \n\tModule: %d \n\tKey: %s\n\tType: %i\n\tNewValue: ",
		config->module, config->key, config->type);
	switch (config->type)
	{
	    case CFGT_INT:
		printf ("%i\n", *((gint*)config->newvalue));
		break;
    	    case CFGT_FLOAT:
		printf ("%f\n", *((gfloat*)config->newvalue));
		break;
    	    case CFGT_STRING:
		printf ("%s\n", config->newvalue ? config->newvalue : " ");
		break;
    	    case CFGT_BOOL:
		printf ("%i\n", *((gboolean*)config->newvalue));
		break;
	    default:
		break;
	}
	printf ("*********************************************************************\n");
	fflush(stdout);
    }
#endif

    switch (config->module)
    {
	case CFGM_BRAILLE:
	    src_process_config_changed_for_braille (config);
	    break;
	case CFGM_KBD_MOUSE:
	    src_process_config_changed_for_keyboard (config);
	    break;
	case CFGM_GNOPI:
	    break;
    	case CFGM_MAGNIFIER:
	    src_process_config_changed_for_magnifier (config); 
	    break;
	case CFGM_SPEECH_VOICE_PARAM:
	    src_speech_proces_voice_config_changed (config);
	    break;
	case CFGM_SPEECH:
	case CFGM_SPEECH_VOICE:
	    src_process_config_changed_for_speech (config);
	    break;
	case CFGM_SRCORE:
	    src_process_config_changed_for_srcore (config);
	    break;
	case CFGM_KEY_PAD:
	    src_process_config_changed_for_control (config);
	    break;
	case CFGM_PRESENTATION:
	    src_process_config_changed_for_presentation (config);
	    break;
    }
}

static void
src_process_keyboard_echo (SREvent *event,
			   unsigned long flags)
{
/*
    gchar *str;
    if (!sre_get_event_data (event, (void**)&str))
	return;
    if (!str)
	return;
    
    fprintf(stderr, "PROCESS KEYBOARD ECHO: %s\n", str);
*/
}
	
static void
src_process_hotkey (SREvent *event,
		    unsigned long flags)
{			
/*
    SRHotkeyData *srhotkey_data;
    
    if (!sre_get_event_data (event, (void**)&srhotkey_data))
	return;
    if (!srhotkey_data)
	return;
    
    fprintf(stderr, "SR: process HotKey: %s%s%s%s\n", srhotkey_data->modifiers & SRHOTKEY_ALT ? "ALT + " : "",
			 srhotkey_data->modifiers & SRHOTKEY_CTRL ? "CTRL + " : "",
			 srhotkey_data->modifiers & SRHOTKEY_SHIFT ? "SHIFT + ": "",
			 srhotkey_data->keystring ? srhotkey_data->keystring : " ");
*/
}
gboolean
src_kb_key_echo ()
{
    gchar *message;
    sru_assert (src_last_key_echo);

    if (!src_use_speech)
	return FALSE;
    message = src_xml_format ("TEXT", src_speech_get_voice ("system"), src_last_key_echo);
    if (message && message[0])
	src_speech_send_chunk (message, SRC_SPEECH_PRIORITY_IDLE);
    g_free (message); 
    
    return TRUE;    
}

gboolean
src_kb_punct_echo ()
{
    gchar *message;
    sru_assert (src_last_punct_echo);

    if (!src_use_speech)
	return FALSE;
    message = src_xml_format ("TEXT", src_speech_get_voice ("system"), src_last_punct_echo);
    if (message && message[0])
	src_speech_send_chunk (message, SRC_SPEECH_PRIORITY_IDLE);
    g_free (message); 
    
    return TRUE;    
}

gboolean
src_kb_space_echo ()
{
    gchar *message;
    sru_assert (src_last_space_echo);

    if (!src_use_speech)
	return FALSE;
    message = src_xml_format ("TEXT", src_speech_get_voice ("system"), src_last_space_echo);
    if (message && message[0])
	src_speech_send_chunk (message, SRC_SPEECH_PRIORITY_IDLE);
    g_free (message); 
    
    return TRUE;    
}

gboolean
src_kb_modifier_echo ()
{
    gchar *message;
    sru_assert (src_last_modifier_echo);

    if (!src_use_speech)
	return FALSE;
    message = src_xml_format ("TEXT", src_speech_get_voice ("system"), src_last_modifier_echo);
    if (message && message[0])
	src_speech_send_chunk (message, SRC_SPEECH_PRIORITY_IDLE);
    g_free (message); 
    
    return TRUE;    
}


gboolean
src_kb_cursor_echo ()
{
    gchar *message;
    sru_assert (src_last_cursor_echo);

    if (!src_use_speech)
	return FALSE;
    message = src_xml_format ("TEXT", src_speech_get_voice ("system"), src_last_cursor_echo);
    if (message && message[0])
	src_speech_send_chunk (message, SRC_SPEECH_PRIORITY_IDLE);
    g_free (message); 
    
    return TRUE;    
}


extern SRCSpeechTextEchoType src_speech_text_echo_type;
static gboolean
src_process_key_ (gchar *key)
{
    sru_assert (key);

    if (src_speech_text_echo_type == SRC_SPEECH_TEXT_ECHO_CHAR)
    {
	src_last_key_echo = key;
	src_cmd_queue_add ("kbd key");
	src_cmd_queue_process ();
	src_last_key_echo = NULL;	
    }
    else if (src_speech_text_echo_type == SRC_SPEECH_TEXT_ECHO_WORD)
    {
	gchar *tmp = src_last_key_echo;
	
	src_last_key_echo = g_strconcat (src_last_key_echo ? src_last_key_echo : "",
					 key,
					 NULL);
	g_free (tmp);	
    }
    else
	sru_assert_not_reached ();
    return TRUE;
}

#define SRC_TP_NONE	0
#define SRC_TP_ALL	1

static gboolean
src_process_punct (gchar *punct)
{
    static gint src_kb_punct_mode = SRC_TP_ALL;

    sru_assert (punct);
    
    g_free (src_last_punct_echo);
    src_last_punct_echo = NULL;
    
    if (src_last_key_echo)
	src_cmd_queue_add ("kbd key");

    
    if (src_kb_punct_mode == SRC_TP_ALL)
    {
	src_last_punct_echo = g_strdup (punct);
	src_cmd_queue_add ("kbd punct");
    }
    
    if (src_last_key_echo || src_last_punct_echo)
	src_cmd_queue_process ();
    
    g_free (src_last_key_echo);
    src_last_key_echo = NULL;
    
    return TRUE;
}


extern SRCSpeechSpacesEchoType src_speech_spaces_echo_type;
static gboolean
src_process_space (gchar *space)
{
    sru_assert (space);
    
    g_free (src_last_space_echo);
    src_last_space_echo = NULL;

    if (src_last_key_echo)
	src_cmd_queue_add ("kbd key");

    if (src_speech_spaces_echo_type == SRC_SPEECH_SPACES_ECHO_ALL)
    {
	src_last_space_echo = g_strdup (space);
	src_cmd_queue_add ("kbd space");
    }

    if (src_last_key_echo || src_last_space_echo)
	src_cmd_queue_process ();

    g_free (src_last_key_echo);
    src_last_key_echo = NULL;
    return TRUE;
}


extern SRCSpeechModifiersEchoType src_speech_modifiers_echo_type;
static gboolean
src_process_modifier (gchar *modifier)
{
    sru_assert (modifier);
    
    if (src_last_modifier_echo)
	g_free (src_last_modifier_echo);
    src_last_modifier_echo = NULL;

    if (src_speech_modifiers_echo_type == SRC_SPEECH_MODIFIERS_ECHO_ALL)
    {
	gchar *part1, *part2;
	gint pos;
    
	part1 = part2 = NULL;
	pos = 0;

	if (strncmp (modifier, "Alt", 3) == 0)
	{
	    part1 = _("alt");
	    pos = 4;
	}
	else if (strncmp (modifier, "Control", 7) == 0)
	{
	    part1 = _("control");
	    pos = 8;
	}
	else if (strncmp (modifier, "Shift", 5) == 0)
	{
	    part1 = _("shift");
	    pos = 6;
        }
	else if (strncmp (modifier, "Caps", 3) == 0)
	{
	    part1 = _("caps lock");
	}
	else if (strncmp (modifier, "Num", 3) == 0)
	{
	    part1 = _("num lock");
	}
	else
	    sru_assert_not_reached ();
    
	if (pos)
	{
	    if (modifier[pos] == 'R')
		part2 = _("right");
	    else if (modifier[pos] == 'L')
		part2 = _("left");
	    else
		sru_assert_not_reached ();
	}
	src_last_modifier_echo = g_strconcat (part1 ? part1 : "", 
					    part2 ? " " : "",
					    part2 ? part2 : "",
					    NULL);

	src_cmd_queue_add ("kbd modifier");
	src_cmd_queue_process ();
    }
    return TRUE;
}

extern SRCSpeechCursorsEchoType src_speech_cursors_echo_type;
static gboolean
src_process_cursor (gchar *cursor)
{
    sru_assert (cursor);
    
    if (src_last_cursor_echo)
	g_free (src_last_cursor_echo);
    src_last_cursor_echo = NULL;

    if (src_last_key_echo)
	src_cmd_queue_add ("kbd key");

    if (src_speech_cursors_echo_type == SRC_SPEECH_CURSORS_ECHO_ALL)
    {
	if (strcmp (cursor, "Home") == 0)
	    src_last_cursor_echo = g_strdup (_("home"));
	else if (strcmp (cursor, "End") == 0)
    	    src_last_cursor_echo = g_strdup (_("end"));
	else if (strcmp (cursor, "Page_Up") == 0)
	    src_last_cursor_echo = g_strdup (_("page up"));
	else if (strcmp (cursor, "Page_Down") == 0)
	    src_last_cursor_echo = g_strdup (_("page down"));
	else if (strcmp (cursor, "Right") == 0)
	    src_last_cursor_echo = g_strdup (_("right"));
	else if (strcmp (cursor, "Left") == 0)
	    src_last_cursor_echo = g_strdup (_("left"));
	else if (strcmp (cursor, "Down") == 0)
	    src_last_cursor_echo = g_strdup (_("down"));
	else if (strcmp (cursor, "Up") == 0)
	    src_last_cursor_echo = g_strdup (_("up"));
	else
	    sru_assert_not_reached ();
	        
	src_cmd_queue_add ("kbd cursor");
    }

    if (src_last_key_echo || src_last_cursor_echo)
	src_cmd_queue_process ();

    g_free (src_last_key_echo);
    src_last_key_echo = NULL;

    return TRUE;
}



static void
src_process_key (SREvent *event,
		 unsigned long flags)
{
    SRHotkeyData *srhotkey_data;

    if (!sre_get_event_data (event, (void**)&srhotkey_data))
	return;
    if (!srhotkey_data)
	return;

    if (!srhotkey_data->keystring)
	return;
    
    if (strncmp (srhotkey_data->keystring, "Alt", 3) == 0 ||
	strncmp (srhotkey_data->keystring, "Control", 7) == 0 ||
	strncmp (srhotkey_data->keystring, "Shift", 5) == 0 ||
	strncmp (srhotkey_data->keystring, "Caps", 3) == 0 ||
	strncmp (srhotkey_data->keystring, "Num", 3) == 0)
    {
    
	src_process_modifier (srhotkey_data->keystring);
    }
    else if (strcmp (srhotkey_data->keystring, "Home") == 0 ||
	     strcmp (srhotkey_data->keystring, "End") == 0 ||
	     strcmp (srhotkey_data->keystring, "Page_Up") == 0 ||
	     strcmp (srhotkey_data->keystring, "Page_Down") == 0 ||
	     strcmp (srhotkey_data->keystring, "Right") == 0 ||
	     strcmp (srhotkey_data->keystring, "Left") == 0 ||
	     strcmp (srhotkey_data->keystring, "Down") == 0 ||
	     strcmp (srhotkey_data->keystring, "Up") == 0)
    {
	src_process_cursor (srhotkey_data->keystring);
    }
}

typedef enum
{
    BST_PS = 0,
    BST_HOS
}SRBrlSensorTypes;

static gboolean
src_translate_sensor_code (gchar *sensor_code,
			   SRBrlSensorTypes *sensor_type,
			   gint *sensor_index)
{
	/* !!! TBI: !!! this table will be loaded for each actual device */
	/* !!! TO DO like this !!! */
/*	static gchar *sensor_codes = {	"HMS00", "HMS01", "HMS02", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00",
					"HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00",
					"HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00",
					"HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00",
					"HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00",
					"HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00",
					"HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00",
					"HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00", "HMS00"
				};
*/

    gchar *sensor_bank = NULL;
    gboolean rv = FALSE;

    /* BAUM only or others that comply with Baum sensor naming convention */
    if (sensor_code)
    {
	sensor_bank = g_strndup(sensor_code, 3);
	/* 
	    DR: could be
	    sensor_bank = sensor_code +3; 
	    !!!!!!!! then atention to g_free
	*/
    }
    if (sensor_bank)
    {
	*sensor_index = atoi(&sensor_code[3]);
	if (g_strcasecmp(sensor_bank, "HMS") == 0)
	{
	    *sensor_type = BST_PS;
	    rv = TRUE;
	}
	else if (g_strcasecmp(sensor_bank, "HOS") == 0)
	{
	    *sensor_type = BST_HOS;
	    rv = TRUE;
	}
	else if (g_strcasecmp(sensor_bank, "LOS") == 0)
	{
	    /* fprintf(stderr, "SR: left optical sensor pressed\n"); */
	}
	else if (g_strcasecmp(sensor_bank, "ROS") == 0)
	{
	    /* fprintf(stderr, "SR: right optical sensor pressed\n"); */
	}
	else
	{
	    /* fprintf(stderr, "SR: this cannot be...\n"); */
	}
	g_free(sensor_bank);
    }
    return rv;
}

SRLong src_text_index;

static void
src_position_sensor_action (gint index)
{
    if (!src_crt_sro)
	return;
	
    src_text_index = index - 4 >= 0 ? index - 4 : 0;

    switch (src_braille_position_sensor)
    {
	case 0:
	    break;
	case 1: 
	    src_cmd_queue_add ("mouse goto current");
	    src_cmd_queue_process ();
	    break;
	case 2:
	    src_cmd_queue_add ("mouse goto current");
	    src_cmd_queue_add ("mouse left click");
	    src_cmd_queue_process ();
	    break;
	case 3:
	    src_cmd_queue_add ("mouse goto current");
	    src_cmd_queue_add ("mouse right click");
	    src_cmd_queue_process ();
	    break;
	case 4:
	    src_cmd_queue_add ("mouse goto current");
/*
	    src_cmd_queue_add ("mouse right click");
	    src_cmd_queue_add ("mouse left click");
*/
	    src_cmd_queue_process ();
	    break;
	case 5:
	    src_cmd_queue_add ("move caret");
	    src_cmd_queue_process ();
	    break;
	case 6:
	    src_cmd_queue_add ("show x coordinate");
	    src_cmd_queue_process ();
	    break;
	case 7:
	    src_cmd_queue_add ("show y coordinate");
	    src_cmd_queue_process ();
	    break;
	case 8:
	    src_cmd_queue_add ("font style");
	    src_cmd_queue_process ();
	    break;
	case 9:
	    src_cmd_queue_add ("font name");
	    src_cmd_queue_process ();
	    break;
	case 10:
	    src_cmd_queue_add ("font size");
	    src_cmd_queue_process ();
	    break;
	case 11:
	    sru_message ("sensor index 10 not implemented");
	    break;
	case 12:
	    src_cmd_queue_add ("text attributes");
	    src_cmd_queue_process ();
	    break;
	
	default:
	    sru_assert_not_reached ();
	    break;
    }	
}

static void
src_optical_sensor_action (gint index)
{
    if (!src_crt_sro)
	return;

    src_text_index = index - 4 >= 0 ? index - 4 : 0;

    switch (src_braille_optical_sensor)
    {
	case 0:
	    break;
	case 1: 
	    src_cmd_queue_add ("mouse goto current");
	    src_cmd_queue_process ();
	    break;
	case 2:
	    src_cmd_queue_add ("mouse goto current");
	    src_cmd_queue_add ("mouse left click");
	    src_cmd_queue_process ();
	    break;
	case 3:
	    src_cmd_queue_add ("mouse goto current");
	    src_cmd_queue_add ("mouse right click");
	    src_cmd_queue_process ();
	    break;
	case 4:
	    src_cmd_queue_add ("mouse goto current");
/*
	    src_cmd_queue_add ("mouse right click");
	    src_cmd_queue_add ("mouse left click");
*/
	    src_cmd_queue_process ();
	    break;
	case 5:
	    src_cmd_queue_add ("move caret");
	    src_cmd_queue_process ();
	    break;
	case 6:
	    src_cmd_queue_add ("show x coordinate");
	    src_cmd_queue_process ();
	    break;
	case 7:
	    src_cmd_queue_add ("show y coordinate");
	    src_cmd_queue_process ();
	    break;
	case 8:
	    src_cmd_queue_add ("font style");
	    src_cmd_queue_process ();
	    break;
	case 9:
	    src_cmd_queue_add ("font name");
	    src_cmd_queue_process ();
	    break;
	case 10:
	    src_cmd_queue_add ("font size");
	    src_cmd_queue_process ();
	    break;
	case 11:
	    sru_message ("sensor index 10 not implemented");
	    break;
	case 12:
	    src_cmd_queue_add ("text attributes");
	    src_cmd_queue_process ();
	    break;
	default:
	    sru_assert_not_reached ();
	    break;
    }	
}


static void
src_process_braille_sensor (gchar *code)
{
    SRBrlSensorTypes type;
    gint index;

    if (!code)
	return;
	
    if (src_translate_sensor_code (code, &type, &index))
    {
	switch (type)
	{
	    case BST_PS:
		src_position_sensor_action (index);
		break;
	    case BST_HOS:
		src_optical_sensor_action (index);
		break;
	    default:
	    	fprintf (stderr, "SR: unsuported sensor code\n");
		break;
	}
    }
}




static void
src_process_sro_event (SREvent *event,
			unsigned long flags)
{
    SRObject *obj;

    if (sre_get_event_data (event, (void**)&obj))
    {
	src_process_sro (obj);
	src_magnifier_start_panning (obj);
    }
/*
    {
	gchar *r;
	sro_get_reason (obj, &r);
	fprintf (stderr, "\n%s", r);
	SR_freeString (r);
    }
*/
}

static void
src_process_window_event (SREvent *event,
			    unsigned long flags)
{
    SRObject *obj;

    if (sre_get_event_data (event, (void**)&obj))
    {
	src_process_window (obj);
    }
}

static void
src_process_tooltip_event (SREvent *event,
			    unsigned long flags)
{
    SRObject *obj;

    if (sre_get_event_data (event, (void**)&obj))
    {
	src_process_tooltip (obj);
    }
}

static void
src_process_mouse_event (SREvent *event,
		 unsigned long flags)
{
    SRPoint *point;

    if (!src_use_magnifier)
	return;
	
    if (sre_get_event_data (event, (void**)&point))
    {
	static gboolean process = FALSE;
	if (flags == 1) /* mouse:rel event */
	{
	    process = TRUE;
	    src_magnifier_stop_panning ();
	}
	else if (process)
	{
	    gchar *magout;
	    magout = g_strdup_printf ("<MAGOUT><ZOOMER ID =\"%s\" tracking=\"mouse\" ROILeft =\"%d\" ROITop =\"%d\" ROIWidth =\"%d\" ROIHeight=\"%d\"></ZOOMER></MAGOUT>",

				"generic_zoomer",
				point->x, point->y, point->x + 1, point->y + 1);
	    if (magout)
	    	src_magnifier_send (magout);
	    g_free (magout);
	    process = FALSE;
	}
    }
}



static void
src_event_sink(SREvent *event_, 
	       unsigned long flags)
{
    SREventType type;
    static GSList *list = NULL;
    static gboolean busy = FALSE;

    if (!event_)
	return;

    list = g_slist_append (list, event_);
    sre_add_reference (event_);

    if (busy)
	return;
    busy = TRUE;
    
    while (list)
    {
	SREvent *event;
	GSList *tmp;
	
	event = (SREvent*) list->data;
	tmp = list;
	list = list->next;
	g_slist_free_1 (tmp);
	
	if (sre_get_type(event, &type))
	{
/*	    static gboolean shutup = TRUE;

	    switch (type)
	    {
		case SR_EVENT_WINDOW:
		    src_speech_shutup_ ();
		    shutup = FALSE;
		    break;
		case SR_EVENT_MOUSE:
		case SR_EVENT_COMMAND_LAYER:
		    break;	    
		default:
		    if (shutup)
			src_speech_shutup_ ();
		    shutup = TRUE;
		    break;		 
	    }
*/	
	    switch (type)
	    {
		case SR_EVENT_CONFIG_CHANGED:
		    src_process_config_changed(event, flags);
	   	    break;
		case SR_EVENT_KEYBOARD_ECHO:
		    src_process_keyboard_echo(event, flags);
	   	    break;
		case SR_EVENT_HOTKEY:
		    src_process_hotkey(event, flags);
	   	    break;
		case SR_EVENT_KEY:
	       	    src_process_key(event, flags);
	   	    break;
		case SR_EVENT_COMMAND_LAYER:
		    src_process_layer_event (event, flags);
		    break;
		case SR_EVENT_COMMAND_LAYER_CHANGED:
	    	    src_process_layer_changed_event (event, flags);
		    break;
		case SR_EVENT_SRO:
		    src_process_sro_event (event, flags);
		    break;
		case SR_EVENT_WINDOW:
		    src_process_window_event (event, flags);
		    break;
		case SR_EVENT_TOOLTIP:
		    src_process_tooltip_event (event, flags);
		    break;
		case SR_EVENT_MOUSE:
		    src_process_mouse_event (event, flags);
		    break;
		default:
		    sru_assert_not_reached ();
		    break;
	    }
	    switch (type)
	    {
		case SR_EVENT_SRO:
		case SR_EVENT_WINDOW:
		    src_ctrl_flat_mode_terminate ();
		    break;
		default:
		    break;
	    }
	}
	sre_release_reference (event);
    }
    busy = FALSE;
}

void
brl_input_event (BrlInEvent *event)
{
    switch (event->event_type)
    {
    	case BIET_KEY:
/*	    fprintf (stderr, "SENSOR: %s\n", event->event_data.key_codes);*/
	    if (event->event_data.key_codes)
		src_ctrl_process_key (event->event_data.key_codes);
	    break;
	case BIET_SENSOR:
	    /* fprintf (stderr, "SENSOR: %s\n", brl_in_event->event_data.sensor_codes); */
	    src_process_braille_sensor (event->event_data.sensor_codes);
	    break;
    	case BIET_SWITCH:
	    /* fprintf (stderr, "SWITCH: %s\n", brl_in_event->event_data.switch_codes); */
	    break;
	default:
	    fprintf (stderr, "UNKNOWN BRAILLE EVENT");
	break;
    }
}

void
brl_xml_input_proc (char* buffer, int len)
{
    brl_in_xml_parse (buffer, len);
}


static void
src_get_defaults ()
{
    /* get braille port*/
    if (MIN_BRAILLE_PORT <= src_braille_port && src_braille_port <= MAX_BRAILLE_PORT)
	SET_BRAILLE_CONFIG_DATA (BRAILLE_PORT_NO, CFGT_INT, &src_braille_port);
    else
    {
	gint default_port = DEFAULT_BRAILLE_PORT_NO;
	GET_BRAILLE_CONFIG_DATA_WITH_DEFAULT (BRAILLE_PORT_NO, CFGT_INT, 
		    &src_braille_port, &default_port);
    }
    
    /* get braille device*/
    if (src_braille_device)
	SET_BRAILLE_CONFIG_DATA (BRAILLE_DEVICE, CFGT_STRING, &src_braille_device);
    else
    {
	gchar *default_device = DEFAULT_BRAILLE_DEVICE;
	GET_BRAILLE_CONFIG_DATA_WITH_DEFAULT (BRAILLE_DEVICE, CFGT_STRING, 
		&src_braille_device, default_device);
    }
    
    /* get braille - position sensor value */
    {
	gint default_sensor = DEFAULT_BRAILLE_POSITION_SENSOR;
	GET_BRAILLE_CONFIG_DATA_WITH_DEFAULT (BRAILLE_POSITION_SENSOR, 
		    CFGT_INT, &src_braille_position_sensor, &default_sensor);
    }
    
    /* get braille - optical sensor value */
    {
	gint default_sensor = DEFAULT_BRAILLE_OPTICAL_SENSOR;
	GET_BRAILLE_CONFIG_DATA_WITH_DEFAULT (BRAILLE_OPTICAL_SENSOR, 
		    CFGT_INT, &src_braille_optical_sensor, &default_sensor);
    }
    

    GET_SRCORE_CONFIG_DATA_WITH_DEFAULT (SRCORE_BRAILLE_ACTIVE, CFGT_BOOL, 
	    &src_use_braille, &src_use_braille);
    GET_SRCORE_CONFIG_DATA_WITH_DEFAULT (SRCORE_BRAILLE_MONITOR_ACTIVE, CFGT_BOOL, 
	    &src_use_braille_monitor, &src_use_braille_monitor);
    GET_SRCORE_CONFIG_DATA_WITH_DEFAULT (SRCORE_SPEECH_ACTIVE, CFGT_BOOL, 
	    &src_use_speech, &src_use_speech);
    GET_SRCORE_CONFIG_DATA_WITH_DEFAULT (SRCORE_MAGNIF_ACTIVE, CFGT_BOOL, 
	    &src_use_magnifier, &src_use_magnifier);

    GET_SRCORE_CONFIG_DATA_WITH_DEFAULT (KEYBOARD_TAKE_MOUSE, CFGT_BOOL, 
	    &src_mouse_take, &src_mouse_take);
    GET_SRCORE_CONFIG_DATA_WITH_DEFAULT (KEYBOARD_SIMULATE_CLICK, CFGT_BOOL, 
	    &src_mouse_click, &src_mouse_click);
	    
}

gchar *src_message;
gboolean 
src_say_message (gchar *message)
{
    if (src_message)
	g_free (src_message);
    src_message = NULL;
    if (message && message[0])
	src_message = g_strdup (message);
    src_cmd_queue_add ("present last message");
    src_cmd_queue_process ();
    
    return TRUE;
}


gboolean 
src_present_last_message ()
{

    if (src_use_speech)
	if (src_message)
	    src_speech_say (src_message, TRUE);
    return TRUE;
}


static void
src_init()
{
    SRLClient client;
    gchar *message = _("Welcome to Gnopernicus");
    gboolean brl_error = FALSE;

    /* initialize SR Utils */
    sru_init();

    /* initialize SRConf */
    srconf_init((SRConfCB)src_event_sink, "/apps/gnopernicus");

    /* layout init */
    src_ctrl_init ();
    
/*    src_speech_count_mode = DEFAULT_SPEECH_REPEAT_TYPE;*/

    src_get_defaults ();

    /* initialize Braille */
    if (src_use_braille)
    {
    	src_use_braille = src_braille_init ();
	if (!src_use_braille)
	    brl_error = TRUE;
    }
	
    if (src_use_braille_monitor)
	src_use_braille_monitor = src_braille_monitor_init ();
 	
    /* initialize Speech */
    if (src_use_speech)
	src_use_speech = src_speech_init ();

    /* initialize Magnifier */
    if (src_use_magnifier)
    {
	src_use_magnifier = src_magnifier_init ();
	if (src_use_magnifier)
	    src_magnifier_create ();
    }
    
    src_presentation_init ();	
    /* initialize SRLow */
    srl_init();

    client.event_proc = (SROnEventProc) src_event_sink;
    src_srl_client = srl_add_client (&client);

    /* initialize keyboard */
    ke_init ((KeyboardEchoCB)&src_event_sink);

    src_layer = NULL;
    src_focused_sro = NULL;
    src_crt_sro = NULL;
    src_crt_tooltip = NULL;
    src_crt_window = NULL;

    src_message = NULL;

    /* and Welcome to Gnopernicus... */
    if (src_use_speech)
	src_say_message (message);
    if (src_use_braille)
	src_braille_show (message);
    if (src_use_braille_monitor)
	src_brlmon_show (message);
    src_mouse_take = DEFAULT_KEYBOARD_TAKE_MOUSE;
    src_mouse_click = DEFAULT_KEYBOARD_SIMULATE_CLICK;

    src_last_key_echo		= NULL;
    src_last_punct_echo 	= NULL;
    src_last_space_echo 	= NULL;
    src_last_modifier_echo 	= NULL;
    src_last_cursor_echo 	= NULL;
    if (brl_error)
	src_say_message (_("braille device can not be initialized"));
    
}

static void
src_terminate()
{
    gboolean exitack = TRUE;
    gchar *message = _("now gnopernicus will gonna exit");

    if (src_focused_sro)
	sro_release_reference (src_focused_sro);
    if (src_crt_sro) 
	sro_release_reference (src_crt_sro);
    if (src_crt_tooltip)
	sro_release_reference (src_crt_tooltip);
    if (src_crt_window)
	sro_release_reference (src_crt_window);

    if (src_use_braille)
	src_braille_show (message);
	
    if (src_use_braille_monitor)
	src_brlmon_show (message);
	
    if (src_use_speech)
	src_say_message (message);
	
    /* terminate SRLow */
    srl_remove_client (src_srl_client);
    srl_terminate ();

    /* terminate Keyboard */
    ke_terminate();

    /* terminate Braille */
    if (src_use_braille)
	src_braille_terminate ();
	
    /*terminate Braille Monitor*/	
    if (src_use_braille_monitor)
	src_brlmon_terminate ();

    /* terminate Magnifier */
    if (src_use_magnifier)
	src_magnifier_terminate (); 

    /* terminate Speech */
    if (src_use_speech)
	src_speech_terminate ();


    /* terminate SRConf */
    SET_SRCORE_CONFIG_DATA(SRCORE_EXIT_ACK_KEY, CFGT_BOOL, &exitack);
    exitack = !exitack;
    SET_SRCORE_CONFIG_DATA(SRCORE_EXIT_ACK_KEY, CFGT_BOOL, &exitack);
    srconf_terminate();

    /* end of SPI usage */
    sru_terminate();	
    src_presentation_terminate ();
    src_ctrl_terminate ();
    if (src_layer)
	g_free (src_layer);
    if (src_message)
	g_free (src_message);
    if (src_last_key_echo)
	g_free (src_last_key_echo);
    if (src_last_punct_echo)
	g_free (src_last_punct_echo);
    if (src_last_space_echo)
	g_free (src_last_space_echo);
    if (src_last_modifier_echo)
	g_free (src_last_modifier_echo);
    if (src_last_cursor_echo)
	g_free (src_last_cursor_echo);
}

gint
main (gint argc, 
      gchar **argv)
{
    gchar *device = NULL;

    struct poptOption poptopt[] = 
    {		
	{"device", 	'd', POPT_ARG_STRING, 	&device,  	'd', "Braille Device", 		"device_name"},
	{"port", 	'p', POPT_ARG_INT, 	&src_braille_port,'p', "Serial port (ttyS)", 	"serial_port_no"},
	{NULL, 		0,0, NULL, 0}
    };
#ifdef ENABLE_NLS
    bindtextdomain (GETTEXT_PACKAGE, GNOPERNICUSLOCALEDIR);
    textdomain (GETTEXT_PACKAGE);
#ifdef HAVE_BIND_TEXTDOMAIN_CODESET
    bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
#endif
#endif    

    fprintf (stderr, "\n**********************");
    fprintf (stderr, "\n* SCREEN READER CORE *");
    fprintf (stderr, "\n**********************\n\n");
    
    src_braille_device = NULL;
    gnome_program_init ("srcore", VERSION,
		    LIBGNOMEUI_MODULE,
		    argc, argv,
		    GNOME_PARAM_POPT_TABLE, poptopt,
		    GNOME_PARAM_HUMAN_READABLE_NAME, _("srcore"),
		    LIBGNOMEUI_PARAM_CRASH_DIALOG, FALSE,
		    NULL);
    sru_log_init ();
	
    if (device)
	{
	    if (src_braille_device) g_free (src_braille_device);
	    src_braille_device = g_strdup (device);
	    g_free (device);
	}
    if (src_braille_port <= MIN_BRAILLE_PORT && src_braille_port >= MAX_BRAILLE_PORT)
	{
	    src_braille_port = DEFAULT_BRAILLE_PORT_NO;
	}

    src_init ();

    sru_entry_loop ();

    src_terminate ();
    
    sru_log_terminate ();
    return EXIT_SUCCESS;
}

gchar*
src_xml_process_string (gchar *str_)
{
    gint len, i, pos;
    gchar *rv, *crt;
    gchar *str;
    
    if (!str_ || !str_[0])
	return NULL;

    str = src_process_string (str_, SRC_SPEECH_COUNT_AUTO, SRC_SPEECH_SPELL_AUTO);

    if (!str)
	return NULL;    
    
    len = strlen (str);
    /* 6 = maximum lengt of xml_ch din translate table */
    crt = rv = (gchar*) g_malloc ((len * 6 + 1) * sizeof (gchar));
    if (!rv)
	return NULL;
	
    for (i = 0, pos = 0; i < len; ++i)
    {
	static struct
	{
	    gchar ch;
	    gchar *xml_ch;
	}translate[] = {
		    {'<',	"&lt;"	},
		    {'>',	"&gt;"	},
		    {'&',	"&amp;"	},
		    {'\'',	"&apos;"},
		    {'\"',	"&quot;"},
		};
	gint j;
	gboolean special = FALSE;
	
	for (j = 0; j < G_N_ELEMENTS (translate); j++)
	{
	    if (str[i] == translate[j].ch)
	    {
	    	crt = g_stpcpy (crt, translate[j].xml_ch);
		special = TRUE;
	    }
	}
	if (!special)
	{
	    *crt = str[i];
	    crt++;
	}
    }
    *crt = '\0';
    g_free (str);
    return rv;
}		

gchar*
src_xml_make_part (gchar *tag,
	    	   gchar *attr,
	           gchar *text)
{
    if (!tag || !text)
	return NULL;
    return g_strconcat ("<", 
			    tag, 
			    attr ? " " : "", attr ? attr : "",
			">",
			text, 
			"</", tag, ">",
			NULL);
}

gchar*
src_xml_format (gchar *tag,
	    	gchar *attr,
	        gchar *text)
{
    gchar *tmp, *rv;;
    
    
    if (!tag || !text)
	return NULL;
    
    tmp = src_xml_process_string (text);
    rv = src_xml_make_part (tag, attr, tmp);
    g_free (tmp);

    return rv;    
}
