// -*- C++ -*-

/* 
 * gnome-util.cc
 *
 * Copyright (C) 1998 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@emccta.com> and
 * Havoc Pennington <hp@emccta.com>.
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-util.h"

static void
spew_cb(GtkObject* obj, gpointer data)
{
  g_debug("Destroying %s %p", gtk_type_name(obj->klass->type), obj);
}

void 
guppi_spew_on_destroy (GtkObject* obj)
{
#ifdef GNOME_ENABLE_DEBUG
  g_return_if_fail(obj != NULL);
  g_return_if_fail(GTK_IS_OBJECT(obj));
  
  gtk_signal_connect(GTK_OBJECT(obj),
                     "destroy",
                     GTK_SIGNAL_FUNC(spew_cb),
                     NULL);
#endif
}

////////////////////

#include "xyactions.h"
#include "baractions.h"
#include "plotstore.h"
#include "markerpaletteedit.h"
#include "gfont.h"

static PlotStore* ps = 0;

PlotStore*
guppi_plot_store()
{
  g_warn_if_fail(ps != 0);
  return ps;
}

static MarkerPaletteEdit* mpe;

MarkerPaletteEdit* 
guppi_marker_palette_edit()
{
  g_warn_if_fail(mpe != 0);
  // OK to do this multiple times
  mpe->set_palette(guppi_marker_palette());
  return mpe;
}

class GnomeFrontend : public Frontend {
public:
  GnomeFrontend();
  virtual VFont* default_font();

private:
  GFont* default_font_;
};

static GnomeFrontend gnomefe;

bool
guppi_gnome_init_library()
{
  g_return_val_if_fail(ps == 0, false); // check for duplicate init

  if (!guppi_init_library())
    return false;

  guppi_register_frontend(&gnomefe);

  ps = new PlotStore;
  mpe = new MarkerPaletteEdit;

  guppi_gnome_init_xy();
  guppi_gnome_init_bar();

  return true;
}

void 
guppi_gnome_shutdown_library()
{
  if (ps != 0)
    {
      delete ps;
      ps = 0;
    }

  if (mpe != 0)
    {
      delete mpe;
      mpe = 0;
    }

  guppi_shutdown_library();
}


void
guppi_set_tooltip(GtkWidget* w, const string& tip)
{
  GtkTooltips* t = gtk_tooltips_new();

  gtk_tooltips_set_tip (t, w, tip.c_str(), NULL);
}

void 
guppi_raise_and_show(GtkWindow* w)
{
  if (w != 0)
    {
      GtkWidget* wid = GTK_WIDGET(w);

      if (!GTK_WIDGET_VISIBLE(wid))
        {
          gtk_widget_show(wid);
        }
      else
        { 
          gdk_window_show(wid->window);
          gdk_window_raise(wid->window);
        }
    }
}

/* This stuff is from gnome-libs, I wrote it there too though. */

/* Callback to display hint in the statusbar when a menu item is 
 * activated. For GnomeAppBar.
 */

static void
put_hint_in_appbar(GtkWidget* menuitem, gpointer data)
{
  gchar* hint =(char*)gtk_object_get_data (GTK_OBJECT(menuitem),
                                           "guppi_appbar_hint");
  GtkWidget* bar = (GtkWidget*)data;

  g_return_if_fail (hint != NULL);
  g_return_if_fail (bar != NULL);
  g_return_if_fail (GNOME_IS_APPBAR(bar));

  gnome_appbar_set_status (GNOME_APPBAR(bar), hint);
}

/* Callback to remove hint when the menu item is deactivated.
 * For GnomeAppBar.
 */
static void
remove_hint_from_appbar(GtkWidget* menuitem, gpointer data)
{
  GtkWidget* bar = (GtkWidget*)data;

  g_return_if_fail (bar != NULL);
  g_return_if_fail (GNOME_IS_APPBAR(bar));

  gnome_appbar_refresh (GNOME_APPBAR(bar));
}

/* Install a hint for a menu item
 */
void
guppi_install_menuitem_hint(GtkWidget* menuitem, GnomeAppBar* bar, 
                            const string& hint)
{
  /* This is mildly fragile; if someone destroys the appbar
     but not the menu, chaos will ensue. */

  gtk_object_set_data_full (GTK_OBJECT(menuitem),
                            "guppi_appbar_hint",
                            g_strdup(hint.c_str()),
                            g_free);
  
  gtk_signal_connect (GTK_OBJECT (menuitem),
                      "select",
                      GTK_SIGNAL_FUNC(put_hint_in_appbar),
                      bar);
  
  gtk_signal_connect (GTK_OBJECT (menuitem),
                      "deselect",
                      GTK_SIGNAL_FUNC(remove_hint_from_appbar),
                      bar);
}


/////////////////// GnomeFrontend

GnomeFrontend::GnomeFrontend()
  : default_font_(0)
{


}

VFont* 
GnomeFrontend::default_font()
{
  if (default_font_ == 0)
    {
      default_font_ = new GFont;
      default_font_->ref(); // default font is never freed
    }
  
  return default_font_;
}
