/* mg-util.c
 * Copyright (C) 2004 Vivien Malerba <malerba@gnome-db.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "mg-defs.h"
#include "mg-util.h"
#include <libgda/libgda.h>
#include "mg-server.h"
#include "mg-query.h"
#include "mg-renderer.h"
#include "mg-entity.h"

/**
 * mg_util_query_execute_modif
 * @query: the #MgQuery to be executed
 * @context: a #MgContext object
 * @ask_confirm_insert:
 * @ask_confirm_update:
 * @ask_confirm_delete:
 * @parent_window:
 * @user_cancelled: a place to store if the user cancelled the query if the choice was given, or %NULL
 * @query_error: a place to store if there was an error, or %NULL
 *
 * Executes @query and displays question and information dialogs if necessary.
 * If a user confirmation was required and the user cancelled the execution, then 
 * the returned value is FALSE.
 *
 * Returns: TRUE if the query was executed.
 */
gboolean
mg_util_query_execute_modif (MgQuery *query, MgContext *context,
			     gboolean ask_confirm_insert,
			     gboolean ask_confirm_update,
			     gboolean ask_confirm_delete,
			     GtkWindow *parent_window,
			     gboolean *user_cancelled,
			     gboolean *query_error)
{
	gchar *sql = NULL;
	MgQueryType qtype;
	gchar *confirm = NULL;
	gboolean do_execute = TRUE;
	gboolean allok = TRUE;
	GError *error = NULL;

	g_return_val_if_fail (query && IS_MG_QUERY (query), FALSE);
	
	sql = mg_renderer_render_as_sql (MG_RENDERER (query), context, 0, &error);
	qtype = mg_query_get_query_type (query);

	switch (qtype) {
	case MG_QUERY_TYPE_INSERT:
		if (ask_confirm_insert)
			confirm = _("Execute the following insertion query ?");
		break;
	case MG_QUERY_TYPE_UPDATE:
		if (ask_confirm_update)
			confirm = _("Execute the following update query ?");
		break;
	case MG_QUERY_TYPE_DELETE:
		if (ask_confirm_delete)
			confirm = _("Execute the following deletion query ?");
		break;
	default:
		g_assert_not_reached ();
	}

	if (user_cancelled)
		*user_cancelled = FALSE;
	if (query_error)
		*query_error = FALSE;

	if (sql) {
		if (confirm) {
			GtkWidget *dlg;
			gint result;
			gchar *msg;
			
			msg = g_strdup_printf (_("<b><big>%s</big></b>\n"
						 "<small>The preferences require a confirmation for the "
						 "following query</small>\n\n%s"), confirm, sql);
			dlg = gtk_message_dialog_new (parent_window, 0,
						      GTK_MESSAGE_QUESTION,
						      GTK_BUTTONS_YES_NO, msg);
			g_free (msg);
			gtk_label_set_use_markup (GTK_LABEL (GTK_MESSAGE_DIALOG (dlg)->label), TRUE);
			result = gtk_dialog_run (GTK_DIALOG (dlg));
			gtk_widget_destroy (dlg);
			do_execute = (result == GTK_RESPONSE_YES);
			if (user_cancelled)
				*user_cancelled = !do_execute;
		}
			
		if (do_execute) {
			MgConf *conf = mg_base_get_conf (MG_BASE (query));
				
#ifdef debug
			g_print ("MODIF SQL: %s\n", sql);
#endif
			mg_server_do_query (mg_conf_get_server (conf), sql, MG_SERVER_QUERY_SQL, &error);
			if (error) {
				GtkWidget *dlg;
				gchar *message;
				
				message = g_strdup (error->message);
				g_error_free (error);
				dlg = gtk_message_dialog_new (parent_window, 0,
							      GTK_MESSAGE_ERROR,
							      GTK_BUTTONS_CLOSE,
							      message);
				g_free (message);
				gtk_dialog_run (GTK_DIALOG (dlg));
				gtk_widget_destroy (dlg);
				allok = FALSE;

				if (query_error)
					*query_error = TRUE;
			}
		}
		else
			allok = FALSE;
		
		g_free (sql);
	}
	else {
		GtkWidget *dlg;
		gchar *message;
		
		if (error) {
			message = g_strdup_printf (_("The following error occured while preparing the query:\n%s"),
						   error->message);
			g_error_free (error);
		}
		else
			message = g_strdup_printf (_("An unknown error occured while preparing the query."));
		
		dlg = gtk_message_dialog_new (parent_window, 0,
					      GTK_MESSAGE_ERROR,
					      GTK_BUTTONS_CLOSE,
					      message);
		g_free (message);
		gtk_dialog_run (GTK_DIALOG (dlg));
		gtk_widget_destroy (dlg);
		allok = FALSE;

		if (query_error)
			*query_error = TRUE;
	}

	return allok;
}

