/* default-display.c
 *
 * Copyright (C) 1999 - 2002 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <string.h>
#include "default-display.h"
#include "data-entry.h"

/* these two functions allocate some memory for the returned string,
   and free the string given as argument */
static struct tm *sqlstr_to_tmstruct_dbspec (gchar * date);
static struct tm *sqlstr_to_tmstruct_locale (gchar * date);

/* the returned char is allocated and the one in argument freed */
static gchar *server_access_escape_chars (gchar * str);
/* the returned char is allocated and the one in argument freed */
static gchar *server_access_unescape_chars (gchar * str);







static void
emit_contents_modified (GObject * obj, GtkWidget * dd)
{
#ifdef debug_signal
	g_print (">> 'CONTENTS_MODIFIED' from emit_contents_modified de=%p\n", dd);
#endif
	g_signal_emit_by_name (G_OBJECT (dd), "contents_modified");
#ifdef debug_signal
	g_print ("<< 'CONTENTS_MODIFIED' from emit_contents_modified\n");
#endif
}

static void 
contents_changed_cb (GObject *obj, GtkWidget *dd)
{
	g_object_set_data (G_OBJECT (dd), "is_set", GINT_TO_POINTER (TRUE));
	emit_contents_modified (obj, dd);
}



/*****************************************************************************/
/*                                                                           */
/* Default functions => Read only                                            */
/*                                                                           */
/*****************************************************************************/
GtkWidget *
widget_from_value_default (const GdaValue * value)
{
	DataEntry *dd;
	GtkWidget *wid, *frame;

	dd = DATA_ENTRY (data_entry_new ());
	data_entry_set_orig_value (dd, value);

	frame = gtk_frame_new (NULL);
	data_entry_pack_default (dd, frame);

	wid = gtk_label_new (_("This data cannot be displayed or modified\n"
			       "NOTE: it is possible to write a plugin for it\n"
			       "and display it normally"));
	gtk_container_add (GTK_CONTAINER (frame), wid);

	gtk_widget_show_all (frame);
	dd->children = g_slist_append (NULL, wid);

	if (value && !gda_value_is_null (value))
		g_object_set_data (G_OBJECT (dd), "is_set", GINT_TO_POINTER (TRUE));
	else
		g_object_set_data (G_OBJECT (dd), "is_set", GINT_TO_POINTER (FALSE));


	return GTK_WIDGET (dd);
}

GdaValue *
value_from_widget_default (GtkWidget * wid)
{
	GdaValue *value;
	gboolean set;

	g_return_val_if_fail (wid && IS_DATA_ENTRY (wid), NULL);

	set = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (wid), "is_set"));

	if (! set)
		value = gda_value_new_null ();
	else {
		switch (DATA_ENTRY (wid)->orig_type) {
		case GDA_VALUE_TYPE_BINARY:
		case GDA_VALUE_TYPE_GEOMETRIC_POINT:
		case GDA_VALUE_TYPE_LIST:
		case GDA_VALUE_TYPE_TIMESTAMP:
		case GDA_VALUE_TYPE_TYPE:
			value = gda_value_copy (DATA_ENTRY (wid)->orig_value);
			break;
		default:
			value = gda_value_new_null ();
		}
	}

	return value;
}

void
widget_update_default (GtkWidget * wid, const GdaValue * value, gboolean value_as_default)
{
	g_return_if_fail (wid && IS_DATA_ENTRY (wid));

	if (value_as_default)
		data_entry_set_orig_value (DATA_ENTRY (wid), value);
	
	if (value && !gda_value_is_null (value))
		g_object_set_data (G_OBJECT (wid), "is_set", GINT_TO_POINTER (TRUE));
	else
		g_object_set_data (G_OBJECT (wid), "is_set", GINT_TO_POINTER (FALSE));

	emit_contents_modified (NULL, wid);
}

gchar *
sql_from_value_default (const GdaValue * value)
{
	gchar *str, *str2, *retval;

	if (!value || gda_value_is_null (value))
		return NULL;

	str = server_resultset_stringify (value);
	if (str) {
		str2 = server_access_escape_chars (str);
		retval = g_strdup_printf ("'%s'", str2);
		g_free (str2);
	}
	else
		retval = NULL;

	return retval;
}

gchar *str_from_value_default (const GdaValue * value)
{
	if (value || gda_value_is_null (value)) 
		return NULL;
	else
		return g_strdup (_("Binary data"));
}

/*****************************************************************************/
/*                                                                           */
/* String functions                                                          */
/*                                                                           */
/*****************************************************************************/
GdaValue *
value_from_widget_string (GtkWidget * wid)
{
	GdaValue *value;
	gboolean set;

	g_return_val_if_fail (wid && IS_DATA_ENTRY (wid), NULL);

	set = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (wid), "is_set"));

	if (!set)
		value = gda_value_new_null ();
	else
		value = gda_value_new_string (gtk_entry_get_text (GTK_ENTRY (DATA_ENTRY (wid)->children->data)));

	return value;
}

GtkWidget *
widget_from_value_string (const GdaValue * value)
{
	DataEntry *dd;
	GtkWidget *wid;

	dd = DATA_ENTRY (data_entry_new ());
	wid = gtk_entry_new ();
	data_entry_pack_default (dd, wid);
	gtk_widget_show (wid);
	dd->children = g_slist_append (NULL, wid);

	if (value)
		widget_update_string (GTK_WIDGET (dd), value, TRUE);

	g_signal_connect (G_OBJECT (wid), "changed",
			  G_CALLBACK (contents_changed_cb), dd);

	return GTK_WIDGET (dd);
}

void
widget_update_string (GtkWidget * dd, const GdaValue * value, gboolean value_as_default)
{
	gchar *str;

	g_return_if_fail (dd && IS_DATA_ENTRY (dd));

	g_signal_handlers_block_by_func (G_OBJECT (DATA_ENTRY (dd)->children->data), 
					 G_CALLBACK (contents_changed_cb), dd);

	if (value_as_default)
		data_entry_set_orig_value (DATA_ENTRY (dd), value);

	if (value && !gda_value_is_null (value)) {
		str = str_from_value_string (value);
		if (str) {
			gtk_entry_set_text (GTK_ENTRY (DATA_ENTRY (dd)->children->data), str);
			g_free (str);
		}
		else 
			gtk_entry_set_text (GTK_ENTRY (DATA_ENTRY (dd)->children->data), "");
		g_object_set_data (G_OBJECT (dd), "is_set", GINT_TO_POINTER (TRUE));
	}
	else {
		gtk_entry_set_text (GTK_ENTRY (DATA_ENTRY (dd)->children->data), "");
		g_object_set_data (G_OBJECT (dd), "is_set", GINT_TO_POINTER (FALSE));
	}
	
	emit_contents_modified (G_OBJECT (DATA_ENTRY (dd)->children->data), dd);
	g_signal_handlers_unblock_by_func (G_OBJECT (DATA_ENTRY (dd)->children->data), 
					   G_CALLBACK (contents_changed_cb), dd);
}

gchar *
sql_from_value_string (const GdaValue * value)
{
	gchar *str, *str2, *retval;

	if ((value->type == GDA_VALUE_TYPE_STRING) &&
	    !gda_value_get_string (value))
		return NULL;

	str = server_resultset_stringify (value);
	if (str) {
		str2 = server_access_escape_chars (str);
		retval = g_strdup_printf ("'%s'", str2);
		g_free (str2);
	}
	else
		retval = NULL;

	return retval;
}


gchar *
str_from_value_string (const GdaValue * value)
{
	if (value) {
		if ((value->type == GDA_VALUE_TYPE_STRING) &&
		    !gda_value_get_string (value))
			return NULL;


		return server_resultset_stringify (value);
	}
	else
		return g_strdup ("");
}

/*****************************************************************************/
/*                                                                           */
/* Functions for the numbers                                                 */
/*                                                                           */
/*****************************************************************************/

GtkWidget *
widget_from_value_number (const GdaValue * value)
{
	DataEntry *dd;
	GtkWidget *wid;

	dd = DATA_ENTRY (data_entry_new ());

	wid = gtk_entry_new ();
	data_entry_pack_default (dd, wid);
	gtk_widget_show (wid);
	dd->children = g_slist_append (NULL, wid);
	if (value)
		widget_update_number (GTK_WIDGET (dd), value, TRUE);
	g_signal_connect (G_OBJECT (wid), "changed",
			  G_CALLBACK (contents_changed_cb), dd);

	return GTK_WIDGET (dd);
}

GdaValue *
value_from_widget_number (GtkWidget * wid)
{
	GdaValue *value;
	gboolean set;

	g_return_val_if_fail (wid && IS_DATA_ENTRY (wid), NULL);

	set = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (wid), "is_set"));

	if (!set)
		value = gda_value_new_null ();
	else {
		GdaNumeric gnum;
		gint64 bigint;
		gint vint;
		gfloat vsingle;
		gdouble vdouble;
		gshort vsmall;
		gchar vtiny;

		switch (DATA_ENTRY (wid)->orig_type) {
		case GDA_VALUE_TYPE_NUMERIC:
			gnum.number = g_strdup (gtk_entry_get_text (GTK_ENTRY (DATA_ENTRY (wid)->children->data)));
			gnum.precision = 10; /* FIXME */
			gnum.width = 10;     /* FIXME */
			value = gda_value_new_numeric (&gnum);
			g_free (gnum.number);
			break;
		case GDA_VALUE_TYPE_BIGINT:
			bigint = atoi (gtk_entry_get_text (GTK_ENTRY (DATA_ENTRY (wid)->children->data)));
			value = gda_value_new_bigint (bigint);
			break;
		case GDA_VALUE_TYPE_DOUBLE:
			vdouble = atof (gtk_entry_get_text (GTK_ENTRY (DATA_ENTRY (wid)->children->data)));
			value = gda_value_new_double (vdouble);
			break;
		case GDA_VALUE_TYPE_INTEGER:
		case GDA_VALUE_TYPE_UNKNOWN:
		default:
			vint = atoi (gtk_entry_get_text (GTK_ENTRY (DATA_ENTRY (wid)->children->data)));
			value = gda_value_new_integer (vint);
			break;
		case GDA_VALUE_TYPE_SINGLE:
			vsingle = atof (gtk_entry_get_text (GTK_ENTRY (DATA_ENTRY (wid)->children->data)));
			value = gda_value_new_single (vsingle);
			break;
		case GDA_VALUE_TYPE_SMALLINT:
			vsmall = atoi (gtk_entry_get_text (GTK_ENTRY (DATA_ENTRY (wid)->children->data)));
			value = gda_value_new_smallint (vsmall);
			break;
		case GDA_VALUE_TYPE_TINYINT:
			vtiny = *gtk_entry_get_text (GTK_ENTRY (DATA_ENTRY (wid)->children->data));
			value = gda_value_new_tinyint (vtiny);
			break;
		}
	}		

	return value;	
}

void
widget_update_number (GtkWidget * dd, const GdaValue * value, gboolean value_as_default)
{
	gchar *str;

	g_signal_handlers_block_by_func (G_OBJECT (DATA_ENTRY (dd)->children->data), 
					 G_CALLBACK (contents_changed_cb), dd);

	if (value_as_default)
		data_entry_set_orig_value (DATA_ENTRY (dd), value);

	if (value && !gda_value_is_null (value)) {
		str = str_from_value_number (value);
		if (str) {
			gtk_entry_set_text (GTK_ENTRY (DATA_ENTRY (dd)->children->data), str);
			g_free (str);
		}
		else
			gtk_entry_set_text (GTK_ENTRY (DATA_ENTRY (dd)->children->data), "");
		g_object_set_data (G_OBJECT (dd), "is_set", GINT_TO_POINTER (TRUE));
	}
	else {
		gtk_entry_set_text (GTK_ENTRY (DATA_ENTRY (dd)->children->data), "");
		g_object_set_data (G_OBJECT (dd), "is_set", GINT_TO_POINTER (FALSE));
	}

	emit_contents_modified (G_OBJECT (DATA_ENTRY (dd)->children->data), dd);
	g_signal_handlers_unblock_by_func (G_OBJECT (DATA_ENTRY (dd)->children->data), 
					   G_CALLBACK (contents_changed_cb), dd);
}

gchar *
sql_from_value_number (const GdaValue * value)
{
	gchar *retval, *ptr;

	retval = server_resultset_stringify (value);
	/* FIXME: take better care of the locales here */
	ptr = retval;
	while (*ptr != '\0') {
		if (*ptr == ',')
			*ptr = '.';
		ptr++;
	}

	return retval;
}

gchar *
str_from_value_number (const GdaValue * value)
{
	return server_resultset_stringify (value);
}


/*****************************************************************************/
/*                                                                           */
/* Functions for the booleans                                                */
/*                                                                           */
/*****************************************************************************/
static void
bool_emit_contents_modified (GObject * obj, GtkWidget * dd)
{
	gboolean set;

	set = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (dd), "is_set"));

	if (set) {
		if (GTK_TOGGLE_BUTTON (obj)->active)
			gtk_label_set_text (GTK_LABEL (GTK_BIN (obj)->child), _("Yes"));
		else
			gtk_label_set_text (GTK_LABEL (GTK_BIN (obj)->child), _("No"));
	}
	else
		gtk_label_set_text (GTK_LABEL (GTK_BIN (obj)->child), _("Unset"));

#ifdef debug_signal
	g_print (">> 'CONTENTS_MODIFIED' from bool_emit_contents_modified de=%p\n", dd);
#endif
	g_signal_emit_by_name (G_OBJECT (dd), "contents_modified");
#ifdef debug_signal
	g_print (">> 'CONTENTS_MODIFIED' from bool_emit_contents_modified\n");
#endif
}

static void
bool_contents_changed_cb (GObject * obj, GtkWidget * dd)
{
	g_object_set_data (G_OBJECT (dd), "is_set", GINT_TO_POINTER (TRUE));
	bool_emit_contents_modified (obj, dd);
}


GtkWidget *
widget_from_value_bool (const GdaValue * value)
{
	DataEntry *dd;
	GtkWidget *wid, *hb;

	dd = DATA_ENTRY (data_entry_new ());

	hb = gtk_hbox_new (FALSE, 0);
	data_entry_pack_default (dd, hb);
	gtk_widget_show (hb);
	wid = gtk_toggle_button_new_with_label (_("No"));
	gtk_box_pack_start (GTK_BOX (hb), wid, FALSE, FALSE, 0);
	gtk_widget_show (wid);
	dd->children = g_slist_append (NULL, wid);

	widget_update_bool (GTK_WIDGET (dd), value, TRUE);

	g_signal_connect (G_OBJECT (wid), "toggled",
			  G_CALLBACK (bool_contents_changed_cb), dd);

	return GTK_WIDGET (dd);
}

GdaValue *
value_from_widget_bool (GtkWidget * wid)
{
	GdaValue *value;
	gboolean set;

	g_return_val_if_fail (wid && IS_DATA_ENTRY (wid), NULL);
	set = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (wid), "is_set"));

	if (!set)
		value = gda_value_new_null ();
	else {
		gboolean bool;

		bool = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (DATA_ENTRY (wid)->children->data));
		value = gda_value_new_boolean (bool);
	}

	return value;	
}

void
widget_update_bool (GtkWidget * dd, const GdaValue * value, gboolean value_as_default)
{
	g_signal_handlers_block_by_func (G_OBJECT (DATA_ENTRY (dd)->children->data), 
					 G_CALLBACK (bool_contents_changed_cb), dd);

	if (value_as_default)
		data_entry_set_orig_value (DATA_ENTRY (dd), value);

	if (value && !gda_value_is_null (value)) {
		if (gda_value_get_boolean (value))
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (DATA_ENTRY (dd)->children->data),
						      TRUE);
		else
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (DATA_ENTRY (dd)->children->data),
						      FALSE);
		g_object_set_data (G_OBJECT (dd), "is_set", GINT_TO_POINTER (TRUE));
	}
	else {

		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (DATA_ENTRY (dd)->children->data), FALSE);
		g_object_set_data (G_OBJECT (dd), "is_set", GINT_TO_POINTER (FALSE));
	}

	bool_emit_contents_modified (G_OBJECT (DATA_ENTRY (dd)->children->data), dd);
	g_signal_handlers_unblock_by_func (G_OBJECT (DATA_ENTRY (dd)->children->data), 
					   G_CALLBACK (bool_contents_changed_cb), dd);
}

gchar *
sql_from_value_bool (const GdaValue * value)
{
	gchar *retval = NULL;

	if (gda_value_is_null (value))
		retval = NULL;
	else {
		if (gda_value_get_boolean (value))
			retval = g_strdup ("'t'");
		else
			retval = g_strdup ("'f'");
	}

	return retval;
}

gchar *
str_from_value_bool (const GdaValue * value)
{
	return server_resultset_stringify (value);
}

/*****************************************************************************/
/*                                                                           */
/* Functions for the dates                                                   */
/*                                                                           */
/*****************************************************************************/
GtkWidget *
widget_from_value_date (const GdaValue * value)
{
	DataEntry *dd;
	GtkWidget *wid, *hb;
	time_t now;

	dd = DATA_ENTRY (data_entry_new ());
	hb = gtk_hbox_new (FALSE, 0);
	data_entry_pack_default (dd, hb);
	gtk_widget_show (hb);

	now = time (NULL);
	wid = gnome_date_edit_new (now, FALSE, FALSE); /* FIXME: does not seem to work properly! */

	gtk_box_pack_start (GTK_BOX (hb), wid, FALSE, FALSE, 0);
	gtk_widget_show (wid);
	dd->children = g_slist_append (NULL, wid);

	widget_update_date (GTK_WIDGET (dd), value, TRUE);

	g_signal_connect (G_OBJECT (wid), "date_changed",
			  G_CALLBACK (contents_changed_cb), dd);

	return GTK_WIDGET (dd);
}

GdaValue *
value_from_widget_date (GtkWidget * wid)
{
	GdaValue *value;
	gboolean set;

	g_return_val_if_fail (wid && IS_DATA_ENTRY (wid), NULL);

	set = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (wid), "is_set"));

	if (!set)
		value = gda_value_new_null ();
	else {
		time_t time;
		struct tm *stm;
		GdaDate date;

		time = gnome_date_edit_get_time (GNOME_DATE_EDIT (DATA_ENTRY (wid)->children->data));
		stm = gmtime (&time);
		date.year = stm->tm_year + 1900;
		date.month = stm->tm_mon + 1;
		date.day = stm->tm_mday;

		value = gda_value_new_date (&date);
	}

	return value;
}


void
widget_update_date (GtkWidget * dd, const GdaValue * value, gboolean value_as_default)
{
	time_t now;
	struct tm *stm;

	if (value_as_default)
		data_entry_set_orig_value (DATA_ENTRY (dd), value);

	if (value && !gda_value_is_null (value)) {
		const GdaDate *date;

		date = gda_value_get_date (value);
		stm = g_new0 (struct tm, 1);
		stm->tm_mday = date->day;
		stm->tm_mon = date->month - 1;
		stm->tm_year = date->year - 1900;
		now = mktime (stm);
		g_free (stm);
		g_object_set_data (G_OBJECT (dd), "is_set", GINT_TO_POINTER (TRUE));
	}
	else {
		now = time (NULL);
		g_object_set_data (G_OBJECT (dd), "is_set", GINT_TO_POINTER (FALSE));
	}

	g_signal_handlers_block_by_func (G_OBJECT (DATA_ENTRY (dd)->children->data), 
					 G_CALLBACK (contents_changed_cb), dd);
	gnome_date_edit_set_time (GNOME_DATE_EDIT (DATA_ENTRY (dd)->children->data), now);

	emit_contents_modified (G_OBJECT (DATA_ENTRY (dd)->children->data), dd);

	g_signal_handlers_unblock_by_func (G_OBJECT (DATA_ENTRY (dd)->children->data), 
					   G_CALLBACK (contents_changed_cb), dd);
}

gchar *
sql_from_value_date (const GdaValue * value)
{
	gchar *str;
	const GdaDate *date;

	date = gda_value_get_date (value);
	str = g_strdup_printf ("'%02d-%02d-%04d'",
			       date->month,
			       date->day,
			       date->year);
	return str;
}

gchar *
str_from_value_date (const GdaValue * value)
{
	return server_resultset_stringify (value);
}


/*****************************************************************************/
/*                                                                           */
/* Functions for the times                                                   */
/*                                                                           */
/*****************************************************************************/

GtkWidget *
widget_from_value_time (const GdaValue * value)
{
	DataEntry *dd;
	GtkWidget *wid, *hb;
	time_t now;

	dd = DATA_ENTRY (data_entry_new ());

	hb = gtk_hbox_new (FALSE, 0);
	data_entry_pack_default (dd, hb);
	gtk_widget_show (hb);

	now = time (NULL);
	wid = gnome_date_edit_new (now, TRUE, FALSE);
	gnome_date_edit_set_flags (GNOME_DATE_EDIT (wid), 
				   GNOME_DATE_EDIT_SHOW_TIME | GNOME_DATE_EDIT_24_HR);

	gtk_box_pack_start (GTK_BOX (hb), wid, FALSE, FALSE, 0);
	gtk_widget_show (wid);
	dd->children = g_slist_append (NULL, wid);

	widget_update_time (GTK_WIDGET (dd), value, TRUE);

	g_signal_connect (G_OBJECT (wid), "time_changed",
			  G_CALLBACK (contents_changed_cb), dd);

	return GTK_WIDGET (dd);
}

GdaValue *
value_from_widget_time (GtkWidget * wid)
{
	GdaValue *value;
	gboolean set;

	g_return_val_if_fail (wid && IS_DATA_ENTRY (wid), NULL);

	set = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (wid), "is_set"));

	if (!set)
		value = gda_value_new_null ();
	else {
		time_t time;
		struct tm *stm;
		GdaTime tim;

		time = gnome_date_edit_get_time (GNOME_DATE_EDIT (DATA_ENTRY (wid)->children->data));
		stm = gmtime (&time);
		tim.hour = stm->tm_hour;
		tim.minute = stm->tm_min;
		tim.second = stm->tm_sec;
		
		value = gda_value_new_time (&tim);
	}

	return value;
}

void
widget_update_time (GtkWidget * dd, const GdaValue * value, gboolean value_as_default)
{
	time_t now;
	struct tm *stm;

	if (value_as_default)
		data_entry_set_orig_value (DATA_ENTRY (dd), value);

	g_signal_handlers_block_by_func (G_OBJECT (DATA_ENTRY (dd)->children->data), 
					 G_CALLBACK (contents_changed_cb), dd);

	if (value && !gda_value_is_null (value)) {
		const GdaTime *tim;

		tim = gda_value_get_time (value);
		stm = g_new0 (struct tm, 1);
		stm->tm_hour = tim->hour;
		stm->tm_min = tim->minute;
		stm->tm_sec = tim->second;
		now = mktime (stm);
		g_free (stm);
		g_object_set_data (G_OBJECT (dd), "is_set", GINT_TO_POINTER (TRUE));
	}
	else {
		now = time (NULL);
		g_object_set_data (G_OBJECT (dd), "is_set", GINT_TO_POINTER (FALSE));
	}

	gnome_date_edit_set_time (GNOME_DATE_EDIT (DATA_ENTRY (dd)->children->data), now);
	emit_contents_modified (G_OBJECT (DATA_ENTRY (dd)->children->data), dd);

	g_signal_handlers_unblock_by_func (G_OBJECT (DATA_ENTRY (dd)->children->data), 
					   G_CALLBACK (contents_changed_cb), dd);
}

gchar *
sql_from_value_time (const GdaValue * value)
{
	gchar *str;
	const GdaTime *tim;

	tim = gda_value_get_time (value);
	str = g_strdup_printf ("'%02d-%02d-%02d'",
			       tim->hour,
			       tim->minute,
			       tim->second);
	return str;
}

gchar *
str_from_value_time (const GdaValue * value)
{
	return server_resultset_stringify (value);
}

/***************************************************************************/
/*                                                                         */
/* Some usefull functions                                                  */
/*                                                                         */
/***************************************************************************/

/* dbspec: 
   Converting MM-DD-YYYY to struct tm */
static struct tm *
sqlstr_to_tmstruct_dbspec (gchar * date)
{
	int day, month, year;
	char *ptr;
	char mdate[11];
	static struct tm stm;

	if ((date == NULL) || (*date == '\0')) {
		return NULL;
	}
	strncpy (mdate, date, 10);
	mdate[10] = '\0';
	ptr = (char *) strtok (mdate, "-");
	month = atoi (ptr);
	if (!(ptr = (char *) strtok (NULL, "-")))
		return NULL;	/* Error */
	day = atoi (ptr);
	if (!(ptr = (char *) strtok (NULL, "-")))
		return NULL;	/* Error */
	year = atoi (ptr);

	stm.tm_mday = day;
	stm.tm_mon = month - 1;
	stm.tm_year = year - 1900;

	return &stm;
}

/* locale:
   it is needed to introduce the way dates are represented in the current 
   locale. 
   At the moment: MM*DD*YYYY is the only supported, * being either -, / or . */
static struct tm *
sqlstr_to_tmstruct_locale (gchar * date)
{
	int day, month, year;
	char *ptr;
	char mdate[11];
	static struct tm stm;

	if ((date == NULL) || (*date == '\0')) {
		return NULL;
	}
	strncpy (mdate, date, 10);
	mdate[10] = '\0';
	ptr = (char *) strtok (mdate, "-/.");
	month = atoi (ptr);
	if (!(ptr = (char *) strtok (NULL, "-/.")))
		return NULL;	/* Error */
	day = atoi (ptr);
	if (!(ptr = (char *) strtok (NULL, "-/.")))
		return NULL;	/* Error */
	year = atoi (ptr);

	stm.tm_mday = day;
	stm.tm_mon = month - 1;
	stm.tm_year = year - 1900;

	return &stm;
}

/* the returned char is allocated and the one in argument freed */
static gchar *
server_access_escape_chars (gchar * str)
{
	gchar *ptr = str, *ret, *retptr;
	gint size;

	/* determination of the new string size */
	size = 1;
	while (*ptr != '\0') {
		if (*ptr == '\'') {
			if (ptr == str)
				size += 2;
			else {
				if (*(ptr - 1) == '\\')
					size += 1;
				else
					size += 2;
			}
		}
		else
			size += 1;
		ptr++;
	}

	ptr = str;
	ret = (gchar *) malloc (sizeof (gchar) * size);
	retptr = ret;
	while (*ptr != '\0') {
		if (*ptr == '\'') {
			if (ptr == str) {
				*retptr = '\\';
				retptr++;
			}
			else if (*(ptr - 1) != '\\') {
				*retptr = '\\';
				retptr++;
			}
		}
		*retptr = *ptr;
		retptr++;
		ptr++;
	}
	*retptr = '\0';
	g_free (str);
	return ret;
}

/* the returned char is allocated and the one in argument freed */
static gchar *
server_access_unescape_chars (gchar * str)
{
	gchar *ptr = str, *ret, *retptr;
	gint size;

	/* determination of the new string size */
	size = 1;
	while (*ptr != '\0') {
		if ((*ptr == '\\') && (*(ptr + 1) == '\'')) {
			ptr++;
		}
		else {
			size += 1;
			ptr++;
		}
	}

	ptr = str;
	ret = (gchar *) malloc (sizeof (gchar) * size);
	retptr = ret;
	while (*ptr != '\0') {
		if ((*ptr == '\\') && (*(ptr + 1) == '\''))
			ptr++;
		else {
			*retptr = *ptr;
			retptr++;
			ptr++;
		}
	}
	*retptr = '\0';
	g_free (str);
	return ret;
}


GSList *
data_display_get_initial_list ()
{
	GSList *list = NULL;
	DataDisplayFns *fns;

	/* 
	 * WARNING: The order here is important, because in the 
	 * sqldata::get_display_fns_from_gda(), the returned value is the first one
	 * which matches the gda type of the given ServerDataType.
	 */

	/* FIXME: the other ones here (only builting ones, not plugins) */

	/* 1 - numerical values */
	fns = g_new0 (DataDisplayFns, 1);
	fns->descr = g_strdup (_("Numerical data representation"));
	fns->plugin_name = NULL;
	fns->plugin_file = NULL;
	fns->lib_handle = NULL;
	fns->version = NULL;
	fns->get_unique_key = NULL;
	fns->nb_gda_type = 7;
	fns->valid_gda_types = g_new0 (GdaValueType, 7);
	fns->valid_gda_types[0] = GDA_VALUE_TYPE_BIGINT;
	fns->valid_gda_types[1] = GDA_VALUE_TYPE_DOUBLE;
	fns->valid_gda_types[2] = GDA_VALUE_TYPE_INTEGER;
	fns->valid_gda_types[3] = GDA_VALUE_TYPE_NUMERIC;
	fns->valid_gda_types[4] = GDA_VALUE_TYPE_SINGLE;
	fns->valid_gda_types[5] = GDA_VALUE_TYPE_SMALLINT;
	fns->valid_gda_types[6] = GDA_VALUE_TYPE_TINYINT;
	fns->expand_widget = FALSE;
	fns->sql_from_value = sql_from_value_number;
	fns->str_from_value = str_from_value_number;
	fns->widget_from_value = widget_from_value_number;
	fns->value_from_widget = value_from_widget_number;
	fns->widget_update = widget_update_number;
	fns->sql_from_value = sql_from_value_number;
	list = g_slist_append (list, fns);

	/* 2 - Boolean values */
	fns = g_new0 (DataDisplayFns, 1);
	fns->descr = g_strdup (_("Boolean data representation"));
	fns->plugin_name = NULL;
	fns->plugin_file = NULL;
	fns->lib_handle = NULL;
	fns->version = NULL;
	fns->get_unique_key = NULL;
	fns->nb_gda_type = 1;
	fns->valid_gda_types = g_new0 (GdaValueType, 1);
	fns->valid_gda_types[0] = GDA_VALUE_TYPE_BOOLEAN;
	fns->expand_widget = FALSE;
	fns->sql_from_value = sql_from_value_bool;
	fns->str_from_value = str_from_value_bool;
	fns->widget_from_value = widget_from_value_bool;
	fns->value_from_widget = value_from_widget_bool;
	fns->widget_update = widget_update_bool;
	fns->sql_from_value = sql_from_value_bool;
	list = g_slist_append (list, fns);

	/* 3 - Dates values */
	fns = g_new0 (DataDisplayFns, 1);
	fns->descr = g_strdup (_("Dates representation"));
	fns->plugin_name = NULL;
	fns->plugin_file = NULL;
	fns->lib_handle = NULL;
	fns->version = NULL;
	fns->get_unique_key = NULL;
	fns->nb_gda_type = 1;
	fns->valid_gda_types = g_new0 (GdaValueType, 1);
	fns->valid_gda_types[0] = GDA_VALUE_TYPE_DATE;
	fns->expand_widget = FALSE;
	fns->sql_from_value = sql_from_value_date;
	fns->str_from_value = str_from_value_date;
	fns->widget_from_value = widget_from_value_date;
	fns->value_from_widget = value_from_widget_date;
	fns->widget_update = widget_update_date;
	fns->sql_from_value = sql_from_value_date;
	list = g_slist_append (list, fns);

	/* 4 - Times values */
	fns = g_new0 (DataDisplayFns, 1);
	fns->descr = g_strdup (_("Dates representation"));
	fns->plugin_name = NULL;
	fns->plugin_file = NULL;
	fns->lib_handle = NULL;
	fns->version = NULL;
	fns->get_unique_key = NULL;
	fns->nb_gda_type = 1;
	fns->valid_gda_types = g_new0 (GdaValueType, 1);
	fns->valid_gda_types[0] = GDA_VALUE_TYPE_TIME;
	fns->expand_widget = FALSE;
	fns->sql_from_value = sql_from_value_time;
	fns->str_from_value = str_from_value_time;
	fns->widget_from_value = widget_from_value_time;
	fns->value_from_widget = value_from_widget_time;
	fns->widget_update = widget_update_time;
	fns->sql_from_value = sql_from_value_time;
	list = g_slist_append (list, fns);

	/* String values */
	fns = g_new0 (DataDisplayFns, 1);
	fns->descr = g_strdup (_("Strings representation"));
	fns->plugin_name = NULL;
	fns->plugin_file = NULL;
	fns->lib_handle = NULL;
	fns->version = NULL;
	fns->get_unique_key = NULL;
	fns->nb_gda_type = 1;
	fns->valid_gda_types = g_new0 (GdaValueType, 1);
	fns->valid_gda_types[0] = GDA_VALUE_TYPE_STRING;
	fns->expand_widget = FALSE;
	fns->sql_from_value = sql_from_value_string;
	fns->str_from_value = str_from_value_string;
	fns->widget_from_value = widget_from_value_string;
	fns->value_from_widget = value_from_widget_string;
	fns->widget_update = widget_update_string;
	fns->sql_from_value = sql_from_value_string;
	list = g_slist_append (list, fns);

	/* LAST - default values */
	fns = g_new0 (DataDisplayFns, 1);
	fns->descr = g_strdup (_("Default data representation (read only)"));
	fns->plugin_name = NULL;
	fns->plugin_file = NULL;
	fns->lib_handle = NULL;
	fns->version = NULL;
	fns->get_unique_key = NULL;
	fns->nb_gda_type = 0;	/* means ALL GDA types */
	fns->valid_gda_types = NULL;	/* means ALL GDA types */
	fns->expand_widget = FALSE;
	fns->sql_from_value = sql_from_value_default;
	fns->str_from_value = str_from_value_default;
	fns->widget_from_value = widget_from_value_default;
	fns->value_from_widget = value_from_widget_default;
	fns->widget_update = widget_update_default;
	fns->sql_from_value = sql_from_value_default;
	list = g_slist_append (list, fns);

	return list;
}

void
data_display_free_display_fns (DataDisplayFns * fns)
{
	if (fns->valid_gda_types)
		g_free (fns->valid_gda_types);
	g_free (fns->descr);
	g_free (fns);
}
