/* mainpagetable.c
 *
 * Copyright (C) 1999 - 2002 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <libgnomedb/gnome-db-stock.h>
#include <libgnomedb/gnome-db-util.h>
#include "mainpagetable.h"
#include "tableedit.h"
#include "query.h"
#include "query-env.h"
#include "query-exec.h"

static void main_page_table_class_init (MainPageTableClass * class);
static void main_page_table_init (MainPageTable * wid);
static void main_page_table_initialize (MainPageTable * wid);
static void main_page_table_finalize (GObject *object);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

typedef struct
{
	ConfManager *conf;
	DbTable     *table;
	Query       *query;
	QueryEnv    *env;
	GtkWidget   *props_dlg;
	GtkWidget   *exec_dlg;
}
Row_Data;


/*
 * static functions 
 */
static void selection_made (GtkWidget * wid,
			    gint row,
			    gint column,
			    GdkEventButton * event, gpointer data);
static gint press_handler_event (GtkWidget * widget,
				 GdkEventButton * event, MainPageTable * mpt);
static void remove_table_cb (GObject   * obj, MainPageTable * wid);
static void edit_table_cb (GObject   * obj, MainPageTable * wid);
static void view_records_button_cb (GObject   * obj, MainPageTable * mpt);
static void database_added_cb (ConfManager *conf, Database *db, MainPageTable * mpt);
static void database_removed_cb (ConfManager *conf, Database *db, MainPageTable * mpt);
static void main_page_table_conn_close_cb (GObject   * obj,
					   MainPageTable * mpt);
static void main_page_table_add_cb (GObject   * obj, DbTable * table,
				    MainPageTable * wid);
static void main_page_table_drop_cb (GObject   * obj, DbTable * table,
				     MainPageTable * wid);
static void main_page_db_updated_cb (Database * db, MainPageTable * mpt);


guint
main_page_table_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (MainPageTableClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) main_page_table_class_init,
			NULL,
			NULL,
			sizeof (MainPageTable),
			0,
			(GInstanceInitFunc) main_page_table_init
		};		

		type = g_type_register_static (GTK_TYPE_VBOX, "MainPageTable", &info, 0);
	}
	return type;
}

static void
main_page_table_class_init (MainPageTableClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	parent_class = g_type_class_peek_parent (class);

	object_class->finalize = main_page_table_finalize;
}

static void
main_page_table_init (MainPageTable * wid)
{
	GtkWidget *sw, *bb;
	gint i;

	/* Toolbar */
	bb = gnome_db_new_toolbar_widget (GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_ICONS);
	gtk_box_pack_start (GTK_BOX (wid), bb, FALSE, TRUE, GNOME_PAD / 2);
	gtk_widget_show (bb);

	wid->remove_table = gtk_toolbar_insert_stock (GTK_TOOLBAR (bb), GTK_STOCK_DELETE,
						      _("Drop selected table"), NULL,
						      GTK_SIGNAL_FUNC (remove_table_cb), wid, -1);
	wid->edit_table = gtk_toolbar_insert_stock (GTK_TOOLBAR (bb), GTK_STOCK_PROPERTIES,
						    _("View selected table properties"), NULL,
						    GTK_SIGNAL_FUNC (edit_table_cb), wid, -1);
	wid->view_table = gtk_toolbar_insert_stock (GTK_TOOLBAR (bb), GNOME_DB_STOCK_TABLES,
						    _("View records for the selected table"), NULL,
						    GTK_SIGNAL_FUNC (view_records_button_cb), wid, -1);
	wid->new_table = gtk_toolbar_insert_stock (GTK_TOOLBAR (bb), GTK_STOCK_NEW,
						   _("Create a new table"), NULL,
						   NULL, NULL, -1); /* FIXME */

	/* Scrolled Window for CList */
	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start (GTK_BOX (wid), sw, TRUE, TRUE, GNOME_PAD);
	gtk_widget_show (sw);

	/* CList */
	wid->clist = gtk_clist_new (4);
	gtk_clist_set_column_title (GTK_CLIST (wid->clist), 0, _("Name"));
	gtk_clist_set_column_title (GTK_CLIST (wid->clist), 1, _("Type"));
	gtk_clist_set_column_title (GTK_CLIST (wid->clist), 2, _("Owner"));
	gtk_clist_set_column_title (GTK_CLIST (wid->clist), 3, _("Comments"));
	gtk_clist_set_selection_mode (GTK_CLIST (wid->clist),
				      GTK_SELECTION_SINGLE);
	for (i = 0; i < 4; i++)
		gtk_clist_set_column_auto_resize (GTK_CLIST (wid->clist), i,
						  TRUE);
	gtk_clist_column_titles_show (GTK_CLIST (wid->clist));
	gtk_clist_column_titles_passive (GTK_CLIST (wid->clist));
	gtk_container_add (GTK_CONTAINER (sw), wid->clist);
	gtk_widget_show (wid->clist);
	g_signal_connect (G_OBJECT (wid->clist), "select_row",
			  G_CALLBACK (selection_made), wid);
	g_signal_connect (G_OBJECT (wid->clist), "button_press_event",
			  G_CALLBACK (press_handler_event), wid);

	wid->sel_row = -1;
}

GtkWidget *
main_page_table_new (ConfManager * conf)
{
	GObject   *obj;
	MainPageTable *wid;

	obj = g_object_new (MAIN_PAGE_TABLE_TYPE, NULL);
	wid = MAIN_PAGE_TABLE (obj);
	wid->conf = conf;

	main_page_table_initialize (wid);

	g_signal_connect (G_OBJECT (conf), "database_added",
			  G_CALLBACK (database_added_cb), wid);

	g_signal_connect (G_OBJECT (conf), "database_removed",
			  G_CALLBACK (database_removed_cb), wid);

	g_signal_connect (G_OBJECT (conf->srv), "conn_closed",
			  G_CALLBACK (main_page_table_conn_close_cb), obj);

	return GTK_WIDGET (obj);
}

static void main_page_table_finalize (GObject *object)
{
	/* FIXME: think about all the Row_Data with Query objects... */
	
	parent_class->finalize (object);
}

static void database_added_cb (ConfManager *conf, Database *db, MainPageTable * mpt)
{
	g_signal_connect (G_OBJECT (conf->db),
			    "table_created",
			    G_CALLBACK (main_page_table_add_cb),
			    mpt);
	g_signal_connect (G_OBJECT (conf->db),
			    "table_dropped",
			    G_CALLBACK (main_page_table_drop_cb),
			    mpt);
	g_signal_connect (G_OBJECT (conf->db), "updated",
			    G_CALLBACK (main_page_db_updated_cb),
			    mpt);
}

static void database_removed_cb (ConfManager *conf, Database *db, MainPageTable * mpt)
{
	/* nothing to do about it */
}


static void
main_page_table_initialize (MainPageTable * wid)
{
	gtk_widget_set_sensitive (wid->remove_table, FALSE);
	gtk_widget_set_sensitive (wid->edit_table, FALSE);
	gtk_widget_set_sensitive (wid->view_table, FALSE);
	gtk_widget_set_sensitive (wid->new_table, FALSE);
	conf_manager_register_sensitive_on_connect (wid->conf,
						    GTK_WIDGET (wid->new_table));
}

void
main_page_table_set_data_dlg (MainPageTable * mpt, guint row, GtkWidget * dlg)
{
	Row_Data *rdata = NULL;

	rdata = (Row_Data *) gtk_clist_get_row_data (GTK_CLIST (mpt->clist),
						     row);
	if (rdata)
		rdata->props_dlg = dlg;
}

GtkWidget *
main_page_table_get_data_dlg (MainPageTable * mpt, guint row)
{
	Row_Data *rdata = NULL;

	rdata = (Row_Data *) gtk_clist_get_row_data (GTK_CLIST (mpt->clist),
						     row);
	if (rdata)
		return rdata->props_dlg;
	else
		return NULL;
}

gint
main_page_table_get_row (MainPageTable * mpt, DbTable * table)
{
	gint i = 0;
	gboolean found = FALSE;
	Row_Data *rdata;

	while ((i < GTK_CLIST (mpt->clist)->rows) && !found) {
		rdata = gtk_clist_get_row_data (GTK_CLIST (mpt->clist), i);
		if (rdata && (rdata->table == table))
			found = TRUE;
		else
			i++;
	}
	if (found)
		return i;
	else
		return -1;
}



static void
selection_made (GtkWidget * wid,
		gint row, gint column, GdkEventButton * event, gpointer data)
{
	MainPageTable *mpt = MAIN_PAGE_TABLE (data);

	mpt->sel_row = row;

	/* set sensitiveness of buttons */
	gtk_widget_set_sensitive (mpt->remove_table, TRUE);
	gtk_widget_set_sensitive (mpt->edit_table, TRUE);
	gtk_widget_set_sensitive (mpt->view_table, TRUE);
}


static gint
press_handler_event (GtkWidget * widget,
		     GdkEventButton * event, MainPageTable * mpt)
{
	gint row, col;

	if (GTK_IS_CLIST (widget)) {
		/* setting the right selection */
		gtk_clist_get_selection_info (GTK_CLIST (widget),
					      event->x, event->y, &row, &col);
		if ((mpt->sel_row != row) && (event->button != 1))
			gtk_clist_select_row (GTK_CLIST (widget), row, col);


		/* depending on what the clicks were */
		if (event->type == GDK_2BUTTON_PRESS) {
			Row_Data *rd;

			rd = (Row_Data *)
				gtk_clist_get_row_data (GTK_CLIST (widget),
							row);
			view_records_button_cb (NULL, mpt);
			/*edit_table_cb(NULL, mpt); */
			return TRUE;
		}

		if (event->button == 3) {
			GtkWidget *menu, *wid;

			menu = gtk_menu_new ();
			wid = gtk_menu_item_new_with_label (_("Properties"));
			g_signal_connect (G_OBJECT (wid), "activate",
					    G_CALLBACK (edit_table_cb),
					    mpt);
			gtk_menu_append (GTK_MENU (menu), wid);
			gtk_widget_show (wid);
			wid = gtk_menu_item_new_with_label (_
							    ("View records"));
			g_signal_connect (G_OBJECT (wid), "activate",
					  G_CALLBACK (view_records_button_cb), mpt);
			gtk_menu_append (GTK_MENU (menu), wid);
			gtk_widget_show (wid);
			gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL,
					NULL, event->button, event->time);
			/* Tell calling code that we have handled this event */
			return TRUE;
		}
	}

	return FALSE;
}

static void remove_table_dlg_response_cb (GtkDialog *dlg, gint reply, MainPageTable * wid);
static void remove_table_dlg_destroy_cb (GtkDialog *dlg, MainPageTable * wid);
static void remove_table_dlg_conn_close_cb (GObject *srv, GtkDialog *dlg);
static void
remove_table_cb (GObject   * obj, MainPageTable * wid)
{
	Row_Data *rdata = NULL;
	gchar *str;
	GtkWidget *dlg;

	if (wid->sel_row >= 0)
		rdata = (Row_Data *)
			gtk_clist_get_row_data (GTK_CLIST (wid->clist),
						wid->sel_row);
	if (rdata) {
		if (rdata->table->is_view)
			str = g_strdup_printf (_("Do you really wish to drop the view '%s'?"),
					       rdata->table->name);
		else
			str = g_strdup_printf (_("Do you really wish to drop the table '%s'?"),
					       rdata->table->name);

		dlg = gtk_message_dialog_new (GTK_WINDOW (rdata->conf->app),
					      0,
					      GTK_MESSAGE_QUESTION,
					      GTK_BUTTONS_YES_NO,
					      str);

		g_signal_connect (G_OBJECT (wid->conf->srv), "conn_to_close",
				  G_CALLBACK (remove_table_dlg_conn_close_cb), dlg);

		g_signal_connect (G_OBJECT (dlg), "response",
				  G_CALLBACK (remove_table_dlg_response_cb), wid);
		g_signal_connect (G_OBJECT (dlg), "destroy",
				  G_CALLBACK (remove_table_dlg_destroy_cb), wid);

		g_free (str);
	}
}

static void
remove_table_dlg_response_cb (GtkDialog *dlg, gint reply, MainPageTable * wid)
{
	Row_Data *rdata = NULL;
	gchar *query;

	if (wid->sel_row >= 0)
		rdata = (Row_Data *)
			gtk_clist_get_row_data (GTK_CLIST (wid->clist),
						wid->sel_row);

	switch (reply) {
	case GTK_RESPONSE_YES:
		if (rdata) {
			ServerResultset *rs;
			if (rdata->table->is_view)
				query = g_strdup_printf ("drop view %s\n",
							 rdata->table->name);
			else
				query = g_strdup_printf ("drop table %s\n",
							 rdata->table->name);
			rs = server_access_do_query (rdata->conf->srv, query, SERVER_ACCESS_QUERY_SQL);
			if (rs)
				g_object_unref (G_OBJECT (rs));
			g_free (query);
			database_refresh (rdata->conf->db, rdata->conf->srv);
		}
		break;
	case GTK_RESPONSE_NO:
		break;
	}
	gtk_widget_destroy (GTK_WIDGET (dlg));
}

static void
remove_table_dlg_destroy_cb (GtkDialog *dlg, MainPageTable * wid)
{
	/* disconnect callback */
	g_signal_handlers_disconnect_by_func (G_OBJECT (wid->conf->srv),
					      G_CALLBACK (remove_table_dlg_conn_close_cb), dlg);
}

static void
remove_table_dlg_conn_close_cb (GObject *srv, GtkDialog *dlg)
{
	gtk_widget_destroy (GTK_WIDGET (dlg));
}


static void
edit_table_cb (GObject   * obj, MainPageTable * wid)
{
	Row_Data *rdata = NULL;
	GtkWidget *dlg;

	if (wid->sel_row >= 0)
		rdata = (Row_Data *)
			gtk_clist_get_row_data (GTK_CLIST (wid->clist),
						wid->sel_row);
	if (rdata) {
		dlg = table_edit_dialog_new (wid->conf, rdata->table);
		if (dlg)
			gtk_widget_show (dlg);
	}
}

static void
view_records_button_cb (GObject   * obj, MainPageTable * mpt)
{
	Row_Data *rdata = NULL;
	gchar *str;

	if (mpt->sel_row >= 0)
		rdata = (Row_Data *)
			gtk_clist_get_row_data (GTK_CLIST (mpt->clist),
						mpt->sel_row);
	if (rdata) {
		QueryExec *qx;

		if (!rdata->exec_dlg) {
			GtkWidget *dlg;
			if (!rdata->query) {	/* build the query here */
				QueryField *qf;
				
				if (rdata->table->is_view)
					str = g_strdup_printf (_("contents of view %s"), rdata->table->name);
				else
					str = g_strdup_printf (_("contents of table %s"), rdata->table->name);
				rdata->query = QUERY (query_new (str, NULL, mpt->conf));
				g_free (str);
				
				/* QueryField */
				qf = query_field_allfields_dbtable_new (rdata->query, rdata->table->name,
									NULL, G_OBJECT (rdata->table));
				qf = QUERY_FIELD (query_field_new (rdata->query, rdata->table->name, 
								   QUERY_FIELD_ALLFIELDS));
				
				query_field_set_is_printed (qf, TRUE);
				query_add_field (QUERY (rdata->query), qf);
				
				rdata->env = QUERY_ENV (query_env_new (rdata->query));
				query_env_set_modif_table (rdata->env, rdata->table);
				rdata->env->actions =
					QUERY_ACTION_INSERT | QUERY_ACTION_REFRESH |
					 QUERY_ACTION_DELETE |
					QUERY_ACTION_COMMIT | QUERY_ACTION_FIRST |
					QUERY_ACTION_LAST | QUERY_ACTION_PREV |
					QUERY_ACTION_NEXT;
			}
#ifdef debug
			query_dump_contents (rdata->query);
#endif
			qx = QUERY_EXEC (query_exec_new (rdata->env));
			dlg = query_exec_get_edit_form (qx, 0);
			if (dlg) {
				rdata->exec_dlg = dlg;
				gtk_widget_show (dlg);
				g_object_add_weak_pointer (G_OBJECT (dlg), (gpointer) &(rdata->exec_dlg));
			}
			else
				query_exec_free (QUERY_EXEC (qx));
		}
		else
			gdk_window_raise (rdata->props_dlg->window);
	}
}


static void drop_associated_query (Row_Data * rdata);
static void
main_page_table_conn_close_cb (GObject   * obj, MainPageTable * mpt)
{
	Row_Data *rdata;
	gint i, nb = GTK_CLIST (mpt->clist)->rows;

	gtk_clist_freeze (GTK_CLIST (mpt->clist));
	for (i = 0; i < nb; i++) {
		rdata = (Row_Data *)
			gtk_clist_get_row_data (GTK_CLIST (mpt->clist), 0);
		drop_associated_query (rdata);
		g_free (rdata);
		gtk_clist_remove (GTK_CLIST (mpt->clist), 0);
	}
	gtk_clist_thaw (GTK_CLIST (mpt->clist));
	gtk_widget_set_sensitive (mpt->remove_table, FALSE);
	gtk_widget_set_sensitive (mpt->edit_table, FALSE);
	gtk_widget_set_sensitive (mpt->view_table, FALSE);
	gtk_widget_set_sensitive (mpt->new_table, FALSE);
	mpt->sel_row = -1;
}


/* this CB is intended to be connected to the "table_created" signal of the 
   Database objects it represents.  So:
   - obj is a Database object
*/
static void comments_table_changed_cb (DbTable * table, Row_Data * rdata);
static void
main_page_table_add_cb (GObject   * obj, DbTable * table,
			MainPageTable * wid)
{
	gchar *col[4];
	gchar *str;
	gint i = 0, j;
	gboolean stop = FALSE;
	Row_Data *rdata;

	col[0] = g_strdup (table->name);
	if (table->is_view)
		col[1] = g_strdup (_("View"));
	else
		col[1] = g_strdup (_("Table"));
	col[2] = g_strdup (table->owner);
	if (table->comments)
		col[3] = g_strdup (table->comments);
	else
		col[3] = g_strdup ("");
	/* where to put it ? */
	while ((i < GTK_CLIST (wid->clist)->rows) && !stop) {
		gtk_clist_get_text (GTK_CLIST (wid->clist), i, 0, &str);
		if (strcmp (str, table->name) > 0) {
			stop = TRUE;
		}
		else
			i++;
	}

	gtk_clist_insert (GTK_CLIST (wid->clist), i, col);
	for (j = 0; j < 4; j++)
		g_free (col[j]);

	rdata = (Row_Data *) g_malloc (sizeof (Row_Data));
	rdata->conf = wid->conf;
	rdata->table = table;
	rdata->query = NULL;
	rdata->env = NULL;
	rdata->props_dlg = NULL;
	rdata->exec_dlg = NULL;
	gtk_clist_set_row_data (GTK_CLIST (wid->clist), i, rdata);
	g_signal_connect (G_OBJECT (table), "comments_changed",
			    G_CALLBACK (comments_table_changed_cb),
			    rdata);
}

static void
comments_table_changed_cb (DbTable * table, Row_Data * rdata)
{
	gint row;
	GtkWidget *clist;

	clist = MAIN_PAGE_TABLE (rdata->conf->tables_page)->clist;
	row = gtk_clist_find_row_from_data (GTK_CLIST (clist), rdata);
	if (row >= 0)
		gtk_clist_set_text (GTK_CLIST (clist), row, 3,
				    table->comments);
}



/* this CB is intended to be connected to the "table_dropped" signal of the 
   Database objects it represents.  So:
   - obj is a Database object
*/

static void
main_page_table_drop_cb (GObject   * obj, DbTable * table,
			 MainPageTable * wid)
{
	gchar *str;
	gint i = 0;
	gboolean found = FALSE;
	Row_Data *rdata;

	while ((i < GTK_CLIST (wid->clist)->rows) && !found) {
		gtk_clist_get_text (GTK_CLIST (wid->clist), i, 0, &str);
		if (strcmp (str, table->name) == 0) {
			found = TRUE;
			rdata = (Row_Data *)
				gtk_clist_get_row_data (GTK_CLIST (wid->clist), i);
			/* the associated query is not valid anymore */
			drop_associated_query (rdata);
			g_free (rdata);
			gtk_clist_remove (GTK_CLIST (wid->clist), i);
			if (wid->sel_row == i) {
				wid->sel_row = -1;
				/* un sensitive buttons */
				gtk_widget_set_sensitive (wid->remove_table,
							  FALSE);
				gtk_widget_set_sensitive (wid->edit_table,
							  FALSE);
				gtk_widget_set_sensitive (wid->view_table,
							  FALSE);
			}
		}
		i++;
	}
}

static void
drop_associated_query (Row_Data * rdata)
{
	if (rdata->env) 
		/* don't free the env, that will be done by the query which holds
		   a ref on it. */
		rdata->env = NULL;

	if (rdata->query) {
		g_object_unref (G_OBJECT (rdata->query));
		rdata->query = NULL;
	}
}


static void
main_page_db_updated_cb (Database * db, MainPageTable * mpt)
{
	gint row;
	Row_Data *rd;
	for (row = 0; row < GTK_CLIST (mpt->clist)->rows; row++) {
		rd = (Row_Data *)
			gtk_clist_get_row_data (GTK_CLIST (mpt->clist), row);
		if (rd->table->is_view)
			gtk_clist_set_text (GTK_CLIST (mpt->clist), row, 1,
					    _("View"));
		else
			gtk_clist_set_text (GTK_CLIST (mpt->clist), row, 1,
					    _("Table"));
		if (rd->table->owner)
			gtk_clist_set_text (GTK_CLIST (mpt->clist), row, 2,
					    rd->table->owner);
		else
			gtk_clist_set_text (GTK_CLIST (mpt->clist), row, 2,
					    "");

		/* the associated query is not valid anymore */
		drop_associated_query (rd);
	}
}
