/* settings.c
 *
 * Copyright (C) 2002 Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <gconf/gconf.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtknotebook.h>
#include <gtk/gtkstock.h>
#include <gtk/gtkpaned.h>
#include <gtk/gtktable.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-uidefs.h>
#include <libgnomeui/gnome-window-icon.h>
#include <libgnomedb/gnome-db-config.h>
#include <libgnomedb/gnome-db-gray-bar.h>
#include <libgnomedb/gnome-db-stock.h>
#include <libgnomedb/gnome-db-util.h>
#include "gnome-db-shortcut.h"
#include "settings.h"
#include "settings-utils.h"

#define ENTRY_REMEMBER_WINDOW_SIZE "/apps/mergeant/Interface/MainWindow/RememberSize"
#define ENTRY_DELETE_CONFIRMATION_DIALOG "/apps/mergeant/Database/DeleteConfirmationDialog"
#define ENTRY_INSERT_CONFIRMATION_DIALOG "/apps/mergeant/Database/InsertConfirmationDialog"
#define ENTRY_SELECT_CONFIRMATION_DIALOG "/apps/mergeant/Database/SelectConfirmationDialog"
#define ENTRY_UPDATE_CONFIRMATION_DIALOG "/apps/mergeant/Database/UpdateConfirmationDialog"

static GtkWidget *settings_dialog = NULL;
static GtkWidget *settings_notebook = NULL;
static GtkWidget *settings_bar = NULL;

/*
 * Interface settings tab
 */

static GtkWidget *
interface_settings_new (void)
{
	GtkWidget *table, *frame, *box, *button;

	table = gnome_db_new_table_widget (4, 4, FALSE);

	/*
	 * Main window options
	 */
	frame = gnome_db_new_frame_widget (_("Confirmation"));
	gtk_table_attach (GTK_TABLE (table), frame, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 2, 2);
	box = gnome_db_new_vbox_widget (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (frame), box);

	button = settings_utils_new_check_button (_("Remember window size"),
						  ENTRY_REMEMBER_WINDOW_SIZE);
	gtk_box_pack_start (GTK_BOX (box), button, FALSE, FALSE, 0);

	return table;
}

/*
 * Database settings tab
 */

static GtkWidget *
database_settings_new (void)
{
	GtkWidget *table, *box, *frame, *cb;

	table = gnome_db_new_table_widget (4, 1, FALSE);

	/*
	 * Database confirmation options
	 */
	frame = gnome_db_new_frame_widget (_("Confirmation"));
	gtk_table_attach (GTK_TABLE (table), frame, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 2, 2);
	box = gnome_db_new_vbox_widget (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (frame), box);

	cb = settings_utils_new_check_button (_("Ask for confirmation on SELECT statements"),
					      ENTRY_SELECT_CONFIRMATION_DIALOG);
	gtk_box_pack_start (GTK_BOX (box), cb, FALSE, FALSE, 0);

	cb = settings_utils_new_check_button (_("Ask for confirmation on INSERT statements"),
					      ENTRY_INSERT_CONFIRMATION_DIALOG);
	gtk_box_pack_start (GTK_BOX (box), cb, FALSE, FALSE, 0);

	cb = settings_utils_new_check_button (_("Ask for confirmation on UPDATE statements"),
					      ENTRY_UPDATE_CONFIRMATION_DIALOG);
	gtk_box_pack_start (GTK_BOX (box), cb, FALSE, FALSE, 0);

	cb = settings_utils_new_check_button (_("Ask for confirmation on DELETE statements"),
					      ENTRY_DELETE_CONFIRMATION_DIALOG);
	gtk_box_pack_start (GTK_BOX (box), cb, FALSE, FALSE, 0);

	return table;
}

/*
 * Settings dialog implementation
 */

static void
dialog_response_cb (GtkDialog *dialog, gint response_id, gpointer user_data)
{
	gtk_widget_hide (settings_dialog);
}

static void
database_settings_cb (GnomeDbShortcut *shortcut,
		      const gchar *label,
		      const gchar *tooltip,
		      gpointer user_data)
{
	gtk_notebook_set_page (GTK_NOTEBOOK (settings_notebook), 1);
	gnome_db_gray_bar_set_text (GNOME_DB_GRAY_BAR (settings_bar), _("Database"));
}

static void
interface_settings_cb (GnomeDbShortcut *shortcut,
		       const gchar *label,
		       const gchar *tooltip,
		       gpointer user_data)
{
	gtk_notebook_set_page (GTK_NOTEBOOK (settings_notebook), 0);
	gnome_db_gray_bar_set_text (GNOME_DB_GRAY_BAR (settings_bar), _("Interface"));
}

void
settings_dialog_display (GtkWidget *parent, ConfManager *conf)
{
	GtkWidget *paned;
	GtkWidget *shortcut;
	GtkWidget *box;

	if (!settings_dialog) {
		/* create the dialog */
		settings_dialog = gtk_dialog_new_with_buttons (
			_("Mergeant settings"), parent, 0,
			GTK_STOCK_CLOSE, GTK_RESPONSE_CANCEL,
			NULL);
		gnome_window_icon_set_from_file (GTK_WINDOW (settings_dialog),
						 PIXMAPDIR "/mergeant.png");
		g_signal_connect (G_OBJECT (settings_dialog), "response",
				  G_CALLBACK (dialog_response_cb), NULL);

		paned = gnome_db_new_hpaned_widget ();
		gtk_box_pack_start (GTK_BOX (GTK_DIALOG (settings_dialog)->vbox),
				    paned, TRUE, TRUE, GNOME_PAD / 2);

		/* shortcut bar */
		shortcut = gnome_db_shortcut_new ();
		gnome_db_shortcut_append (GNOME_DB_SHORTCUT (shortcut),
					  _("Interface"), _("Interface"),
					  GTK_STOCK_PREFERENCES,
					  (GnomeDbShortcutSelectFunc) interface_settings_cb,
					  NULL);
		gnome_db_shortcut_append (GNOME_DB_SHORTCUT (shortcut),
					  _("Database"), _("Database"),
					  GNOME_DB_STOCK_DATABASE,
					  (GnomeDbShortcutSelectFunc) database_settings_cb,
					  NULL);
		gtk_paned_add1 (GTK_PANED (paned), shortcut);
		gtk_widget_show (shortcut);

		/* the title bar and the notebook */
		box = gnome_db_new_vbox_widget (FALSE, 0);
		settings_bar = gnome_db_gray_bar_new (_("Interface"));
		gtk_widget_show (settings_bar);
		gtk_box_pack_start (GTK_BOX (box), settings_bar, FALSE, FALSE, GNOME_PAD / 2);

		settings_notebook = gnome_db_new_notebook_widget ();
		gtk_notebook_set_show_tabs (GTK_NOTEBOOK (settings_notebook), FALSE);
		gtk_notebook_set_show_border (GTK_NOTEBOOK (settings_notebook), FALSE);
		gtk_box_pack_start (GTK_BOX (box), settings_notebook, TRUE, TRUE, GNOME_PAD / 2);

		gtk_notebook_append_page (GTK_NOTEBOOK (settings_notebook),
					  interface_settings_new (),
					  gtk_label_new (_("Interface")));
		gtk_notebook_append_page (GTK_NOTEBOOK (settings_notebook),
					  database_settings_new (),
					  gtk_label_new (_("Database")));
		
		gtk_paned_add2 (GTK_PANED (paned), box);
	}

	gtk_widget_show (settings_dialog);
}

/*
 * Accessor functions for configuration settings
 */

gboolean
settings_get_delete_confirmation_dialog (void)
{
	return gnome_db_config_get_boolean (ENTRY_DELETE_CONFIRMATION_DIALOG);
}

void
settings_set_delete_confirmation_dialog (gboolean confirm)
{
	gnome_db_config_set_boolean (ENTRY_DELETE_CONFIRMATION_DIALOG, confirm);
}

gboolean
settings_get_insert_confirmation_dialog (void)
{
	return gnome_db_config_get_boolean (ENTRY_INSERT_CONFIRMATION_DIALOG);
}

void
settings_set_insert_confirmation_dialog (gboolean confirm)
{
	gnome_db_config_set_boolean (ENTRY_INSERT_CONFIRMATION_DIALOG, confirm);
}

gboolean
settings_get_select_confirmation_dialog (void)
{
	return gnome_db_config_get_boolean (ENTRY_SELECT_CONFIRMATION_DIALOG);
}

void
settings_set_select_confirmation_dialog (gboolean confirm)
{
	gnome_db_config_set_boolean (ENTRY_SELECT_CONFIRMATION_DIALOG, confirm);
}

gboolean
settings_get_update_confirmation_dialog (void)
{
	return gnome_db_config_get_boolean (ENTRY_UPDATE_CONFIRMATION_DIALOG);
}

void
settings_set_update_confirmation_dialog (gboolean confirm)
{
	gnome_db_config_set_boolean (ENTRY_UPDATE_CONFIRMATION_DIALOG, confirm);
}
