/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team <pan@rebelbase.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <config.h>
#include <errno.h>

#include <gnome.h>

#include <pan/debug.h>
#include <pan/file-grouplist.h>
#include <pan/fnmatch.h>
#include <pan/pan-glib-extensions.h>
#include <pan/serverlist.h>

#include <pan/articlelist.h>
#include <pan/dialogs/dialogs.h>
#include <pan/globals.h>
#include <pan/grouplist.h>
#include <pan/group-ui.h>
#include <pan/gui.h>
#include <pan/prefs.h>
#include <pan/queue.h>
#include <pan/task-group-count.h>
#include <pan/task-bodies.h>
#include <pan/task-headers.h>
#include <pan/task-grouplist.h>
#include <pan/util.h>

#include "xpm/folder.xpm"
#include "xpm/envelope.xpm"

static void grouplist_update_ui_thread (void* data);
static void group_menu_popup (GdkEventButton * event);
static void grouplist_filter_changed_cb (void);
static void grouplist_refresh_filter (void);

extern GtkWidget   * ttips;
static GnomePixmap * folder_pixmap           = NULL;
static GnomePixmap * group_pixmap            = NULL;

static int           group_mode              = GROUP_ALL;
static GtkWidget   * group_clist_menu        = NULL;
static GtkWidget   * folder_clist_menu       = NULL;
static Server      * my_server               = NULL;
static GtkWidget   * grouplist_mode_menu     = NULL;
static GtkWidget   * groupname_filter_entry  = NULL;
static gchar       * groupname_filter_string = NULL;

/**
 * This lock must be gotten _before_ the gui lock, and released
 * _after_ the gui lock is released.  Otherwise deadlock may occur.
 */
static GStaticMutex  grouplist_ui_mutex      = G_STATIC_MUTEX_INIT;

PanCallback * grouplist_server_set_callback = NULL;
PanCallback * grouplist_group_selection_changed = NULL;

static gchar*
grouplist_get_group_readable_name (const Group * group)
{
	gchar * retval;
	GString * str = g_string_new (group_get_readable_name(group));

	if (group_is_moderated (group))
		g_string_sprintfa (str, _(" (Moderated)"));
	if (group_is_read_only (group))
		g_string_sprintfa (str, _(" (Read-Only)"));

	retval = str->str;
	g_string_free (str, FALSE);
	return retval;
}

static Server *
grouplist_get_visible_server (void)
{
	return group_mode==GROUP_FOLDERS
		? serverlist_get_named_server(INTERNAL_SERVER_NAME)
		: my_server;
}

Server *
grouplist_get_server (void)
{
	return my_server;
}

static void
group_dialog_clicked_cb (GnomeDialog * dialog, int num, Server * server)
{
	if (num==0) /* yes, get list */
		queue_add (TASK(task_grouplist_new (server,
		                                    TRUE,
		                                    GROUPLIST_ALL)));
}

static void
grouplist_set_server (Server * server)
{
	Server * old_server = my_server;
	debug_enter ("grouplist_set_server");


	g_return_if_fail (server != NULL);
	if (my_server == server)
		return;

	/* set the active list */
	my_server = server;
	pan_callback_call (grouplist_server_set_callback, my_server, NULL);

	if (grouplist_get_visible_server() == old_server) {
		pan_lock ();
		gtk_clist_clear (GTK_CLIST(Pan.group_clist));
		pan_unlock ();
	}

	if (file_grouplist_exists (my_server))
	{
		grouplist_update_ui (my_server);
	}
	else
	{
		GtkWidget * w;
		gchar * s = g_strdup_printf (
			_("We don't have a group list for server \"%s\".\n"
			  "Shall we get one?"), server->name);

		pan_lock ();
		w = gnome_message_box_new (s,
			GNOME_MESSAGE_BOX_QUESTION,
			GNOME_STOCK_BUTTON_YES,
			GNOME_STOCK_BUTTON_NO, NULL);
		gtk_signal_connect (GTK_OBJECT(w), "clicked",
		                    GTK_SIGNAL_FUNC(group_dialog_clicked_cb),
				    server);
		gnome_dialog_set_parent (GNOME_DIALOG(w),
		                         GTK_WINDOW(Pan.window));
		gtk_widget_show_all (w);
		pan_unlock ();

		g_free (s);
	}

	debug_exit ("grouplist_set_server");
}

/*****
******
*****/

/*
 * This thing is frustrating... basically we're trying to reconcile
 * the desired behavior with GtkCList's quirks.  The desired behavior:
 *
 * (1) If the user selects a range, don't activate the group
 *     because it's more than likely that the user is selecting
 *     the range to perform a menu popup operation.
 *
 * (2) If a group is single-clicked and single-click prefs are on,
 *     or if a group was double-clicked, activate it.
 *
 * The problem with (1) is that the selection callback fires once per
 * selected item, so there's no way to tell if the first callback is
 * part of a range being selected or not.  So the workaround here is
 * to push the work off to an idle function that gets invoked after
 * all the selection callbacks are done, then query the grouplist for
 * selection info and fire the grouplist selection pan_callback.
 * (2) isn't as hard to do -- the * variable `button_click_count' is
 * used to keep information from the * keypress callback so that we can
 * use it in the selection callback.
 */
static gint button_click_count = -1;
static gint mb = -1;

static gint
grouplist_selection_changed_cb (gpointer call_object,
                                gpointer call_arg,
                                gpointer user_data)
{
	GPtrArray * groups = (GPtrArray*)call_arg;
	Group * single_group = NULL;
	debug_enter ("grouplist_selection_changed_cb");

	/* load the specified group */
	if (groups!=NULL && groups->len==1)
		single_group = GROUP(g_ptr_array_index(groups,0));

	if ((single_group!=NULL)
		&& ((button_click_count==2) /* double click */
			|| (button_click_count==1 /*&& Pan.viewmode==GUI_PANED*/
					&& (  (mb==2 && grouplist_mb2_action==CLICK_ACTION_LOAD)
					    ||(mb==1 && grouplist_mb1_action==CLICK_ACTION_LOAD)))))
	{
		gui_page_set (ARTICLELIST_PAGE, Pan.article_ctree);
		articlelist_set_group_thread (single_group);
	}

	/* reset the click count for next time... */
	button_click_count = -1;
	mb = -1;

	debug_exit ("grouplist_selection_changed_cb");
	return 0;
}

static void
grouplist_button_press (GtkWidget *widget, GdkEventButton * event)
{
	debug_enter ("grouplist_button_press");
	switch (event->button)
	{
		case 1:
		case 2:
			mb = event->button;
			if (event->type == GDK_2BUTTON_PRESS)
				button_click_count = 2;
			else
				button_click_count = 1;
			break;
		case 3:
#if 0
			if (grouplist_get_selected_group() == NULL) {
				gint row=0, col=0;
				if (gtk_clist_get_selection_info(GTK_CLIST(widget), event->x, event->y, &row, &col))
					gtk_clist_select_row (GTK_CLIST(widget), row, 0);
			}
#endif
			group_menu_popup (event);
			break;
	}
	debug_exit ("grouplist_button_press");
}

static void
grouplist_key_press (GtkWidget     * widget,
                     GdkEventKey   * event,
                     gpointer        user_data)
{
	debug_enter ("grouplist_key_press");
	switch (event->keyval)
	{
		case GDK_Up: case GDK_Down:
		case GDK_Left: case GDK_Right:
		case GDK_Page_Up: case GDK_Page_Down:
			button_click_count = 1;
			break;
		default:
			button_click_count = -1;
	} 
	debug_exit ("grouplist_key_press");
}

static gboolean select_callback_pending = FALSE;

static gint
select_row_cb_idle (gpointer data)
{
	GPtrArray * groups;
	debug_enter ("select_row_cb_idle");

       	groups = grouplist_get_selected_groups ();
	pan_callback_call (grouplist_group_selection_changed,
	                   Pan.group_clist,
			   groups);
	select_callback_pending = FALSE;
	g_ptr_array_free (groups, TRUE);

	debug_exit ("select_row_cb_idle");
	return FALSE;
}
static void
select_row_cb (GtkWidget  * widget,
               int          row,
               int          col,
               GdkEvent   * event)
{
	debug_enter ("select_row_cb");

	if (!select_callback_pending)
	{
		select_callback_pending = TRUE;
		gtk_idle_add (select_row_cb_idle, NULL);
	}

	debug_exit ("select_row_cb");
}

/*****
******
*****/

void
grouplist_set_selected_groups (const GPtrArray * a)
{
	GtkCList * clist;
	debug_enter ("grouplist_set_selected_groups");

	if (a != NULL)
	{
		guint i;

		clist = GTK_CLIST(Pan.group_clist);
		g_static_mutex_lock (&grouplist_ui_mutex);
		pan_lock ();
		gtk_clist_freeze (clist);
		gtk_clist_unselect_all (clist);
		for (i=0; i!=a->len; ++i)
		{
			const Group * g = GROUP(g_ptr_array_index(a,i));
			const int row = gtk_clist_find_row_from_data (clist, (gpointer)g);
			gtk_clist_select_row (clist, row, 0);
			if (!gtk_clist_row_is_visible (clist, row))
			{
				gfloat valign = 0.5;
				gfloat halign = 0.0;
				gtk_clist_moveto (clist, row, 0, (double)valign, (double)halign);
			}
		}
		gtk_clist_thaw (clist);
		pan_unlock ();
		g_static_mutex_unlock (&grouplist_ui_mutex);
	}

	debug_exit ("grouplist_set_selected_groups");
}

Group*
grouplist_get_selected_group (void)
{
	GList * l;
	GtkCList * clist;
	Group * retval = NULL;
	debug_enter ("grouplist_get_selected_group");

	pan_lock ();
	clist = GTK_CLIST(Pan.group_clist);
	l = clist->selection;
	if (l != NULL)
		retval = GROUP(gtk_clist_get_row_data(clist, GPOINTER_TO_INT(l->data)));
	pan_unlock ();

	debug_exit ("grouplist_get_selected_group");
	return retval;
}

GPtrArray*
grouplist_get_selected_groups (void)
{
	GtkCList * clist = GTK_CLIST(Pan.group_clist);
	GPtrArray * retval = g_ptr_array_new ();
	GList * l = NULL;
	debug_enter ("grouplist_get_selected_groups");

	/* get the group selected items; otherwise, 
	   use the articlelist's group */
	pan_lock ();
	for (l=clist->selection; l!=NULL; l=l->next)
		g_ptr_array_add (retval, gtk_clist_get_row_data (clist, GPOINTER_TO_INT (l->data)));
	pan_unlock ();

	if (!retval->len) {
		Group * group = articlelist_get_group ();
		if (group != NULL)
			g_ptr_array_add (retval, group);
	}

	debug_exit ("grouplist_get_selected_groups");
	return retval;
}

/*****
******
*****/

static void
group_download_all_articles (Group * group)
{
	Task * task = NULL;
	GPtrArray * a = NULL;

	g_return_if_fail (group != NULL);
	g_return_if_fail (group->_articles != NULL);

	group_ref_articles (group, NULL);
	a = group_get_article_array (group);
	if (a->len)
		task = TASK(task_bodies_new (group, a));
	if (task != NULL)
		queue_add (task);
	group_unref_articles (group, NULL);

	/* cleanup */
	g_ptr_array_free (a, TRUE);
}

void
grouplist_remove_row (const Group *group)
{
	debug_enter ("grouplist_remove_row");

	if (group->server == grouplist_get_visible_server())
	{
		GtkCList *list = GTK_CLIST(Pan.group_clist);
		int row;

		g_static_mutex_lock (&grouplist_ui_mutex);
		pan_lock();
		row = gtk_clist_find_row_from_data (list, (gpointer)group);
		if (row != -1)
			gtk_clist_remove (list, row);
		pan_unlock();
		g_static_mutex_unlock (&grouplist_ui_mutex);
	}

	debug_exit ("grouplist_remove_row");
}
static void
grouplist_update_selected_rows (void)
{
	GPtrArray * groups;
	debug_enter ("grouplist_update_selected_rows");

       	groups = grouplist_get_selected_groups ();
	if (groups->len)
		grouplist_update_groups ((const Group**)groups->pdata, groups->len);
	g_ptr_array_free (groups, TRUE);

	debug_exit ("grouplist_update_selected_rows");
}
static void
grouplist_process_selected (int type)
{
	gint i;
	GPtrArray * groups;
	debug_enter ("grouplist_process_selected");

	/* find out which group(s) we're to operate on */
	groups = grouplist_get_selected_groups ();
	if (groups->len==0) {
		Group * group = articlelist_get_group ();
		if (group != NULL)
			g_ptr_array_add (groups, group);
	}

	/* iterate through the groups */
	for (i=0; i!=groups->len; ++i)
	{
		Group * group = GROUP(g_ptr_array_index(groups,i));

		switch (type)
		{
			case 1: /* subscribe */
				group_set_subscribed (group, TRUE);
				break;

			case 2: /* unsubscribe */
				group_set_subscribed (group, FALSE);
				if (group_mode==GROUP_SUBSCRIBED)
					grouplist_remove_row (group);
				if (group->article_qty != 0)
					group_empty_dialog (group);
				break;

			case 3: /* download all */
				queue_add (TASK(task_headers_new(group, HEADERS_ALL)));
				break;

			case 4: /* download new */
				queue_add (TASK(task_headers_new(group, HEADERS_NEW)));
				break;

			case 5: /* clean/empty */
				group_empty_dialog (group);
				break;

			case 6: /* group properties */
				dialog_group_props (group);
				break;

			case 10: /* download all bodies */
				group_download_all_articles (group);
				break;

			default:
				break;
		}
	}

	/* refresh the ui */
	grouplist_update_selected_rows ();

	/* cleanup */
	g_ptr_array_free (groups, TRUE);
	debug_exit ("grouplist_process_selected");
}

void
grouplist_selected_subscribe (void)
{
	grouplist_process_selected (1);
}
void
grouplist_selected_unsubscribe (void)
{
	grouplist_process_selected (2);
}
void
grouplist_selected_empty (void)
{
	grouplist_process_selected (5);
}
void
grouplist_selected_properties (void)
{
	grouplist_process_selected (6);
}

void
grouplist_selected_update_count_info (void)
{
	GPtrArray * a;
	debug_enter ("grouplist_selected_update_count_info");

       	a = grouplist_get_selected_groups ();
	if (a->len)
		queue_add (TASK(task_group_count_new (
			(Group**)a->pdata, a->len)));
	else
		g_ptr_array_free (a, TRUE);

	debug_exit ("grouplist_selected_update_count_info");
}

void
grouplist_selected_mark_read (void)
{
	GPtrArray * a = grouplist_get_selected_groups ();
	guint i;
	debug_enter ("grouplist_selected_mark_read");

        for (i=0; i<a->len; ++i)
	{
		Group * g = GROUP(g_ptr_array_index(a,i));
		group_mark_all_read (g, TRUE);
	}
	g_ptr_array_free (a, TRUE);

	debug_exit ("grouplist_selected_mark_read");
}

/*****
******
*****/

static void
delete_groups_dialog_cb (gint reply, gpointer data)
{
	GPtrArray * a;
	debug_enter ("delete_groups_dialog_cb");

	a = (GPtrArray*) data;
	if (reply == 0 /*yes*/)
	{
		guint i;
		GtkCList * clist = GTK_CLIST(Pan.group_clist);

		pan_lock ();
		gtk_clist_freeze (clist);
		pan_unlock ();

		for (i=0; i!=a->len; ++i)
		{
			Group * g = GROUP(g_ptr_array_index(a,i));
			server_destroy_group (g->server, g);
		}

		pan_lock ();
		gtk_clist_thaw (clist);
		pan_unlock ();
	}

	g_ptr_array_free (a, TRUE);
	debug_exit ("delete_groups_dialog_cb");
}

void
grouplist_selected_destroy (void)
{
	GPtrArray * a = grouplist_get_selected_groups ();
	if (a->len != 0)
	{
		gchar * txt = g_strdup_printf (_("Are you sure you want to delete these %d groups/folders and their articles?"), (gint)a->len);
		pan_lock ();
		gnome_question_dialog_parented (txt,
		                                delete_groups_dialog_cb,
		                                a,
		                                GTK_WINDOW(Pan.window));
		pan_unlock ();
		g_free (txt);
	}
	else g_ptr_array_free (a, TRUE);
}

/*****
******
*****/

/**
***  Process all the commands to download
***  [new | all | sample] [articles | headers]
**/

static void
grouplist_start_task_headers (GPtrArray * a,
                              HeaderDownloadType type,
                              gboolean bodies)
{
	guint i;
	debug_enter ("grouplist_start_task_headers");

	/* sample all the selected groups */
	for (i=0; i!=a->len; ++i)
	{
		Group * group = GROUP(g_ptr_array_index(a,i));
		PanObject * o = bodies
			? task_headers_new_with_bodies (group, type)
			: task_headers_new (group, type);
		queue_add (TASK(o));
	}

	/* refresh the ui */
	grouplist_update_selected_rows ();

	debug_exit ("grouplist_start_task_headers");
}

void
grouplist_selected_download_dialog (void)
{
	guint i;
	GPtrArray * a;
	debug_enter ("grouplist_selected_download_dialog");

	/* sample all the selected groups */
       	a = grouplist_get_selected_groups ();
	for (i=0; i!=a->len; ++i) {
		Group * group = GROUP(g_ptr_array_index(a,i));
		if (!group_is_folder(group))
			dialog_download_headers (group);
	}

	/* update ui */
	grouplist_update_selected_rows ();

	/* cleanup */
	g_ptr_array_free (a, TRUE);
	debug_exit ("grouplist_selected_download_dialog");
}

void
grouplist_selected_download_sample (void)
{
	GPtrArray * a;
	debug_enter ("grouplist_selected_download_sample");

       	a = grouplist_get_selected_groups ();
	grouplist_start_task_headers (a, HEADERS_SAMPLE, FALSE);
	g_ptr_array_free (a, TRUE);

	debug_exit ("grouplist_selected_download_sample");
}
void
grouplist_selected_download_sample_bodies (void)
{
	GPtrArray * a;
	debug_enter ("grouplist_selected_download_sample_bodies");

       	a = grouplist_get_selected_groups ();
	grouplist_start_task_headers (a, HEADERS_SAMPLE, TRUE);
	g_ptr_array_free (a, TRUE);

	debug_exit ("grouplist_selected_download_sample_bodies");
}
void
grouplist_selected_download_all (void)
{
	GPtrArray * a;
	debug_enter ("grouplist_selected_download_all");

       	a = grouplist_get_selected_groups ();
	grouplist_start_task_headers (a, HEADERS_ALL, FALSE);
	g_ptr_array_free (a, TRUE);

	debug_exit ("grouplist_selected_download_all");
}
void
grouplist_selected_download_all_bodies (void)
{
	GPtrArray * a;
	debug_enter ("grouplist_selected_download_all_bodies");

       	a = grouplist_get_selected_groups ();
	grouplist_start_task_headers (a, HEADERS_ALL, TRUE);
	g_ptr_array_free (a, TRUE);

	debug_exit ("grouplist_selected_download_all_bodies");
}
void
grouplist_selected_download_new (void)
{
	GPtrArray * a;
	debug_enter ("grouplist_selected_download_new");

       	a = grouplist_get_selected_groups ();
	grouplist_start_task_headers (a, HEADERS_NEW, FALSE);
	g_ptr_array_free (a, TRUE);

	debug_exit ("grouplist_selected_download_new");
}
void
grouplist_selected_download_new_bodies (void)
{
	GPtrArray * a;
	debug_enter ("grouplist_selected_download_new_bodies");

       	a = grouplist_get_selected_groups ();
	grouplist_start_task_headers (a, HEADERS_NEW, TRUE);
	g_ptr_array_free (a, TRUE);

	debug_exit ("grouplist_selected_download_new_bodies");
}
void
grouplist_subscribed_download_new (void)
{
	GPtrArray * a;
	debug_enter ("grouplist_subscribed_download_new");

	a = server_get_groups (my_server, SERVER_GROUPS_SUBSCRIBED);
	grouplist_start_task_headers (a, HEADERS_NEW, FALSE);
	g_ptr_array_free (a, TRUE);

	debug_exit ("grouplist_subscribed_download_new");
}
void
grouplist_subscribed_download_new_bodies (void)
{
	GPtrArray * a;
	debug_enter ("grouplist_subscribed_download_new_bodies");

	a = server_get_groups (my_server, SERVER_GROUPS_SUBSCRIBED);
	grouplist_start_task_headers (a, HEADERS_NEW, TRUE);
	g_ptr_array_free (a, TRUE);

	debug_exit ("grouplist_subscribed_download_new_bodies");
}

/*****
******
*****/


void
grouplist_get_all (void)
{
	if (my_server == NULL)
		pan_error_dialog ("No server selected.");
	else
		queue_add (TASK(task_grouplist_new (my_server,
		                                    TRUE,
		                                    GROUPLIST_ALL)));
}

void
grouplist_get_new (void)
{
	if (my_server == NULL)
		pan_error_dialog ("No server selected.");
	else
		queue_add (TASK(task_grouplist_new (my_server,
		                                    TRUE,
		                                    GROUPLIST_NEW)));
}

/*****
******
*****/



void
grouplist_select_all (void)
{
	pan_lock();
	gtk_clist_select_all (GTK_CLIST (Pan.group_clist));
	pan_unlock();
}

/*****
******
*****/

static void
grouplist_set_view_mode (GtkWidget *widget, gpointer data)
{
	GtkWidget *menu = NULL;
	debug_enter ("grouplist_set_view_mode");

	group_mode = GPOINTER_TO_INT (data);
	grouplist_update_ui (grouplist_get_visible_server());

        pan_lock ();
        menu = gtk_option_menu_get_menu (GTK_OPTION_MENU(grouplist_mode_menu));
        gtk_object_ref (GTK_OBJECT(menu));
        gtk_option_menu_remove_menu (GTK_OPTION_MENU (grouplist_mode_menu));
	switch (group_mode) {
		case GROUP_ALL:
			gtk_menu_set_active (GTK_MENU (menu), 0);
			break;
		case GROUP_SUBSCRIBED:
			gtk_menu_set_active (GTK_MENU (menu), 1);
			break;
		case GROUP_NEW:
			gtk_menu_set_active (GTK_MENU (menu), 2);
			break;
		case GROUP_FOLDERS:
			gtk_menu_set_active (GTK_MENU (menu), 3);
			break;
		default: break;
	}
        gtk_option_menu_set_menu (GTK_OPTION_MENU(grouplist_mode_menu), menu);
        gtk_object_unref (GTK_OBJECT(menu));
        pan_unlock ();

	debug_exit ("grouplist_set_view_mode");
}

typedef struct
{
	gchar * name;
	int mode;
}
GrouplistModeMenuStruct;

static GtkWidget*
grouplist_mode_menu_create (void)
{
	GtkWidget *option_menu = gtk_option_menu_new();
	GtkWidget *menu = gtk_menu_new ();
	int index = 0;
	int i;
	GrouplistModeMenuStruct foo[] = {
		{NULL, GROUP_ALL},
		{NULL, GROUP_SUBSCRIBED},
		{NULL, GROUP_NEW},
		{NULL, GROUP_FOLDERS}
	};
	const int row_qty = sizeof(foo) / sizeof(foo[0]);
	foo[0].name = _("All Groups");
	foo[1].name = _("Subscribed");
	foo[2].name = _("New Groups");
	foo[3].name = _("Folders");

	for (i=0; i<row_qty; ++i) {
		GtkWidget *item = gtk_menu_item_new_with_label (foo[i].name);
		gtk_widget_show (item);
		gtk_menu_append (GTK_MENU (menu), item);
		gtk_signal_connect (GTK_OBJECT(item), "activate",
				    (GtkSignalFunc)grouplist_set_view_mode,
				    GINT_TO_POINTER(foo[i].mode));
		if (group_mode == foo[i].mode)
			index = i;
	}

	gtk_menu_set_active (GTK_MENU(menu), index);
        gtk_option_menu_set_menu (GTK_OPTION_MENU (option_menu), menu);
	gtk_widget_show_all (GTK_WIDGET(option_menu));

	return option_menu;
}

/*****
******  CALLBACKS
*****/

static gint
grouplist_server_groups_added_cb (gpointer server, gpointer groups, gpointer foo)
{
	Server * visible;

	debug_enter ("grouplist_server_groups_added_cb");

	visible = grouplist_get_visible_server ();
	if (visible!=NULL && visible==SERVER(server))
		grouplist_update_ui (visible);

	debug_exit ("grouplist_server_groups_added_cb");
	return 0;
}

static gint
grouplist_server_groups_removed_cb (gpointer server,
                                    gpointer groups,
                                    gpointer foo)
{
	Server * visible;

	debug_enter ("grouplist_server_groups_removed_cb");

	visible = grouplist_get_visible_server ();
	if (visible!=NULL && visible==SERVER(server)) {
		GPtrArray * a = (GPtrArray*) groups;
		pan_g_ptr_array_foreach (a, (GFunc)grouplist_remove_row, NULL);
	}

	debug_exit ("grouplist_server_groups_removed_cb");
	return 0;
}

static gint
grouplist_group_changed_cb (gpointer group_data, gpointer unused1, gpointer unused2)
{
	const Group * group = GROUP(group_data);
	if (group->server == grouplist_get_visible_server())
		grouplist_update_groups (&group, 1);
	return 0;
}

static gint
grouplist_server_activated_cb (gpointer server_data, gpointer unused1, gpointer unused2)
{
	grouplist_set_server (SERVER(server_data));
	return 0;
}

/*****
******
*****/

void
grouplist_shutdown_module (void)
{
	/* stop listening... */
	pan_callback_remove (serverlist_get_server_activated_callback(),
	                     grouplist_server_activated_cb, NULL);
	pan_callback_remove (group_get_group_changed_callback(),
	                     grouplist_group_changed_cb, NULL);
	pan_callback_remove (grouplist_group_selection_changed,
	                     grouplist_selection_changed_cb, NULL);
	pan_callback_remove (server_get_groups_added_callback(),
	                     grouplist_server_groups_added_cb, NULL);
	pan_callback_remove (server_get_groups_removed_callback(),
	                     grouplist_server_groups_removed_cb, NULL);

	/* save state for next time */
	gnome_config_set_int ("/Pan/State/group_mode", group_mode);

	/* remember server so that we can default to it next time */
	if (my_server!=NULL && is_nonempty_string(my_server->name))
		gnome_config_set_string ("/Pan/State/Server", my_server->name);

	gnome_config_sync ();
}

gpointer
grouplist_create (void)
{
	GtkWidget * w;
        GtkWidget * vbox;
        GtkWidget * hbox;
	GtkWidget * toolbar;
	GtkCList * clist;
	char * titles[5];
	debug_enter ("grouplist_create");

	titles[0] = "";
	titles[1] = _("Groups");
	titles[2] = _("Unread");
	titles[3] = _("Total");
	titles[4] = _("Description");

	/* create callbacks */
	grouplist_group_selection_changed = pan_callback_new ();
	grouplist_server_set_callback = pan_callback_new ();

	/* register callbacks */
	pan_callback_add (serverlist_get_server_activated_callback(),
	                  grouplist_server_activated_cb, NULL);
	pan_callback_add (group_get_group_changed_callback(),
	                  grouplist_group_changed_cb, NULL);
	pan_callback_add (grouplist_group_selection_changed,
	                  grouplist_selection_changed_cb, NULL);
	pan_callback_add (server_get_groups_added_callback(),
	                  grouplist_server_groups_added_cb, NULL);
	pan_callback_add (server_get_groups_removed_callback(),
	                  grouplist_server_groups_removed_cb, NULL);

       	toolbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL,GTK_TOOLBAR_TEXT);

	vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER(vbox), GNOME_PAD_SMALL); 
	hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);

	group_mode = gnome_config_get_int ("/Pan/State/group_mode=1");
	if ( group_mode!=GROUP_ALL
	     && group_mode!=GROUP_SUBSCRIBED
	     && group_mode!=GROUP_NEW
	     && group_mode!=GROUP_FOLDERS)
		group_mode = GROUP_ALL;
	
	Pan.group_clist = gtk_clist_new_with_titles (5, titles);
	clist = GTK_CLIST(Pan.group_clist);
	clist->button_actions[1] = clist->button_actions[0];
	widget_set_font (GTK_WIDGET(Pan.group_clist), grouplist_font);
	gtk_clist_set_column_justification (clist, 2, GTK_JUSTIFY_RIGHT);
	gtk_clist_set_column_justification (clist, 3, GTK_JUSTIFY_RIGHT);


	w = gtk_label_new (_("Groups"));
	gtk_toolbar_append_widget(GTK_TOOLBAR(toolbar), w, "", "");
	gtk_toolbar_append_space (GTK_TOOLBAR(toolbar));

	grouplist_mode_menu = grouplist_mode_menu_create();
	gtk_box_pack_start (GTK_BOX(hbox), grouplist_mode_menu, FALSE, FALSE, 0);

	w = gtk_label_new (_("Find:"));
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);

	groupname_filter_entry = w = gtk_entry_new ();
	gtk_signal_connect (GTK_OBJECT (w), "activate",
			    GTK_SIGNAL_FUNC(grouplist_filter_changed_cb), w);
	gtk_signal_connect (GTK_OBJECT (w), "focus-out-event",
			    GTK_SIGNAL_FUNC(grouplist_filter_changed_cb), w);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w,
		_("Type in a group search string and press ENTER.  "
		  "Wildcards are allowed."), "");
	gtk_toolbar_set_space_size(GTK_TOOLBAR(toolbar), 100);
	gtk_widget_show_all (toolbar);
	gtk_box_pack_start (GTK_BOX(hbox), w, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	w = gtk_scrolled_window_new (NULL, NULL);

	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (w),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER(w), Pan.group_clist);
        gtk_box_pack_start (GTK_BOX (vbox), w, TRUE, TRUE, 0);

	gtk_signal_connect (GTK_OBJECT (Pan.group_clist),
			    "select_row",
			    GTK_SIGNAL_FUNC (select_row_cb), NULL);

	gtk_signal_connect (GTK_OBJECT (Pan.group_clist),
			    "unselect_row",
			    GTK_SIGNAL_FUNC (select_row_cb), NULL);

	gtk_signal_connect (GTK_OBJECT (Pan.group_clist),
			    "button_press_event",
			    GTK_SIGNAL_FUNC (grouplist_button_press), NULL);
	gtk_signal_connect (GTK_OBJECT (Pan.group_clist), "key_press_event",
	                    GTK_SIGNAL_FUNC (grouplist_key_press),
	                    NULL);


	gtk_clist_set_column_width     (clist, 0, 16);
	gtk_clist_set_column_min_width (clist, 0, 16);
	gtk_clist_set_column_max_width (clist, 0, 16);

	gtk_clist_set_column_width (clist, 1, 170);
	gtk_clist_set_column_width (clist, 2, 36);
	gtk_clist_set_column_width (clist, 3, 33);
	gtk_clist_set_column_width (clist, 4, 350);

/* JEL: this works well in paned mode, but not so well in notebook mode, we
   need to find a solution that will cover both ... maybe splitting the justification
   parts into the _construct funcs for both ...
	gtk_clist_set_column_justification (clist, 0, GTK_JUSTIFY_LEFT );
	gtk_clist_set_column_justification (clist, 1, GTK_JUSTIFY_RIGHT );
	gtk_clist_set_column_justification (clist, 2, GTK_JUSTIFY_RIGHT );
*/
	gtk_clist_set_selection_mode (clist, GTK_SELECTION_EXTENDED);

	/* Create the Popup Menu while we are here. */
	group_clist_menu = gnome_popup_menu_new (group_menu);	
	folder_clist_menu = gnome_popup_menu_new (folder_menu);	

	/* Lets have the Pixmap ready too */
	folder_pixmap = (GnomePixmap*) gnome_pixmap_new_from_xpm_d (folder_xpm);
	group_pixmap = (GnomePixmap*) gnome_pixmap_new_from_xpm_d (envelope_xpm);

	/* bootstrap */
	if (1) {
		my_server = serverlist_get_active_server ();
		grouplist_update_ui_thread (NULL);
	}

	gtk_widget_show_all (vbox);
	debug_exit ("grouplist_create");
	return vbox;
}

static void
grouplist_filter_changed_cb (void)
{
	grouplist_refresh_filter ();
}

/*****
******
*****/

static void
grouplist_add_group_nolock (const Group * group)
{
	gint row;
	GnomePixmap * pixmap;
	gchar total_buf[16] = { '\0' };
	gchar unread_buf[16] = { '\0' };
	const gint total = group->article_qty;
	const gint unread = MAX(0, total - group->article_read_qty);
	gchar * text[5];
	gchar * name;

	/* commatize the totals */
	commatize_ulong (total, total_buf);
	commatize_ulong (unread, unread_buf);

	/* look for special qualities of the group */
	name = grouplist_get_group_readable_name (group);
       
	text[0] = "";
	text[1] = name;
	text[2] = unread_buf;
	text[3] = total_buf;
	text[4] = (gchar*)(group->description?group->description:"");

	/* add the row */
	row = gtk_clist_prepend (GTK_CLIST(Pan.group_clist), (char**)text);
	gtk_clist_set_row_data (GTK_CLIST(Pan.group_clist), row, (gpointer)group);

	/* add the pixmap, if any */
	pixmap = NULL;
	if (group_is_folder (group))
		pixmap = folder_pixmap;
	else if (group_is_subscribed (group))
		pixmap = group_pixmap;
	if (pixmap != NULL)
		gtk_clist_set_pixmap (GTK_CLIST(Pan.group_clist), row, 0,
		                      pixmap->pixmap,
		                      pixmap->mask);

	/* cleanup */
	g_free (name);
}

static gchar*
grouplist_update_describe (const StatusItem *item)
{
	return g_strdup (_("Updating Group List"));
}

static void
grouplist_update_ui_thread (void * unused)
{
	gchar * title;
	GtkCList * clist = GTK_CLIST(Pan.group_clist);
	GPtrArray * groups;
	GPtrArray * sel;
	StatusItem *item = NULL;
	int i;
	debug_enter ("grouplist_update_ui_thread");

	/**
	***  Get the list of groups that we're going to add.
	***
	***  Note that we temporarily turn off the groups_added
	***  callback -- the groups may be loaded as a result of
	***  our call, and we don't want to update the clist twice.
	***  This is a temporary workaround that should dissapear
	***  in 0.10.x
	**/
	pan_callback_remove (server_get_groups_added_callback(),
	                     grouplist_server_groups_added_cb, NULL);
	if (group_mode == GROUP_SUBSCRIBED)
	{
		groups = my_server==NULL
			? g_ptr_array_new ()
			: server_get_groups (my_server, SERVER_GROUPS_SUBSCRIBED);
	}
	else if (group_mode == GROUP_ALL)
	{
		groups = my_server==NULL
			? g_ptr_array_new ()
			: server_get_groups (my_server, SERVER_GROUPS_ALL);
	}
	else if (group_mode == GROUP_FOLDERS)
	{
		Server * s = serverlist_get_named_server(INTERNAL_SERVER_NAME);
		groups = s==NULL
			? g_ptr_array_new()
			: server_get_groups (s, SERVER_GROUPS_ALL);
	}
	else if (group_mode == GROUP_NEW)
	{
		groups = g_ptr_array_new ();
		if (my_server != NULL) {
			guint i;
			GPtrArray * a = server_get_groups (my_server, SERVER_GROUPS_ALL);
			for (i=0; i<a->len; ++i) {
				Group * g = GROUP(g_ptr_array_index(a,i));
				if (group_is_new (g))
					g_ptr_array_add (groups, g);
			}
			g_ptr_array_free (a, TRUE);
		}
	}
	else
	{
		pan_warn_if_reached ();
		groups = g_ptr_array_new ();
	}
	pan_callback_add (server_get_groups_added_callback(),
	                  grouplist_server_groups_added_cb, NULL);

        /* create a new status item to get sort/thread messages */
	item = STATUS_ITEM(status_item_new(grouplist_update_describe));
	status_item_set_active (item, TRUE);
	status_item_emit_init_steps (item, groups->len);
	status_item_emit_status (item, _("Updating Group List"));

	/* set the groups column title */
	switch (group_mode) {
		case GROUP_SUBSCRIBED: title = _("Subscribed Groups");  break;
		case GROUP_ALL:        title = _("All Groups");         break;
		case GROUP_NEW:        title = _("New Groups");         break;
		case GROUP_FOLDERS:    title = _("Folders");            break;
		default:               title = _("Bug in Code");        break;
	}

       	sel = grouplist_get_selected_groups ();

	/* update the list */
	g_static_mutex_lock (&grouplist_ui_mutex);
	pan_lock ();
	gtk_clist_freeze (clist);
	gtk_clist_clear (clist);
	gtk_clist_set_column_title (clist, 1, title);

	/* add the groups/folders */
	for (i=groups->len-1; i>=0; --i)
	{
		Group * group = GROUP(g_ptr_array_index (groups, i));

		if (!(i%512)) {
			pan_unlock ();
			status_item_emit_set_step (item, groups->len-i);
			pan_lock ();
		}

		if (is_nonempty_string(groupname_filter_string) && fnmatch(groupname_filter_string,group->name,PAN_CASEFOLD))
			continue;

		grouplist_add_group_nolock (group);

	}
	gtk_clist_thaw (clist);
	pan_unlock();
	g_static_mutex_unlock (&grouplist_ui_mutex);

	/* update the selection */
	grouplist_set_selected_groups (sel);

	/* clean out the status item */
	status_item_set_active (item, FALSE);
	pan_object_unref(PAN_OBJECT(item));

	/* cleanup */
	g_ptr_array_free (sel, TRUE);
	g_ptr_array_free (groups, TRUE);
	debug_exit ("grouplist_update_ui_thread");
}

void
grouplist_update_ui (Server * server)
{
	debug_enter ("grouplist_update_ui");

	if (server!=NULL && server==grouplist_get_visible_server())
	{
		pthread_t thread;

		pthread_create (&thread, NULL, (void*)grouplist_update_ui_thread, NULL);
		pthread_detach (thread);
	}

	debug_exit ("grouplist_update_ui");
}

/***
****
****
***/

/**
 * @group: Group who's row is to be updated.
 *
 * Update the CList row for a group to show it's message counts or
 * any other relative data (subscribed/on disk) for that group.
 **/
static void
grouplist_update_group_nolock (const Group * group)
{
	GtkCList * list;
	gint row;

	/* sanity check */
	g_return_if_fail (group != NULL);

	list = GTK_CLIST (Pan.group_clist);
       	row = gtk_clist_find_row_from_data (list,(gpointer)group);
	if (row != -1)
	{
		gchar buf[16];
		gchar * name;
		const gint total = group->article_qty;
		const gint unread = MAX(0, total-group->article_read_qty);

		/* icon column */
		if (group_is_subscribed (group))
			gtk_clist_set_pixmap (
				GTK_CLIST(Pan.group_clist), row, 0,
				group_pixmap->pixmap,
				group_pixmap->mask);
		else if (!group_is_folder(group))
			gtk_clist_set_text (GTK_CLIST(Pan.group_clist),
					    row, 0, "");

		/* name column */
		name = grouplist_get_group_readable_name (group);
		gtk_clist_set_text (list, row, 1, name);
		g_free (name);

		/* total column */
		commatize_ulong (total, buf);
		gtk_clist_set_text (list, row, 3, buf);

		/* unread column */
		commatize_ulong (unread, buf);
		gtk_clist_set_text (list, row, 2, buf);
	}
}

static gint
grouplist_update_groups_idle (gpointer data)
{
	guint i;
	GPtrArray * a = (GPtrArray*)data;
	GtkCList * list = GTK_CLIST (Pan.group_clist);
	debug_enter ("grouplist_update_groups_idle");

	g_static_mutex_lock (&grouplist_ui_mutex);
	pan_lock_unconditional ();
	gtk_clist_freeze (list);
	for (i=0; i!=a->len; ++i)
		grouplist_update_group_nolock (GROUP(g_ptr_array_index(a,i)));
	gtk_clist_thaw (list);
	pan_unlock_unconditional ();
	g_static_mutex_unlock (&grouplist_ui_mutex);

	/* cleanup */
	g_ptr_array_free (a, TRUE);
	debug_exit ("grouplist_update_groups_idle");
	return 0;
}

void
grouplist_update_groups (const Group ** groups, int group_qty)
{
	gint i;
	GPtrArray * a;
	Server * server;
	debug_enter ("grouplist_update_groups");

	g_return_if_fail (groups != NULL);
	g_return_if_fail (group_qty >= 1);

	/* build an array of groups that we might need to update */
	a = g_ptr_array_new ();
	server = grouplist_get_visible_server ();
	for (i=0; i<group_qty; ++i)
		if (groups[i]!=NULL && groups[i]->server==server)
			g_ptr_array_add (a, (gpointer)groups[i]);

	/* if we have any that we should update, queue it */
	if (a->len != 0) {
		pan_lock ();
		gtk_idle_add (grouplist_update_groups_idle, a);
		pan_unlock ();
	} else {
		g_ptr_array_free (a, TRUE);
	}

	debug_exit ("grouplist_update_groups");
}

/***
****
****
***/

static void
grouplist_internal_select_row (gint      row,
                               gboolean  activate,
                               gboolean  lock)
{
        GtkCList* list = GTK_CLIST(Pan.group_clist);

	if (activate)
		button_click_count = 2;

	if (lock) pan_lock ();
	gtk_clist_freeze (list);
	gtk_clist_unselect_all (list);
	gtk_clist_select_row (list,row,-1);
	gtk_clist_thaw (list);
	if (lock) pan_unlock ();
}

static gboolean
grouplist_select_row_if_unread (int row,
                                gboolean activate,
                                gboolean lock)
{
        GtkCList * list = GTK_CLIST(Pan.group_clist);
	Group * g = gtk_clist_get_row_data(list,row);
	const gulong total = g->article_qty;
	const gulong read = g->article_read_qty;
	gboolean retval = FALSE;

	if (read < total) {
		grouplist_internal_select_row (row, activate, lock);
		retval = TRUE;
	}

	return retval;
}
static void
grouplist_next_unread_group (gboolean activate)
{
        GtkCList* list;
	int row_qty, i, row=0;

	pan_lock ();
	list = GTK_CLIST(Pan.group_clist);
        row_qty = list->rows;

	if (row_qty != 0)
	{
		/* get the first row to check */
		row = 0;
		if (list->selection) {
			int sel = GPOINTER_TO_INT(list->selection->data);
			row = (sel + 1) % row_qty;
		}

		/* walk through */
		for ( i=0; i<row_qty; ++i, row=++row%row_qty)
			if (grouplist_select_row_if_unread (row, activate, 0))
				break;
	}

	pan_unlock ();
}
void
grouplist_select_next_unread_group (void)
{
	grouplist_next_unread_group (FALSE);
}
void
grouplist_activate_next_unread_group (void)
{
	grouplist_next_unread_group (TRUE);
}
static void
grouplist_next_group (gboolean activate)
{
        GtkCList* list;
	int row_qty;
	int row = 0;

	pan_lock ();
	list = GTK_CLIST(Pan.group_clist);
        row_qty = list->rows;

	if (row_qty != 0)
	{
		/* get the first row to check */
		row = 0;
		if (list->selection) {
			int sel = GPOINTER_TO_INT(list->selection->data);
			row = (sel + 1) % row_qty;
		}

		grouplist_internal_select_row (row, activate, 0);
	}

	pan_unlock ();
}
void
grouplist_select_next_group (void)
{
	grouplist_next_group (FALSE);
}
void
grouplist_activate_next_group (void)
{
	grouplist_next_group (TRUE);
}

/***
****
***/

static void
group_menu_popup (GdkEventButton *event)
{
	gboolean selected = grouplist_get_selected_group() != NULL;
	GList * l = NULL;
	debug_enter ("group_menu_popup");

	if (group_mode == GROUP_FOLDERS) /* We are dealing with folders */
	{
		pan_lock ();
		l=GTK_MENU_SHELL(folder_clist_menu)->children;
		/* l is new folder */
		l=l->next; /* delete folder */
		gtk_widget_set_sensitive (GTK_WIDGET(l->data), selected);

		/* show the menu */
		gtk_menu_popup (GTK_MENU(folder_clist_menu),
			NULL, NULL, NULL, NULL,
			event->button, event->time);
		pan_unlock ();
	}
	else /* We must be dealing with real groups */
	{
		pan_lock ();

		l=GTK_MENU_SHELL(group_clist_menu)->children;  /* download */
		gtk_widget_set_sensitive (GTK_WIDGET(l->data), selected);
		l=l->next; /* l is separator */
		l=l->next; /* l is get new headers in subscribed */
		gtk_widget_set_sensitive (GTK_WIDGET(l->data), TRUE);
		l=l->next; /* l is get new headers in selected */
		gtk_widget_set_sensitive (GTK_WIDGET(l->data), selected);
		l=l->next; /* l is get all headers in selected */
		gtk_widget_set_sensitive (GTK_WIDGET(l->data), selected);
		l=l->next; /* l is sample for selected */
		gtk_widget_set_sensitive (GTK_WIDGET(l->data), selected);
		l=l->next; /* separator */
		l=l->next; /* l is get new headers & bodies in subscribed */
		gtk_widget_set_sensitive (GTK_WIDGET(l->data), TRUE);
		l=l->next; /* l is get new headers & bodies in selected */
		gtk_widget_set_sensitive (GTK_WIDGET(l->data), selected);
		l=l->next; /* l is get all headers & bodies in selected */
		gtk_widget_set_sensitive (GTK_WIDGET(l->data), selected);
		l=l->next; /* l is sample new headers & bodies in selected */
		gtk_widget_set_sensitive (GTK_WIDGET(l->data), selected);
		l=l->next; /* separator */
		l=l->next; /* refresh article count */
		gtk_widget_set_sensitive (GTK_WIDGET(l->data), selected);
		l=l->next; /* catch up */
		gtk_widget_set_sensitive (GTK_WIDGET(l->data), selected);

		l=l->next; /* separator */
		l=l->next; /* subscribe */
		gtk_widget_set_sensitive (GTK_WIDGET(l->data), selected);
		l=l->next; /* unsubscribe */
		gtk_widget_set_sensitive (GTK_WIDGET(l->data), selected);

		l=l->next; /* separator */
		l=l->next; /* destroy group(s) */
		gtk_widget_set_sensitive (GTK_WIDGET(l->data), selected);
		l=l->next; /* empty group */
		gtk_widget_set_sensitive (GTK_WIDGET(l->data), selected);
		l=l->next; /* properties */
		gtk_widget_set_sensitive (GTK_WIDGET(l->data), selected);

		/* show the menu */
		gtk_menu_popup (GTK_MENU(group_clist_menu),
				NULL, NULL, NULL, NULL,
				event->button, event->time);
		pan_unlock ();
	} 

	debug_exit ("group_menu_popup");
}

static void
grouplist_refresh_filter (void)
{
	gchar * new_filter;
        gchar * pch;

	pan_lock ();
	pch  = gtk_editable_get_chars (GTK_EDITABLE(groupname_filter_entry), 0, -1);
	pan_unlock ();

	/* build the new filter string */
	new_filter = NULL;
	if (is_nonempty_string (pch)) {
		if (strchr(pch,'*') != NULL)
			new_filter = g_strdup (pch);
		else 
			new_filter = g_strdup_printf("*%s*",pch);
	}

	/* if it differs from the old filter string, refresh */
	if (pan_strcmp (new_filter, groupname_filter_string)) {
		replace_gstr (&groupname_filter_string, g_strdup(new_filter));
		grouplist_update_ui (grouplist_get_visible_server());
	}

	/* cleanup */
	g_free (pch);
	g_free (new_filter);
}
