/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Copyright (C) 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <errno.h>
#include <string.h>

#include <pan/group.h>
#include <pan/pan-i18n.h>
#include <pan/log.h>
#include <pan/newsrc.h>
#include <pan/newsrc-port.h>
#include <pan/pan-glib-extensions.h>
#include <pan/util-file.h>

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

/*********************
**********************  Private Function Prototypes
*********************/

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PUBLIC ROUTINES
************/

/*****
******
*****/

void
newsrc_import (Server          * server,
               const gchar     * filename_in,
               gboolean          subscribed_only)
{
	GArray * body;
	GString * line;
	gchar * filename;
	const gchar * pch;
	gint tot_groups = 0;
	GPtrArray * groups_to_add;

	g_return_if_fail (server!=NULL);
	g_return_if_fail (is_nonempty_string(filename_in));
	filename = pan_substitute (filename_in, "~", g_get_home_dir());
	g_return_if_fail (file_exists(filename));

       	groups_to_add = g_ptr_array_new ();
	line = g_string_new (NULL);
	body = read_file (filename);
	pch = NULL;
	if (body==NULL || body->len<1)
		log_add_va (LOG_ERROR, _("Error reading file `%s'"), filename);
	else
		pch = body->data;

	/* march through each line of newsrc... */
	while (get_next_token_g_str (pch, '\n', &pch, line))
	{
		gchar * pch;

		/* get the group name */
		pch = strpbrk (line->str, "!:");
		if (pch!=NULL)
		{
			Newsrc * n;
		       	gboolean subscribed;
			gchar * name;
			Group * g;

			if (subscribed_only && *pch!=':')
				continue;

			++tot_groups;
			name = g_strndup (line->str, pch-line->str);

			/* get the group which this line is for. */
			g = server_get_named_group (server, name);
			if (g == NULL) {
				g = group_new (name);
				g_ptr_array_add (groups_to_add, g);
			}

			/* import the newsrc line */
			n = group_get_newsrc (g);
			subscribed = FALSE;
			newsrc_import_line (n, &subscribed, line->str);
			group_set_article_range (g, n->group_low, n->group_high);
			group_set_subscribed (g, subscribed);
			g_free (name);
		}
	}
	server_add_groups (server, (Group**)groups_to_add->pdata, groups_to_add->len, NULL, NULL);

	log_add_va (LOG_INFO, _("Imported %d groups (%d new) from `%s'"), tot_groups, (int)groups_to_add->len, filename);

	/* cleanup */
	g_free (filename);
	g_array_free (body, TRUE);
	g_string_free (line, TRUE);
	g_ptr_array_free (groups_to_add, TRUE);
}

/*****
******
*****/

void
newsrc_export (Server          * server,
               const gchar     * filename_in,
               gboolean          subscribed_only,
               gboolean          backup_first)
{
	FILE * fp;
	gchar * tmpfile;
	gchar * filename;
	gint tot_groups = 0;
	gint sub_groups = 0;

	/* sanity clause */
	g_return_if_fail (server!=NULL);
	g_return_if_fail (is_nonempty_string(filename_in));
	filename = pan_substitute (filename_in, "~", g_get_home_dir());
	filename_in = NULL;

	/* open a temp file to write to */
	fp = NULL;
	tmpfile = pan_make_temp (&fp);
	if (fp == NULL) {
		log_add_va (LOG_ERROR, _("Couldn't open tmp file `%s' for writing newsrc file: %s"),
		                       filename,
		                       g_strerror(errno));
		g_free (filename);
		return;
	}

	/* write the newsrc file */
	if (1) {
		gint i;
		ServerGroupsType type = subscribed_only?SERVER_GROUPS_SUBSCRIBED:SERVER_GROUPS_ALL;
		GString * s = g_string_sized_new (4096);
		GPtrArray * a = server_get_groups (server, type);
		for (i=0; i<a->len; ++i) {
			Group * g = GROUP(g_ptr_array_index(a, i));
			Newsrc * n = group_has_newsrc(g) ? group_get_newsrc(g) : NULL;
			++tot_groups;
			if (group_is_subscribed(g))
				++sub_groups;
			g_string_truncate (s, 0);
			newsrc_export_line (n, g->name, group_is_subscribed(g), s);
			fprintf (fp, "%s\n", s->str);
		}
		fclose (fp);
		g_ptr_array_free (a, TRUE);
	}

	/* backup? */
	if (backup_first) {
		gchar * backup = g_strdup_printf ("%s.bak", filename);
		rename (filename, backup);
		g_free (backup);

	}

	/* move the temp file to newsrc */
	rename (tmpfile, filename);

	/* log */
	log_add_va (LOG_INFO, _("Exported %d groups (%d subscribed) to `%s'"),
	                      tot_groups, sub_groups, filename);

	/* cleanup */
	g_free (filename);
	g_free (tmpfile);
}
