///
// Copyright (C) 2002 - 2004, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "imageframe.h"
#include "fileerrors.h"
#include <sstream>
#include <fstream>
#include <libxml++/libxml++.h>
#include "util/tempfile.h" 
#include "util/warning.h"
#include "util/stringutil.h"
#include "util/filesys.h"
#include "ps/misc.h"
#include "ps/wineps.h"
#include "document.h"

bool ImageFrame::is_postscript(std::string filename) {
  std::ifstream in(filename.c_str());
  char magic[5]; // the magic number is four bytes long
  return in && in.get(magic, 5) 
    && (std::string(magic) == "%!PS" || PS::check_windows_magic(magic));
}

ImageFrame::ImageFrame(Group *parent, 
			 const std::string& assoc, float w, float h) :
  CachedFrame(parent, w, h, "Image " + basename(assoc)),
  association(assoc), filewatcher(association)
{
  parsed_file = Tempfile::find_new_name();
  filewatcher.modified_signal.connect(slot(*this, 
					   &ImageFrame::on_file_modified));
}

ImageFrame::ImageFrame(Group *parent, const xmlpp::Element& node,
		       const FileContext &context) :
  CachedFrame(parent, node), filewatcher(association)
{
  filewatcher.modified_signal.connect(slot(*this, 
					   &ImageFrame::on_file_modified));

  if(!node.get_attribute("type") 
     || node.get_attribute("type")->get_value() != "image")
    throw Error::FrameType("Bad image frame type: "
			    + node.get_attribute("type")->get_value());

  if(const xmlpp::Attribute* file = node.get_attribute("file"))
    set_association(context.from(file->get_value()));
  else {
    set_association("");
    warning << "No \"file\" attribute found in image frame" << std::endl;
  }

  parsed_file = Tempfile::find_new_name();
  read_size(); // try to read bounding box from file
}

ImageFrame::~ImageFrame() {
  if(!parsed_file.empty())
    try { unlink(parsed_file); }
    catch(const std::exception& err) {
      warning << "Failed to clean up: " << err.what() << std::endl;
    }
}

std::string ImageFrame::getTypeName() const { return "image"; }

xmlpp::Element *ImageFrame::save(xmlpp::Element& parent_node,
				 const FileContext &context) const 
{
  xmlpp::Element *node = CachedFrame::save(parent_node, context);
  node->set_attribute("type", "image");
  node->set_attribute("file", context.to(association));
  return node;
}


namespace {
  Vector parse(PS::WinEPSFilter &in, std::ostream &out, 
	       const float& scale_x, const float& scale_y) {
    float x1, y1;		// width and height of actual file.
    float _width, _height;
    bool boundingbox_found = false;
    bool transf_performed = false;
    
    std::string tmpline;
    while(getline(in, tmpline)) {
      if(starts_with(tmpline, "%%BoundingBox:")) {
	std::istringstream tmp(tmpline.c_str());
	std::string word;
	float x2, y2;
	if(tmp >> word >> x1 >> y1 >> x2 >> y2) {
	  boundingbox_found = true;
	  _width = x2 - x1;  _height = y2 - y1;
	  out << "%%BoundingBox: 0 0 "
	      << _width*scale_x << ' ' << _height*scale_y << '\n';
	}
      } else if(!starts_with(tmpline, '%') && !transf_performed) {
	out << tmpline << "\n\n";
	transf_performed = true;
	if(!boundingbox_found)
	  throw BasicFrame::GenPicError(BasicFrame::GENERATION, 
					   "No BoundingBox found");
	else {
	  out << scale_x << ' ' << scale_y << " scale\n"
	      << -x1 << ' ' << -y1 << " translate\n\n";
	}
      } else out << tmpline << '\n';
    }
    return Vector(_width, _height);
  }
}

void ImageFrame::print(std::ostream &out, bool grayscale) const {
  if(association.empty()) {
    CachedFrame::print(out);
    return;
  }
  
  out << "save\n"
      << "/showpage {} def\n"
      << PS::Concat(get_matrix())
      << "0 setgray 0 setlinecap 1 setlinewidth\n"
      << "0 setlinejoin 10 setmiterlimit [] 0 setdash newpath\n";
  if(true) //FIXME!!! if(level2)
    out << "false setoverprint false setstrokeadjust\n";

  // redefine color operators
  // NOTE: does not work for images
  // NOTE: this does not work for CIE-based or special color spaces
  if(grayscale) {
    out << "/setrgbcolor {setrgbcolor currentgray setgray} bind def\n"
	<< "/sethsbcolor {sethsbcolor currentgray setgray} bind def\n"
	<< "/setcmykcolor {setcmykcolor currentgray setgray} bind def\n"
	<< "/setcolor {setcolor currentgray setgray} bind def\n";
  }

  out << "%%BeginDocument: " << association << '\n';
  PS::WinEPSFilter in(association.c_str());
  if(in) {
    try {
      parse(in, out, 1.0, 1.0);
    } catch(GenPicError e){
      throw Error::Print(get_name() + ": " + e.what());
    }
  } else
    throw Error::Print(get_name() + ": Couldn't read " + association);
  out << "%%EndDocument\n"
      << "restore" << std::endl;
}

void ImageFrame::set_association(const std::string &s) {
  if(association == s) 
    return;
  
  association = s;
  props_changed_signal(this);
  // Todo:  Shouldn't the filewatcher signal that the object is changed when
  // we call its set_file?  If so, we shouldn't signal ready_to_draw here.
  ready_to_draw_signal(this);

  filewatcher.set_file(association);
}

void ImageFrame::generate_picture(FileCallback callback) {
  if(association.empty())
    throw GenPicError(ASSOCIATION, "No associated file");

  {
    PS::WinEPSFilter in(association);
    std::ofstream out(parsed_file.c_str());
    if(!in)
      throw GenPicError(ASSOCIATION, "Couldn't open " + association);
    if(!out)
      throw GenPicError(GENERATION, "Couldn't write to " + parsed_file);
    
    const Matrix &m = get_matrix();
    
    // Ghostscript needs a showpage
    out << "%!PS\n"
	<< "save\n"
	<< "/showpage {} def\n";
    Vector size = parse(in, out, m.sc_x(), m.sc_y());
    width = size.x; height = size.y;
    out << "restore\n"
	<< "showpage" << std::endl;
  }
  callback(parsed_file);
}

void ImageFrame::on_file_modified() {
  /// \todo notify the viewent, so it can clear the image
  object_changed_signal(this);
  ready_to_draw_signal(this);
}

// Note: as long as this is only called in the constructor there is no need to
// emit a geometry_changed_signal
void ImageFrame::read_size() {
  width = 100; height = 100; 
  // Must set the size to something, so we can see the frame even on failure.
  if(!association.empty()) {
    PS::WinEPSFilter in(association);
    std::string tmpline;
    while(getline(in, tmpline)) {
      if(starts_with(tmpline, "%%BoundingBox:")) {
	std::istringstream tmp(tmpline.c_str());
	std::string word;
	float x1, y1, x2, y2;
	if(tmp >> word >> x1 >> y1 >> x2 >> y2) {
	  width = x2 - x1;  height = y2 - y1;
	  return;
	}
	else
	  warning << "Bad bounding box in " << association << std::endl;
      }
    }
  }
}
