///
// Copyright (C) 2002 - 2004, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "propertiesdialog.h"
#include "propbase.h"
#include "widget/subpanel.h"
#include "widget/wmisc.h"
#include "util/matrix.h"
#include "util/stringutil.h"
#include "util/warning.h"
#include <gtkmm/label.h>
#include <gtkmm/separator.h>
#include <gtkmm/entry.h>
#include <gtkmm/checkbutton.h>
#include <gtkmm/stock.h>
#include <gtkmm/sizegroup.h>
#include <algorithm>
#include "pptcore.h"
#include "widget/spinner.h"
#include "document/basicframe.h"
#include "lengthunits.h"
#include "config.h"


// The basic properties that apply to all Pagent's.
// Currently that means some kind of name, and the position and size of the
// object.
class PropBasic : public PropBase {
public:
  // | name:  | "name of object"             |
  // -----------------------------------------
  // | left:  | "left"  | width:  | "width"  |
  // | right: | "right" | height: | "height" |
  PropBasic(const std::string &default_unit)
    :PropBase("_Basic"), lock(false), object(0),
     e_left(0, true, &length_units, default_unit),
     e_bottom(0, true, &length_units, default_unit), 
     e_width(0, true, &length_units, default_unit),
     e_height(0, true, &length_units, default_unit), 
     e_rotate(0, true, &angle_units), 
     e_margin(0, true, &length_units, default_unit),
     c_flow("Text avoids object", 0), c_locked("Locked", 0)
  {
      Gtk::Label *label;
      Gtk::Box *line = manage(new Gtk::HBox(false, double_space));
      line->pack_start(*(label = manage(new Gtk::Label("Object _name:", 
						       0.0, 0.5, true))),
		       Gtk::PACK_SHRINK);
      label->set_mnemonic_widget(e_name);
      line->pack_start(e_name);
      pack_start(*line, Gtk::PACK_SHRINK);
      
      SubPanel *box = manage(new SubPanel("Geometry"));
      Glib::RefPtr<Gtk::SizeGroup>  sizegroup = 
	Gtk::SizeGroup::create(Gtk::SIZE_GROUP_HORIZONTAL);

      line = manage(new Gtk::HBox(false, double_space));
      sizegroup->add_widget(*(label = manage(new Gtk::Label("_Left:", 
							    0.0, 0.5, true))));
      line->pack_start(*label, Gtk::PACK_SHRINK);
      line->pack_start(e_left, Gtk::PACK_SHRINK);
      label->set_mnemonic_widget(e_left);

      sizegroup->add_widget(*(label = manage(new Gtk::Label("_Width:",
							    0.0, 0.5, true))));
      line->pack_start(*label, Gtk::PACK_SHRINK);
      line->pack_start(e_width, Gtk::PACK_SHRINK);
      label->set_mnemonic_widget(e_width);
      box->pack_start(*line, Gtk::PACK_SHRINK);
      
      line = manage(new Gtk::HBox(false, double_space));
      sizegroup->add_widget(*(label = manage(new Gtk::Label("_Bottom:", 
							    0.0, 0.5, true))));
      line->pack_start(*label, Gtk::PACK_SHRINK);
      line->pack_start(e_bottom, Gtk::PACK_SHRINK);
      label->set_mnemonic_widget(e_bottom);

      sizegroup->add_widget(*(label = manage(new Gtk::Label("_Height:",
							    0.0, 0.5, true))));
      line->pack_start(*label, Gtk::PACK_SHRINK);
      line->pack_start(e_height, Gtk::PACK_SHRINK);
      label->set_mnemonic_widget(e_height);
      box->pack_start(*line, Gtk::PACK_SHRINK);
      
      line = manage(new Gtk::HBox(false, double_space));
      sizegroup->add_widget(*(label = manage(new Gtk::Label("R_otate:",
							    0.0, 0.5, true))));
      line->pack_start(*label, Gtk::PACK_SHRINK);
      e_rotate.limits(-180.0, 180.0);
      line->pack_start(e_rotate, Gtk::PACK_SHRINK);
      label->set_mnemonic_widget(e_rotate);
      
      box->pack_start(*line, Gtk::PACK_SHRINK);
      pack_start(*box, Gtk::PACK_SHRINK);
      
      pack_start(hsep2, Gtk::PACK_SHRINK);
      
      Gtk::HBox *flow_box = manage(new Gtk::HBox(false, double_space));
      flow_box->pack_start(c_locked, Gtk::PACK_SHRINK, triple_space);
      flow_box->pack_start(c_flow, Gtk::PACK_SHRINK, 0);
      c_flow.signal_toggled().connect
	(slot(*this, &PropBasic::on_c_flow_toggle));
      flow_box->pack_start(e_margin, Gtk::PACK_SHRINK, 0);
      pack_start(*flow_box, Gtk::PACK_SHRINK);

      set_sensitive(false);
    }
  void on_c_flow_toggle() {
    e_margin.set_sensitive(c_flow.get_active());
  }
  void on_geometry_changed(Pagent* pagent) {
    update();
  }
  void on_props_changed(Pagent* pagent) {
    update();
  }
  void setObject(Pagent* pagent) {
    geometry_changed_connection.disconnect();
    props_changed_connection.disconnect();
    object = pagent;
    if(object) {
      geometry_changed_connection = object->geometry_changed_signal.connect
	(slot(*this, &PropBasic::on_geometry_changed));
      props_changed_connection = object->props_changed_signal.connect
	(slot(*this, &PropBasic::on_props_changed));
    }
    set_sensitive(object);
    if(is_visible())
      update();
  }
  void update() {
    if(lock) // don't update while applying
      return;
    if(!object) {
      e_name.set_text("");
    } else {
      e_name.set_text(object->get_name());

      c_locked.set_active(object->get_lock());
      
      // geometry
      const Matrix &m = object->get_matrix();
      float width = object->get_width() * m.sc_x();
      float height = object->get_height() * m.sc_y();
      e_left.set(m.tr_x());
      e_bottom.set(m.tr_y());
      e_width.set(width);
      e_height.set(height);
      e_rotate.set(Matrix::rad2deg(m.rot()));

      c_flow.set_active(object->get_flow_around());
      on_c_flow_toggle(); // set_active doesn't emit a signal
      e_margin.set(object->get_obstacle_margin());
    }
  }
  void apply() {
    lock = true; // don't update while applying
    
    object->set_name(e_name.get_text());
    object->set_lock(c_locked.get_active());
    object->set_obstacle_margin(e_margin.get());
    object->set_flow_around(c_flow.get_active());

    object->set_rotation(Matrix::deg2rad(e_rotate.get()));
    object->set_translation(Vector(e_left.get(), e_bottom.get()));

    // resize if possible, scale otherwise
    if(object->is_resizable())
      object->set_size(e_width.get(), e_height.get());
    else {
      object->set_scaling(e_width.get() / object->get_width(), 
			  e_height.get() / object->get_height());
    }

    object->ready_to_draw_signal(object);
    lock = false;
    update();
  }

private:
  bool lock;
  Pagent* object;
  Gtk::HSeparator hsep1, hsep2;
  Gtk::Entry e_name;
  Spinner e_left, e_bottom, e_width, e_height, e_rotate, e_margin;
  Gtk::CheckButton c_flow, c_locked;
  SigC::Connection geometry_changed_connection, props_changed_connection;
};

// - - - - back to the actual PropertiesDialog implementation - - - -

PropertiesDialog *PropertiesDialog::_instance = 0;

PropertiesDialog &PropertiesDialog::instance() {
  if(!_instance)
    _instance = new PropertiesDialog();
  return *_instance;
}

PropertiesDialog::PropertiesDialog()
  : DialogWrap("Object Properties"),
    document(0),
    apply_button(Gtk::Stock::APPLY), close_button(Gtk::Stock::CLOSE)
{
  //  set_border_width(border_width);
  get_vbox()->pack_start(book);

  PropBase* prop = new PropBasic(config.LengthUnit.values.front());
  pages.push_back(prop);
  book.append_page(*prop, prop->getLabel());
  for(PptCore::MetaMap::const_iterator
	i = core.m_begin(); i != core.m_end(); ++i) {
    if(PropBase* prop = i->second->getProp()) {
      pages.push_back(prop);
      book.append_page(*prop, prop->getLabel());
    }
  }
  
  get_action_area()->pack_start(apply_button, false, false);
  get_action_area()->pack_start(close_button, false, false);
  apply_button.signal_clicked().connect
    (slot(*this, &PropertiesDialog::apply));
  close_button.signal_clicked().connect
    (slot(*this, &PropertiesDialog::hide));

  // listen to selection change signals:
  Document::selection_changed_signal.connect
    (slot(*this, &PropertiesDialog::select_change));
}

PropertiesDialog::~PropertiesDialog() {}

void
PropertiesDialog::show_raise() {
  show_all();
  DialogWrap::show_raise();
  
  update();
}

void
PropertiesDialog::apply() {
  if(document)
    for_each(pages.begin(), pages.end(), std::mem_fun(&PropBase::apply));
}

void
PropertiesDialog::set_document(DocRef document_) {
  document = document_;
  update();
}

void
PropertiesDialog::update() {
  Document::Selection all_selected;
  if(document) 
    all_selected = document->selected();

  // Note: If there is more than one object selected, no properties are shown.
  // Maybe we should show the properties that are common
  // to all selected objects.
  Pagent* pagent = (all_selected.size()!=1 ? 0
		    : all_selected.front());

  apply_button.set_sensitive(pagent);

  for(std::vector<PropBase*>::const_iterator i = pages.begin();
      i != pages.end(); ++i)
    (*i)->setObject(pagent);
}

void
PropertiesDialog::select_change(DocRef doc) {
  if(document == doc)
    update();
}
