///
// Copyright (C) 2002 - 2004, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "pdf.h"
#include "fonts/fontmetrics.hh"
#include "fonts/fontmanager.hh"
#include "util/filesys.h"
#include <string>
#include <map>
#include <iomanip>
#include <iostream>
#include <stdexcept>
#include <glibmm/convert.h>
#include <util/stringutil.h>
#include <fstream>

namespace PDF {

  Object::~Object() {
    // Delete managed parts:
    for(Managed::iterator i=managed.begin(); i!=managed.end(); i++)
      delete *i;
  }

  std::ostream& operator << (std::ostream& out, Object& obj) {
    return obj.write(out);
  }
  
  class Header: public Object {
  private:
    struct {int major, minor;} version;
  public:
    Header() {}
    void set_version(int major_version=1, int minor_version=0)
    {version.major=major_version; version.minor=minor_version;}
    
    std::ostream& write(std::ostream& out) {
      out << "%PDF-" << version.major << '.' << version.minor << "\n"
	// Binary (>127) junk comment to make sure that e.g. ftp programs 
	// treat the file as binary data:
	  << "%\xff\xff\xff\xff\n";
      
      return out;
    }
  };
  
  ReferencedObject::ReferencedObject(Ref* ref, Object *object)
    :obj_(object), ref_(ref), offset_(0)
  {}
  
  Ref* ReferencedObject::get_ref() {
    if(!ref_)
      throw std::runtime_error("get_ref for object with no ref yet");
    return ref_;
  }
  
  std::streampos ReferencedObject::get_offset() const {
    if(offset_ <= 0)
      throw std::logic_error("Tried to get offset of unwritten object");
    return offset_;
  }
  
  std::ostream&
  ReferencedObject::write(std::ostream& out) {
    offset_ = out.tellp();
    out << ref_->get_num() << ' ' << ref_->get_generation() << " obj\n"
	<< *obj_
	<< "endobj\n";
    
    return out;
  }
  
  XRefs::XRefs(Document *document)
    : document_(document)
  {}
  
  ReferencedObject* 
  XRefs::add_object(Object* object) {
    refs.push_back(manage(new Ref(refs.size()+1, 0u))); 
    ReferencedObject* ro = new ReferencedObject(refs.back(), object);
    objects.push_back(ro); 

    return ro;
  }
  
  std::streampos
  XRefs::get_xref_offset() { return xref_offset; }
  
  Ref::Num
  XRefs::get_num_of_refs() const {return refs.size();}

  class Name: public Object {
  public:
    explicit Name(std::string name_)
      : name(name_) 
    {}
    bool operator<(Name x) const {return this->name<x.name;} 
    const std::string &get_name() const {return name;}
    std::ostream& write(std::ostream& out) {
      out << '/' << name;
      return out;
    }

  private:
    std::string name;
  };

  template<typename C>
  class BasicObject : public Object {
  public:
    explicit BasicObject(C value)
      : value_(value) 
    {}
    std::ostream& write(std::ostream& out) {
      return out << value_;
    }
  private:
    C value_;
  };
  
  typedef BasicObject<int> Integer;
  typedef BasicObject<float> Real;
  
  class Array: public Object {
  public:
    Array() {}
    
    Object *push_back(Object *object)
      // Todo: manage(object); ???
      {items.push_back(object); return object;}
    
    std::ostream& write(std::ostream& out) {
      Items::iterator i=items.begin();
      out << '[' << **i;
      while(++i != items.end())
	out << ' ' << **i;
      out << ']';
      return out;
    }
    
  private:
    typedef std::vector<Object*> Items;
    Items items;
  };

  Array* rectangle(int llx, int lly, int urx, int ury) {
    Array* result = new Array;
    result->push_back(new Integer(llx));
    result->push_back(new Integer(lly));
    result->push_back(new Integer(urx));
    result->push_back(new Integer(ury));
    return result;
  }
  
  Dictionary::Dictionary() {}
  
  Object *
  Dictionary::set_entry(std::string name, Object *object) 
    {
      entries[name]=object; 
      return object;
    } 
  
  const Object*
  Dictionary::get_entry(const std::string& name) const {
    Entries::const_iterator i = entries.find(name);
    if(i != entries.end())
      return i->second;
    else
      return 0;
  }
  
  std::ostream&
  Dictionary::write(std::ostream& out) {
    out << "<<\n";
    
    for(Entries::const_iterator i=entries.begin(); i!=entries.end(); i++) {
      out << '/' << i->first << ' ' << *i->second << '\n';
    }
    out << ">>\n";
    return out;
  }
  
  std::ostream& Stream::write(std::ostream& out) {
      const std::string streamdata = data_.str();
      set_entry("Length", manage(new Integer(streamdata.length())));
      Dictionary::write(out);
      out << "stream\n"
	  << streamdata
	  << "\nendstream\n";
      return out;
  }
  
  class Page: public Dictionary {
  public:
    Page(int width, int height, Ref* parent, Document *document);
    
    Content *get_content();
    
    /**
     * Make sure a font is loaded in the document, return the font resource
     * name for this page.
     */
    std::string getFont(const font::FontInfo& font);
    
    void set_parent(Ref *ref);
    
  protected:
    Document* document_;
  private:
    Content *content;  
    Dictionary* font_resource;
    int num_reffed_fonts;
  };

  Content::Content(Page* page)
    : page_(page), last_xpos(0), last_ypos(0), 
      cur_charspace(0), cur_wordspace(0)
  { 
    data() << "BT\n";
  }
  
  std::ostream& Content::write(std::ostream& out) {
    commitText();
    data() << "ET";
    return Stream::write(out);
  }
  void Content::selectfont(const font::FontInfo& font) { 
    commitText();
    data() << '/' << page_->getFont(font) << ' ' << font.getSize() << " Tf\n"; 
  }
  void Content::setgray(float gray) { 
    commitText();
    data() << gray << " g\n"; 
  }
  void Content::moveto(float xpos, float ypos) 
  {
    commitText();
    data() << (xpos-last_xpos) << ' ' << (ypos-last_ypos) << " Td\n";
    last_xpos=xpos; last_ypos=ypos;
  }
  void Content::textRise(float rise) {
    commitText();
    data() << rise << " Ts\n";
  }
  
  void Content::setWordSpace(const float& w) {
    if(w != cur_wordspace) {
      commitText();
      cur_wordspace = w;
      data() << w << " Tw\n";
    }
  }
  void Content::setCharSpace(const float& w) {
    if(w != cur_charspace) {
      commitText();
      cur_charspace = w;
      data() << w << " Tc\n";
    }
  }
  
  void Content::show(const Glib::ustring& s) {
    std::ostringstream encoded;
    for(Glib::ustring::const_iterator i = s.begin(); i != s.end(); ++i) {
      if(*i == '\\')
	encoded << "\\\\";
      if(*i == '(')
	encoded << "\\(";
      else if(*i == ')')
	encoded << "\\)";
      else try {
	/// \todo Use a CID-keyed font rather than the standard encoding, as
	/// that seems to be the only way to cover all glyps.
	Glib::ustring t(1, *i);
	encoded << Glib::convert(t.raw(), "MacRoman", "UTF-8");
      } catch(Glib::ConvertError e) {
	// PDF doesn't seem to have anything like glyphshow in postscript.
	encoded << '?';
      }
    }
    textbuf << " (" << encoded.str() << ')';
  } 
  
  void Content::whitespace() {
    textbuf << " ( )";
  }
  
  void Content::whitespace(float width) {
    textbuf << ' ' << ((int) (-100*width));
  }
  
  void Content::commitText() {
    if(!textbuf.str().empty()) {
      data() << '[' << textbuf.str() << "] TJ\n";
      textbuf.str("");
    }
  }
  Page::Page(int width, int height, Ref* parent, Document *document)
    : document_(document), content(0), num_reffed_fonts(0)
    {
      set_entry("Type", manage(new Name("Page")));
      set_entry("Parent", parent);
      
      Dictionary *resources=manage(new Dictionary());
      font_resource = manage(new Dictionary());
      resources->set_entry("Font", font_resource);
      set_entry("Resources", resources);

      Array *mediabox=manage(new Array);
      mediabox->push_back(manage(new Integer(0)));
      mediabox->push_back(manage(new Integer(0)));
      mediabox->push_back(manage(new Integer(width)));
      mediabox->push_back(manage(new Integer(height)));
      set_entry("MediaBox", mediabox);
    }
  
  Content* Page::get_content() {
      if(!content)
	{
	  content=manage(new Content(this));
	  ReferencedObject* ro = document_->get_xrefs()->add_object(content);
	  set_entry("Contents", ro->get_ref());
	}
      return content;
    }
    
  std::string Page::getFont(const font::FontInfo& font) {
    ReferencedObject* rfont = document_->getFontObject(font);
    if(!rfont)
      throw std::runtime_error("Font \"" + font.getName() + "\" not found");
    Dictionary& fd = dynamic_cast<Dictionary&>(*rfont->get_obj());
    
    const std::string resname
      = dynamic_cast<const Name&>(*fd.get_entry("Name")).get_name();
    font_resource->set_entry(resname, rfont->get_ref());
    return resname;
  }
    
  void Page::set_parent(Ref *ref)
    {
      set_entry("Parent", ref);
    }

  class Trailer: public Object {
  public:
    Trailer() {}
    void set_num_of_refs(long num_of_refs_) {num_of_refs=num_of_refs_;}
    void set_catalog_ref(Ref* catalog_ref_) {catalog_ref=catalog_ref_;}
    void set_xref_offset(long xref_offset_) {xref_offset=xref_offset_;}    
    
    std::ostream& write(std::ostream& out) {
      Dictionary dict;
      Integer num_of_refs_object(num_of_refs);
      dict.set_entry("Size", &num_of_refs_object);
      dict.set_entry("Root", catalog_ref);
      out << "trailer\n"
	  << dict
	  << "startxref \n" << xref_offset << " \n%%EOF\n";
      return out;
    }

  private:
    long num_of_refs;
    Ref* catalog_ref;
    std::streampos xref_offset;
  };

  Document::Document()
    : xrefs(this), page_tree_root(xrefs.add_object(new Dictionary()))
  {}

  std::ostream& XRefs::write(std::ostream& out) {
    
    for(Objects::iterator i=objects.begin(); i!=objects.end(); i++)
      out << **i;
    
    xref_offset = out.tellp();
    
    out << "xref \n0 " << (refs.size()+1) << " \n"
	<< std::setw(10) << std::setfill('0') << 0 << " " 
	<< std::setw(0) << std::setfill(' ') << "65535 f \n";
    for(Objects::iterator i=objects.begin(); i!=objects.end(); i++)
      out << std::setw(10) << std::setfill('0') << (*i)->get_offset() 
	  << ' '
	  << std::setw(0) << std::setfill(' ') << "00000 n \n";
    
    return out;
  }
  
  Content* Document::add_page(int width, int height)
  {
    ///\todo There might be a page tree, its not allways just a simple array.
    Page* page = manage(new Page(width, height, page_tree_root->get_ref(),
				 this));
    pages.push_back(xrefs.add_object(page));
    return page->get_content();
  }

  std::ostream& Document::write(std::ostream& out) {
    {
      Header header;
      header.set_version();
      header.write(out);
    }
    
    Dictionary catalog;
    ReferencedObject* rcatalog = xrefs.add_object(&catalog);
    
    catalog.set_entry("Type", manage(new Name("Catalog")));
    catalog.set_entry("Pages", page_tree_root->get_ref());

    Dictionary& root(dynamic_cast<Dictionary&>(*page_tree_root->get_obj()));
    root.set_entry("Type",  manage(new Name("Pages")));
    root.set_entry("Count", manage(new Integer(pages.size())));
    Array kids;
    for(Pages::iterator i=pages.begin(); i!=pages.end(); i++)
      kids.push_back((*i)->get_ref());
    root.set_entry("Kids", &kids);

    xrefs.write(out);
    
    Trailer trailer;
    trailer.set_num_of_refs(xrefs.get_num_of_refs());
    trailer.set_catalog_ref(rcatalog->get_ref());
    trailer.set_xref_offset(xrefs.get_xref_offset());
    trailer.write(out);
    
    return out;
  }

};

// The PDF Reference counts bits as LSB=1.
inline int PdfBit(int pos) {
  return 1 << (pos-1);
}

PDF::Dictionary* createFontDict(const font::FontInfo& fontinfo) {
  const std::string fontname = fontinfo.getName();
  
  // First the basics, needed for all fonts.
  PDF::Dictionary *dict = new PDF::Dictionary();
  dict->set_entry("Type",     new PDF::Name("Font"));
  dict->set_entry("Subtype",  new PDF::Name("Type1"));
  dict->set_entry("BaseFont", new PDF::Name(fontname));
  dict->set_entry("Encoding", new PDF::Name("MacRomanEncoding"));
  
  // If it's one of the 14 standard fonts, we don't have to do more.
  if(!((fontname == "Times-Roman")  || (fontname == "Times-Bold") ||
       (fontname == "Times-Italic") || (fontname == "Times-BoldItalic") ||
       (fontname == "Helvetica")    || (fontname == "Helvetica-BoldOblique") ||
       (fontname == "Helvetica-Oblique") || (fontname == "Helvetica-Bold") ||
       (fontname == "Courier")        || (fontname == "Courier-Bold") ||
       (fontname == "Courier-Oblique")|| (fontname == "Courier-BoldOblique") ||
       (fontname == "Symbol")         || (fontname == "ZapfDingbats"))) {
    
    PDF::Array* widths = new PDF::Array;
    const font::Metrics& metrics(fontinfo.getMetrics());
    for(int i = 0; i <= 254; ++i) {
      try {
	std::string t;
	t.append(1, (char)i);
	Glib::ustring converted = Glib::convert(t, "UTF-8", "MacRoman");
	widths->push_back(new PDF::Integer
			  ((int)(1000*metrics.getWidth(converted))));
      } catch(Glib::ConvertError e) {
	std::cerr << "Failed to convert char #" << i << std::endl;
	// PDF doesn't seem to have anything like glyphshow in postscript.
	widths->push_back(new PDF::Integer(0));
      }
    }
    dict->set_entry("FirstChar", new PDF::Integer(0));
    dict->set_entry("LastChar", new PDF::Integer(254));
    dict->set_entry("Widths", widths);
    

    // And then the FontDescriptor
    PDF::Dictionary* descriptor = new PDF::Dictionary;
    descriptor->set_entry("Type", new PDF::Name("FontDescriptor"));
    descriptor->set_entry("FontName", new PDF::Name(fontname));
    float italicangle = metrics.getItalicAngle();
    descriptor->set_entry("Flags", new PDF::Integer
			  (PdfBit(1) * 0 /* Todo: fixed pitch */ +
			   PdfBit(2) * 1 /* Todo: Serif */ + 
			   PdfBit(3) * 0 /* Todo: Symbolic */ +
			   PdfBit(4) * 0 /* Todo: Script */ +
			   PdfBit(6) * 1 /* !Symbolic */ +
			   PdfBit(7) * (italicangle < -3) /* arbitrary */ +
			   PdfBit(17) * 0 /* AllCap */ +
			   PdfBit(18) * 0 /* SmallCap */ +
			   PdfBit(19) * 0 /* don't force bold */));
    descriptor->set_entry("FontBBox", PDF::rectangle(metrics.getBBox(0),
						     metrics.getBBox(1),
						     metrics.getBBox(2),
						     metrics.getBBox(3)));
    descriptor->set_entry("ItalicAngle", new PDF::Real(italicangle));
    descriptor->set_entry("Ascent", new PDF::Integer
			  (int(1000 * metrics.getAscender())));
    descriptor->set_entry("Descent", new PDF::Integer
			  (int(1000 * metrics.getDescender())));
    descriptor->set_entry("CapHeight", new PDF::Integer
			  (int(1000 * metrics.getCapHeight())));
    descriptor->set_entry("XHeight", new PDF::Integer
			  (int(1000 * metrics.getExHeight())));
    descriptor->set_entry("StemV", new PDF::Integer(0)); // Todo

    // Todo: Actually include the font.  But then I need to compute the three
    // lengths!
//     const std::string fontfilename =
//       fontinfo.getFontManager()->getFontFile(fontname);
//     if(fontfilename.empty())
//       throw std::runtime_error("Couldn't find font file for " + fontname);
//     else if(suffix(fontfilename) == "pfa") {
//       PDF::Stream* fontstream = new PDF::Stream;
//       //fontstream->set_entry("Length1", ...);
//       // Ignore up to "%!"
//       // Copy up to, inclusive "currentfile eexec", lenght1
//       // Copy to, exclusive "000...", length2
//       // Copy rest, length3
//       std::ifstream in(fontfilename.c_str());
//       fontstream->data() << in.rdbuf();
//       descriptor->set_entry("FontFile", fontstream);
//     }
//     dict->set_entry("FontDescriptor", descriptor);
    // Optional: ToUnicode
  }
  
  return dict;
}

PDF::ReferencedObject* 
PDF::Document::getFontObject(const font::FontInfo& font) {
  if(ReferencedObject* used = used_font[font.getName()])
    return used;

  if(Dictionary* fdict = createFontDict(font)) {
    fdict->set_entry("Name", new PDF::Name("F" + tostr(used_font.size()+1)));
    ReferencedObject* used = xrefs.add_object(manage(fdict));
    used_font[font.getName()] = used;
    return used;
  }
  throw std::runtime_error("Failed to get font \"" + font.getName() + "\"");
}


#ifdef STAND_ALONE
int main(int argc, char *argv[])
{
  PDF::Document document;
  document.add_page(200, 300);
  document.selectfont(20);
  document.moveto(20, 20);
  document.show("Hello, World!");
  std::cerr<<document.write(&std::cout)<<" bytes written"<<std::endl;
}
#endif
