/*
 * Copyright (C) 2009 Nokia Corporation, all rights reserved.
 *
 * Author: Zeeshan Ali (Khattak) <zeeshanak@gnome.org>
 *                               <zeeshan.ali@nokia.com>
 *
 * This file is part of Rygel.
 *
 * Rygel is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Rygel is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "rygel-didl-lite-writer.h"
#include <stdlib.h>
#include <string.h>
#include <gee/arraylist.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <gee/collection.h>
#include "rygel-media-item.h"
#include "rygel-media-container.h"
#include <string.h>




struct _RygelDIDLLiteWriterPrivate {
	RygelHTTPServer* http_server;
};

#define RYGEL_DIDL_LITE_WRITER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), RYGEL_TYPE_DIDL_LITE_WRITER, RygelDIDLLiteWriterPrivate))
enum  {
	RYGEL_DIDL_LITE_WRITER_DUMMY_PROPERTY
};
static void rygel_didl_lite_writer_serialize_item (RygelDIDLLiteWriter* self, RygelMediaItem* item, GError** error);
static void rygel_didl_lite_writer_serialize_container (RygelDIDLLiteWriter* self, RygelMediaContainer* container, GError** error);
static char* rygel_didl_lite_writer_get_protocol_for_uri (RygelDIDLLiteWriter* self, const char* uri, GError** error);
static GUPnPDIDLLiteResource* _gupnp_didl_lite_resource_dup (GUPnPDIDLLiteResource* self);
static GeeArrayList* rygel_didl_lite_writer_get_transcoded_resources (RygelDIDLLiteWriter* self, RygelMediaItem* item, GeeArrayList* orig_res_list, GError** error);
static gboolean rygel_didl_lite_writer_http_res_present (RygelDIDLLiteWriter* self, GeeArrayList* res_list);
static GeeArrayList* rygel_didl_lite_writer_get_original_res_list (RygelDIDLLiteWriter* self, RygelMediaItem* item, GError** error);
static GUPnPDIDLLiteResource rygel_didl_lite_writer_create_res (RygelDIDLLiteWriter* self, RygelMediaItem* item, const char* uri, GError** error);
static gpointer rygel_didl_lite_writer_parent_class = NULL;
static void rygel_didl_lite_writer_finalize (GObject* obj);
static int _vala_strcmp0 (const char * str1, const char * str2);



GQuark rygel_didl_lite_writer_error_quark (void) {
	return g_quark_from_static_string ("rygel_didl_lite_writer_error-quark");
}


RygelDIDLLiteWriter* rygel_didl_lite_writer_construct (GType object_type, RygelHTTPServer* http_server) {
	RygelDIDLLiteWriter * self;
	RygelHTTPServer* _tmp1;
	RygelHTTPServer* _tmp0;
	g_return_val_if_fail (http_server != NULL, NULL);
	self = g_object_newv (object_type, 0, NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->http_server = (_tmp1 = (_tmp0 = http_server, (_tmp0 == NULL) ? NULL : g_object_ref (_tmp0)), (self->priv->http_server == NULL) ? NULL : (self->priv->http_server = (g_object_unref (self->priv->http_server), NULL)), _tmp1);
	return self;
}


RygelDIDLLiteWriter* rygel_didl_lite_writer_new (RygelHTTPServer* http_server) {
	return rygel_didl_lite_writer_construct (RYGEL_TYPE_DIDL_LITE_WRITER, http_server);
}


void rygel_didl_lite_writer_serialize (RygelDIDLLiteWriter* self, RygelMediaObject* media_object, GError** error) {
	GError * inner_error;
	g_return_if_fail (self != NULL);
	g_return_if_fail (media_object != NULL);
	inner_error = NULL;
	if (RYGEL_IS_MEDIA_ITEM (media_object)) {
		rygel_didl_lite_writer_serialize_item (self, RYGEL_MEDIA_ITEM (media_object), &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			return;
		}
	} else {
		if (RYGEL_IS_MEDIA_CONTAINER (media_object)) {
			rygel_didl_lite_writer_serialize_container (self, RYGEL_MEDIA_CONTAINER (media_object), &inner_error);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				return;
			}
		} else {
			inner_error = g_error_new_literal (RYGEL_DIDL_LITE_WRITER_ERROR, RYGEL_DIDL_LITE_WRITER_ERROR_UNSUPPORTED_OBJECT, "Unable to serialize unsupported object");
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				return;
			}
		}
	}
}


static void rygel_didl_lite_writer_serialize_item (RygelDIDLLiteWriter* self, RygelMediaItem* item, GError** error) {
	GError * inner_error;
	gboolean _tmp0;
	gboolean _tmp1;
	gboolean _tmp2;
	GeeArrayList* res_list;
	GeeArrayList* trans_res_list;
	g_return_if_fail (self != NULL);
	g_return_if_fail (item != NULL);
	inner_error = NULL;
	gupnp_didl_lite_writer_start_item ((GUPnPDIDLLiteWriter*) self, ((RygelMediaObject*) item)->id, ((RygelMediaObject*) ((RygelMediaObject*) item)->parent)->id, NULL, FALSE);
	/* Add fields */
	gupnp_didl_lite_writer_add_string ((GUPnPDIDLLiteWriter*) self, "title", GUPNP_DIDL_LITE_WRITER_NAMESPACE_DC, NULL, ((RygelMediaObject*) item)->title);
	gupnp_didl_lite_writer_add_string ((GUPnPDIDLLiteWriter*) self, "class", GUPNP_DIDL_LITE_WRITER_NAMESPACE_UPNP, NULL, item->upnp_class);
	_tmp0 = FALSE;
	if (item->author != NULL) {
		_tmp0 = _vala_strcmp0 (item->author, "") != 0;
	} else {
		_tmp0 = FALSE;
	}
	if (_tmp0) {
		gupnp_didl_lite_writer_add_string ((GUPnPDIDLLiteWriter*) self, "creator", GUPNP_DIDL_LITE_WRITER_NAMESPACE_DC, NULL, item->author);
		if (g_str_has_prefix (item->upnp_class, RYGEL_MEDIA_ITEM_VIDEO_CLASS)) {
			gupnp_didl_lite_writer_add_string ((GUPnPDIDLLiteWriter*) self, "author", GUPNP_DIDL_LITE_WRITER_NAMESPACE_UPNP, NULL, item->author);
		} else {
			if (g_str_has_prefix (item->upnp_class, RYGEL_MEDIA_ITEM_MUSIC_CLASS)) {
				gupnp_didl_lite_writer_add_string ((GUPnPDIDLLiteWriter*) self, "artist", GUPNP_DIDL_LITE_WRITER_NAMESPACE_UPNP, NULL, item->author);
			}
		}
	}
	if (item->track_number >= 0) {
		gupnp_didl_lite_writer_add_int ((GUPnPDIDLLiteWriter*) self, "originalTrackNumber", GUPNP_DIDL_LITE_WRITER_NAMESPACE_UPNP, NULL, item->track_number);
	}
	_tmp1 = FALSE;
	if (item->album != NULL) {
		_tmp1 = _vala_strcmp0 (item->album, "") != 0;
	} else {
		_tmp1 = FALSE;
	}
	if (_tmp1) {
		gupnp_didl_lite_writer_add_string ((GUPnPDIDLLiteWriter*) self, "album", GUPNP_DIDL_LITE_WRITER_NAMESPACE_UPNP, NULL, item->album);
	}
	_tmp2 = FALSE;
	if (item->date != NULL) {
		_tmp2 = _vala_strcmp0 (item->date, "") != 0;
	} else {
		_tmp2 = FALSE;
	}
	if (_tmp2) {
		gupnp_didl_lite_writer_add_string ((GUPnPDIDLLiteWriter*) self, "date", GUPNP_DIDL_LITE_WRITER_NAMESPACE_DC, NULL, item->date);
	}
	/* Add resource data */
	res_list = rygel_didl_lite_writer_get_original_res_list (self, item, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return;
	}
	/* Now get the transcoded/proxy URIs */
	trans_res_list = rygel_didl_lite_writer_get_transcoded_resources (self, item, res_list, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(res_list == NULL) ? NULL : (res_list = (g_object_unref (res_list), NULL));
		return;
	}
	{
		GeeIterator* _trans_res_it;
		_trans_res_it = gee_iterable_iterator ((GeeIterable*) trans_res_list);
		while (gee_iterator_next (_trans_res_it)) {
			GUPnPDIDLLiteResource _tmp4 = {0};
			GUPnPDIDLLiteResource* _tmp3 = {0};
			GUPnPDIDLLiteResource _tmp5 = {0};
			GUPnPDIDLLiteResource trans_res;
			trans_res = (_tmp5 = (gupnp_didl_lite_resource_copy (&(*(_tmp3 = (GUPnPDIDLLiteResource*) gee_iterator_get (_trans_res_it))), &_tmp4), _tmp4), (_tmp3 == NULL) ? NULL : (_tmp3 = (g_free (_tmp3), NULL)), _tmp5);
			gupnp_didl_lite_writer_add_res ((GUPnPDIDLLiteWriter*) self, &trans_res);
			gupnp_didl_lite_resource_destroy (&trans_res);
		}
		(_trans_res_it == NULL) ? NULL : (_trans_res_it = (g_object_unref (_trans_res_it), NULL));
	}
	/* Add the original resources in the end */
	{
		GeeIterator* _res_it;
		/* Add the original resources in the end */
		_res_it = gee_iterable_iterator ((GeeIterable*) res_list);
		/* Add the original resources in the end */
		while (gee_iterator_next (_res_it)) {
			GUPnPDIDLLiteResource _tmp7 = {0};
			GUPnPDIDLLiteResource* _tmp6 = {0};
			GUPnPDIDLLiteResource _tmp8 = {0};
			GUPnPDIDLLiteResource res;
			/* Add the original resources in the end */
			res = (_tmp8 = (gupnp_didl_lite_resource_copy (&(*(_tmp6 = (GUPnPDIDLLiteResource*) gee_iterator_get (_res_it))), &_tmp7), _tmp7), (_tmp6 == NULL) ? NULL : (_tmp6 = (g_free (_tmp6), NULL)), _tmp8);
			gupnp_didl_lite_writer_add_res ((GUPnPDIDLLiteWriter*) self, &res);
			gupnp_didl_lite_resource_destroy (&res);
		}
		(_res_it == NULL) ? NULL : (_res_it = (g_object_unref (_res_it), NULL));
	}
	/* End of item */
	gupnp_didl_lite_writer_end_item ((GUPnPDIDLLiteWriter*) self);
	(res_list == NULL) ? NULL : (res_list = (g_object_unref (res_list), NULL));
	(trans_res_list == NULL) ? NULL : (trans_res_list = (g_object_unref (trans_res_list), NULL));
}


static void rygel_didl_lite_writer_serialize_container (RygelDIDLLiteWriter* self, RygelMediaContainer* container, GError** error) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (container != NULL);
	gupnp_didl_lite_writer_start_container ((GUPnPDIDLLiteWriter*) self, ((RygelMediaObject*) container)->id, ((RygelMediaObject*) ((RygelMediaObject*) container)->parent)->id, (gint) container->child_count, FALSE, FALSE);
	gupnp_didl_lite_writer_add_string ((GUPnPDIDLLiteWriter*) self, "class", GUPNP_DIDL_LITE_WRITER_NAMESPACE_UPNP, NULL, "object.container.storageFolder");
	gupnp_didl_lite_writer_add_string ((GUPnPDIDLLiteWriter*) self, "title", GUPNP_DIDL_LITE_WRITER_NAMESPACE_DC, NULL, ((RygelMediaObject*) container)->title);
	/* End of Container */
	gupnp_didl_lite_writer_end_container ((GUPnPDIDLLiteWriter*) self);
}


static char* rygel_didl_lite_writer_get_protocol_for_uri (RygelDIDLLiteWriter* self, const char* uri, GError** error) {
	GError * inner_error;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (uri != NULL, NULL);
	inner_error = NULL;
	if (g_str_has_prefix (uri, "http")) {
		return g_strdup ("http-get");
	} else {
		if (g_str_has_prefix (uri, "file")) {
			return g_strdup ("internal");
		} else {
			if (g_str_has_prefix (uri, "rtsp")) {
				/* FIXME: Assuming that RTSP is always accompanied with RTP over UDP*/
				return g_strdup ("rtsp-rtp-udp");
			} else {
				inner_error = g_error_new (RYGEL_DIDL_LITE_WRITER_ERROR, RYGEL_DIDL_LITE_WRITER_ERROR_UNKNOWN_URI_TYPE, "Failed to probe protocol for URI %s", uri);
				if (inner_error != NULL) {
					g_propagate_error (error, inner_error);
					return NULL;
				}
			}
		}
	}
}


static GUPnPDIDLLiteResource* _gupnp_didl_lite_resource_dup (GUPnPDIDLLiteResource* self) {
	GUPnPDIDLLiteResource* dup;
	dup = g_new0 (GUPnPDIDLLiteResource, 1);
	gupnp_didl_lite_resource_copy (self, dup);
	return dup;
}


/* FIXME: We only proxy URIs through our HTTP server for now*/
static GeeArrayList* rygel_didl_lite_writer_get_transcoded_resources (RygelDIDLLiteWriter* self, RygelMediaItem* item, GeeArrayList* orig_res_list, GError** error) {
	GError * inner_error;
	GeeArrayList* resources;
	char* uri;
	GUPnPDIDLLiteResource res;
	char* _tmp1;
	GeeArrayList* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (item != NULL, NULL);
	g_return_val_if_fail (orig_res_list != NULL, NULL);
	inner_error = NULL;
	resources = gee_array_list_new (GUPNP_TYPE_DIDL_LITE_RESOURCE, (GBoxedCopyFunc) _gupnp_didl_lite_resource_dup, g_free, g_direct_equal);
	if (rygel_didl_lite_writer_http_res_present (self, orig_res_list)) {
		return resources;
	}
	/* Create the HTTP URI*/
	uri = rygel_http_server_create_http_uri_for_item (self->priv->http_server, item);
	res = rygel_didl_lite_writer_create_res (self, item, uri, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(resources == NULL) ? NULL : (resources = (g_object_unref (resources), NULL));
		uri = (g_free (uri), NULL);
		return NULL;
	}
	_tmp1 = NULL;
	res.protocol = (_tmp1 = g_strdup ("http-get"), res.protocol = (g_free (res.protocol), NULL), _tmp1);
	gee_collection_add ((GeeCollection*) resources, &res);
	_tmp2 = NULL;
	return (_tmp2 = resources, uri = (g_free (uri), NULL), gupnp_didl_lite_resource_destroy (&res), _tmp2);
}


static gboolean rygel_didl_lite_writer_http_res_present (RygelDIDLLiteWriter* self, GeeArrayList* res_list) {
	gboolean present;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (res_list != NULL, FALSE);
	present = FALSE;
	{
		GeeIterator* _res_it;
		_res_it = gee_iterable_iterator ((GeeIterable*) res_list);
		while (gee_iterator_next (_res_it)) {
			GUPnPDIDLLiteResource* res;
			res = (GUPnPDIDLLiteResource*) gee_iterator_get (_res_it);
			if (_vala_strcmp0 ((*res).protocol, "http-get") == 0) {
				present = TRUE;
				(res == NULL) ? NULL : (res = (g_free (res), NULL));
				break;
			}
			(res == NULL) ? NULL : (res = (g_free (res), NULL));
		}
		(_res_it == NULL) ? NULL : (_res_it = (g_object_unref (_res_it), NULL));
	}
	return present;
}


static GeeArrayList* rygel_didl_lite_writer_get_original_res_list (RygelDIDLLiteWriter* self, RygelMediaItem* item, GError** error) {
	GError * inner_error;
	GeeArrayList* resources;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (item != NULL, NULL);
	inner_error = NULL;
	resources = gee_array_list_new (GUPNP_TYPE_DIDL_LITE_RESOURCE, (GBoxedCopyFunc) _gupnp_didl_lite_resource_dup, g_free, g_direct_equal);
	{
		GeeIterator* _uri_it;
		_uri_it = gee_iterable_iterator ((GeeIterable*) item->uris);
		while (gee_iterator_next (_uri_it)) {
			char* uri;
			GUPnPDIDLLiteResource res;
			uri = (char*) gee_iterator_get (_uri_it);
			res = rygel_didl_lite_writer_create_res (self, item, uri, &inner_error);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				uri = (g_free (uri), NULL);
				(_uri_it == NULL) ? NULL : (_uri_it = (g_object_unref (_uri_it), NULL));
				(resources == NULL) ? NULL : (resources = (g_object_unref (resources), NULL));
				return NULL;
			}
			gee_collection_add ((GeeCollection*) resources, &res);
			uri = (g_free (uri), NULL);
			gupnp_didl_lite_resource_destroy (&res);
		}
		(_uri_it == NULL) ? NULL : (_uri_it = (g_object_unref (_uri_it), NULL));
	}
	return resources;
}


static GUPnPDIDLLiteResource rygel_didl_lite_writer_create_res (RygelDIDLLiteWriter* self, RygelMediaItem* item, const char* uri, GError** error) {
	GError * inner_error;
	GUPnPDIDLLiteResource _tmp0 = {0};
	GUPnPDIDLLiteResource res;
	char* _tmp2;
	const char* _tmp1;
	char* _tmp4;
	const char* _tmp3;
	char* _tmp7;
	inner_error = NULL;
	res = (memset (&_tmp0, 0, sizeof (GUPnPDIDLLiteResource)), _tmp0);
	gupnp_didl_lite_resource_reset (&res);
	_tmp2 = NULL;
	_tmp1 = NULL;
	res.uri = (_tmp2 = (_tmp1 = uri, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), res.uri = (g_free (res.uri), NULL), _tmp2);
	_tmp4 = NULL;
	_tmp3 = NULL;
	res.mime_type = (_tmp4 = (_tmp3 = item->mime_type, (_tmp3 == NULL) ? NULL : g_strdup (_tmp3)), res.mime_type = (g_free (res.mime_type), NULL), _tmp4);
	res.size = item->size;
	res.duration = item->duration;
	res.bitrate = item->bitrate;
	res.sample_freq = item->sample_freq;
	res.bits_per_sample = item->bits_per_sample;
	res.n_audio_channels = item->n_audio_channels;
	res.width = item->width;
	res.height = item->height;
	res.color_depth = item->color_depth;
	/* Protocol info */
	if (res.uri != NULL) {
		char* protocol;
		char* _tmp6;
		const char* _tmp5;
		protocol = rygel_didl_lite_writer_get_protocol_for_uri (self, res.uri, &inner_error);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			gupnp_didl_lite_resource_destroy (&res);
			return;
		}
		_tmp6 = NULL;
		_tmp5 = NULL;
		res.protocol = (_tmp6 = (_tmp5 = protocol, (_tmp5 == NULL) ? NULL : g_strdup (_tmp5)), res.protocol = (g_free (res.protocol), NULL), _tmp6);
		protocol = (g_free (protocol), NULL);
	}
	/* DLNA related fields */
	_tmp7 = NULL;
	res.dlna_profile = (_tmp7 = g_strdup ("MP3"), res.dlna_profile = (g_free (res.dlna_profile), NULL), _tmp7);
	/* FIXME */
	if (g_str_has_prefix (item->upnp_class, RYGEL_MEDIA_ITEM_IMAGE_CLASS)) {
		res.dlna_flags = res.dlna_flags | GUPNP_DLNA_FLAG_INTERACTIVE_TRANSFER_MODE;
	} else {
		res.dlna_flags = res.dlna_flags | GUPNP_DLNA_FLAG_STREAMING_TRANSFER_MODE;
	}
	if (res.size > 0) {
		res.dlna_operation = GUPNP_DLNA_OPERATION_RANGE;
		res.dlna_flags = res.dlna_flags | GUPNP_DLNA_FLAG_BACKGROUND_TRANSFER_MODE;
	}
	return res;
}


static void rygel_didl_lite_writer_class_init (RygelDIDLLiteWriterClass * klass) {
	rygel_didl_lite_writer_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (RygelDIDLLiteWriterPrivate));
	G_OBJECT_CLASS (klass)->finalize = rygel_didl_lite_writer_finalize;
}


static void rygel_didl_lite_writer_instance_init (RygelDIDLLiteWriter * self) {
	self->priv = RYGEL_DIDL_LITE_WRITER_GET_PRIVATE (self);
}


static void rygel_didl_lite_writer_finalize (GObject* obj) {
	RygelDIDLLiteWriter * self;
	self = RYGEL_DIDL_LITE_WRITER (obj);
	(self->priv->http_server == NULL) ? NULL : (self->priv->http_server = (g_object_unref (self->priv->http_server), NULL));
	G_OBJECT_CLASS (rygel_didl_lite_writer_parent_class)->finalize (obj);
}


GType rygel_didl_lite_writer_get_type (void) {
	static GType rygel_didl_lite_writer_type_id = 0;
	if (rygel_didl_lite_writer_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (RygelDIDLLiteWriterClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) rygel_didl_lite_writer_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (RygelDIDLLiteWriter), 0, (GInstanceInitFunc) rygel_didl_lite_writer_instance_init, NULL };
		rygel_didl_lite_writer_type_id = g_type_register_static (GUPNP_TYPE_DIDL_LITE_WRITER, "RygelDIDLLiteWriter", &g_define_type_info, 0);
	}
	return rygel_didl_lite_writer_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




