/*
 * Copyright (C) 2008, 2009 Nokia Corporation, all rights reserved.
 * Copyright (C) 2006, 2007, 2008 OpenedHand Ltd.
 *
 * Author: Zeeshan Ali (Khattak) <zeeshanak@gnome.org>
 *                               <zeeshan.ali@nokia.com>
 *         Jorn Baayen <jorn.baayen@gmail.com>
 *
 * This file is part of Rygel.
 *
 * Rygel is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Rygel is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include <rygel-http-server.h>
#include <gee/arraylist.h>
#include <libsoup/soup.h>
#include <libgupnp-av/gupnp-av.h>
#include <gee/collection.h>
#include <rygel-http-response.h>
#include <rygel-media-item.h>
#include <rygel-media-object.h>
#include <rygel-seekable-response.h>
#include <rygel-live-response.h>




struct _RygelHTTPServerPrivate {
	char* path_root;
	GUPnPContext* context;
	GeeArrayList* responses;
};

#define RYGEL_HTTP_SERVER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), RYGEL_TYPE_HTTP_SERVER, RygelHTTPServerPrivate))
enum  {
	RYGEL_HTTP_SERVER_DUMMY_PROPERTY
};
#define RYGEL_HTTP_SERVER_SERVER_PATH_PREFIX "/RygelHTTPServer"
static void _rygel_http_server_server_handler_soup_server_callback (SoupServer* server, SoupMessage* msg, const char* path, GHashTable* query, SoupClientContext* client, gpointer self);
static char* rygel_http_server_create_uri_for_path (RygelHTTPServer* self, const char* path);
static void _rygel_http_server_on_response_ended_rygel_http_response_ended (RygelLiveResponse* _sender, gpointer self);
static void rygel_http_server_stream_from_gst_source (RygelHTTPServer* self, GstElement* src, SoupMessage* msg, GError** error);
static void rygel_http_server_serve_uri (RygelHTTPServer* self, const char* uri, SoupMessage* msg, RygelSeek* seek, gsize size, GError** error);
static void rygel_http_server_on_response_ended (RygelHTTPServer* self, RygelHTTPResponse* response);
static void rygel_http_server_server_handler (RygelHTTPServer* self, SoupServer* server, SoupMessage* msg, const char* server_path, GHashTable* query, SoupClientContext* soup_client);
static void rygel_http_server_handle_item_request (RygelHTTPServer* self, SoupMessage* msg, const char* item_id);
static void rygel_http_server_add_item_headers (RygelHTTPServer* self, SoupMessage* msg, RygelMediaItem* item, RygelSeek* seek);
static inline void _dynamic_set_tcp_timeout0 (GstElement* obj, gint64 value);
static void rygel_http_server_handle_streaming_item (RygelHTTPServer* self, SoupMessage* msg, RygelMediaItem* item);
static void rygel_http_server_handle_interactive_item (RygelHTTPServer* self, SoupMessage* msg, RygelMediaItem* item, RygelSeek* seek);
static RygelSeek* rygel_http_server_parse_range (RygelHTTPServer* self, SoupMessage* message, RygelMediaItem* item, GError** error);
static gpointer rygel_http_server_parent_class = NULL;
static void rygel_http_server_finalize (GObject* obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static gint _vala_array_length (gpointer array);
static int _vala_strcmp0 (const char * str1, const char * str2);


static void g_cclosure_user_marshal_VOID__POINTER_POINTER (GClosure * closure, GValue * return_value, guint n_param_values, const GValue * param_values, gpointer invocation_hint, gpointer marshal_data);
static void g_cclosure_user_marshal_VOID__STRING_POINTER (GClosure * closure, GValue * return_value, guint n_param_values, const GValue * param_values, gpointer invocation_hint, gpointer marshal_data);

GQuark rygel_http_server_error_quark (void) {
	return g_quark_from_static_string ("rygel_http_server_error-quark");
}


static void _rygel_http_server_server_handler_soup_server_callback (SoupServer* server, SoupMessage* msg, const char* path, GHashTable* query, SoupClientContext* client, gpointer self) {
	rygel_http_server_server_handler (self, server, msg, path, query, client);
}


RygelHTTPServer* rygel_http_server_construct (GType object_type, GUPnPContext* context, const char* name) {
	RygelHTTPServer * self;
	GUPnPContext* _tmp1;
	GUPnPContext* _tmp0;
	GeeArrayList* _tmp2;
	char* _tmp3;
	g_return_val_if_fail (context != NULL, NULL);
	g_return_val_if_fail (name != NULL, NULL);
	self = g_object_newv (object_type, 0, NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->context = (_tmp1 = (_tmp0 = context, (_tmp0 == NULL) ? NULL : g_object_ref (_tmp0)), (self->priv->context == NULL) ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL)), _tmp1);
	_tmp2 = NULL;
	self->priv->responses = (_tmp2 = gee_array_list_new (RYGEL_TYPE_HTTP_RESPONSE, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_direct_equal), (self->priv->responses == NULL) ? NULL : (self->priv->responses = (g_object_unref (self->priv->responses), NULL)), _tmp2);
	_tmp3 = NULL;
	self->priv->path_root = (_tmp3 = g_strconcat (RYGEL_HTTP_SERVER_SERVER_PATH_PREFIX "/", name, NULL), self->priv->path_root = (g_free (self->priv->path_root), NULL), _tmp3);
	soup_server_add_handler (gupnp_context_get_server (context), self->priv->path_root, _rygel_http_server_server_handler_soup_server_callback, g_object_ref (self), g_object_unref);
	return self;
}


RygelHTTPServer* rygel_http_server_new (GUPnPContext* context, const char* name) {
	return rygel_http_server_construct (RYGEL_TYPE_HTTP_SERVER, context, name);
}


void rygel_http_server_destroy (RygelHTTPServer* self) {
	g_return_if_fail (self != NULL);
	soup_server_remove_handler (gupnp_context_get_server (self->priv->context), self->priv->path_root);
}


static char* rygel_http_server_create_uri_for_path (RygelHTTPServer* self, const char* path) {
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (path != NULL, NULL);
	return g_strdup_printf ("http://%s:%u%s%s", gupnp_context_get_host_ip (self->priv->context), gupnp_context_get_port (self->priv->context), self->priv->path_root, path);
}


char* rygel_http_server_create_http_uri_for_item (RygelHTTPServer* self, RygelMediaItem* item) {
	char* escaped;
	char* query;
	char* _tmp0;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (item != NULL, NULL);
	escaped = g_uri_escape_string (((RygelMediaObject*) item)->id, "", TRUE);
	query = g_strdup_printf ("?itemid=%s", escaped);
	_tmp0 = NULL;
	return (_tmp0 = rygel_http_server_create_uri_for_path (self, query), escaped = (g_free (escaped), NULL), query = (g_free (query), NULL), _tmp0);
}


static void _rygel_http_server_on_response_ended_rygel_http_response_ended (RygelLiveResponse* _sender, gpointer self) {
	rygel_http_server_on_response_ended (self, _sender);
}


static void rygel_http_server_stream_from_gst_source (RygelHTTPServer* self, GstElement* src, SoupMessage* msg, GError** error) {
	GError * inner_error;
	RygelLiveResponse* response;
	g_return_if_fail (self != NULL);
	g_return_if_fail (src != NULL);
	g_return_if_fail (msg != NULL);
	inner_error = NULL;
	response = rygel_live_response_new (gupnp_context_get_server (self->priv->context), msg, "RygelLiveResponse", src, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		(src == NULL) ? NULL : (src = (gst_object_unref (src), NULL));
		return;
	}
	rygel_live_response_start (response);
	g_signal_connect_object ((RygelHTTPResponse*) response, "ended", (GCallback) _rygel_http_server_on_response_ended_rygel_http_response_ended, self, 0);
	gee_collection_add ((GeeCollection*) self->priv->responses, (RygelHTTPResponse*) response);
	(response == NULL) ? NULL : (response = (g_object_unref (response), NULL));
	(src == NULL) ? NULL : (src = (gst_object_unref (src), NULL));
}


static void rygel_http_server_serve_uri (RygelHTTPServer* self, const char* uri, SoupMessage* msg, RygelSeek* seek, gsize size, GError** error) {
	GError * inner_error;
	RygelSeekableResponse* response;
	g_return_if_fail (self != NULL);
	g_return_if_fail (uri != NULL);
	g_return_if_fail (msg != NULL);
	inner_error = NULL;
	response = rygel_seekable_response_new (gupnp_context_get_server (self->priv->context), msg, uri, seek, size, &inner_error);
	if (inner_error != NULL) {
		g_propagate_error (error, inner_error);
		return;
	}
	g_signal_connect_object ((RygelHTTPResponse*) response, "ended", (GCallback) _rygel_http_server_on_response_ended_rygel_http_response_ended, self, 0);
	gee_collection_add ((GeeCollection*) self->priv->responses, (RygelHTTPResponse*) response);
	(response == NULL) ? NULL : (response = (g_object_unref (response), NULL));
}


static void rygel_http_server_on_response_ended (RygelHTTPServer* self, RygelHTTPResponse* response) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (response != NULL);
	/* Remove the response from our list. */
	gee_collection_remove ((GeeCollection*) self->priv->responses, response);
}


static void rygel_http_server_server_handler (RygelHTTPServer* self, SoupServer* server, SoupMessage* msg, const char* server_path, GHashTable* query, SoupClientContext* soup_client) {
	gboolean _tmp0;
	char* _tmp2;
	char* _tmp1;
	gboolean _tmp3;
	char* item_id;
	g_return_if_fail (self != NULL);
	g_return_if_fail (server != NULL);
	g_return_if_fail (msg != NULL);
	g_return_if_fail (server_path != NULL);
	g_return_if_fail (soup_client != NULL);
	_tmp0 = FALSE;
	_tmp2 = NULL;
	_tmp1 = NULL;
	if ((_tmp3 = _vala_strcmp0 (_tmp2 = (g_object_get (msg, "method", &_tmp1, NULL), _tmp1), "HEAD") != 0, _tmp2 = (g_free (_tmp2), NULL), _tmp3)) {
		char* _tmp5;
		char* _tmp4;
		_tmp5 = NULL;
		_tmp4 = NULL;
		_tmp0 = _vala_strcmp0 (_tmp5 = (g_object_get (msg, "method", &_tmp4, NULL), _tmp4), "GET") != 0;
		_tmp5 = (g_free (_tmp5), NULL);
	} else {
		_tmp0 = FALSE;
	}
	if (_tmp0) {
		/* We only entertain 'HEAD' and 'GET' requests */
		soup_message_set_status (msg, (guint) SOUP_STATUS_BAD_REQUEST);
		return;
	}
	item_id = NULL;
	if (query != NULL) {
		char* _tmp7;
		const char* _tmp6;
		_tmp7 = NULL;
		_tmp6 = NULL;
		item_id = (_tmp7 = (_tmp6 = (const char*) g_hash_table_lookup (query, "itemid"), (_tmp6 == NULL) ? NULL : g_strdup (_tmp6)), item_id = (g_free (item_id), NULL), _tmp7);
	}
	if (item_id == NULL) {
		soup_message_set_status (msg, (guint) SOUP_STATUS_NOT_FOUND);
		item_id = (g_free (item_id), NULL);
		return;
	}
	rygel_http_server_handle_item_request (self, msg, item_id);
	item_id = (g_free (item_id), NULL);
}


static void rygel_http_server_handle_item_request (RygelHTTPServer* self, SoupMessage* msg, const char* item_id) {
	GError * inner_error;
	RygelMediaItem* item;
	RygelMediaItem* _tmp1;
	RygelMediaItem* _tmp0;
	RygelSeek* seek;
	char* _tmp5;
	char* _tmp4;
	gboolean _tmp6;
	g_return_if_fail (self != NULL);
	g_return_if_fail (msg != NULL);
	g_return_if_fail (item_id != NULL);
	inner_error = NULL;
	item = NULL;
	/* Signal the requestion for an item*/
	_tmp1 = NULL;
	_tmp0 = NULL;
	g_signal_emit_by_name (self, "item-requested", item_id, &_tmp0);
	item = (_tmp1 = _tmp0, (item == NULL) ? NULL : (item = (rygel_media_object_unref (item), NULL)), _tmp1);
	if (item == NULL) {
		g_warning ("rygel-http-server.vala:137: Requested item '%s' not found\n", item_id);
		soup_message_set_status (msg, (guint) SOUP_STATUS_NOT_FOUND);
		(item == NULL) ? NULL : (item = (rygel_media_object_unref (item), NULL));
		return;
	}
	seek = NULL;
	{
		RygelSeek* _tmp2;
		RygelSeek* _tmp3;
		_tmp2 = rygel_http_server_parse_range (self, msg, item, &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == RYGEL_HTTP_SERVER_ERROR) {
				goto __catch1_rygel_http_server_error;
			}
			goto __finally1;
		}
		_tmp3 = NULL;
		seek = (_tmp3 = _tmp2, (seek == NULL) ? NULL : (seek = (g_object_unref (seek), NULL)), _tmp3);
	}
	goto __finally1;
	__catch1_rygel_http_server_error:
	{
		GError * err;
		err = inner_error;
		inner_error = NULL;
		{
			g_warning ("rygel-http-server.vala:147: %s", err->message);
			soup_message_set_status (msg, (guint) err->code);
			(err == NULL) ? NULL : (err = (g_error_free (err), NULL));
			(item == NULL) ? NULL : (item = (rygel_media_object_unref (item), NULL));
			(seek == NULL) ? NULL : (seek = (g_object_unref (seek), NULL));
			return;
		}
	}
	__finally1:
	if (inner_error != NULL) {
		(item == NULL) ? NULL : (item = (rygel_media_object_unref (item), NULL));
		(seek == NULL) ? NULL : (seek = (g_object_unref (seek), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return;
	}
	/* Add headers*/
	rygel_http_server_add_item_headers (self, msg, item, seek);
	_tmp5 = NULL;
	_tmp4 = NULL;
	if ((_tmp6 = _vala_strcmp0 (_tmp5 = (g_object_get (msg, "method", &_tmp4, NULL), _tmp4), "HEAD") == 0, _tmp5 = (g_free (_tmp5), NULL), _tmp6)) {
		/* Only headers requested, no need to send contents*/
		soup_message_set_status (msg, (guint) SOUP_STATUS_OK);
		(item == NULL) ? NULL : (item = (rygel_media_object_unref (item), NULL));
		(seek == NULL) ? NULL : (seek = (g_object_unref (seek), NULL));
		return;
	}
	if (item->res.size > 0) {
		rygel_http_server_handle_interactive_item (self, msg, item, seek);
	} else {
		rygel_http_server_handle_streaming_item (self, msg, item);
	}
	(item == NULL) ? NULL : (item = (rygel_media_object_unref (item), NULL));
	(seek == NULL) ? NULL : (seek = (g_object_unref (seek), NULL));
}


static void rygel_http_server_add_item_headers (RygelHTTPServer* self, SoupMessage* msg, RygelMediaItem* item, RygelSeek* seek) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (msg != NULL);
	g_return_if_fail (item != NULL);
	if (item->res.mime_type != NULL) {
		soup_message_headers_append (msg->response_headers, "Content-Type", item->res.mime_type);
	}
	if (item->res.size >= 0) {
		char* _tmp0;
		_tmp0 = NULL;
		soup_message_headers_append (msg->response_headers, "Content-Length", _tmp0 = g_strdup_printf ("%li", item->res.size));
		_tmp0 = (g_free (_tmp0), NULL);
	}
	if ((item->res.dlna_operation & GUPNP_DLNA_OPERATION_RANGE) == GUPNP_DLNA_OPERATION_RANGE) {
		soup_message_headers_append (msg->response_headers, "Accept-Ranges", "bytes");
	}
	if (item->res.size > 0) {
		gint64 first_byte;
		gint64 last_byte;
		char* _tmp7;
		char* _tmp6;
		char* _tmp5;
		char* _tmp4;
		char* _tmp3;
		char* _tmp2;
		char* _tmp1;
		char* _tmp8;
		char* content_range;
		first_byte = 0LL;
		last_byte = 0LL;
		if (seek != NULL) {
			first_byte = rygel_seek_get_start (seek);
			last_byte = rygel_seek_get_stop (seek);
		} else {
			first_byte = (gint64) 0;
			last_byte = (gint64) (item->res.size - 1);
		}
		/* Content-Range: bytes START_BYTE-STOP_BYTE/TOTAL_LENGTH*/
		_tmp7 = NULL;
		_tmp6 = NULL;
		_tmp5 = NULL;
		_tmp4 = NULL;
		_tmp3 = NULL;
		_tmp2 = NULL;
		_tmp1 = NULL;
		_tmp8 = NULL;
		content_range = (_tmp8 = g_strconcat (_tmp6 = g_strconcat (_tmp5 = g_strconcat (_tmp3 = g_strconcat (_tmp2 = g_strconcat ("bytes ", _tmp1 = g_strdup_printf ("%lli", first_byte), NULL), "-", NULL), _tmp4 = g_strdup_printf ("%lli", last_byte), NULL), "/", NULL), _tmp7 = g_strdup_printf ("%li", item->res.size), NULL), _tmp7 = (g_free (_tmp7), NULL), _tmp6 = (g_free (_tmp6), NULL), _tmp5 = (g_free (_tmp5), NULL), _tmp4 = (g_free (_tmp4), NULL), _tmp3 = (g_free (_tmp3), NULL), _tmp2 = (g_free (_tmp2), NULL), _tmp1 = (g_free (_tmp1), NULL), _tmp8);
		soup_message_headers_append (msg->response_headers, "Content-Range", content_range);
		content_range = (g_free (content_range), NULL);
	}
}


static inline void _dynamic_set_tcp_timeout0 (GstElement* obj, gint64 value) {
	g_object_set (obj, "tcp-timeout", value, NULL);
}


static void rygel_http_server_handle_streaming_item (RygelHTTPServer* self, SoupMessage* msg, RygelMediaItem* item) {
	GError * inner_error;
	const char* _tmp0;
	char* uri;
	GstElement* src;
	g_return_if_fail (self != NULL);
	g_return_if_fail (msg != NULL);
	g_return_if_fail (item != NULL);
	inner_error = NULL;
	_tmp0 = NULL;
	uri = (_tmp0 = item->res.uri, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0));
	src = NULL;
	if (uri != NULL) {
		GstElement* _tmp1;
		/* URI provided, try to create source element from it*/
		_tmp1 = NULL;
		src = (_tmp1 = gst_element_make_from_uri (GST_URI_SRC, uri, NULL), (src == NULL) ? NULL : (src = (gst_object_unref (src), NULL)), _tmp1);
	} else {
		GstElement* _tmp3;
		GstElement* _tmp2;
		/* No URI provided, ask for source element directly*/
		_tmp3 = NULL;
		_tmp2 = NULL;
		g_signal_emit_by_name (self, "need-stream-source", item, &_tmp2);
		src = (_tmp3 = _tmp2, (src == NULL) ? NULL : (src = (gst_object_unref (src), NULL)), _tmp3);
	}
	if (src == NULL) {
		g_warning ("rygel-http-server.vala:219: Failed to create source element for item: %s\n", ((RygelMediaObject*) item)->id);
		soup_message_set_status (msg, (guint) SOUP_STATUS_NOT_FOUND);
		uri = (g_free (uri), NULL);
		(src == NULL) ? NULL : (src = (gst_object_unref (src), NULL));
		return;
	}
	/* For rtspsrc since some RTSP sources takes a while to start
	 transmitting*/
	_dynamic_set_tcp_timeout0 (src, (gint64) 60000000);
	{
		GstElement* _tmp4;
		/* Then start the gst stream*/
		_tmp4 = NULL;
		rygel_http_server_stream_from_gst_source (self, (_tmp4 = src, (_tmp4 == NULL) ? NULL : gst_object_ref (_tmp4)), msg, &inner_error);
		if (inner_error != NULL) {
			goto __catch2_g_error;
			goto __finally2;
		}
	}
	goto __finally2;
	__catch2_g_error:
	{
		GError * error;
		error = inner_error;
		inner_error = NULL;
		{
			g_critical ("rygel-http-server.vala:233: Error in attempting to start streaming %s: %s", uri, error->message);
			(error == NULL) ? NULL : (error = (g_error_free (error), NULL));
		}
	}
	__finally2:
	if (inner_error != NULL) {
		uri = (g_free (uri), NULL);
		(src == NULL) ? NULL : (src = (gst_object_unref (src), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return;
	}
	uri = (g_free (uri), NULL);
	(src == NULL) ? NULL : (src = (gst_object_unref (src), NULL));
}


static void rygel_http_server_handle_interactive_item (RygelHTTPServer* self, SoupMessage* msg, RygelMediaItem* item, RygelSeek* seek) {
	GError * inner_error;
	const char* _tmp0;
	char* uri;
	g_return_if_fail (self != NULL);
	g_return_if_fail (msg != NULL);
	g_return_if_fail (item != NULL);
	inner_error = NULL;
	_tmp0 = NULL;
	uri = (_tmp0 = item->res.uri, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0));
	if (uri == NULL) {
		g_warning ("rygel-http-server.vala:245: Requested item '%s' didn't provide a URI\n", ((RygelMediaObject*) item)->id);
		soup_message_set_status (msg, (guint) SOUP_STATUS_NOT_FOUND);
		uri = (g_free (uri), NULL);
		return;
	}
	{
		rygel_http_server_serve_uri (self, uri, msg, seek, (gsize) item->res.size, &inner_error);
		if (inner_error != NULL) {
			goto __catch3_g_error;
			goto __finally3;
		}
	}
	goto __finally3;
	__catch3_g_error:
	{
		GError * error;
		error = inner_error;
		inner_error = NULL;
		{
			g_warning ("rygel-http-server.vala:253: Error in attempting to serve %s: %s", uri, error->message);
			soup_message_set_status (msg, (guint) SOUP_STATUS_NOT_FOUND);
			(error == NULL) ? NULL : (error = (g_error_free (error), NULL));
		}
	}
	__finally3:
	if (inner_error != NULL) {
		uri = (g_free (uri), NULL);
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return;
	}
	uri = (g_free (uri), NULL);
}


/* Parses the HTTP Range header on @message and sets:
     *
     * @offset to the requested offset (left unchanged if none specified),
     * @length to the requested length (left unchanged if none specified).
     *
     * Both @offset and @length are expected to be initialised to their default
     * values. Throws a #HTTPServerError in case of error.
     *
     * Returns %true a range header was found, false otherwise. */
static RygelSeek* rygel_http_server_parse_range (RygelHTTPServer* self, SoupMessage* message, RygelMediaItem* item, GError** error) {
	GError * inner_error;
	char* range;
	gint range_tokens_size;
	gint range_tokens_length1;
	char** range_tokens;
	RygelSeek* seek;
	char* _tmp1;
	const char* _tmp0;
	char** _tmp4;
	char** _tmp3;
	gboolean _tmp5;
	RygelSeek* _tmp6;
	const char* _tmp7;
	char* first_byte;
	const char* _tmp8;
	char* last_byte;
	RygelSeek* _tmp13;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (message != NULL, NULL);
	g_return_val_if_fail (item != NULL, NULL);
	inner_error = NULL;
	range = NULL;
	range_tokens = (range_tokens_length1 = 0, NULL);
	seek = NULL;
	_tmp1 = NULL;
	_tmp0 = NULL;
	range = (_tmp1 = (_tmp0 = soup_message_headers_get (message->request_headers, "Range"), (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), range = (g_free (range), NULL), _tmp1);
	if (range == NULL) {
		RygelSeek* _tmp2;
		_tmp2 = NULL;
		return (_tmp2 = seek, range = (g_free (range), NULL), range_tokens = (_vala_array_free (range_tokens, range_tokens_length1, (GDestroyNotify) g_free), NULL), _tmp2);
	}
	/* We have a Range header. Parse.*/
	if (!g_str_has_prefix (range, "bytes=")) {
		inner_error = g_error_new (RYGEL_HTTP_SERVER_ERROR, RYGEL_HTTP_SERVER_ERROR_INVALID_RANGE, "Invalid Range '%s'", range);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			range = (g_free (range), NULL);
			range_tokens = (_vala_array_free (range_tokens, range_tokens_length1, (GDestroyNotify) g_free), NULL);
			(seek == NULL) ? NULL : (seek = (g_object_unref (seek), NULL));
			return NULL;
		}
	}
	_tmp4 = NULL;
	_tmp3 = NULL;
	range_tokens = (_tmp4 = _tmp3 = g_strsplit (g_utf8_offset_to_pointer (range, (glong) 6), "-", 2), range_tokens = (_vala_array_free (range_tokens, range_tokens_length1, (GDestroyNotify) g_free), NULL), range_tokens_length1 = _vala_array_length (_tmp3), range_tokens_size = range_tokens_length1, _tmp4);
	_tmp5 = FALSE;
	if (range_tokens[0] == NULL) {
		_tmp5 = TRUE;
	} else {
		_tmp5 = range_tokens[1] == NULL;
	}
	if (_tmp5) {
		inner_error = g_error_new (RYGEL_HTTP_SERVER_ERROR, RYGEL_HTTP_SERVER_ERROR_INVALID_RANGE, "Invalid Range '%s'", range);
		if (inner_error != NULL) {
			g_propagate_error (error, inner_error);
			range = (g_free (range), NULL);
			range_tokens = (_vala_array_free (range_tokens, range_tokens_length1, (GDestroyNotify) g_free), NULL);
			(seek == NULL) ? NULL : (seek = (g_object_unref (seek), NULL));
			return NULL;
		}
	}
	_tmp6 = NULL;
	seek = (_tmp6 = rygel_seek_new (GST_FORMAT_BYTES, (gint64) 0, (gint64) (item->res.size - 1)), (seek == NULL) ? NULL : (seek = (g_object_unref (seek), NULL)), _tmp6);
	/* Get first byte position*/
	_tmp7 = NULL;
	first_byte = (_tmp7 = range_tokens[0], (_tmp7 == NULL) ? NULL : g_strdup (_tmp7));
	if (g_unichar_isdigit (g_utf8_get_char (g_utf8_offset_to_pointer (first_byte, 0)))) {
		rygel_seek_set_start (seek, g_ascii_strtoll (first_byte, NULL, 0));
	} else {
		if (_vala_strcmp0 (first_byte, "") != 0) {
			inner_error = g_error_new (RYGEL_HTTP_SERVER_ERROR, RYGEL_HTTP_SERVER_ERROR_INVALID_RANGE, "Invalid Range '%s'", range);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				range = (g_free (range), NULL);
				range_tokens = (_vala_array_free (range_tokens, range_tokens_length1, (GDestroyNotify) g_free), NULL);
				(seek == NULL) ? NULL : (seek = (g_object_unref (seek), NULL));
				first_byte = (g_free (first_byte), NULL);
				return NULL;
			}
		}
	}
	/* Get last byte position if specified*/
	_tmp8 = NULL;
	last_byte = (_tmp8 = range_tokens[1], (_tmp8 == NULL) ? NULL : g_strdup (_tmp8));
	if (g_unichar_isdigit (g_utf8_get_char (g_utf8_offset_to_pointer (last_byte, 0)))) {
		rygel_seek_set_stop (seek, g_ascii_strtoll (last_byte, NULL, 0));
	} else {
		if (_vala_strcmp0 (last_byte, "") != 0) {
			inner_error = g_error_new (RYGEL_HTTP_SERVER_ERROR, RYGEL_HTTP_SERVER_ERROR_INVALID_RANGE, "Invalid Range '%s'", range);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				range = (g_free (range), NULL);
				range_tokens = (_vala_array_free (range_tokens, range_tokens_length1, (GDestroyNotify) g_free), NULL);
				(seek == NULL) ? NULL : (seek = (g_object_unref (seek), NULL));
				first_byte = (g_free (first_byte), NULL);
				last_byte = (g_free (last_byte), NULL);
				return NULL;
			}
		}
	}
	if (item->res.size > 0) {
		gboolean _tmp9;
		gboolean _tmp10;
		_tmp9 = FALSE;
		if (rygel_seek_get_start (seek) > item->res.size) {
			_tmp9 = TRUE;
		} else {
			_tmp9 = rygel_seek_get_length (seek) > item->res.size;
		}
		/* shouldn't go beyond actual length of media*/
		if (_tmp9) {
			inner_error = g_error_new (RYGEL_HTTP_SERVER_ERROR, RYGEL_HTTP_SERVER_ERROR_OUT_OF_RANGE, "Range '%s' not setsifiable", range);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				range = (g_free (range), NULL);
				range_tokens = (_vala_array_free (range_tokens, range_tokens_length1, (GDestroyNotify) g_free), NULL);
				(seek == NULL) ? NULL : (seek = (g_object_unref (seek), NULL));
				first_byte = (g_free (first_byte), NULL);
				last_byte = (g_free (last_byte), NULL);
				return NULL;
			}
		}
		_tmp10 = FALSE;
		if (rygel_seek_get_start (seek) == 0) {
			_tmp10 = rygel_seek_get_length (seek) == item->res.size;
		} else {
			_tmp10 = FALSE;
		}
		/* No need to seek if whole stream is requested*/
		if (_tmp10) {
			RygelSeek* _tmp11;
			_tmp11 = NULL;
			return (_tmp11 = NULL, range = (g_free (range), NULL), range_tokens = (_vala_array_free (range_tokens, range_tokens_length1, (GDestroyNotify) g_free), NULL), (seek == NULL) ? NULL : (seek = (g_object_unref (seek), NULL)), first_byte = (g_free (first_byte), NULL), last_byte = (g_free (last_byte), NULL), _tmp11);
		}
	} else {
		if (rygel_seek_get_start (seek) == 0) {
			RygelSeek* _tmp12;
			/* Might be an attempt to get the size, in which case it's not
			 an error. Just don't seek.*/
			_tmp12 = NULL;
			return (_tmp12 = NULL, range = (g_free (range), NULL), range_tokens = (_vala_array_free (range_tokens, range_tokens_length1, (GDestroyNotify) g_free), NULL), (seek == NULL) ? NULL : (seek = (g_object_unref (seek), NULL)), first_byte = (g_free (first_byte), NULL), last_byte = (g_free (last_byte), NULL), _tmp12);
		} else {
			inner_error = g_error_new (RYGEL_HTTP_SERVER_ERROR, RYGEL_HTTP_SERVER_ERROR_UNACCEPTABLE, "Partial download not applicable for item %s", ((RygelMediaObject*) item)->id);
			if (inner_error != NULL) {
				g_propagate_error (error, inner_error);
				range = (g_free (range), NULL);
				range_tokens = (_vala_array_free (range_tokens, range_tokens_length1, (GDestroyNotify) g_free), NULL);
				(seek == NULL) ? NULL : (seek = (g_object_unref (seek), NULL));
				first_byte = (g_free (first_byte), NULL);
				last_byte = (g_free (last_byte), NULL);
				return NULL;
			}
		}
	}
	_tmp13 = NULL;
	return (_tmp13 = seek, range = (g_free (range), NULL), range_tokens = (_vala_array_free (range_tokens, range_tokens_length1, (GDestroyNotify) g_free), NULL), first_byte = (g_free (first_byte), NULL), last_byte = (g_free (last_byte), NULL), _tmp13);
}


static void rygel_http_server_class_init (RygelHTTPServerClass * klass) {
	rygel_http_server_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (RygelHTTPServerPrivate));
	G_OBJECT_CLASS (klass)->finalize = rygel_http_server_finalize;
	g_signal_new ("need_stream_source", RYGEL_TYPE_HTTP_SERVER, G_SIGNAL_RUN_LAST, 0, NULL, NULL, g_cclosure_user_marshal_VOID__POINTER_POINTER, G_TYPE_NONE, 2, RYGEL_TYPE_MEDIA_ITEM, G_TYPE_POINTER);
	g_signal_new ("item_requested", RYGEL_TYPE_HTTP_SERVER, G_SIGNAL_RUN_LAST, 0, NULL, NULL, g_cclosure_user_marshal_VOID__STRING_POINTER, G_TYPE_NONE, 2, G_TYPE_STRING, G_TYPE_POINTER);
}


static void rygel_http_server_instance_init (RygelHTTPServer * self) {
	self->priv = RYGEL_HTTP_SERVER_GET_PRIVATE (self);
}


static void rygel_http_server_finalize (GObject* obj) {
	RygelHTTPServer * self;
	self = RYGEL_HTTP_SERVER (obj);
	self->priv->path_root = (g_free (self->priv->path_root), NULL);
	(self->priv->context == NULL) ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL));
	(self->priv->responses == NULL) ? NULL : (self->priv->responses = (g_object_unref (self->priv->responses), NULL));
	G_OBJECT_CLASS (rygel_http_server_parent_class)->finalize (obj);
}


GType rygel_http_server_get_type (void) {
	static GType rygel_http_server_type_id = 0;
	if (rygel_http_server_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (RygelHTTPServerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) rygel_http_server_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (RygelHTTPServer), 0, (GInstanceInitFunc) rygel_http_server_instance_init, NULL };
		rygel_http_server_type_id = g_type_register_static (G_TYPE_OBJECT, "RygelHTTPServer", &g_define_type_info, 0);
	}
	return rygel_http_server_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
	g_free (array);
}


static gint _vala_array_length (gpointer array) {
	int length;
	length = 0;
	if (array) {
		while (((gpointer*) array)[length]) {
			length++;
		}
	}
	return length;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}



static void g_cclosure_user_marshal_VOID__POINTER_POINTER (GClosure * closure, GValue * return_value, guint n_param_values, const GValue * param_values, gpointer invocation_hint, gpointer marshal_data) {
	typedef void (*GMarshalFunc_VOID__POINTER_POINTER) (gpointer data1, gpointer arg_1, gpointer arg_2, gpointer data2);
	register GMarshalFunc_VOID__POINTER_POINTER callback;
	register GCClosure * cc;
	register gpointer data1, data2;
	cc = (GCClosure *) closure;
	g_return_if_fail (n_param_values == 3);
	if (G_CCLOSURE_SWAP_DATA (closure)) {
		data1 = closure->data;
		data2 = param_values->data[0].v_pointer;
	} else {
		data1 = param_values->data[0].v_pointer;
		data2 = closure->data;
	}
	callback = (GMarshalFunc_VOID__POINTER_POINTER) (marshal_data ? marshal_data : cc->callback);
	callback (data1, rygel_value_get_media_object (param_values + 1), g_value_get_pointer (param_values + 2), data2);
}


static void g_cclosure_user_marshal_VOID__STRING_POINTER (GClosure * closure, GValue * return_value, guint n_param_values, const GValue * param_values, gpointer invocation_hint, gpointer marshal_data) {
	typedef void (*GMarshalFunc_VOID__STRING_POINTER) (gpointer data1, const char* arg_1, gpointer arg_2, gpointer data2);
	register GMarshalFunc_VOID__STRING_POINTER callback;
	register GCClosure * cc;
	register gpointer data1, data2;
	cc = (GCClosure *) closure;
	g_return_if_fail (n_param_values == 3);
	if (G_CCLOSURE_SWAP_DATA (closure)) {
		data1 = closure->data;
		data2 = param_values->data[0].v_pointer;
	} else {
		data1 = param_values->data[0].v_pointer;
		data2 = closure->data;
	}
	callback = (GMarshalFunc_VOID__STRING_POINTER) (marshal_data ? marshal_data : cc->callback);
	callback (data1, g_value_get_string (param_values + 1), g_value_get_pointer (param_values + 2), data2);
}



