/* xdaliclock - a melting digital clock
 * Copyright (c) 1991-2010 Jamie Zawinski <jwz@jwz.org>
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  No representations are made about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <sys/time.h>
#include <time.h>

#include "xdaliclock.h"

typedef unsigned short POS;
typedef unsigned char BOOL;

#ifdef BUILTIN_FONTS

/* static int use_builtin_font; */

struct raw_number {
  const unsigned char *bits;
  POS width, height;
};

#endif /* BUILTIN_FONTS */

#ifdef BUILTIN_FONTS

# include "zeroB.xbm"
# include "oneB.xbm"
# include "twoB.xbm"
# include "threeB.xbm"
# include "fourB.xbm"
# include "fiveB.xbm"
# include "sixB.xbm"
# include "sevenB.xbm"
# include "eightB.xbm"
# include "nineB.xbm"
# include "colonB.xbm"
# include "slashB.xbm"

static struct raw_number numbers_B [] = {
  { zeroB_bits,  zeroB_width,  zeroB_height  },
  { oneB_bits,   oneB_width,   oneB_height   },
  { twoB_bits,   twoB_width,   twoB_height   },
  { threeB_bits, threeB_width, threeB_height },
  { fourB_bits,  fourB_width,  fourB_height  },
  { fiveB_bits,  fiveB_width,  fiveB_height  },
  { sixB_bits,   sixB_width,   sixB_height   },
  { sevenB_bits, sevenB_width, sevenB_height },
  { eightB_bits, eightB_width, eightB_height },
  { nineB_bits,  nineB_width,  nineB_height  },
  { colonB_bits, colonB_width, colonB_height },
  { slashB_bits, slashB_width, slashB_height },
  { 0, }
};

# include "zeroC.xbm"
# include "oneC.xbm"
# include "twoC.xbm"
# include "threeC.xbm"
# include "fourC.xbm"
# include "fiveC.xbm"
# include "sixC.xbm"
# include "sevenC.xbm"
# include "eightC.xbm"
# include "nineC.xbm"
# include "colonC.xbm"
# include "slashC.xbm"

static struct raw_number numbers_C [] = {
  { zeroC_bits,  zeroC_width,  zeroC_height  },
  { oneC_bits,   oneC_width,   oneC_height   },
  { twoC_bits,   twoC_width,   twoC_height   },
  { threeC_bits, threeC_width, threeC_height },
  { fourC_bits,  fourC_width,  fourC_height  },
  { fiveC_bits,  fiveC_width,  fiveC_height  },
  { sixC_bits,   sixC_width,   sixC_height   },
  { sevenC_bits, sevenC_width, sevenC_height },
  { eightC_bits, eightC_width, eightC_height },
  { nineC_bits,  nineC_width,  nineC_height  },
  { colonC_bits, colonC_width, colonC_height },
  { slashC_bits, slashC_width, slashC_height },
  { 0, }
};

# include "zeroD.xbm"
# include "oneD.xbm"
# include "twoD.xbm"
# include "threeD.xbm"
# include "fourD.xbm"
# include "fiveD.xbm"
# include "sixD.xbm"
# include "sevenD.xbm"
# include "eightD.xbm"
# include "nineD.xbm"
# include "colonD.xbm"
# include "slashD.xbm"

static struct raw_number numbers_D [] = {
  { zeroD_bits,  zeroD_width,  zeroD_height  },
  { oneD_bits,   oneD_width,   oneD_height   },
  { twoD_bits,   twoD_width,   twoD_height   },
  { threeD_bits, threeD_width, threeD_height },
  { fourD_bits,  fourD_width,  fourD_height  },
  { fiveD_bits,  fiveD_width,  fiveD_height  },
  { sixD_bits,   sixD_width,   sixD_height   },
  { sevenD_bits, sevenD_width, sevenD_height },
  { eightD_bits, eightD_width, eightD_height },
  { nineD_bits,  nineD_width,  nineD_height  },
  { colonD_bits, colonD_width, colonD_height },
  { slashD_bits, slashD_width, slashD_height },
  { 0, }
};

# include "zeroE.xbm"
# include "oneE.xbm"
# include "twoE.xbm"
# include "threeE.xbm"
# include "fourE.xbm"
# include "fiveE.xbm"
# include "sixE.xbm"
# include "sevenE.xbm"
# include "eightE.xbm"
# include "nineE.xbm"
# include "colonE.xbm"
# include "slashE.xbm"

static struct raw_number numbers_E [] = {
  { zeroE_bits,  zeroE_width,  zeroE_height  },
  { oneE_bits,   oneE_width,   oneE_height   },
  { twoE_bits,   twoE_width,   twoE_height   },
  { threeE_bits, threeE_width, threeE_height },
  { fourE_bits,  fourE_width,  fourE_height  },
  { fiveE_bits,  fiveE_width,  fiveE_height  },
  { sixE_bits,   sixE_width,   sixE_height   },
  { sevenE_bits, sevenE_width, sevenE_height },
  { eightE_bits, eightE_width, eightE_height },
  { nineE_bits,  nineE_width,  nineE_height  },
  { colonE_bits, colonE_width, colonE_height },
  { slashE_bits, slashE_width, slashE_height },
  { 0, }
};

# include "zeroF.xbm"
# include "oneF.xbm"
# include "twoF.xbm"
# include "threeF.xbm"
# include "fourF.xbm"
# include "fiveF.xbm"
# include "sixF.xbm"
# include "sevenF.xbm"
# include "eightF.xbm"
# include "nineF.xbm"
# include "colonF.xbm"
# include "slashF.xbm"

static struct raw_number numbers_F [] = {
  { zeroF_bits,  zeroF_width,  zeroF_height  },
  { oneF_bits,   oneF_width,   oneF_height   },
  { twoF_bits,   twoF_width,   twoF_height   },
  { threeF_bits, threeF_width, threeF_height },
  { fourF_bits,  fourF_width,  fourF_height  },
  { fiveF_bits,  fiveF_width,  fiveF_height  },
  { sixF_bits,   sixF_width,   sixF_height   },
  { sevenF_bits, sevenF_width, sevenF_height },
  { eightF_bits, eightF_width, eightF_height },
  { nineF_bits,  nineF_width,  nineF_height  },
  { colonF_bits, colonF_width, colonF_height },
  { slashF_bits, slashF_width, slashF_height },
  { 0, }
};

#endif /* BUILTIN_FONTS */

#undef countof
#define countof(x) (sizeof((x))/sizeof(*(x)))

/* Number of horizontal segments/line.  Enlarge this if you are trying
   to use a font that is too "curvy" for XDaliClock to cope with.
   This code was sent to me by Dan Wallach <c169-bg@auriga.berkeley.edu>.
   I'm highly opposed to ever using statically-sized arrays, but I don't
   really feel like hacking on this code enough to clean it up.
 */
#ifndef MAX_SEGS_PER_LINE
# define MAX_SEGS_PER_LINE 5
#endif

struct scanline {
  POS left[MAX_SEGS_PER_LINE], right[MAX_SEGS_PER_LINE];
};

struct frame {
  struct scanline scanlines [1]; /* scanlines are contiguous here */
};


/* The runtime settings (some initialized from system prefs, but changable.)
 */
struct render_state {
  unsigned int last_secs;
  int char_width, char_height, colon_width;
  struct frame *base_frames [12];	/* all digits */
  struct frame *orig_frames [8];	/* what was there */
  struct frame *current_frames [8];	/* current intermediate animation */
  struct frame *target_frames [8];	/* where we are going */
  int           target_digits [8];	/* where we are going */
  struct frame *empty_frame;
  struct frame *empty_colon;
};


static struct frame *
make_empty_frame (int width, int height)
{
  int size = sizeof (struct frame) + (sizeof (struct scanline) * height);
  struct frame *frame;
  int x, y;

  frame = (struct frame *) calloc (size, 1);
  for (y = 0; y < height; y++)
    for (x = 0; x < MAX_SEGS_PER_LINE; x++)
      frame->scanlines[y].left [x] = frame->scanlines[y].right [x] = width / 2;
  return frame;
}


static struct frame *
copy_frame (dali_config *c, struct frame *from)
{
  struct render_state *state = c->render_state;
  int height = state->char_height;
  int size = sizeof (struct frame) + (sizeof (struct scanline) * height);
  struct frame *to = (struct frame *) calloc (size, 1);
  int y;
  for (y = 0; y < height; y++)
    to->scanlines[y] = from->scanlines[y];  /* copies the whole struct */
  return to;
}


static struct frame *
number_to_frame (const unsigned char *bits, int width, int height)
{
  int x, y;
  struct frame *frame;
  POS *left, *right;

  frame = make_empty_frame (width, height);

  for (y = 0; y < height; y++)
    {
      int seg, end;
      x = 0;
# define GETBIT(bits,x,y) \
         (!! ((bits) [((y) * ((width+7) >> 3)) + ((x) >> 3)] \
              & (1 << ((x) & 7))))

      left = frame->scanlines[y].left;
      right = frame->scanlines[y].right;

      for (seg = 0; seg < MAX_SEGS_PER_LINE; seg++)
        left [seg] = right [seg] = width / 2;

      for (seg = 0; seg < MAX_SEGS_PER_LINE; seg++)
        {
          for (; x < width; x++)
            if (GETBIT (bits, x, y)) break;
          if (x == width) break;
          left [seg] = x;
          for (; x < width; x++)
            if (! GETBIT (bits, x, y)) break;
          right [seg] = x;
        }

      for (; x < width; x++)
        if (GETBIT (bits, x, y))
          {
            fprintf (stderr, "%s: font is too curvy\n", progname);
            /* Increase MAX_SEGS_PER_LINE and recompile. */
            exit (-1);
          }

      /* If there were any segments on this line, then replicate the last
         one out to the end of the line.  If it's blank, leave it alone,
         meaning it will be a 0-pixel-wide line down the middle.
       */
      end = seg;
      if (end > 0)
        for (; seg < MAX_SEGS_PER_LINE; seg++)
          {
            left [seg] = left [end-1];
            right [seg] = right [end-1];
          }

# undef GETBIT
    }

  return frame;
}


static int
pick_font_size (dali_config *c, unsigned int *w_ret, unsigned int *h_ret)
{
#ifdef BUILTIN_FONTS
  int nn, cc;
  int f;
  unsigned int w, h;
  unsigned int ww = (w_ret ? *w_ret : c->width);
  unsigned int hh = (h_ret ? *h_ret : c->height);

  switch (c->time_mode)
    {
    case SS:     nn = 2, cc = 0; break;
    case HHMM:   nn = 4, cc = 1; break;
    case HHMMSS: nn = 6, cc = 2; break;
    default:   abort(); break;
    }

  if      ((w = ((numbers_B[0].width * nn) +
                 (numbers_B[10].width * cc))) <= ww &&
           ((h = numbers_B[0].height) <= hh))
    f = 4;
  else if ((w = ((numbers_C[0].width * nn) +
                 (numbers_C[10].width * cc))) <= ww &&
           ((h = numbers_C[0].height) <= hh))
    f = 3;
  else if ((w = ((numbers_D[0].width * nn) +
                 (numbers_D[10].width * cc))) <= ww &&
           ((h = numbers_D[0].height) <= hh))
    f = 2;
  else if ((w = ((numbers_E[0].width * nn) +
                 (numbers_E[10].width * cc))) <= ww &&
           ((h = numbers_E[0].height) <= hh))
    f = 1;
  else
    {
      w = ((numbers_F[0].width * nn) +
           (numbers_F[10].width * cc));
      h = numbers_F[0].height;
      f = 0;
    }

  w += 2;  /* Leave a single pixel margin in the overall bitmap */
  h += 2;

  w = ((w + 7) / 8) * 8;  /* round up to byte */

#else  /* !BUILTIN_FONTS */
  int w = 0, h = 0, f = 0;
#endif /* !BUILTIN_FONTS */
  
  if (w_ret) *w_ret = w;
  if (h_ret) *h_ret = h;
  return f;
}


static void
init_numbers (dali_config *c)
{
  struct render_state *state = c->render_state;
  int i;
#ifdef BUILTIN_FONTS
  struct raw_number *raw;

  int size = pick_font_size (c, NULL, NULL);
  switch (size)
    {
    case 0: raw = numbers_F; break;
    case 1: raw = numbers_E; break;
    case 2: raw = numbers_D; break;
    case 3: raw = numbers_C; break;
    case 4: raw = numbers_B; break;
    default: abort(); break;
    }

  state->char_width  = raw[0].width;
  state->char_height = raw[0].height;
  state->colon_width = raw[10].width;

  state->empty_frame = make_empty_frame (raw[0].width,  raw[0].height);
  state->empty_colon = make_empty_frame (raw[10].width, raw[10].height);

  for (i = 0; i < countof(state->base_frames); i++)
    state->base_frames [i] =
      number_to_frame (raw[i].bits, raw[i].width, raw[i].height);
#endif /* BUILTIN_FONTS */

  memset (state->orig_frames,    0, sizeof(state->orig_frames));
  memset (state->current_frames, 0, sizeof(state->current_frames));
  memset (state->target_frames,  0, sizeof(state->target_frames));

  for (i = 0; i < countof(state->current_frames); i++) {
    int colonic_p = (i == 2 || i == 5);
    int cw = raw[colonic_p ? 10 : 0].width;
    int ch = raw[0].height;
    state->orig_frames[i]    = make_empty_frame (cw, ch);
    state->current_frames[i] = make_empty_frame (cw, ch);
    state->target_frames[i]  = make_empty_frame (cw, ch);
  }

  for (i = 0; i < countof(state->target_digits); i++)
    state->target_digits[i] = -1;

  memset (c->bitmap, 0, c->height * (c->width >> 3));
}


static void
free_numbers (dali_config *c)
{
  struct render_state *state = c->render_state;
  int i;
# define FREEIF(x) do { if ((x)) { free((x)); (x) = 0; } } while (0)
# define FREELOOP(x) do { \
    for (i = 0; i < countof ((x)); i++) FREEIF ((x)[i]); } while (0)

  FREELOOP (state->base_frames);
  FREELOOP (state->orig_frames);
  FREELOOP (state->current_frames);
  FREELOOP (state->target_frames);
  FREEIF (state->empty_frame);
  FREEIF (state->empty_colon);

# undef FREELOOP
# undef FREEIF
}


static void
fill_target_digits (dali_config *c, unsigned long time)
{
  struct render_state *state = c->render_state;
  struct tm *tm = localtime ((time_t *) &time);

  int i;
  int h = tm->tm_hour;
  int m = tm->tm_min;
  int s = tm->tm_sec;
  int D = tm->tm_mday;
  int M = tm->tm_mon;
  int Y = tm->tm_year % 100;

  int twelve_p = c->twelve_hour_p;

  if (c->countdown)
    {
      long delta = ((unsigned long) c->countdown) - time;
      if (delta < 0) delta = -delta;
      s = delta % 60;
      m = (delta / 60) % 60;
      h = (delta / (60 * 60)) % 100;
      twelve_p = 0;
    }

  if (twelve_p) 
    {
      if (h > 12) { h -= 12; }
      else if (h == 0) { h = 12; }
    }

  for (i = 0; i < countof(state->target_digits); i++)
    state->target_digits[i] = -1;

  if (c->test_hack)
    {
      int a = (c->test_hack >= '0' && c->test_hack <= '9'
               ? c->test_hack - '0'
               : -1);
      state->target_digits [0] = a;
      state->target_digits [1] = a;
      state->target_digits [2] = 10;
      state->target_digits [3] = a;
      state->target_digits [4] = a;
      state->target_digits [5] = 10;
      state->target_digits [6] = a;
      state->target_digits [7] = a;
      c->test_hack = 0;
    }
  else if (!c->display_date_p)
    {
      switch (c->time_mode)
        {
        case SS:
          state->target_digits[0] = (s / 10);
          state->target_digits[1] = (s % 10);
          break;
        case HHMM:
          state->target_digits[0] = (h / 10);
          state->target_digits[1] = (h % 10);
          state->target_digits[2] = 10;		/* colon */
          state->target_digits[3] = (m / 10);
          state->target_digits[4] = (m % 10);
          if (twelve_p && state->target_digits[0] == 0)
            state->target_digits[0] = -1;
          break;
        case HHMMSS:
          state->target_digits[0] = (h / 10);
          state->target_digits[1] = (h % 10);
          state->target_digits[2] = 10;		/* colon */
          state->target_digits[3] = (m / 10);
          state->target_digits[4] = (m % 10);
          state->target_digits[5] = 10;		/* colon */
          state->target_digits[6] = (s / 10);
          state->target_digits[7] = (s % 10);
          if (twelve_p && state->target_digits[0] == 0)
            state->target_digits[0] = -1;
          break;
        default: 
          abort();
        }
    }
  else	/* date mode */
    {
      switch (c->date_mode) 
        {
        case MMDDYY:
          switch (c->time_mode) {
          case SS:
            state->target_digits[0] = (D / 10);
            state->target_digits[1] = (D % 10);
            break;
          case HHMM:
            state->target_digits[0] = (M / 10);
            state->target_digits[1] = (M % 10);
            state->target_digits[2] = 11;		/* dash */
            state->target_digits[3] = (D / 10);
            state->target_digits[4] = (D % 10);
            break;
          case HHMMSS:
            state->target_digits[0] = (M / 10);
            state->target_digits[1] = (M % 10);
            state->target_digits[2] = 11;		/* dash */
            state->target_digits[3] = (D / 10);
            state->target_digits[4] = (D % 10);
            state->target_digits[5] = 11;		/* dash */
            state->target_digits[6] = (Y / 10);
            state->target_digits[7] = (Y % 10);
            break;
          default:
            abort();
          }
          break;
        case DDMMYY:
          switch (c->time_mode) {
          case SS:
            state->target_digits[0] = (D / 10);
            state->target_digits[1] = (D % 10);
            break;
          case HHMM:
            state->target_digits[0] = (D / 10);
            state->target_digits[1] = (D % 10);
            state->target_digits[2] = 11;		/* dash */
            state->target_digits[3] = (M / 10);
            state->target_digits[4] = (M % 10);
            break;
          case HHMMSS:
            state->target_digits[0] = (D / 10);
            state->target_digits[1] = (D % 10);
            state->target_digits[2] = 11;		/* dash */
            state->target_digits[3] = (M / 10);
            state->target_digits[4] = (M % 10);
            state->target_digits[5] = 11;		/* dash */
            state->target_digits[6] = (Y / 10);
            state->target_digits[7] = (Y % 10);
            break;
          default:
            abort();
          }
          break;
        case YYMMDD:
          switch (c->time_mode) {
          case SS:
            state->target_digits[0] = (D / 10);
            state->target_digits[1] = (D % 10);
            break;
          case HHMM:
            state->target_digits[0] = (M / 10);
            state->target_digits[1] = (M % 10);
            state->target_digits[2] = 11;		/* dash */
            state->target_digits[3] = (D / 10);
            state->target_digits[4] = (D % 10);
            break;
          case HHMMSS:
            state->target_digits[0] = (Y / 10);
            state->target_digits[1] = (Y % 10);
            state->target_digits[2] = 11;		/* dash */
            state->target_digits[3] = (M / 10);
            state->target_digits[4] = (M % 10);
            state->target_digits[5] = 11;		/* dash */
            state->target_digits[6] = (D / 10);
            state->target_digits[7] = (D % 10);
            break;
          default:
            abort();
          }
          break;
        default:
          abort();
        }
    }
}


static void
draw_horizontal_line (dali_config *c, int x1, int x2, int y, BOOL black_p)
{
  unsigned char *scanline;
  if (x1 == x2) return;
  if (y > c->height) return;
  if (x1 > c->width) x1 = c->width;
  if (x2 > c->width) x2 = c->width;
  if (x1 > x2)
    {
      int swap = x1;
      x1 = x2;
      x2 = swap;
    }

  scanline = c->bitmap + (y * (c->width >> 3));
  
#define BIGENDIAN
  
#ifdef BIGENDIAN
  if (black_p)
    for (; x1 < x2; x1++)
      scanline[x1>>3] |= 1 << (7 - (x1 & 7));
  else
    for (; x1 < x2; x1++)
      scanline[x1>>3] &= ~(1 << (7 - (x1 & 7)));
#else  /* !BIGENDIAN */
  if (black_p)
    for (; x1 < x2; x1++)
      scanline[x1>>3] |= 1 << (x1 & 7);
  else
    for (; x1 < x2; x1++)
      scanline[x1>>3] &= ~(1 << (x1 & 7));
#endif /* !BIGENDIAN */
}


static int
draw_frame (dali_config *c, struct frame *frame, int x, int y, int colonic_p)
{
  struct render_state *state = c->render_state;
  int px, py;
  int cw = (colonic_p ? state->colon_width : state->char_width);

  for (py = 0; py < state->char_height; py++)
    {
      struct scanline *line = &frame->scanlines [py];
      int last_right = 0;

      for (px = 0; px < MAX_SEGS_PER_LINE; px++)
        {
          if (px > 0 &&
              (line->left[px] == line->right[px] ||
               (line->left [px] == line->left [px-1] &&
                line->right[px] == line->right[px-1])))
            continue;

          /* Erase the line between the last segment and this segment.
           */
          draw_horizontal_line (c,
                                x + last_right,
                                x + line->left [px],
                                y + py,
                                0);

          /* Draw the line of this segment.
           */
          draw_horizontal_line (c,
                                x + line->left [px],
                                x + line->right[px],
                                y + py,
                                1);

          last_right = line->right[px];
        }

      /* Erase the line between the last segment and the right edge.
       */
      draw_horizontal_line (c,
                            x + last_right,
                            x + cw,
                            y + py,
                            0);
    }
  return cw;
}


static void draw_clock (dali_config *c);

static void
start_sequence (dali_config *c, unsigned long time)
{
  struct render_state *state = c->render_state;
  int i;

  /* Move the (old) current_frames into the (new) orig_frames,
     since that's what's on the screen now. 
     Frames are freed as they expire out of orig_frames.
   */
  for (i = 0; i < countof (state->current_frames); i++)
    {
      if (state->orig_frames[i]) 
        free (state->orig_frames[i]);
      state->orig_frames[i]    = state->current_frames[i];
      state->current_frames[i] = state->target_frames[i];
      state->target_frames[i]  = 0;
    }

  /* generate new target_digits */
  fill_target_digits (c, time);

  /* Fill the (new) target_frames from the (new) target_digits. */
  for (i = 0; i < countof (state->target_frames); i++)
    {
      int colonic_p = (i == 2 || i == 5);
      state->target_frames[i] =
        copy_frame (c,
                    (state->target_digits[i] == -1
                     ? (colonic_p ? state->empty_colon : state->empty_frame)
                     : state->base_frames[state->target_digits[i]]));
    }

  /* Render the current frame. */
  draw_clock (c);
}


static void
one_step (dali_config *c,
          struct frame *orig_frame,
          struct frame *current_frame,
          struct frame *target_frame,
          unsigned int msecs)
{
  struct render_state *state = c->render_state;
  struct scanline *orig   =    &orig_frame->scanlines [0];
  struct scanline *curr   = &current_frame->scanlines [0];
  struct scanline *target =  &target_frame->scanlines [0];
  int i = 0, x;

  for (i = 0; i < state->char_height; i++)
    {
# define STEP(field) \
         (curr->field = (orig->field \
                         + (((int) (target->field - orig->field)) \
                            * (int) msecs / 1000)))

      for (x = 0; x < MAX_SEGS_PER_LINE; x++)
        {
          STEP (left [x]);
          STEP (right[x]);
        }
      orig++;
      curr++;
      target++;
# undef STEP
    }
}


static void
tick_sequence (dali_config *c)
{
  struct render_state *state = c->render_state;
  int i;

  struct timeval now;
  struct timezone tzp;
  gettimeofday (&now, &tzp);
  unsigned long secs = now.tv_sec;
  unsigned long msecs = now.tv_usec / 1000;

  if (!state->last_secs)
    state->last_secs = secs;   /* fading in! */
  else if (secs != state->last_secs) 
    {
      /* End of the animation sequence; fill target_frames with the
         digits of the current time. */
      start_sequence (c, secs);
      state->last_secs = secs;
    }

  /* Linger for about 1/10th second at the end of each cycle. */
  msecs *= 1.2;
  if (msecs > 1000) msecs = 1000;

  /* Construct current_frames by interpolating between
     orig_frames and target_frames. */
  for (i = 0; i < countof (state->current_frames); i++)
    one_step (c,
              state->orig_frames[i],
              state->current_frames[i],
              state->target_frames[i],
              msecs);
}


static void
draw_clock (dali_config *c)
{
  struct render_state *state = c->render_state;
  int x, y, i, nn, cc;

  switch (c->time_mode)
    {
    case SS:     nn = 2, cc = 0; break;
    case HHMM:   nn = 4, cc = 1; break;
    case HHMMSS: nn = 6, cc = 2; break;
    default:     abort(); break;
    }

  x = y = 0;
  for (i = 0; i < nn+cc; i++) 
    {
      int colonic_p = (i == 2 || i == 5);
      x += draw_frame (c, state->current_frames[i], x, y, colonic_p);
    }
}


void
render_init (dali_config *c)
{
  if (c->render_state) abort();
  c->render_state = (struct render_state *)
    calloc (1, sizeof (struct render_state));
  init_numbers (c);
}

void
render_free (dali_config *c)
{
  if (!c->render_state) abort();
  free_numbers (c);
  free (c->render_state);
  c->render_state = 0;
}

void
render_once (dali_config *c)
{
  if (! c->render_state) abort();
  if (! c->bitmap) abort();
  tick_sequence (c);
  draw_clock (c);
}

void
render_bitmap_size (dali_config *c, unsigned int *w_ret, unsigned int *h_ret)
{
  pick_font_size (c, w_ret, h_ret);
}
