/*
 * Setup.cs - Setup curses prior to any other operations and shutdown after.
 *
 * Copyright (C) 2001  Southern Storm Software, Pty Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

namespace Curses
{

using System;

public sealed class Setup
{
	// Internal state.
	private static bool initialized = false;
	private static bool initSucceeded;

	/// <summary>
	/// <para>Initialize curses and prepare for terminal operations.</para>
	/// </summary>
	///
	/// <returns>
	/// <para><see langword="true"/> if curses could be initialized,
	/// or <see langword="false"/> otherwise.</para>
	/// </returns>
	///
	/// <remarks>
	/// <para>It is safe to call this method multiple times.  Subsequent
	/// calls will return the previous result.</para>
	///
	/// <para>The curses library will be automatically initialized if
	/// the application tries to access window or drawing operations
	/// prior to calling this method.</para>
	///
	/// <para>It is recommended that applications call this method at
	/// startup and then exit with a suitable error message if it
	/// returns <see langword="false"/>.</para>
	/// </remarks>
	public static bool Initialize()
			{
				lock(typeof(Setup))
				{
					if(initialized)
					{
						// We've already been called at least once.
						return initSucceeded;
					}
					initSucceeded = (Curses.Native.CursesHelpInitialize() != 0);
					Misc.dockedCursor = false;
					initialized = true;
					return initSucceeded;
				}
			}

	/// <summary>
	/// <para>Shutdown curses and prepare for program exit.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>After this method has been called, it is possible to
	/// re-initialize curses with the <c>Initialize</c> method.</para>
	/// </remarks>
	public static void Shutdown()
			{
				lock(typeof(Setup))
				{
					if(initialized)
					{
						if(initSucceeded)
						{
							Curses.Native.CursesHelpTerminate();
							initSucceeded = false;
						}
						initialized = false;
					}
				}
			}

	/// <summary>
	/// <para>Shutdown curses and exit from the program.</para>
	/// </summary>
	public static void Exit(int exitCode)
			{
				Shutdown();
				Environment.Exit(exitCode);
			}

} // class Setup

} // namespace Curses
