//
// System.Web.Security.MembershipUser
//
// Authors:
//	Ben Maurer (bmaurer@users.sourceforge.net)
//
// (C) 2003 Ben Maurer
//

//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//

#if NET_2_0
namespace System.Web.Security
{
	[Serializable]
	public class MembershipUser
	{
		string providerName;
		string name;
		object providerUserKey;
		string email;
		string passwordQuestion;
		string comment;
		bool isApproved;
		bool isLockedOut;
		DateTime creationDate;
		DateTime lastLoginDate;
		DateTime lastActivityDate;
		DateTime lastPasswordChangedDate;
		DateTime lastLockoutDate;
		
		protected MembershipUser ()
		{
		}
		
		public MembershipUser (string providerName, string name, object providerUserKey, string email,
			string passwordQuestion, string comment, bool isApproved, bool isLockedOut,
			DateTime creationDate, DateTime lastLoginDate, DateTime lastActivityDate,
			DateTime lastPasswordChangedDate, DateTime lastLockoutDate)
		{
			this.providerName = providerName;
			this.name = name;
			this.providerUserKey = providerUserKey;
			this.email = email;
			this.passwordQuestion = passwordQuestion;
			this.comment = comment;
			this.isApproved = isApproved;
			this.isLockedOut = isLockedOut;
			this.creationDate = creationDate;
			this.lastLoginDate = lastLoginDate;
			this.lastActivityDate = lastActivityDate;
			this.lastPasswordChangedDate = lastPasswordChangedDate;
			this.lastLockoutDate = lastLockoutDate;
		}
		
		public virtual bool ChangePassword (string oldPassword, string newPassword)
		{
			bool success = Provider.ChangePassword (UserName, oldPassword, newPassword);
			if (success)
				lastPasswordChangedDate = DateTime.Now;
			
			return success;
		}
		
		public virtual bool ChangePasswordQuestionAndAnswer (string password, string newPasswordQuestion, string newPasswordAnswer)
		{
			bool success = Provider.ChangePasswordQuestionAndAnswer (UserName, password, newPasswordQuestion, newPasswordAnswer);
			if (success)
				passwordQuestion = newPasswordQuestion;
			
			return success;
		}
		
		public virtual string GetPassword ()
		{
			return GetPassword (null);
		}
		
		public virtual string GetPassword (string answer)
		{
			return Provider.GetPassword (UserName, answer);
		}
		
		public virtual string ResetPassword ()
		{
			return ResetPassword (null);
		}
		
		public virtual string ResetPassword (string answer)
		{
			string newPass = Provider.ResetPassword (UserName, answer);
			if (newPass != null)
				lastPasswordChangedDate = DateTime.Now;
			
			return newPass;
		}
		
		public virtual string Comment {
			get { return comment; }
			set { comment = value; }
		}
		
		public virtual DateTime CreationDate {
			get { return creationDate; }
		}
		
		public virtual string Email {
			get { return email; }
			set { email = value; }
		}
		
		public virtual bool IsApproved {
			get { return isApproved; }
			set { isApproved = value; }
		}
		
		public virtual bool IsLockedOut {
			get { return isLockedOut; }
		}
		
		public bool IsOnline {
			get {
				return LastActivityDate > DateTime.Now - TimeSpan.FromMinutes (Membership.UserIsOnlineTimeWindow);  
			}
		}
		
		public virtual DateTime LastActivityDate {
			get { return lastActivityDate; }
			set { lastActivityDate = value; }
		}
		
		public virtual DateTime LastLoginDate {
			get { return lastLoginDate; }
			set { lastLoginDate = value; }
		}
		
		public virtual DateTime LastPasswordChangedDate {
			get { return lastPasswordChangedDate; }
		}
		
		public virtual DateTime LastLockoutDate {
			get { return lastLockoutDate; }
		}
		
		public virtual string PasswordQuestion {
			get { return passwordQuestion; }
		}
		
		public virtual string ProviderName {
			get { return providerName; }
		}
		
		public virtual string UserName {
			get { return name; }
		}
		
		public virtual object ProviderUserKey {
			get { return providerUserKey; }
		}
		
		public override string ToString ()
		{
			return UserName;
		}
		
		public virtual bool UnlockUser ()
		{
			if (Provider.UnlockUser (UserName)) {
				isLockedOut = false;
				return true;
			}
			return false;
		}
		
		MembershipProvider Provider {
			get {
				MembershipProvider p = Membership.Providers [ProviderName];
				if (p == null) throw new InvalidOperationException ("Membership provider '" + ProviderName + "' not found.");
				return p;
			}
		}
	}
}
#endif

