# GNU Enterprise Application Server - Class Repository
#
# Copyright 2001-2009 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 3, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: repository.py 9953 2009-10-11 18:50:17Z reinhard $

from gnue.common.apps import errors
from gnue.common.utils.CaselessDict import CaselessDict

from gnue.appserver import data

# =============================================================================
# Exceptions
# =============================================================================

class ModuleNotFoundError (errors.ApplicationError):
  def __init__ (self, module):
    msg = u_("Module '%s' not found in class repository") % module
    errors.ApplicationError.__init__ (self, msg)

class ClassNotFoundError (errors.ApplicationError):
  def __init__ (self, classname):
    msg = u_("Class '%s' not found in class repository") % classname
    errors.ApplicationError.__init__ (self, msg)

class FilterNotFoundError (errors.ApplicationError):
  def __init__ (self, classname):
    msg = u_("Filter '%s' not found in class repository") % classname
    errors.ApplicationError.__init__ (self, msg)

class PropertyNotFoundError (errors.ApplicationError):
  def __init__ (self, name, classname):
    msg = u_("Class '%(classname)s' has no property '%(property)s'") \
          % {"classname": classname, "property": name}
    errors.ApplicationError.__init__ (self, msg)

class ProcedureNotFoundError (errors.ApplicationError):
  def __init__ (self, name, classname):
    msg = u_("Class '%(classname)s' has no procedure '%(procedure)s'") \
          % {"classname": classname, "procedure": name}
    errors.ApplicationError.__init__ (self, msg)

class ParameterNotFoundError (errors.ApplicationError):
  def __init__ (self, parameter, procedure):
    msg = u_("Procedure '%(procedure)s' has no parameter '%(parameter)s'") \
          % {"procedure": procedure, "parameter": parameter}
    errors.ApplicationError.__init__ (self, msg)

class LabelNotFoundError (errors.ApplicationError):
  def __init__ (self, name, prname):
    msg = u_("Property '%(prname)s' has no label '%(name)s'") \
          % {"prname": prname, "name": name}
    errors.ApplicationError.__init__ (self, msg)

class ProcLabelNotFoundError (errors.ApplicationError):
  def __init__ (self, name, prname):
    msg = u_("Procedure '%(prname)s' has no label '%(name)s'") \
          % {"prname": prname, "name": name}
    errors.ApplicationError.__init__ (self, msg)

class ClassLabelNotFoundError (errors.ApplicationError):
  def __init__ (self, language, classname):
    msg = u_("Class '%(class)s' has no label for language '%(language)s'") \
        % {'class': classname, 'language': language}
    errors.ApplicationError.__init__ (self, msg)

class MissingFilterClassError (errors.ApplicationError):
  def __init__ (self, classname, filterid):
    msg = u_("Filter '%(filter)s' defined in class '%(classname)s' not found "
             "in class repository") \
          % {'classname': classname, 'filter': filterid}
    errors.ApplicationError.__init__ (self, msg)

class ValidationError (errors.ApplicationError):
  pass

class TypeNameError (ValidationError):
  def __init__ (self, typename):
    msg = u_("'%s' is not a valid type") % typename
    ValidationError.__init__ (self, msg)

class TypeFormatError (ValidationError):
  pass

class ParameterValidationError (ValidationError):
  def __init__ (self, procedure, parameter, error):
    msg = u_("Error validating parameter '%(parameter)s' of procedure "
             "'%(procedure)s': %(errors)s") \
          % {'procedure': procedure, 'parameter': parameter, 'error': error}
    ValidationError.__init__ (self, msg)

class InvalidNameError (errors.ApplicationError):
  def __init__ (self, name):
    msg = u_("'%s' is not a valid, fully qualified identifier") % name
    errors.ApplicationError.__init__ (self, name)

class CircularFilterError (errors.ApplicationError):
  def __init__ (self):
    msg = u_("The filters have circular references")
    errors.ApplicationError.__init__ (self, msg)

class NotAReferenceError (errors.ApplicationError):
  def __init__ (self, classname, propertyname):
    msg = u_("'%(property)s' of class '%(class)s' is not a reference property")\
        % {'class': classname, 'property': propertyname}
    errors.ApplicationError.__init__ (self, msg)

# =============================================================================
# This class implements the class repository
# =============================================================================

class Repository:
  """
  This class provides access to the class repository of AppServer. To load the
  repository from the backend database call the method 'load'. After this the
  following instance variables are available:

  @ivar modules: dictionary with all modules available. The dictionary has both,
      the modulenames and their gnue_id's as keys.
  @ivar classes: dictionary with all classes available. The dictionary has
      both, the classnames and their gnue_id's as keys.
  @ivar filters: dictionary with all filter classes available. The dictionary
      uses the 'gnue_id' as keys.
  """

  # ---------------------------------------------------------------------------
  # Create a new repository instance
  # ---------------------------------------------------------------------------

  def __init__ (self, connections):
    """
    Create a new repository instance.

    @param connections: L{gnue.common.datasources.GConnections} instance
    """

    self._connections = connections
    self._database    = gConfig ('connection')

    self.modules = None
    self.classes = None
    self.filters = None
    self.filterOrder = None


  # ---------------------------------------------------------------------------
  # Load the repository into modules and classes
  # ---------------------------------------------------------------------------

  def load (self):
    """
    This function reloads the whole class repository, which is available
    through the properties 'modules' and 'classes'.
    """

    modules    = CaselessDict (ModuleNotFoundError)
    classes    = CaselessDict (ClassNotFoundError)
    filters    = CaselessDict (FilterNotFoundError)

    properties = {}
    procedures = {}

    try:
      # First load all raw data from the gnue_* tables
      self.__loadRawData ()

      # Wrap all modules
      for (key, value) in self.__rawModules.items ():
        module = Module (value)
        modules [key] = module
        modules [module.fullName] = module

      # Wrap all classes and install them into the apropriate module
      for (key, value) in self.__rawClasses.items ():
        aClass = Class (value, self.__rawModules)
        module = aClass.gnue_module
        modules [module]._installClass (aClass)

        classes [key]             = aClass
        classes [aClass.fullName] = aClass

        # If a class has a filter defined, we add it to the filter dictionary,
        # which will be completed right after loading all classes
        if aClass.gnue_filter is not None:
          if not self.__rawClasses.has_key (aClass.gnue_filter):
            raise MissingFilterClassError, (aClass.fullName, aClass.gnue_filter)

          filters [aClass.gnue_filter] = None

      # After all classes are loaded, complete the filter dictionary
      for key in filters.keys ():
        filters [key] = classes [key]

      # Wrap all properties and install them into the apropriate class
      for (key, value) in self.__rawProperties.items ():
        aProperty = Property (value, self.__rawModules)
        classes [aProperty.gnue_class]._installProperty (aProperty)
        properties [key] = aProperty

      # Wrap all procedures and link them into a lookup dictionary first
      for (key, value) in self.__rawProcedures.items ():
        aProcedure = Procedure (value, self.__rawModules)
        procedures [key] = aProcedure

      # Now create all parameters and install them to their procedures.
      for (key, value) in self.__rawParameters.items ():
        aParameter = Parameter (value)
        procedures [aParameter.gnue_procedure]._installParameter (aParameter)

      # Create the labels and install them into their properties or procedures
      for (key, value) in self.__rawLabels.items ():
        aLabel = Label (value)

        if aLabel.gnue_property:
          properties [aLabel.gnue_property]._installLabel (aLabel)
        else:
          procedures [aLabel.gnue_procedure]._installLabel (aLabel)

      # After having all parameters installed to their procedures, we can then
      # run their second level initialization. Here it'll be determined wether
      # a procedure is a calculated field or not. Finally the procedures will
      # be installed to their corresponding class.
      for item in procedures.values ():
        item.secondaryInit (self.__rawModules)
        classes [item.gnue_class]._installProcedure (item)

      # Now finalize and verify all properties and procedures
      for item in properties.values ():
        item.finalize (classes)

      # The verification of procedures implicitly runs a type-check for all
      # calculated properties.
      map (verifyProcedure, procedures.values ())

      # To finish the build process, all references (unicode-strings) are
      # replaced by the corresponding wrapper instance. This way a
      # 'someClass.gnue_module.fullName' will work.
      for item in modules.values ():
        item.updateLinks (modules, classes)

    except:
      raise

    else:
      # We keep referernces to the old dictionaries before replacing them, so
      # we can release them later. This eliminates a time gap between having
      # and old and a new repository available.
      oldmodules = self.modules
      oldclasses = self.classes
      oldfilters = self.filters

      self.modules = modules
      self.classes = classes
      self.filters = filters

      # finally, if we had an old repository, release it
      if oldmodules is not None:
        self.release (oldmodules, oldclasses, oldfilters)

    self.filterOrder = self.__orderFilters ()


  # ---------------------------------------------------------------------------
  # Release resources of the repository
  # ---------------------------------------------------------------------------

  def release (self, modules = None, classes = None, filters = None):
    """
    Release the current repository dictionaries so garbage collection still
    works fine.

    @param modules: module dictionary to release
    @param classes: class dictionary to release
    @param filters: filter dictionary to release
    """

    if modules is None: modules = self.modules
    if classes is None: classes = self.classes
    if filters is None: filters = self.filters

    for item in modules.values ():
      item.updateLinks (None, None, True)

    filters.clear ()
    classes.clear ()
    modules.clear ()


  # ---------------------------------------------------------------------------
  #
  # ---------------------------------------------------------------------------

  def getFilterDefinition (self, language):
    """
    """

    result = []
    dcon   = data.connection (self._connections, self._database)

    try:
      for fk in self.filterOrder:
        result.append (self.filters [fk]._getFilterDefinition (language, dcon))

    finally:
      dcon.close ()

    return result


  # ---------------------------------------------------------------------------
  # Load the raw data from all repository tables
  # ---------------------------------------------------------------------------

  def __loadRawData (self):
    """
    This function reads all repository tables into dictionaries.
    """

    self._connection = data.connection (self._connections, self._database)

    try:
      self.__rawModules = self.__loadTable (u'gnue_module', [u'gnue_name'])
      self.__rawClasses = self.__loadTable (u'gnue_class', \
                                [u'gnue_module', u'gnue_name', u'gnue_filter'])
      self.__rawProperties = self.__loadTable (u'gnue_property', \
                  [u'gnue_module', u'gnue_class', u'gnue_name', u'gnue_length',
                   u'gnue_scale', u'gnue_nullable', u'gnue_type',
                   u'gnue_comment'])
      self.__rawProcedures = self.__loadTable (u'gnue_procedure', \
                [u'gnue_module', u'gnue_class', u'gnue_name', u'gnue_language',
                 u'gnue_length', u'gnue_nullable', u'gnue_scale', u'gnue_type',
                 u'gnue_comment', u'gnue_code'])

      self.__rawParameters = self.__loadTable (u'gnue_parameter', \
                [u'gnue_procedure', u'gnue_name', u'gnue_type', u'gnue_length',
                 u'gnue_scale', u'gnue_comment'])

      self.__rawLabels = self.__loadTable (u'gnue_label', \
          [u'gnue_property', u'gnue_procedure', u'gnue_language', u'gnue_page',
           u'gnue_label', u'gnue_position', u'gnue_search', u'gnue_info'])

    finally:
      self._connection.close ()


  # ---------------------------------------------------------------------------
  # Load fields from a given table
  # ---------------------------------------------------------------------------

  def __loadTable (self, table, fields):
    """
    This function reads all rows from a table into a dictionary.

    @param table: name of the table to load data from.
    @param fields: sequence of fields to load per record.

    @return: dictionary with the primary key as key and another dictionary as
        value, using the fieldnames as keys.
    """

    result    = {}
    contents  = {None: (table, None, None, fields)}
    resultSet = self._connection.query (contents, None, [{'name': u'gnue_id'}])

    try:
      for rec in resultSet:
        row = result.setdefault (rec.getField (u'gnue_id'), {})

        for field in fields:
          row [field] = rec.getField (field)

    finally:
      resultSet.close ()

    return result


  # ---------------------------------------------------------------------------
  # Create an ordered list of filters
  # ---------------------------------------------------------------------------

  def __orderFilters (self):
    
    # Build a dependency tree of all filters
    result  = []
    filters = {}

    for item in self.filters.values ():
      rec = filters.setdefault (item.gnue_id, [])
      if item.gnue_filter:
        rec.append (item.gnue_filter.gnue_id)

    # and transform that tree into a list starting with independent filters
    while filters:
      addition = []

      for (fid, deps) in filters.items ():
        # if a filter has no other dependencies, add it to the result and
        # remove all other references to it
        if not len (deps):
          addition.append (fid)

          for ref in filters.values ():
            if fid in ref:
              ref.remove (fid)

          del filters [fid]

      # Finally, if nothing was added to the result, but there are still
      # filters left in the tree we have a circular reference problem
      if not len (addition) and len (filters):
        raise CircularFilterError

      result.extend (addition)

    return result



# =============================================================================
# Base class implementing a repository element
# =============================================================================

class BaseItem:
  """
  Base class for repository items. The optionally given dictionary will be
  transferred into the instances __dict__ attribute.
  """

  # ---------------------------------------------------------------------------
  # Create a new instance of a repository item
  # ---------------------------------------------------------------------------

  def __init__ (self, classname, predefined = None):
    """
    Create a new repository item instance.

    @param classname: name of the class wrapped by this item
    @param predefined: dictionary with predefined values
    """

    self.classname = classname

    if predefined is not None:
      self.__dict__.update (predefined)


  # ---------------------------------------------------------------------------
  # Dictionary emulation for wrapper instances
  # ---------------------------------------------------------------------------

  def __getitem__ (self, attr):
    """
    Emulate a dictionary access to the wrapper instance.

    @param attr: attribute to return
    @return: value of the attribute 'attr'
    """

    return getattr (self, attr)


  # ---------------------------------------------------------------------------
  # Truth value testing
  # ---------------------------------------------------------------------------

  def __nonzero__ (self):

    return True



# =============================================================================
# MixIn class for classes having gnue_labels
# =============================================================================

class LabelsMixIn:

  # ---------------------------------------------------------------------------
  # Add a label to the labels dictionary
  # ---------------------------------------------------------------------------

  def _installLabel (self, aLabel):
    """
    Add a label to the list of labels in the proper language.

    @param aLabel: L{Label} to be added
    """

    self.labels [aLabel.gnue_language] = aLabel


  # ---------------------------------------------------------------------------
  # Get all labels usable within filter grouped by their language
  # ---------------------------------------------------------------------------

  def getFilterLabels (self, language):

    usable = {}

    for (lang, label) in self.labels.items ():
      if label.order:
        usable [lang] = label

    for lang in getLanguages (language):
      if lang.lower () in usable:
        return usable [lang.lower ()]

    return None



# =============================================================================
# This class implements a wrapper for module items of the repository
# =============================================================================

class Module (BaseItem):
  """
  A module object in the repository.

  @ivar classes: a caseless dictionary of all classes of the module. Each class
    will be listed in this dictionary by it's gnue_id as well as it's full name.
  @ivar fullname: fully qualified name of the module
  """

  # ---------------------------------------------------------------------------
  # Create a new module item of the repository
  # ---------------------------------------------------------------------------

  def __init__ (self, predefined = None):
    """
    Create a new module item.

    @param predefined: dictionary with predefined ('builtin') values
    """

    BaseItem.__init__ (self, u'gnue_module', predefined)

    self.classes  = CaselessDict (ClassNotFoundError)
    self.fullName = self.gnue_name


  # ---------------------------------------------------------------------------
  # Install a class into the module's class list
  # ---------------------------------------------------------------------------

  def _installClass (self, aClass):
    """
    Add a class element to the class dictionary.

    @param aClass: L{Class} instance to be added
    """

    self.classes [aClass.gnue_id]  = aClass
    self.classes [aClass.fullName] = aClass


  # ---------------------------------------------------------------------------
  # Update links of a module item
  # ---------------------------------------------------------------------------

  def updateLinks (self, modules, classes, unlink = False):
    """
    Update links of a module item by updating the links of all it's classes.

    @param modules: dictionary with all modules available
    @param classes: dictionary with all classes available
    @param unlink: if set to True, references will be cleared
    """

    for item in self.classes.values ():
      item.updateLinks (modules, classes, unlink)



# =============================================================================
# This class implements a class item of the repository
# =============================================================================

class Class (BaseItem):
  """
  A class object in the repository.

  @ivar fullName: fully qualified name (including the module name)
  @ivar table: name of the underlying table in the backend database
  @ivar properties: caseless dictionary with all properties of the class. Every
    property is accessible either by it's gnue_id or it's full name. This
    dictionary also contains I{calculated properties}
  @ivar procedures: caseless dictionary with all procedures of the class. Every
    procedure is accessible either by it's gnue_id or it's full name
  @ivar masters: caseless dictionary of all the master classes
    (master-/detail-relation). This dictionary contains the full name of the
    master classes as keys, and a sequence of those properties referencing that
    master class.
  @ivar OnInit: list of L{Procedure} objects containing OnInit triggers of the
    class
  @ivar OnChange: list of L{Procedure} objects containing OnChange triggers of
    the class
  @ivar OnValidate: list of L{Procedure} objects containing OnValidate triggers
    of the class
  @ivar OnDelete: list of L{Procedure} objects containing OnDelete triggers of
    the class
  """

  # ---------------------------------------------------------------------------
  # Create a new class item of the repository
  # ---------------------------------------------------------------------------

  def __init__ (self, predefined, moduleLookup):
    """
    Create a new class item.

    @param predefined: dictionary with predefined ('builtin') values
    @param moduleLookup: (raw) lookup-dictionary with all modules available
    """

    BaseItem.__init__ (self, u'gnue_class', predefined)

    moduleName    = moduleLookup [self.gnue_module] ['gnue_name']
    self.fullName = createName (moduleName, self.gnue_name)
    self.table    = self.fullName

    self.properties = CaselessDict (PropertyNotFoundError, self.fullName)
    self.procedures = CaselessDict (ProcedureNotFoundError, self.fullName)
    self.masters    = CaselessDict (ClassNotFoundError)

    self.labels       = CaselessDict (ClassLabelNotFoundError, self.fullName)
    self.labels ['C'] = self.fullName

    self.OnInit     = []
    self.OnChange   = []
    self.OnValidate = []
    self.OnDelete   = []


  # ---------------------------------------------------------------------------
  # Install a new property to the property dictionary of the class
  # ---------------------------------------------------------------------------

  def _installProperty (self, aProperty):
    """
    Add a property to the property dictionary

    @param aProperty: Property instance to be installed
    """

    self.properties [aProperty.gnue_id]  = aProperty
    self.properties [aProperty.fullName] = aProperty


  # ---------------------------------------------------------------------------
  # Install a new procedure to the procedure dictionary of the class
  # ---------------------------------------------------------------------------

  def _installProcedure (self, aProcedure):
    """
    Add a procedure to the procedure dictionary. If the procedure is a
    'calculated property', also add it to the property dictionary. If a
    procedure is a trigger, add it to the apropriate trigger sequence.

    @param aProcedure: procedure to be installed
    """

    self.procedures [aProcedure.gnue_id]  = aProcedure
    self.procedures [aProcedure.fullName] = aProcedure

    if aProcedure.isCalculated:
      aProperty = CalculatedProperty (aProcedure)
      self._installProperty (aProperty)

    # If the procedure is an OnInit trigger add it to the OnInit sequence. If 
    # the trigger is defined by the classes' module, it will be the first
    # trigger to fire.
    if aProcedure.gnue_name.upper () == 'ONINIT':
      if aProcedure.gnue_module == self.gnue_module:
        self.OnInit.insert (0, aProcedure)
      else:
        self.OnInit.append (aProcedure)

    # If the procedure is an OnValidate trigger add it to the trigger sequence.
    # The OnValidate trigger defined by the classes' module will be the last
    # one fired.
    if aProcedure.gnue_name.upper () == 'ONVALIDATE':
      if aProcedure.gnue_module == self.gnue_module:
        self.OnValidate.append (aProcedure)
      else:
        self.OnValidate.insert (0, aProcedure)

    # If the procedure is an OnChange trigger add it to the trigger sequence.
    if aProcedure.gnue_name.upper () == 'ONCHANGE':
      self.OnChange.append (aProcedure)

    # If the procedure is an OnDelete trigger add it to the trigger sequence.
    if aProcedure.gnue_name.upper () == 'ONDELETE':
      self.OnDelete.append (aProcedure)


  # ---------------------------------------------------------------------------
  # Add a given class as master of the class
  # ---------------------------------------------------------------------------

  def addMasterClass (self, aProperty, aMaster):
    """
    Add a given class to the dictionary of master-classes.

    @param aProperty: name of the property holding the pointer to master-class
    @param aMaster: L{Class} instance to be added
    """

    self.masters.setdefault (aMaster.fullName, []).append (aProperty)


  # ---------------------------------------------------------------------------
  # Update all links within a class wrapper
  # ---------------------------------------------------------------------------

  def updateLinks (self, modules, classes, unlink = False):
    """
    Updates links of a L{Class} item by updating the links of all it's
    properties and procedures.

    @param modules: dictionary with all modules available
    @param unlink: if set to True, references will be cleared
    """

    if unlink:
      self.gnue_module = None
      self.gnue_filter = None

      self.masters.clear ()

      del self.OnInit [:]
      del self.OnChange [:]
      del self.OnValidate [:]
      del self.OnDelete [:]

    else:
      if not isinstance (self.gnue_module, Module):
        self.gnue_module = modules [self.gnue_module]

      if self.gnue_filter and not isinstance (self.gnue_filter, Class):
        self.gnue_filter = classes [self.gnue_filter]

    for item in self.properties.values () + self.procedures.values ():
      item.updateLinks (self, modules, unlink)


  # ---------------------------------------------------------------------------

  def getFilterLabels (self, language):
    """
    """

    props = {}
    for item in self.properties.values ():
      if item.gnue_id in props:
        continue

      flabels = item.getFilterLabels (language)
      if flabels:
        props [item.gnue_id] = flabels

    if not props:
      labels = [Label ({u'gnue_id': None,
                     u'gnue_property': self.properties [u'gnue_id'],
                     u'gnue_procedure': None,
                     u'gnue_label': u'gnue_id',
                     u'gnue_language': u'C',
                     u'gnue_search': 0,
                     u'gnue_info': None,
                     u'gnue_page': None,
                     u'gnue_position': 0})]
    else:
      labels = props.values ()

    order = [(l.order, l.gnue_label, l) for l in labels]
    order.sort ()

    return [label for (pos, text, label) in order]

    
  # ---------------------------------------------------------------------------

  def getClassLabel (self, language):

    for lang in getLanguages (language):
      result = self.labels.get (lang)
      if result:
        return result

    return self.fullName


  # ---------------------------------------------------------------------------
  # Find a property or procedure by it's fully qualified name
  # ---------------------------------------------------------------------------

  def findItem (self, itemname):
    """
    Find a property of a procedure of the given name. If the name contains
    references (sepearted by a point) these references will be resolved.

    @param itemname: fully qualified name of the property or procedure to find
    @returns: the property or procedure instance of the found item

    @raises PropertyNotFoundError: if no such property or procedure exists.
    """

    parts   = itemname.split ('.', 1)
    current = parts.pop (0)

    if self.properties.has_key (current):
      if self.properties [current].isReference and parts:
        return self.properties [current].referencedClass.findItem (parts [0])

      elif parts:
        raise NotAReferenceError, (self.fullName, current)

      else:
        return self.properties [current]

    elif self.procedures.has_key (current):
      return self.procedures [current]

    else:
      raise PropertyNotFoundError, (current, self.fullName)



  # ---------------------------------------------------------------------------

  def _getFilterDefinition (self, language, dataCon):
    """
    """

    labels = self.getFilterLabels (language)

    master = None
    fields = [l.gnue_property.fullName for l in labels]
    if self.gnue_filter:
      master = self.gnue_filter.fullName
      fields.append (master)

    data      = {}
    contents  = {None: (self.table, None, None, fields)}
    resultSet = dataCon.query (contents, None, [{'name': u'gnue_id'}])

    try:
      for rec in resultSet:
        if master:
          group = data.setdefault (rec.getField (master), [])
        else:
          group = data.setdefault (None, [])

        row = {}
        for field in fields:
          if field != master:
            row [field] = rec.getField (field)

        group.append (row)

    finally:
      resultSet.close ()


    return ((self.fullName, self.getClassLabel (language)), \
        [(l.gnue_label, l.search, l.gnue_property.fullName) for l in labels],
        master, data)


# =============================================================================
# This class wraps a property item of the repository
# =============================================================================

class Property (LabelsMixIn, BaseItem):
  """
  A property object in the repository.

  @ivar fullName: fully qualified name of the property
  @ivar column: name of the column in the backend this property is bound to
  @ivar isCalculated: if True, the property is a calculated field, based on a
    L{Procedure}. In this case column is set to None
  @ivar isReference: True, if the property is a reference to another L{Class}.
  @ivar referencedClass: for a reference property this attribute contains the
    L{Class} instance of the referenced class (= master class)
  @ivar fullType: full datatype as used by gnue
  @ivar dbFullType: full datatype as used at the backend
  @ivar dbType: base datatype as used at the backend
  @ivar dbLength: length used at the backend
  @ivar dbScale: scale used at the backend
  """
  
  # ---------------------------------------------------------------------------
  # Create a new property item
  # ---------------------------------------------------------------------------

  def __init__ (self, predefined, moduleLookup):
    """
    Create a new property item.

    @param predefined: dictionary with predefined ('builtin') values
    @param moduleLookup: (raw) lookup-dictionary with all modules available
    """

    updateTypeInfo (predefined)
    BaseItem.__init__ (self, u'gnue_property', predefined)

    moduleName        = moduleLookup [self.gnue_module]['gnue_name']
    self.fullName     = createName (moduleName, self.gnue_name)
    self.column       = self.fullName
    self.labels       = CaselessDict (LabelNotFoundError, self.fullName)

    self.isCalculated    = False
    self.isReference     = False
    self.referencedClass = None


  # ---------------------------------------------------------------------------
  # Update the links of a property item
  # ---------------------------------------------------------------------------

  def updateLinks (self, aClass, modules, unlink = False):
    """
    Update the links of a property item to it's L{Module} and L{Class}. If
    unlink is set the references are cleared.

    @param aClass: L{Class} instance the property belongs to
    @param modules: module dictionary with all modules available
    @param unlink: this boolean Flag determines wether to establish links or to
        break them
    """

    if unlink:
      self.gnue_module = None
      self.gnue_class  = None
      self.referencedClass = None

    else:
      self.gnue_class = aClass
      if not isinstance (self.gnue_module, Module):
        self.gnue_module = modules [self.gnue_module]

    for item in self.labels.values ():
      item.updateLinks (self, unlink)


  # ---------------------------------------------------------------------------
  # Finalize the definition of a property
  # ---------------------------------------------------------------------------

  def finalize (self, classes):
    """
    Verify the type of the property and set the reference flags for rerference
    properties.

    @param classes: dictionary with all classes available
    """

    r = verifyType (self.gnue_type, self.gnue_length, self.gnue_scale, classes)

    self.isReference     = r is not None
    self.referencedClass = r

    # If the property is a reference to another class, the referenced class
    # is a master of this class
    if r is not None:
      classes [self.gnue_class].addMasterClass (self.fullName, r)

    # a 'gnue_id' property defines the class labels (if any)
    parent = classes [self.gnue_class]
    if self.fullName == u'gnue_id':
      for (lang, label) in self.labels.items ():
        parent.labels [lang] = label.gnue_label


# =============================================================================
# This class wraps a calculated property (= special procedures)
# =============================================================================

class CalculatedProperty (LabelsMixIn, BaseItem):
  """
  A calculated property object in the repository. A calculated property is a
  special kind of a L{Procedure}.

  @ivar fullName: fully qualified name of the property
  @ivar column: allways None
  @ivar isReference: allways False
  @ivar referencedClass: allways None
  @ivar isCalculated: allways True
  @ivar procedure: the L{Procedure} instance defining the property
  @ivar fullType: full datatype as used by gnue
  @ivar dbFullType: full datatype as used at the backend
  @ivar dbType: base datatype as used at the backend
  @ivar dbLength: length used at the backend
  @ivar dbScale: scale used at the backend
  """

  # ---------------------------------------------------------------------------
  # Create a new calculated property
  # ---------------------------------------------------------------------------

  def __init__ (self, aProcedure):
    """
    Create a new calculated property item.

    @param aProcedure: procedure to bind as calculated property
    """

    predefined = {
      'gnue_id'      : aProcedure.gnue_id,
      'gnue_name'    : aProcedure.calcName,
      'gnue_type'    : aProcedure.gnue_type,
      'gnue_length'  : aProcedure.gnue_length,
      'gnue_scale'   : aProcedure.gnue_scale,
      'gnue_module'  : aProcedure.gnue_module,
      'gnue_class'   : aProcedure.gnue_class,
      'gnue_nullable': aProcedure.gnue_nullable}
    updateTypeInfo (predefined)

    BaseItem.__init__ (self, u'gnue_procedure', predefined)

    self.fullName  = aProcedure.calcFullName
    self.column    = None

    self.isReference     = False
    self.referencedClass = None
    self.isCalculated    = True
    self.procedure       = aProcedure
    self.labels          = CaselessDict (LabelNotFoundError, self.fullName)

    # Make sure to have all labels of the procedure bound to the calculated
    # field as well
    for item in aProcedure.labels.values ():
      self._installLabel (item)


  # ---------------------------------------------------------------------------
  # Update the links of a calculated property item
  # ---------------------------------------------------------------------------

  def updateLinks (self, aClass, modules, unlink = False):
    """
    If in unlink mode this function clears the pointer to the bound procedure.

    @param aClass: L{Class} instance the property belongs to
    @param modules: module dictionary with all modules available
    @param unlink: this boolean Flag determines wether to establish links or to
        break them
    """

    if unlink:
      self.gnue_module = None
      self.gnue_class  = None
      self.procedure   = None

    else:
      self.gnue_class = aClass
      if not isinstance (self.gnue_module, Module):
        self.gnue_module = modules [self.gnue_module]

    for item in self.labels.values ():
      item.updateLinks (self, unlink)



# =============================================================================
# This class wraps a procedure item of the repository
# =============================================================================

class Procedure (LabelsMixIn, BaseItem):
  """
  A procedure object in the repository.

  @ivar fullName: fully qualified name of the procedure
  @ivar isCalculated: if True, this procedure matches all requirements of a
    calculated property.
  @ivar calcFullName: fully qualified name of the calculated property
  @ivar calcName: name of the calculated property withouth any module name
  @ivar parameters: caseless dictionary with all L{Parameter}s of the procedure
  """

  # ---------------------------------------------------------------------------
  # Create a new procedure wrapper
  # ---------------------------------------------------------------------------

  def __init__ (self, predefined, moduleLookup):
    """
    Create a new procedure item.

    @param predefined: dictionary with predefined ('builtin') values
    @param moduleLookup: (raw) lookup-dictionary with all modules available
    """

    BaseItem.__init__ (self, u'gnue_procedure', predefined)

    moduleName    = moduleLookup [self.gnue_module]['gnue_name']
    self.fullName = createName (moduleName, self.gnue_name)

    self.isCalculated = False
    self.calcFullName = None
    self.calcName     = None

    self.parameters = CaselessDict (ParameterNotFoundError)
    self.labels     = CaselessDict (ProcLabelNotFoundError, self.fullName)


  # ---------------------------------------------------------------------------
  # Second level initialization for a procedure
  # ---------------------------------------------------------------------------

  def secondaryInit (self, moduleLookup):
    """
    Check wether a procedure matches all requirements of a calculated property

    @param moduleLookup: lookup dictionary for modules
    """

    self.isCalculated = self.gnue_type is not None and \
        not len (self.parameters) and self.gnue_name [:3].lower () == 'get'

    if self.isCalculated:
      self.calcName     = self.gnue_name [3:]
      moduleName        = moduleLookup [self.gnue_module]['gnue_name']
      self.calcFullName = createName (moduleName, self.calcName)


  # ---------------------------------------------------------------------------
  # Install a parameter to the procedures parameter dictionary
  # ---------------------------------------------------------------------------

  def _installParameter (self, aParameter):
    """
    Add a L{Parameter} to the procedure's parameter dictionary.

    @param aParameter: the parameter to be installed.
    """

    self.parameters [aParameter.gnue_id]  = aParameter
    self.parameters [aParameter.fullName] = aParameter


  # ---------------------------------------------------------------------------
  # Update the links of a procedure item
  # ---------------------------------------------------------------------------

  def updateLinks (self, aClass, modules, unlink = False):
    """
    Update the links of a procedure item to it's module and class. If unlink is
    set the references are cleared.

    @param aClass: L{Class} instance the procedure belongs to
    @param modules: module dictionary with all modules available
    @param unlink: this boolean Flag determines wether to establish links or to
        break them
    """

    if unlink:
      self.gnue_module = None
      self.gnue_class  = None

    else:
      self.gnue_class = aClass
      if not isinstance (self.gnue_module, Module):
        self.gnue_module = modules [self.gnue_module]

    for param in self.parameters.values ():
      param.updateLinks (self, unlink)

    for item in self.labels.values ():
      item.updateLinks (self, unlink)



# =============================================================================
# This class implements a parameter item of the repository
# =============================================================================

class Parameter (BaseItem):
  """
  A parameter object of the repository.

  @ivar fullName: fully qualified name of the paramter
  @ivar dbType: the type of the parameter in the backend database
  """

  # ---------------------------------------------------------------------------
  # Create a new parameter wrapper instance
  # ---------------------------------------------------------------------------

  def __init__ (self, predefined):
    """
    Create a new parameter item.

    @param predefined: dictionary with predefined ('builtin') values
    """

    BaseItem.__init__ (self, u'gnue_parameter', predefined)

    self.fullName = self.gnue_name

    if self.gnue_type in NONREF_TYPES:
      self.dbType = self.gnue_type
    else:
      self.dbType = REF_TYPE


  # ---------------------------------------------------------------------------
  # Update links
  # ---------------------------------------------------------------------------

  def updateLinks (self, aProcedure, unlink = False):
    """
    This functions updates the link to the owning procedure.

    @param aProcedure: the owning procedure
    @param unlink: if set to True the reference will be cleared
    """

    if unlink:
      self.gnue_procedure = None

    else:
      self.gnue_procedure = aProcedure


# =============================================================================
# Label class
# =============================================================================

class Label (BaseItem):
  """
  A label object in the repository.

  @ivar order: the search order of a label which is either gnue_search or if
    that is None gnue_info
  @ivar search: True, if gnue_search is not None
  """

  # ---------------------------------------------------------------------------
  # Create a new label item
  # ---------------------------------------------------------------------------

  def __init__ (self, predefined):
    """
    Create a new label item.

    @param predefined: dictionary with predefined ('builtin') values
    """

    BaseItem.__init__ (self, u'gnue_label', predefined)

    self.order  = self.gnue_search or self.gnue_info
    self.search = self.gnue_search is not None


  # ---------------------------------------------------------------------------
  # Update links
  # ---------------------------------------------------------------------------

  def updateLinks (self, aParent, unlink = False):
    """
    Update the link to the owning property or procedure of a label

    @param aParent: the owning property or procedure
    @param unlink: if set to True the reference will be cleared
    """

    if unlink:
      self.gnue_procedure = None
      self.gnue_property  = None

    else:
      if self.gnue_property:
        self.gnue_property  = aParent
      else:
        self.gnue_procedure = aParent


# =============================================================================
# Type checking support
# =============================================================================

NONREF_TYPES = {'boolean': True, 'date'  : True, 'datetime': True,
                'number' : True, 'string': True, 'time'    : True}
BASE_TYPES   = {'id': True}
BASE_TYPES.update (NONREF_TYPES)

NOLS_TYPES   = {'id': 1, 'date': 1, 'time': 1, 'datetime': 1, 'boolean': 1}

REF_TYPE     = "string"
REF_LENGTH   = 32
REF_SCALE    = 0


# -----------------------------------------------------------------------------
# Check if a combination of typename, length and scale is valid
# -----------------------------------------------------------------------------

def verifyBasetype (typename, length, scale):
  """
  Verify a given typename with length and scale. If this combination makes no
  sense a TypeFormatError will be raised. If typename is no valid base type a
  TypeNameError will be raised.

  @param typename: name of the datatype
  @param length: length of the datatype
  @param scale: scale of the datatype

  @raises TypeFormatError: the combination of length and scale is not valid
  @raises TypeNameError: typename is not a valid base type
  """

  if not BASE_TYPES.has_key (typename):
    raise TypeNameError, (typename)

  # A string type must not have a scale
  if typename == 'string':
    if scale:
      raise TypeFormatError, u_("string does not support 'scale'")

  # All of the following types must not have length nor scale
  if NOLS_TYPES.has_key (typename):
    if length:
      raise TypeFormatError, u_("%s does not support 'length'") % typename
    if scale:
      raise TypeFormatError, u_("%s does not support 'scale'") % typename

  # A number must have at least a 'length'
  if typename == 'number':
    if not length:
      raise TypeFormatError, u_("number without 'length'")


# -----------------------------------------------------------------------------
# Verify a given type
# -----------------------------------------------------------------------------

def verifyType (typename, length, scale, classes):
  """
  This function verifies a given type, length and scale combination, optionally
  using the given class dictionary for lookups of references.

  @param typename: name of the datatype
  @param length: length of the datatype
  @param scale: scale of the datatype
  @param classes: class dictionary to check for reference types

  @return: If 'typename' is a reference type this function returns the class
      definition of this reference type, otherwise None

  @raises TypeFormatError: the type, length and scale combination is not valid
  """

  if classes.has_key (typename):
    if length:
      raise TypeFormatError, u_("Reference types must not have a 'length'")

    if scale:
      raise TypeFormatError, u_("Reference types must not have a 'scale'")

    return classes [typename]

  else:
    verifyBasetype (typename, length, scale)
    return None


# -----------------------------------------------------------------------------
# Verify a procedure definition
# -----------------------------------------------------------------------------

def verifyProcedure (aProc):
  """
  This function checks the resulttype of a procedure definition, and all
  parameter types (if available).

  @param aProc: procedure wrapper item to be checked
  """

  # If a result type is specified, check it
  if aProc.gnue_type is not None:
    verifyBasetype (aProc.gnue_type, aProc.gnue_length, aProc.gnue_scale)

  else:
    # otherwise there must not be anything concerning a result type
    if aProc.gnue_length:
      raise TypeFormatError, u_("%s: Procedure has no result, but a 'length' "
                                "is specified.") % aProc.fullName
    if aProc.gnue_scale:
      raise TypeFormatError, u_("%s: Procedure has no result, but a 'scale' "
                                "is specified.") % aProc.fullName

  # verify all given parameter types
  for pa in aProc.parameters.values ():
    try:
      verifyBasetype (pa.gnue_type, pa.gnue_length, pa.gnue_scale)

    except ValidationError, vErr:
      raise ParameterValidationError, \
          (aProc.fullName, pa.fullName, vErr.message)


# -----------------------------------------------------------------------------
# Create a type information dictionary for a given wrapper item
# -----------------------------------------------------------------------------

def updateTypeInfo (item):
  """
  This function updates all type information in a given dictionary. It assumes
  to find the keys 'gnue_type', 'gnue_length' and 'gnue_scale' in the
  dictionary. After updating the dictionary will contain the following
  additional keys: fullType, dbFullType, dbType, dbLength and dbScale

  @param item: dictionary to update type information
  """

  gType  = dbType   = fullType = item ['gnue_type']
  length = dbLength = item ['gnue_length']
  scale  = dbScale  = item ['gnue_scale']

  if gType in ["string", "number"]:
    if gType == "number" and length and scale:
      fullType = "%s(%d,%d)" % (gType, length, scale)

    elif length:
      fullType = "%s(%d)" % (gType, length)

  # build database specific type information
  if not NONREF_TYPES.has_key (gType):
    (dbType, dbLength, dbScale) = (REF_TYPE, REF_LENGTH, REF_SCALE)

  dbFullType = dbType
  if dbType in ["string", "number"]:
    if dbType == "number" and dbLength and dbScale:
      dbFullType = "%s(%d,%d)" % (dbType, dbLength, dbScale)

    elif dbLength:
      dbFullType = "%s(%d)" % (dbType, dbLength)

  item.update ({'fullType'  : fullType,
                'dbFullType': dbFullType,
                'dbType'    : dbType,
                'dbLength'  : dbLength,
                'dbScale'   : dbScale})


# -----------------------------------------------------------------------------
# Create a fully qualified name from namespace and identifier
# -----------------------------------------------------------------------------

def createName (namespace, identifier):
  """
  This function creates a fully qualified name from namespace and identifier.
  If the result is not a valid name, an InvalidNameError will be raised.

  @param namespace: the namespace to use
  @param identifier: the identifier to use

  @return: fully qualified name: 'namespace_idendifier'
  """

  result = "%s_%s" % (namespace, identifier)

  if len (result.split ('_')) != 2:
    raise InvalidNameError, result

  return result


# -----------------------------------------------------------------------------
# Split a fully qualified name into namespace and identifier
# -----------------------------------------------------------------------------

def splitName (name):
  """
  This function splits the given name into namespace and identifier parts. If
  name does not contain a namespace at all, it will be left empty. If the
  resulting tuple has more than two parts an InvalidNameError will be raised.

  @param name: the name to be split into namespace and identifier

  @return: tuple (namespace, identifier) where namespace could be an empty
      string if name had no namespace information at all
  """

  parts = name.split ('_')
  items = len (parts)

  if items == 1:
    result = ('', name)

  elif items == 2:
    result = tuple (parts)

  else:
    raise InvalidNameError, name

  return result


# -----------------------------------------------------------------------------
# Create a list of languages in descending order
# -----------------------------------------------------------------------------

def getLanguages (language):
  """
  Create an ordered list of languages to fetch labels for. The list starts with
  the most specific language and contains at least 'C'. The language 'de_AT'
  will result in a list ['de_AT', 'de', 'C'].

  @param language: language (locale) to create a list for, e.g. 'de_AT'
  @return: list of language-codes in descending order
  """

  result = []

  if language is not None:
    if '_' in language:
      result = [language.split ('_') [0]]

    if not language in result:
      result.insert (0, language)

  if not 'C' in result:
    result.append ('C')

  return result
