# GNU Enterprise Forms - GF Object Hierarchy - Box
#
# Copyright 2001-2009 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 3, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: GFGrid.py 10016 2009-10-28 14:41:59Z reinhard $
"""
Logical box support
"""

from gnue.forms.GFObjects.GFTabStop import GFTabStop
from gnue.forms.GFObjects.GFGridLine import GFGridLine
from gnue.forms.GFObjects.GFContainer import GFContainer

__all__ = ['GFGrid']

# =============================================================================
# <vbox>
# =============================================================================

class GFGrid(GFContainer):

    # -------------------------------------------------------------------------
    # Attributes
    # -------------------------------------------------------------------------

    label = None


    # -------------------------------------------------------------------------
    # Constructor
    # -------------------------------------------------------------------------

    def __init__(self, parent=None):
        GFContainer.__init__(self, parent, "GFGrid")


    # -------------------------------------------------------------------------
    # Initialisation
    # -------------------------------------------------------------------------

    def _phase_1_init_(self):
        GFContainer._phase_1_init_(self)

        self.__rows = int(getattr(self, "rows", 1))
        self.walk(self.__init_rows_walker)
        self._block._rows = self.__rows
        self._block.register_scrollbar(self)

    # -------------------------------------------------------------------------

    def __init_rows_walker(self, item):
        if (item is not self) and isinstance(item, (GFTabStop, GFGridLine)):
            item._rows = self.__rows
            item._gap = len(self._children) - 1


    # -------------------------------------------------------------------------
    # UI event handlers
    # -------------------------------------------------------------------------

    def _event_rows_changed(self, new_rows):
        """
        Notify the grid that the number of visible rows has changed due to a
        user interface event (the user resized the grid).
        """

        old_rows = self.__rows
        self.__rows = new_rows
        self.walk(self.__rows_walker)
        self._block._event_rows_changed(new_rows)

    # -------------------------------------------------------------------------

    def __rows_walker(self, item):
        if (item is not self) and isinstance(item, (GFTabStop, GFGridLine)):
            item.rows_changed(self.__rows)


    # -------------------------------------------------------------------------
    # Public methods
    # -------------------------------------------------------------------------

    def line_up(self):
        """
        Move the database cursor behind this grid up one record.
        """

        self._block.prev_record()

    # -------------------------------------------------------------------------

    def line_down(self):
        """
        Move the database cursor behind this grid down one record.
        """

        self._block.next_record()

    # -------------------------------------------------------------------------

    def page_up(self):
        """
        Scroll the grid up (backwards) by one screen page.
        """

        self._block._event_scroll_delta(-(self.__rows-1))

    # -------------------------------------------------------------------------

    def page_down(self):
        """
        Scroll the grid down (forward) by one screen page.
        """

        self._block._event_scroll_delta(self.__rows-1)


    # -------------------------------------------------------------------------
    # Call the UI widget to adjust the scrollbar to a given position
    # -------------------------------------------------------------------------

    def adjust_scrollbar(self, current, count):
        """
        Update the UI scrollbar to reflect a new position or record-count.

        @param current: current first visible record number in the block
        @param count: number of records in the block
        """

        if self.uiWidget is not None:
            self.uiWidget._ui_adjust_scrollbar_(current, self.__rows, count)
