/*
   XGFontEnumerator

   NSFontManager helper for GNUstep GUI X/GPS Backend

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author: Ovidiu Predescu <ovidiu@bx.logicnet.ro>
   Date: February 1997
   A completely rewritten version of the original source of Scott Christley.
   Modified:  Fred Kiefer <FredKiefer@gmx.de>
   Date: Febuary 2000
   Added some X calls and changed the overall structure
 
   This file is part of the GNUstep GUI X/GPS Library.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA.
*/

#include <config.h>
#include <stdio.h>

#include <AppKit/GSFontInfo.h>
#include <gnustep/xgps/XGContext.h>
#include <gnustep/xgps/XGContextPrivate.h>

#define stringify_it(X) #X

// X atoms used to query a font
static Atom XA_WEIGHT_NAME;
static Atom XA_SLANT;
static Atom XA_SETWIDTH_NAME;
static Atom XA_ADD_STYLE_NAME;
static Atom XA_PIXEL_SIZE;
static Atom XA_RESOLUTION_X;
static Atom XA_RESOLUTION_Y;
static Atom XA_SPACING;
static Atom XA_AVERAGE_WIDTH;
static Atom XA_CHARSET_REGISTRY;
static Atom XA_CHARSET_ENCODING;
static Atom XA_FACE_NAME;

static NSMutableDictionary* creationDictionary;

/*
 * Initialise the X atoms we are going to use
 */
BOOL XGInitAtoms(Display *dpy)
{
  if (!dpy)
    {
      NSDebugLog(@"No Display opened in XGInitAtoms");      
      return NO;
    }

  XA_WEIGHT_NAME = XInternAtom(dpy, "WEIGHT_NAME", False);
  XA_SLANT = XInternAtom(dpy, "SLANT", False);
  XA_SETWIDTH_NAME = XInternAtom(dpy, "SETWIDTH_NAME", False);
  XA_ADD_STYLE_NAME = XInternAtom(dpy, "ADD_STYLE_NAME", False);
  XA_PIXEL_SIZE = XInternAtom(dpy, "PIXEL_SIZE", False);
  XA_RESOLUTION_X = XInternAtom(dpy, "RESOLUTION_X", False);
  XA_RESOLUTION_Y = XInternAtom(dpy, "RESOLUTION_Y", False);
  XA_SPACING = XInternAtom(dpy, "SPACING", False);
  XA_AVERAGE_WIDTH = XInternAtom(dpy, "AVERAGE_WIDTH", False);
  XA_CHARSET_REGISTRY = XInternAtom(dpy, "CHARSET_REGISTRY", False);
  XA_CHARSET_ENCODING = XInternAtom(dpy, "CHARSET_ENCODING", False);
  XA_FACE_NAME = XInternAtom(dpy, "FACE_NAME", False);

  return YES;
}

/*
 * Read an X Font property of type unsigned long
 */
static unsigned long XGFontPropULong(Display *dpy, 
				     XFontStruct *font_struct,
				     Atom atom)
{
  unsigned long lvalue;
  
  if (XGetFontProperty(font_struct, atom, &lvalue))
    return lvalue;
  else
    return 0;
}

/*
 * Read an X Font property of type string
 */
static NSString *XGFontPropString(Display *dpy, 
				  XFontStruct *font_struct,
				  Atom atom)
{
  unsigned long	lvalue;
  char		*value;
  NSString	*ret = nil;

  if (XGetFontProperty(font_struct, atom, &lvalue) && dpy)
    {
      value = XGetAtomName(dpy, lvalue);
      if (value != NULL)
	{
	  // We convert all props to lowercase so comparing is easier
	  ret = [[NSString stringWithCString: value] lowercaseString];
	  XFree(value);
	}
    }
  
  return ret;
}

NSString *XGFontName(Display *dpy, XFontStruct *font_struct)
{
  return XGFontPropString(dpy, font_struct, XA_FONT);
}

float XGFontPointSize(Display *dpy, XFontStruct *font_struct)
{
  float size = 12.0;
  long pointSize;

  /*
   * We use pixel size here not point size, which is about 10 times its size.
   * Perhaps we will change that later on!
   */
  pointSize = XGFontPropULong(dpy, font_struct, XA_PIXEL_SIZE);
  if (pointSize != 0)
    {
      size = (float) pointSize;
    }

  return size;
}

BOOL XGFontIsFixedPitch(Display *dpy, XFontStruct *font_struct)
{
  /* Is this font fixed pitch? default, NO */
  BOOL fixedFont = NO;
  NSString *spacing;

  /*
   * We could also check the value of MONOSPACED, but I have never seen it set.
   * The way is is done here looks strange to me.
   */
  spacing = XGFontPropString(dpy, font_struct, XA_SPACING);
  if (spacing != nil)
    {
      /* Only proportional fonts are not fixed pitch... */
      if ([spacing cString][0] != 'p')
	fixedFont = YES;
    }

  /*
   * We could calculate the pitch from the XLFD but that does not seem to be
   * saved. If we can't get the property, say no and cope.
   */
  return fixedFont;
}

NSString *XGFontFamily(Display *dpy, XFontStruct *font_struct)
{
  NSString *family;

  family = XGFontPropString(dpy, font_struct, XA_FAMILY_NAME);
  if (family == nil)
    return @"Unknown";	// FIXME: We should return the font name instead
  
  return [family capitalizedString];
}

// Compute the integer weight from a string
static int XGWeightForString(const char *weight)
{
  // FIXME: We should use a dictionary here!
  if (!weight || !strlen(weight))
    return 5;
  else if (!strcasecmp(weight, "ultralight"))
    return 1;
  else if (!strcasecmp(weight, "thin"))
    return 2;
  else if (!strcasecmp(weight, "light") || !strcasecmp(weight, "extralight"))
    return 3;
  else if (!strcasecmp(weight, "book"))
    return 4;
  else if (!strcasecmp(weight, "regular") || !strcasecmp(weight, "plain")
    || !strcasecmp(weight, "display") || !strcasecmp(weight, "roman")
    || !strcasecmp(weight, "semilight"))
    return 5;
  else if (!strcasecmp(weight, "medium"))
    return 6;
  else if (!strcasecmp(weight, "demi") || !strcasecmp(weight, "demibold"))
    return 7;
  else if (!strcasecmp(weight, "semi") || !strcasecmp(weight, "semibold"))
    return 8;
  else if (!strcasecmp(weight, "bold"))
    return 9;
  else if (!strcasecmp(weight, "extra") || !strcasecmp(weight, "extrabold"))
    return 10;
  else if (!strcasecmp(weight, "heavy") || !strcasecmp(weight, "heavyface"))
    return 11;
  else if (!strcasecmp(weight, "black") || !strcasecmp(weight, "super") 
    || !strcasecmp(weight, "ultrabold"))
    return 12;
  else if (!strcasecmp(weight, "ultra") || !strcasecmp(weight, "ultrablack") 
    || !strcasecmp(weight, "fat"))
    return 13;
  else if (!strcasecmp(weight, "extrablack") || !strcasecmp(weight, "obese") 
    || !strcasecmp(weight, "nord"))
    return 14;
 
  NSDebugLog(@"Unknown font weight %s", weight);
  return 5;
} 

// Get the weight of a X font
int XGWeightOfFont(Display *dpy, XFontStruct *info)
{
  int		w = 5;
  NSString	*string = XGFontPropString(dpy, info, XA_WEIGHT_NAME);

  if (string != nil)
    {
      w = XGWeightForString([string cString]);
    }

  return w;
}

// Get the traits of a X font
NSFontTraitMask XGTraitsOfFont(Display *dpy, XFontStruct *info)
{
  NSFontTraitMask	mask = 0;
  NSString		*string;
  int			w = XGWeightOfFont(dpy, info);
  
  if (w >= 9)
    mask |= NSBoldFontMask;
  else
    mask |= NSUnboldFontMask;
  
  if (XGFontIsFixedPitch(dpy, info))
    mask |= NSFixedPitchFontMask;
  
  string = XGFontPropString(dpy, info, XA_SLANT);
  if (string != nil)
    {
      char c = [string cString][0];
      
      if (c == 'o' || c == 'i')
	mask |= NSItalicFontMask;
      else
	mask |= NSUnitalicFontMask;
    }
  
  string = XGFontPropString(dpy, info, XA_CHARSET_REGISTRY);
  if (string != nil)
    {
      if (![string isEqualToString: @"iso8859"])
	mask |= NSNonStandardCharacterSetFontMask;
    }
  string = XGFontPropString(dpy, info, XA_CHARSET_ENCODING);
  if (string != nil)
    {
      if (![string isEqualToString: @"1"])
	mask |= NSNonStandardCharacterSetFontMask;
    }
  
  string = XGFontPropString(dpy, info, XA_SETWIDTH_NAME);
  if (string != nil)
    {
      if ([string isEqualToString: @"narrow"])
	mask |= NSNarrowFontMask;
    }
    
  string = XGFontPropString(dpy, info, XA_SPACING);
  if (string != nil)
    {
      if ([string isEqualToString: @"c"])
	mask |= NSCondensedFontMask;
    }
  
  //FIXME: How can I find out about the other traits?
  /*
    unsigned long weight = XGFontPropULong(dpy, info, XA_WEIGHT);
  */

  return mask;
}


// Fills in the size into an creation string to make it an X font name
NSString *XGXFontName(NSString *fontName, float size)
{
  NSString *creationName = [creationDictionary objectForKey: fontName];

  if (creationName != nil)
    return [NSString stringWithFormat: creationName, (int)size];
  else
    // if the desired XLFD does not exist we fudge it with 9x15.
    return  @"9x15";
}

@implementation XGFontEnumerator

static NSString		*cacheName;
static NSDictionary	*cache;

static BOOL
loadCache(XGFontEnumerator *self, BOOL async)
{
  id		o;
  NSString	*path;
  NSDictionary	*env = [[NSProcessInfo processInfo] environment];

  if (cacheName == nil)
    {
      NSFileManager	*mgr;
      BOOL		flag;

      /*
       * The home directory for per-user information is given by
       * the GNUSTEP_USER_ROOT environment variable, or is assumed
       * to be the 'GNUstep' subdirectory of the users home directory.
       */
      if (!env || !(path = [env objectForKey: @"GNUSTEP_USER_ROOT"]))
	{
	  path = [NSHomeDirectory() stringByAppendingPathComponent: @"GNUstep"];
	}
      mgr = [NSFileManager defaultManager];
      if ([mgr fileExistsAtPath: path isDirectory: &flag] == NO || flag == NO)
	{
	  NSLog(@"Home directory '%@' not available!", path);
	  return NO;
	}
      path = [path stringByAppendingPathComponent: @"Library"];
      if ([mgr fileExistsAtPath: path] == NO)
	{
	  [mgr createDirectoryAtPath: path attributes: nil];
	}
      if ([mgr fileExistsAtPath: path isDirectory: &flag] == NO || flag == NO)
	{
	  NSLog(@"Library directory '%@' not available!", path);
	  return NO;
	}
      path = [path stringByAppendingPathComponent: @"Fonts"];
      if ([mgr fileExistsAtPath: path] == NO)
	{
	  [mgr createDirectoryAtPath: path attributes: nil];
	}
      if ([mgr fileExistsAtPath: path isDirectory: &flag] == NO || flag == NO)
	{
	  NSLog(@"Fonts directory '%@' not available!", path);
	  return NO;
	}
      path = [path stringByAppendingPathComponent: @"Cache"];
      if ([mgr fileExistsAtPath: path] == NO)
	{
	  [mgr createDirectoryAtPath: path attributes: nil];
	}
      if ([mgr fileExistsAtPath: path isDirectory: &flag] == NO || flag == NO)
	{
	  NSLog(@"Fonts directory '%@' not available!", path);
	  return NO;
	}
      cacheName = [path stringByAppendingPathComponent: 
	[NSString stringWithCString: XDisplayName(0)]];
      RETAIN(cacheName);
    }

  o = [NSUnarchiver unarchiveObjectWithFile: cacheName];
  if (o == nil)
    {
      NSTask	*task;
   
      if (async == NO)
	{
	  NSLog(@"No font cache available - building new one - this may take several seconds (or minutes on a slow machine with lots of fonts)");
	}
      if (!env || !(path = [env objectForKey: @"GNUSTEP_SYSTEM_ROOT"]))
	{
	  path = [NSString stringWithCString: 
			     stringify_it(GNUSTEP_INSTALL_PREFIX)];
	}
      path = [path stringByAppendingPathComponent: @"Tools"];
      path = [path stringByAppendingPathComponent: @"font_cacher"];
      task = [NSTask launchedTaskWithLaunchPath: path
				      arguments: nil];
      if (task == nil || async == YES)
	{
	  return NO;
	}
      [task waitUntilExit];
      o = [NSUnarchiver unarchiveObjectWithFile: cacheName];
      if (o == nil)
	{
	  NSLog(@"Error - font cache doesn't exist");
	}
    }
  ASSIGN(cache, o);
  if (cache != nil)
    {
      self->allFontNames = [cache objectForKey: @"AllFontNames"];
      self->allFontFamilies = [cache objectForKey: @"AllFontFamilies"];
      creationDictionary = [cache objectForKey: @"CreationDictionary"];
    }
  return YES;
}

- (void) enumerateFontsAndFamilies
{
  Display *dpy = [XGContext currentXDisplay];

  if (cache == nil)
    {
      loadCache(self, NO);
    }
  XGInitAtoms(dpy);
}

- initWithFontManager: manager
{
  [super init];
  [self enumerateFontsAndFamilies];
  return self;
}

- (NSArray*) availableFonts
{
  NSMutableArray *fontNames;
  NSEnumerator	*enumerator;
  NSString	*name;

  fontNames = [NSMutableArray arrayWithCapacity: [allFontNames count]];
  enumerator = [allFontNames objectEnumerator];
  while ((name = [enumerator nextObject]) != nil)
    {
      [fontNames addObject: name];
    }

  return fontNames;
}

- (NSArray*) availableFontFamilies
{
  return [allFontFamilies allKeys];
}

- (NSArray*) availableMembersOfFontFamily: (NSString*)family
{
  int		i;
  NSArray	*fontsList;
  NSMutableArray *fontDefs;
  
  fontsList = [allFontFamilies objectForKey: family];
  if (fontsList == nil)
    return nil;
  fontDefs = [NSMutableArray array];

  for (i = 0; i < [fontsList count]; i++)
    {
      [fontDefs addObject: [fontsList objectAtIndex: i]];
    }

  return fontDefs;
}

@end
