/*
   Font cacher for GNUstep GUI X/GPS Backend

   Copyright (C) 2000 Free Software Foundation, Inc.

   Author:  Fred Kiefer <FredKiefer@gmx.de> and Richard Frith-Macdonald
   Date: Febuary 2000
 
   This file is part of the GNUstep GUI X/GPS Library.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA.
*/

#include <config.h>
#include <stdio.h>
#include <Foundation/Foundation.h>
#include <AppKit/AppKit.h>
#include <X11/Xatom.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#define	stringify_it(X)	#X
#define	makever(X) stringify_it(X)

extern BOOL XGInitAtoms(Display *dpy);
extern NSString *XGFontName(Display *dpy, XFontStruct *font_struct);
extern NSString *XGFontFamily(Display *dpy, XFontStruct *font_struct);
extern int XGWeightOfFont(Display *dpy, XFontStruct *info);
extern NSFontTraitMask XGTraitsOfFont(Display *dpy, XFontStruct *info);

// Find a suitable type face name for a full X font name, which 
// has already been split into parts seperated by -
//-sony-fixed-medium-r-normal--16-150-75-75-c-80-iso8859-1
// becomes Normal
static NSString *
XGFaceNameFromParts(NSArray *parts)
{
  NSString *faceName;
  //NSString *family = [[parts objectAtIndex: 2] capitalizedString];
  NSString *face = [[parts objectAtIndex: 3] capitalizedString];
  NSString *slant = [[parts objectAtIndex: 4] capitalizedString];
  NSString *weight = [[parts objectAtIndex: 5] capitalizedString];
  NSString *add = [[parts objectAtIndex: 6] capitalizedString];
 
  if ([face length] == 0 || [face isEqualToString: @"Medium"])
    faceName = @"";
  else
    faceName = face;

  if ([slant isEqualToString: @"I"])
    faceName = [NSString stringWithFormat: @"%@%@", faceName, @"Italic"];
  else if ([slant isEqualToString: @"O"])
    faceName = [NSString stringWithFormat: @"%@%@", faceName, @"Oblique"];

  if ([weight length] != 0 && ![weight isEqualToString: @"Normal"])
    {
      if ([faceName length] != 0)
	faceName = [NSString stringWithFormat: @"%@-%@", faceName, weight];
      else
	faceName = weight;
    }

  if ([add length] != 0)
    {
      if ([faceName length] != 0)
	faceName = [NSString stringWithFormat: @"%@-%@", faceName, add];
      else
	faceName = add;
    }

  if ([faceName length] == 0)
    faceName = @"Normal";
  
  return faceName;
}

// Build up an X font creation string
// -sony-fixed-medium-r-normal--16-150-75-75-c-80-iso8859-1
// becomes 
// -*-fixed-medium-r-normal--%d-*-*-*-c-*-iso8859-1
static NSString*
XGCreationNameFromParts(NSArray *parts)
{
  NSString *creationName;
  
  creationName = [NSString stringWithFormat: 
			     @"-%@-%@-%@-%@-%@-%@-%@-%@-%@-%@-%@-%@-%@-%@", 
			   @"*", 
			   [parts objectAtIndex: 2], 
			   [parts objectAtIndex: 3], 
			   [parts objectAtIndex: 4], 
			   [parts objectAtIndex: 5], 
			   [parts objectAtIndex: 6],
			   @"%d", // place holder for integer size
			   @"*",//[parts objectAtIndex: 8], 
			   @"*",//[parts objectAtIndex: 9], 
			   @"*",//[parts objectAtIndex: 10], 
			   [parts objectAtIndex: 11], 
			   @"*",//[parts objectAtIndex: 12], 
			   [parts objectAtIndex: 13], 
			   [parts objectAtIndex: 14]];
  
  return creationName;
}

static int fontDefSorter(NSArray *el1, NSArray *el2, void *context)
{
  // This is not exactly the order the OpenStep specification states.
  NSFontTraitMask t1 = [[el1 objectAtIndex: 3] unsignedIntValue];
  NSFontTraitMask t2 = [[el2 objectAtIndex: 3] unsignedIntValue];
  int w1 = [[el1 objectAtIndex: 2] intValue];
  int w2 = [[el2 objectAtIndex: 2] intValue];

  if (t1 < t2)
    return NSOrderedAscending;
  else if (t2 < t1)
    return NSOrderedDescending;
  else if (w1 < w2)
    return NSOrderedAscending;
  else if(w2 < w1)
    return NSOrderedDescending;
      
  return NSOrderedSame;
}

int
main(int argc, char **argv)
{
  NSData		*data;
  NSString		*path;
  NSString		*cache;
  NSArray		*args;
  NSDictionary		*env;
  NSFileManager		*mgr;
  NSMutableSet		*allFontNames;
  NSMutableDictionary	*allFontFamilies;
  NSMutableDictionary	*creationDictionary;
  Display		*dpy;
  NSMutableDictionary	*xFontDictionary;
  NSMutableArray	*allFonts;
  NSMutableSet		*knownFonts;
  NSEnumerator		*enumerator;
  id			key;
  int			nnames = 10000;
  int			available = nnames+1;
  int			i;
  char			**fonts;
  BOOL			flag;
  CREATE_AUTORELEASE_POOL(pool);

  args = [[NSProcessInfo processInfo] arguments];
  if ([args containsObject: @"--help"])
    {
      NSLog(@"This tool caches system font information in $(GNUSTEP_USER_ROOT)/Library/Fonts/Cache/<XDisplayName>");
      return 0;
    }
  if ([args containsObject: @"--version"])
    {
      NSLog(@"%@ version %s", [[args objectAtIndex: 0] lastPathComponent],
	makever(GNUSTEP_VERSION));
      return 0;
    }

  dpy = XOpenDisplay(NULL);
  if (dpy == 0)
    {
      NSLog(@"Unable to open X display - no font information available");
      RELEASE(pool);
      return 1;
    }

  XGInitAtoms(dpy);

  /*
   *        The home directory for per-user information is given by
   *        the GNUSTEP_USER_ROOT environment variable, or is assumed
   *        to be the 'GNUstep' subdirectory of the users home directory.
   */
  env = [[NSProcessInfo processInfo] environment];
  if (!env || !(path = [env objectForKey: @"GNUSTEP_USER_ROOT"]))
    {
      path = [NSHomeDirectory() stringByAppendingPathComponent: @"GNUstep"];
    }
  mgr = [NSFileManager defaultManager];
  if ([mgr fileExistsAtPath: path isDirectory: &flag] == NO || flag == NO)
    {
      return 1;
    }
  path = [path stringByAppendingPathComponent: @"Library"];
  if ([mgr fileExistsAtPath: path] == NO)
    {
      [mgr createDirectoryAtPath: path attributes: nil];
    }
  if ([mgr fileExistsAtPath: path isDirectory: &flag] == NO || flag == NO)
    {
      NSLog(@"Library directory '%@' not available!", path);
      return 1;
    }
  path = [path stringByAppendingPathComponent: @"Fonts"];
  if ([mgr fileExistsAtPath: path] == NO)
    {
      [mgr createDirectoryAtPath: path attributes: nil];
    }
  if ([mgr fileExistsAtPath: path isDirectory: &flag] == NO || flag == NO)
    {
      NSLog(@"Fonts directory '%@' not available!", path);
      return 1;
    }
  path = [path stringByAppendingPathComponent: @"Cache"];
  if ([mgr fileExistsAtPath: path] == NO)
    {
      [mgr createDirectoryAtPath: path attributes: nil];
    }
  if ([mgr fileExistsAtPath: path isDirectory: &flag] == NO || flag == NO)
    {
      NSLog(@"Fonts directory '%@' not available!", path);
      return 1;
    }

  allFontNames = [NSMutableSet setWithCapacity: 1000];
  allFontFamilies = [NSMutableDictionary dictionaryWithCapacity: 1000];
  creationDictionary = [NSMutableDictionary dictionaryWithCapacity: 1000];
  xFontDictionary = [NSMutableDictionary dictionaryWithCapacity: 1000];
  allFonts = [NSMutableArray arrayWithCapacity: 1000];
  knownFonts = [NSMutableSet setWithCapacity: 1000];

  /* Get list of all fonts */
  for (;;) 
    {
      fonts = XListFonts(dpy, "*", nnames, &available);
      if (fonts == NULL || available < nnames)
	break;
      /* There are more fonts then we expected, so just start 
	 again and request more */
      XFreeFontNames(fonts);
      nnames = available * 2;
    }

  if (fonts == NULL) 
    {
      NSDebugLog(@"No fonts found");
      return 0;
    }

  NSDebugFLog(@"Fonts loaded, now the loop.  Available: %d", available);

  for (i = 0; i < available; i++) 
    {
      char	*name = fonts[i];
      NSString	*fontName;
      NSString	*alias = [[NSString stringWithCString: name] lowercaseString];
      
      [allFonts addObject: alias];
      /*
       * First time we find this font. A font may be found more than 
       * once, when the XFontPath has the same directory twice.
       * Somehow this is the case on my machine.
       */
      if ([xFontDictionary objectForKey: alias] == nil)
	{
	  XFontStruct	*info = XLoadQueryFont(dpy, name);
	  NSString	*family;
	  NSString	*baseName;
	  NSString	*face;
	  NSString	*creationName;
	  NSArray	*parts;
	  
	  if (info == 0) 
	    {
	      NSDebugLog(@"No information for font %s", name);
	      continue;
	    }

	  fontName = XGFontName(dpy, info);
	  if (fontName == nil)
	    {
	      NSDebugLog(@"No font Name in info for font %s", name);
	      XFreeFontInfo(NULL, info, 1);      
	      continue;
	    }

	  if ([alias isEqualToString: fontName] == NO)
	    {
	      // We have got an alias name, store it
	      [xFontDictionary setObject: fontName forKey: alias];
	    }

	  // Use the normal function to keep the names consistent
	  family = XGFontFamily(dpy, info);	  

	  parts = [fontName componentsSeparatedByString: @"-"];
	  if ([parts count] == 15)
	    {
	      face = XGFaceNameFromParts(parts);
	      if ([face length] == 0 || [face isEqualToString: @"Normal"])
		{
		  baseName = family;
		}
	      else
		baseName = [NSString stringWithFormat: @"%@-%@", family, face];

	      creationName = XGCreationNameFromParts(parts);
	    }
	  else
	    {
	      baseName = [fontName capitalizedString];
	      face = @"Normal";
	      creationName = fontName;
	    }

	  // Store the alias to baseName
	  [xFontDictionary setObject: baseName forKey: fontName];
	  // it might already have been found with another size
	  if ([knownFonts member: baseName] == nil)
	    {
	      int		weight;
	      NSFontTraitMask	traits;
	      NSMutableArray	*fontDefs;
	      NSMutableArray	*fontDef;

	      // the first time we find that base font.
	      
	      // Store the font name
	      [knownFonts addObject: baseName];
	      [creationDictionary setObject: creationName forKey: baseName];
	      
	      weight = XGWeightOfFont(dpy, info);
	      traits = XGTraitsOfFont(dpy, info);

	      [allFontNames addObject: baseName];

	      // Store the family name and information
	      fontDefs = [allFontFamilies objectForKey: family];
	      if (fontDefs == nil)
		{
		  fontDefs = [NSMutableArray array];
		  [allFontFamilies setObject: fontDefs forKey: family];
		}
	      // Fill the structure for the font
	      fontDef = [NSMutableArray arrayWithCapacity: 4];  
	      [fontDef addObject: baseName];
	      [fontDef addObject: face];
	      [fontDef addObject: [NSNumber numberWithInt: weight]];
	      [fontDef addObject: [NSNumber numberWithUnsignedInt: traits]];
	      [fontDefs addObject: fontDef];
	    }
	  // Release the font structure
	  XFreeFontInfo (NULL, info, 1);
	}
      if (i && i % 100 == 0)
        NSDebugLog(@"Finished %d", i);
    }
  
  XFreeFontNames(fonts);
  
  // Now sort the fonts of each family
  enumerator = [allFontFamilies keyEnumerator];
  while ((key = [enumerator nextObject])) 
    {
      NSMutableArray *fontDefs = [allFontFamilies objectForKey: key];
      
      [fontDefs sortUsingFunction: fontDefSorter context: nil];
    }
  

  // define a creation string for alias names
  enumerator = [xFontDictionary keyEnumerator];
  while ((key = [enumerator nextObject])) 
    {
      id name = key;
      id creationName;
      
      while ((name != nil)
	&& (creationName = [creationDictionary objectForKey: name]) == nil)
	{
	  name = [xFontDictionary objectForKey: name];
	}

      if (creationName != nil)
	{
	  [creationDictionary setObject: creationName forKey: key];
	}
    }
  [xFontDictionary removeAllObjects];
  [xFontDictionary setObject: allFontNames forKey: @"AllFontNames"];
  [xFontDictionary setObject: allFontFamilies forKey: @"AllFontFamilies"];
  [xFontDictionary setObject: creationDictionary forKey: @"CreationDictionary"];
  data = [NSArchiver archivedDataWithRootObject: xFontDictionary];
  cache = [path stringByAppendingPathComponent:
    [NSString stringWithCString: XDisplayName(0)]];
  [data writeToFile: cache atomically: YES];

  RELEASE(pool);
  return 0;
}

