/* PIPS version 1.01: Parallel Information Processing System 
   Copyright (C) 1994, 95, 96 Free Software Foundation, Inc.

This file is part of GNU PIPS.

GNU PIPS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU PIPS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU PIPS; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.
*/

/*
 * Name          : nc2pnm.c
 * Author        : Frank Heimes
 * Institution   : Imperial College of Science, Technology, and Medicine
 * Written on    : 7 Sep 1994
 * Modified on   : 
 * Synopsis      :
 *
 * Usage:
 *   nc2pnm [-h] ncName [-v varName] [-i imgName] [-t imgType] [n]
 *
 * Where:
 *   -h         Prints this help text.
 *   ncName     Name of netCDF file to process, must be first parameter.
 *   varName    Name of variable to extract. Default is all variables.
 *   imgName    Name of the imagefile(s) to create.
 *              The image number is appended to 'imgName'.
 *              If no imgName is given, then its set to 'varName'.
 *   imgType    Image type to create. Default is to extract this from
 *              the variable attribute or data type. Possible values:
 *                pbmascii, pgmascii, ppmascii, pgmraw, ppmraw  
 *   n          Number of the image to extract from the variable.
 *              If no number is given, all images will be extracted.
 * Description:
 *   nc2pnm opens ncName and copies the specified images from the specified
 *   variable to one or more image files in the specified format.
 *   If 'varName' is omitted, all variables are extracted.
 *   If 'imgName' is omitted, the variablename is used. Hence if no variable
 *   name is given but an image name, only the first variable is extracted.
 */

#include "pips.h"   /* PIPS system header file  */

/*
 * Global definitions
 */

/*
 * Description of program for display on stderr by pipsPrintError()
 */
char *programName = "nc2pnm";
char *description[] = {
  "  nc2pnm [-h] ncName [-v varName] [-i imgName] [-t imgType] [n] ",
  " ",
  "Where:",
  "  -h         Prints this help text.",
  "  ncName     Name of netCDF file to process, must be first parameter.",
  "  varName    Name of variable to extract. Default is all variables.",
  "  imgName    Name of the imagefile(s) to create.",
  "             The image number is appended to 'imgName'.",
  "             If no imgName is given, then its set to 'varName'.",
  "  imgType    Image type to create. Default is to extract this from",
  "             The variable attribute or data type. Possible values:",
  "               pbmascii, pgmascii, ppmascii, pgmraw, ppmraw ", 
  "  n          Number of the image to extract from the variable.",
  "             If no number is given, all images will be extracted.",
  " ",
  "Description:",
  "  nc2pnm opens ncName and copies the specified images to one or more",
  "  image files in the specified format.",
  ""
}; /* end description */


/******************************************************************************
 *   M A I N
 ******************************************************************************
 */
#ifdef __STDC__

int main (int argc, char *argv[])

#else

main (argc, argv)
    int argc;
    char *argv[];

#endif
{
  long i;                      /* index variable                             */
  int ncId = -1;               /* netcdf file Id                             */
  int imgType = PipsUnknown;   /* Image type, the user has specified         */
  char *ncName     = NULL;     /* Name of netcdf file                        */
  char varName [MAX_NC_NAME];  /* Name of variable to create                 */
  char *imgName    = NULL;     /* Name of image file to copy to netcdf file  */
  char *position   = NULL;     /* Used to 'protect' strlen() from a NULL arg */
  long imageNumber = -1;       /* Number of image to extract (-1 means ALL)  */
  char *history  = NULL;       /* Saved command line parameters              */
  int histIndex;               /* Variable for constructing history string   */

/******************************************************************************
 *   A N A L Y Z E   C O M M A N D   L I N E   P A R A M E T E R S
 ******************************************************************************
 */
/*
 * If no parameters given, complain about missing netcdf filename
 */
  if (argc < 2)
	pipsPrintError (-1, PipsNetcdfMissing, NULL, PipsExit);

/*
 * If the -h option is given, print explanation and quit
 */
  if (strncmp ("-h", argv[1], 2) == 0)
    pipsPrintError (-1, PipsOk, NULL, PipsExit);

/*
 * First argument is expected to be the name of the netcdf file
 */
  ncName = argv[1];

/*
 * Examine argv[i]: Store a number in 'imageNumber',
 * and look for parameters '-v', '-t', '-i'
 */
  varName [0] = 0;
  varName [MAX_NC_NAME-1] = 0;
  for (i=2; i < argc; i++)
    if (! sscanf (argv[i], "%lu", &imageNumber))
      if (argv[i][0]) {
        if (i+1 >= argc) 
          pipsPrintError (-1, PipsSinglePara, argv[i], PipsExit);
        else
          switch (argv[i][1]) {
/*
 * '-v' option : store next parameter in 'varName'
 */
          case 'v' : strncpy (varName, argv[++i], MAX_NC_NAME-1);
                     break;
/*
 * '-t' option : store next parameter in 'imgType
 */
          case 't' : position = strstr (
                       "ppmraw  pgmraw          ppmasciipgmasciipbmascii",
                       argv[++i]);
                     if (position)
                       imgType = strlen (position) / 8;
                     else
                       pipsPrintError (-1, PipsBadParameter, argv[i],PipsExit);
                     break;
/*
 * '-i' option : store next parameter in 'imgName'
 */
          case 'i' : imgName = argv[++i];
                     break;
          default  : pipsPrintError (-1, PipsBadParameter, argv[i], PipsExit);
    } /* end switch */
  }  /* end if ! */


/******************************************************************************
 *  O P E N   N E T C D F   F I L E ,   E X T R A C T   I M A G E S 
 ******************************************************************************
 */

/*
 * Prevent error messages (error codes are evaluated for further calls)
 */
  ncopts = NC_NOERR;

/*
 * Open an EXISTING netCDF file and go in define mode
 */
  ncId = ncopen (ncName, NC_WRITE);
  if (ncId == -1)
    pipsPrintError (-1, PipsErrorFileRead, ncName, PipsExit);
  ncredef (ncId);

/*
 * Save the command line parameters in the history attribute (concatenated)
 */
  for (histIndex = 0; histIndex < argc; histIndex++) {
    pipsStrApp (&history, argv[histIndex]);
    pipsStrApp (&history, " ");
  } /* end for */
  pipsHistory (ncId, history);
  free (history);
  ncendef (ncId);

/*
 * Prevent error messages (error codes are evaluated for further calls)
 */
  ncopts = NC_NOERR;

/*
 * Extract the images
 */
  pipsExtPnm (ncId, varName, imageNumber, imgName, imgType);

/*
 * Commit changes and close netCDF-file
 */
  if (ncclose (ncId) == -1)
    pipsPrintError (-1, PipsErrorFileWrite, ncName, PipsExit);
  return (PipsOk);
} /* end main */


/* end nc2pnm.c */
