/* PIPS version 1.01: Parallel Information Processing System 
   Copyright (C) 1994, 95, 96 Free Software Foundation, Inc.

This file is part of GNU PIPS.

GNU PIPS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU PIPS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU PIPS; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.
*/

/*
 * Name          : ncprocess.c
 * Author        : Frank Heimes
 * Institution   : Imperial College of Science, Technology, and Medicine
 * Written on    : 7 Sep 1994
 * Modified on   : 
 * Synopsis      :
 *
 * Usage:
 *  ncprocess [-h] ncName parameters
 *
 * Where:
 *   -h           Prints this help text.
 *   ncName       Name of netCDF file to process, must be first parameter.
 *   parameters   Parameters passed to the data segmentation module
 *
 * Description:
 *   ncprocess opens 'ncName', spawns tasks on all processors in the
 *   virtual parallel machine and processes the data specified by
 *   'parameters' in parallel.
 */

#include "pips.h"   /* PIPS system header file  */
#include "pvm3.h"   /* PVM system header file   */

/*
 * Global definitions
 */
extern int parameterNumber;        /* Defined in module with pipsJobManager */

/******************************************************************************
 *   M A I N
 ******************************************************************************
 */
#ifdef __STDC__

int main (int argc, char *argv[])

#else

main (argc, argv)
    int argc;
    char *argv[];

#endif
{

/*
 * Variables for netcdf file
 */
  int  ncId      = -1;          /* netcdf file Id                            */
  char *ncName   = NULL;        /* Name of netcdf file                       */
  char *history  = NULL;        /* Saved command line parameters             */
  int histIndex;               /* Variable for constructing history string   */

/*
 * Variables for PVM (initialisation connects to PVM)
 */
  int i;              /* index for tid                                       */
  int parent;         /* my creator's task id                                */
  int ntask;          /* Number of children to spawn                         */
  int *tid;           /* Children task ids (tasks that run 'pipsProcPar')    */

/*
 * Exit if connection to PVM is not possible
 */
  if (pvm_mytid () < 0)
    pipsPrintError (ncId, PipsPvmInactive, NULL, PipsExit);

/*
 * Get parent's task id
 */
  parent = pvm_parent();

/*
 * Determine, if I'm the first or spawned task; first runs pipsTaskmanager(),
 * the others do parallel data processing with pipsProcPar().
 * NOTE: Checking if (parent == pvmNoParent) doesn't work because the first 
 * task could have been spawned by 'pvm' or 'xpvm'
 */
  if ((argc > 1) && (strcmp (PipsChildTag, argv[1]) == 0)) {  /* I'm a child */

/*
 * Check, if properly spawned by a parent process (who runs pipsTaskManager())
 */
    if (parent == PvmNoParent) {
      pvm_exit ();
      pipsPrintError (ncId, PipsPrivatePara, PipsChildTag, PipsExit);
    } /* end if */

/******************************************************************************
 *  C H I L D T A S K   :   D A T A P R O C E S S I N G
 ******************************************************************************
 * CHILD: Continue receiving, processing in parallel and sending data
 */
    pvm_initsend (PvmDataDefault);   /* Needed for first pvm_send() */
    do {

/*
 * Tell jobManager() I'm waiting for work to do
 */
      pvm_send (parent, PipsWaiting);

/*
 * Receive data from jobManager(), process it and send it back
 */
      if (pvm_recv (parent, PipsCommence) > 0)
        pipsProcPar ();
    } while (1);                            /* my parent's gonna kill me !   */
  } else {                                  /* I'm not a child, but a parent */

/******************************************************************************
 *  P A R E N T T A S K   :   D A T A S E G M E N T A T I O N
 *****************************************************************************/
/******************************************************************************
 *   A N A L Y Z E   C O M M A N D   L I N E   P A R A M E T E R S
 ******************************************************************************
 */
/*
 * If no parameters given, complain about missing netcdf filename
 */
    if (argc < 2) {
      pvm_exit ();
      pipsPrintError (-1, PipsNetcdfMissing, NULL, PipsExit);
    } /* end if */

/*
 * If the -h option is given, print explanation and quit
 */
    if (strncmp ("-h", argv[1], 2) == 0) {
      pvm_exit ();
      pipsPrintError (-1, PipsOk, NULL, PipsExit);
    } /* end if */

/*
 * First argument is expected to be the name of the netcdf file
 */
    ncName = argv[1];

/*
 * Check if minimum number of parameters is given
 */
  if (argc - parameterNumber < 2) {
    pvm_exit ();
    pipsPrintError (-1, PipsNoOption, NULL, PipsExit);
  }

/******************************************************************************
 *  O P E N   N E T C D F   F I L E
 ******************************************************************************
 * Prevent error messages (error codes are evaluated for further calls)
 */
    ncopts = NC_NOERR;

/*
 * Open an EXISTING netCDF file and go in define mode
 */
    ncId = ncopen (ncName, NC_WRITE);
    if (ncId == -1) {
      pvm_exit ();
      pipsPrintError (-1, PipsErrorFileRead, ncName, PipsExit);
    } /* end if */
    ncredef (ncId);

/*
 * Save the command line parameters in the history attribute (concatenated)
 */
    for (histIndex = 0; histIndex < argc; histIndex++) {
      pipsStrApp (&history, argv[histIndex]);
      pipsStrApp (&history, " ");
    } /* end for */
    pipsHistory (ncId, history);
    free (history);
    ncendef (ncId);

/*
 * Redirect all output to console that launched ncprocess
 */
    pvm_catchout (stdout);
    pvm_catchout (stderr);

/*
 * Spawn tasks, return number and tid's of successfully spawned tasks
 */
    pipsTaskManager (argv [0], &ntask, &tid);

/*
 * Start processing, if at least one task could be spawned
 */
    if (ntask > 0) {
/*
 * Call the jobManager to do the data segmentation
 * and pass it the remaining command line parameters
 */
      pipsJobManager (ntask, tid, ncId, argc-2, &(argv[2]));

/*
 * Shut down remote tasks
 */
      for (i=0; i<ntask; i++)
        pvm_kill (tid[i]);
    } /* end if ntask */

/*
 * Free all allocated memory
 */
    free (tid);

/*
 * Commit changes and close netCDF-file
 */
    if (ncclose (ncId) == -1)
      pipsPrintError (-1, PipsErrorFileWrite, ncName, PipsExit);
  } /* end else if I'm the parent */
  pvm_exit();
  return (PipsOk);
} /* end main */


/* end ncprocess.c */
