/* Source Installer, Copyright (c) 2005 Claudio Fontana

   preferences.c - user choices saved in the preferences file

   This program is free software; you can redistribute it and/or modify
       it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
       (at your option) any later version.

   This program is distributed in the hope that it will be useful,
       but WITHOUT ANY WARRANTY; without even the implied warranty of
       MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
       GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
       along with this program (look for the file called COPYING);
       if not, write to the Free Software Foundation, Inc.,
           51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

       You can contact the author (Claudio Fontana) by sending a mail
       to claudio@gnu.org
*/

#include "src_stdinc.h"
#include "preferences.h"

struct srcinst_parsed_data _keywords_preferences[SRCINST_PREFERENCES_NKEYS] = {
  { "ADD_DIRECTORY", 0, "%s" },
  { "INSTALL", 0, "%d" },
  { "KEEP_SOURCE", 0, "%d" },
  { "MANUAL_CONFIGURE", 0, "%d" },
  { "PREFIX", 0, "%s" },
  { "SRC_COMPRESS", 0, "%s" },
  { "STRIP", 0, "%d" }
};

static void _init_keywords_preferences(struct _preferences* pref) {
  struct srcinst_parsed_data* data = _keywords_preferences;
  data[SRCINST_PKEY_ADD_DIRECTORY].variable = &(pref->add_directory);
  data[SRCINST_PKEY_INSTALL].variable = &(pref->install);
  data[SRCINST_PKEY_KEEP_SOURCE].variable = &(pref->keep_source);
  data[SRCINST_PKEY_MANUAL_CONFIGURE].variable = &(pref->manual_configure);
  data[SRCINST_PKEY_PREFIX].variable = &(pref->prefix);
  data[SRCINST_PKEY_SRC_COMPRESS].variable = &(pref->src_compress);
  data[SRCINST_PKEY_STRIP].variable = &(pref->strip);
}

void _init_preferences(struct _preferences* p) {
  memset(p, 0, sizeof(struct _preferences));
  p->prefix = strdup("/usr/local");
  p->manual_configure = 1;
  p->keep_source = 1;
  p->install = 1;
  p->strip = 0;
  p->src_compress = SRCINST_COMP_GZ;
  p->add_directory = 0;
}

void _free_preferences(struct _preferences* p) {
  if (p->prefix) { free(p->prefix); p->prefix = 0; }
  p->manual_configure = 0;
  p->keep_source = 0;
  p->install = 0;
  p->strip = 0;
  p->src_compress = 0;
  if (p->add_directory) { free(p->add_directory); p->add_directory = 0; }
}

int _parse_preferences(struct _preferences* p, char* k, char* s) {
  int i;

  /* find matching keyword entry */
    
  if ((i = srcinst_bsearch(_keywords_preferences, SRCINST_PREFERENCES_NKEYS, k, &srcinst_get_parsed_data_key)) == -1)
    return 0;

  switch (i) {

  case SRCINST_PKEY_SRC_COMPRESS:
    {
      SRCINST_COMP fmt;
      if ((fmt = srcinst_str_to_compression_format(s)) != SRCINST_COMP_UNKNOWN)
	p->src_compress = fmt;
      else
	srcinst_warning(SRCINST_ERR_PARSER, "invalid compression format", s);
      return 1;
    }
    break;

  default:
    return srcinst_parse(_keywords_preferences + i, s);
  }
  assert(0);
  return 0;
}

int _load_preferences(struct _preferences* p) {
  char* filename; int rv;

  _init_keywords_preferences(p);
  filename = _srcinst_state.files[SRCINST_FILE_CFG];
  rv = srcinst_load_parse_file(p, filename, &_parse_preferences);

  if (!p->install && !p->keep_source) {
    srcinst_warning(SRCINST_ERR_PARSER, "INSTALL and KEEP_SOURCE are both disabled", 0);
  }
  return rv;
}

int _save_preferences(struct _preferences* p) {
  char* filename; FILE* f; int i;
  _init_keywords_preferences(p);
  filename = _srcinst_state.files[SRCINST_FILE_CFG];
  
  f = fopen(filename, "w");
  free(filename);

  if (!f)
    return 0;

  for (i = 0; i < SRCINST_PREFERENCES_NKEYS; i++) {
    struct srcinst_parsed_data* data = _keywords_preferences + i;

    switch (i) {
    case SRCINST_PKEY_SRC_COMPRESS:
      {
	char* str;
	if (!(str = srcinst_compression_format_to_str(p->src_compress))) {
	  srcinst_warning(SRCINST_ERR_PARSER, "invalid compression format", 0);
	  str = "gz";
	}
	fprintf(f, "%s: %s\n", data->key, str);
      }
    break;

    default:

      switch (data->format[1]) {
	
      case '%':
	/* nop */
	break;

      case 's':
	if (*((char**)data->variable))
	  fprintf(f, "%s: %s\n", data->key, *((char**)(data->variable)));
	break;

      case 'd':
	fprintf(f, "%s: %d\n", data->key, *((char*)(data->variable)));
	break;
	
      default:
	assert(0);
      }
    }
  }
  
  i = ferror(f);
  fclose(f);
  return !i;
}


