/* Source Installer, Copyright (c) 2006 Claudio Fontana

 mf_features.c - makefile features

 This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3 of the License, or
     (at your option) any later version.

 This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
     along with this program (look for the file called COPYING);
     if not, write to the Free Software Foundation, Inc.,
         51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

     You can contact the author (Claudio Fontana) by sending a mail
     to claudio@gnu.org
*/

#include "src_stdinc.h"
#include "mf_features.h"
#include "actions.h"

static int _detect_features_from_make(struct _makefile_features *);
static int _detect_features_from_makefiles(struct _makefile_features *);
static int _test_makefile_features(struct _makefile_features *mf,
				   char *mfile);

static struct _makefile_features *_new_makefile_features(char *makefile)
{

    struct _makefile_features *mf;
    mf = srcinst_malloc(sizeof(struct _makefile_features));

    mf->makefile = makefile;
    mf->features[SRCINST_FEATURE_DESTDIR].pattern = "\\$[({]?DESTDIR[)}]?";
    mf->features[SRCINST_FEATURE_DESTDIR].supported = 0;
    mf->features[SRCINST_FEATURE_INSTALL_ROOT].pattern =
	"\\$[({]?INSTALL_ROOT[)}]?";
    mf->features[SRCINST_FEATURE_INSTALL_ROOT].supported = 0;
    mf->features[SRCINST_FEATURE_PREFIX].pattern = "\\$[({]?prefix[)}]?";
    mf->features[SRCINST_FEATURE_PREFIX].supported = 0;
    mf->features[SRCINST_FEATURE_PREFIX_UCASE].pattern =
	"\\$[({]?PREFIX[)}]?";
    mf->features[SRCINST_FEATURE_PREFIX_UCASE].supported = 0;
    mf->features[SRCINST_FEATURE_UNINSTALL].pattern = "uninstall *:";
    mf->features[SRCINST_FEATURE_UNINSTALL].supported = 0;
    mf->features[SRCINST_FEATURE_STRIP].pattern = "install-strip *:";
    mf->features[SRCINST_FEATURE_STRIP].supported = 0;

    return mf;
}

void _free_makefile_features(struct _makefile_features *mf)
{

    if (mf->makefile) {
	free(mf->makefile);
	mf->makefile = 0;
    }

    free(mf);
}

/* the main action */

struct _makefile_features *_detect_makefile_features(void)
{
    struct _makefile_features *mf;
    char *makefile;
    int rv1, rv2 = 0;

    if (!(makefile = _get_makefile()))
	return 0;

    mf = _new_makefile_features(makefile);

    rv1 = _detect_features_from_make(mf); /* try method 1 */
    rv2 = _detect_features_from_makefiles(mf); /* and try method 2 */

    if (!rv1 && !rv2) {
	/* none of the methods was successful */
	_free_makefile_features(mf);
	return 0;
    }

    return mf;
}


/* detect features from Make
 * 
 * these functions deal with detecting supported features
 * using Make process invocation, using the POSIX options
 * -q -p to print all rules and variables.
 * Unfortunately, non-GNU Make flavours do not really
 * print all of them (they do not recurse),
 * so we hope for method 2 (detect from Makefiles).
 */

static SRCINST_REGEX _mf_preg[SRCINST_FEATURE_N];

static void _mf_callback_expect_pattern(char *line)
{
    int i;

    for (i = 0; i < SRCINST_FEATURE_N; i++) {
	if (_mf_preg[i] && srcinst_exec_regex(_mf_preg[i], line, 0)) {
	    srcinst_free_regex(_mf_preg[i]);
	    _mf_preg[i] = 0;	/* free, and set to 0 to mean found */
	}
    }
}

static int _detect_features_from_make(struct _makefile_features *mf)
{
    SRCINST_SPAWN_ID id;
    int i, log_out, log_err;
    struct _callback_node *call_out;

    if (!(id = srcinst_spawn(PRG_make, "-f", mf->makefile, "-q", "-p", 0))) {
	return 0;
    }

    for (i = 0; i < SRCINST_FEATURE_N; i++) {
	_mf_preg[i] = srcinst_init_regex(mf->features[i].pattern, -1);
    }

    if ((log_out = _srcinst_state.log_stdout)) {
	_srcinst_state.log_stdout = 0;
    }

    if ((call_out = _srcinst_state.stdout_callbacks.first)) {
	_srcinst_state.stdout_callbacks.first = 0;
    }

    if ((log_err = _srcinst_state.log_stderr)) {
	_srcinst_state.log_stderr = 0;
    }

    srcinst_register_stdout_callback(&_mf_callback_expect_pattern);
    (void) srcinst_wait(id);
    srcinst_unregister_stdout_callback(&_mf_callback_expect_pattern);

    if (log_out) {
	_srcinst_state.log_stdout = 1;
    }

    if (log_err) {
	_srcinst_state.log_stderr = 1;
    }

    if (call_out) {
	_srcinst_state.stdout_callbacks.first = call_out;
    }

    for (i = 0; i < SRCINST_FEATURE_N; i++) {
	if (!_mf_preg[i]) {
	    mf->features[i].supported = 1;
	} else {
	    srcinst_free_regex(_mf_preg[i]);
	}
    }

    return 1;
}


/* detect features from Makefiles
 * 
 * this method looks directly at the Makefiles. This does not take
 * into account include directives and other things, so we might 
 * miss stuff. Hopefully method 1 already catched most of the stuff. */

static int _detect_features_from_makefiles(struct _makefile_features *mf)
{

    if (!(_test_makefile_features(mf, mf->makefile)))
	return 0;

    if (1) {
	struct srcinst_string_list makefiles;
	char *makefile_names[2];

	srcinst_init_string_list(&makefiles);
	makefile_names[0] = mf->makefile;
	makefile_names[1] = 0;

	if (_find_files_aux(&makefiles, ".", makefile_names)) {
	    int i;
	    for (i = 0; i < makefiles.count; i++) {
		if (!_test_makefile_features(mf, makefiles.strings[i]))
		    break;
	    }
	}
	srcinst_free_string_list(&makefiles);
    }
    return 1;
}

static int _test_makefile_features(struct _makefile_features *mf,
				   char *mfile)
{
    int i;
    FILE *m;
    off_t m_size;
    char *buffer;

    if ((m_size = srcinst_file_size(mfile)) == (off_t) - 1) {
	return 0;
    }

    buffer = srcinst_malloc(m_size + 1);

    if (!(m = fopen(mfile, "r"))) {
	free(buffer);
	return 0;
    }

    if ((fread(buffer, 1, m_size, m)) != m_size) {
	free(buffer);
	fclose(m);
	return 0;
    }

    fclose(m);
    buffer[m_size] = 0;

    for (i = 0; i < SRCINST_FEATURE_N; i++) {
	SRCINST_REGEX preg;
	preg = srcinst_init_regex(mf->features[i].pattern, -1);

	if (srcinst_exec_regex(preg, buffer, 0))
	    mf->features[i].supported = 1;

	srcinst_free_regex(preg);
    }

    free(buffer);
    return 1;
}

