/*
 *   Copyright (C) 1997, 1998, 1999 Loic Dachary
 *
 *   This program is free software; you can redistribute it and/or modify it
 *   under the terms of the GNU General Public License as published by the
 *   Free Software Foundation; either version 2, or (at your option) any
 *   later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif /* HAVE_MALLOC_H */
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif /* HAVE_STDLIB_H */
#ifdef HAVE_DMALLOC_H
#include <dmalloc.h>
#endif /* HAVE_DMALLOC_H */

#include <hash.h>
#include <getopttools.h>
#include <cookies.h>
#include <mysql.h>
#include <sqlutil.h>
#include <uri.h>
#include <split.h>
#include <salloc.h>

static int verbose = 0;

static struct option long_options[] =
{
  /* These options set a flag. */
  {"verbose_cookies", 0, &verbose, 1},
  {0, 0, 0, COOKIES_OPTIONS}
};

static cookies_entry_t* cookies_parse(cookies_t* cookies, uri_t* url_object, char* cookie);
static void cookies_load_1(cookies_t* cookies, cookies_entry_t* entry);
static void cookies_save_entry(cookies_t* cookies);
static void cookies_decode(cookies_t* cookies, MYSQL_RES *res, MYSQL_ROW row);
static cookies_t* params_alloc();

struct option* cookies_options(struct option options[])
{
  return long_options;
}

cookies_t* cookies_alloc(int argc, char** argv, struct option options[])
{
  cookies_t* params = params_alloc();

  optind = 0;
  while(1) {
    /* `getopt_long' stores the option index here. */
    int option_index = 0;
    int c;
    int found = 1;

    c = getopt_long_only(argc, argv, "", options, &option_index);

    /* Detect the end of the options. */
    if (c == -1)
      break;
     
    switch (c)
      {
      case 0:
	/* If this option set a flag, do nothing else now. */
	
	if (options[option_index].flag != 0)
	  break;
	if(!strcmp(options[option_index].name, "")) {
	  break;
	}
	found = 0;
	break;
      default:
	fprintf(stderr, "option parse error %c, 0x%x\n", c & 0xff, c);
	exit(1);
      }
    if(found) {
      hash_alloc_insert(params->options, (char*)options[option_index].name, strdup(optarg ? optarg : " "));
    }
  }

  return params;
}

void cookies_load(cookies_t* cookies, char* url, char* cookie)
{
  static uri_t* url_object = 0;
  int url_length = strlen(url);

  if(verbose) fprintf(stderr, "cookies_load: begin\n");

  /*
   * Build an url object from url
   */
  if(!url_object) {
    url_object = uri_alloc(url, url_length);
  } else {
    uri_realloc(url_object, url, url_length);
  }

  if(!strncasecmp(uri_scheme(url_object), "http", 4)) {
    cookies_entry_t* entry = cookies_parse(cookies, url_object, cookie);

    /*
     * Ignore unparsable cookies.
     */
    if(!entry)
      return;

    /*
     * If already in cache
     */
    if(!strcmp(cookies->current.netloc, entry->netloc) &&
       !strcmp(cookies->current.path, entry->path)) {
      /*
       * Cookie did not change, do nothing.
       */
      if(!strcmp(cookies->current.in, entry->in)) {
	if(verbose) fprintf(stderr, "cookies_load: reuse current %s\n", cookie);
      } else {
	/*
	 * Cookie changed, remember the fact
	 */
	strcpy(cookies->current.in, entry->in);
	strcpy(cookies->current.out, entry->out);
	cookies->current.info |= COOKIES_INFO_CHANGED;
	if(verbose) fprintf(stderr, "cookies_load: update current %s\n", cookie);
      }
    } else {
      /*
       * Cookie is not in cache, load it from base, if possible.
       */
      if(verbose) fprintf(stderr, "cookies_load: search cookies table %s\n", cookie);
      cookies_load_1(cookies, entry);
    }
  }
}

char* cookies_match(cookies_t* cookies, uri_t* url_object)
{
  char* netloc = uri_netloc(url_object);
  char* path = url_object->path ? url_object->path : "";

  /*
   * If we already know that this netloc has no cookie,
   * just return.
   */
  if(!strcmp(cookies->netloc_without_cookies, netloc)) {
    if(verbose) fprintf(stderr, "cookies_match: %s has no cookies (cached)\n", netloc);
  } else if(!strcmp(cookies->current.netloc, netloc) &&
	    !strncmp(cookies->current.path, path, strlen(cookies->current.path))) {
    /*
     * If already in cache. No specific support for
     * multiple path in cookies. Should rescan or keep an internal
     * table of cookies netloc/path pairs.
     */
    if(verbose) fprintf(stderr, "cookies_match: use cached %s for %s %s\n", cookies->current.out, netloc, path);
    return cookies->current.out;
  } else {
    webbase_t* base = cookies->base;
    static char* query = 0;
    static int query_size = 0;
    MYSQL_RES *res;
    MYSQL_ROW row;
    static char* found = 0;
    static int found_size = 0;
    int found_length = -1;

    static_alloc(&query, &query_size, 128 + strlen(netloc));
    static_alloc(&found, &found_size, 64);
    found[0] = '\0';

    sprintf(query, "select path from cookies where netloc = '%s'", netloc);
    if(verbose) fprintf(stderr, "cookies_match: query = %s\n", query);
    smysql_query(&base->mysql, query);
    res = smysql_store_result(&base->mysql);
    if(mysql_num_rows(res) == 0) {
      if(verbose) fprintf(stderr, "cookies_match: found nothing\n");
      strcpy(cookies->netloc_without_cookies, netloc);
    } else {
      while((row = mysql_fetch_row(res))) {
	char* cookie_path = row[0];
	int cookie_path_length = strlen(cookie_path);
	/*
	 * Select the most specific path found that match
	 * the path of the URL.
	 */
	if(verbose) fprintf(stderr, "cookies_match: consider %s\n", cookie_path);
	if(cookie_path_length > found_length &&
	   !strncmp(cookie_path, path, cookie_path_length)) {
	  if(verbose) fprintf(stderr, "cookies_match: better match\n");
	  found_length = cookie_path_length;
	  static_alloc(&found, &found_size, found_length + 1);
	  strcpy(found, cookie_path);
	}
      }
    }
    mysql_free_result(res);

    /*
     * We know that the entry exist and match the URL, load it.
     */
    if(found_length >= 0) {
      cookies_entry_t entry;
      if(verbose) fprintf(stderr, "cookies_match: best match for %s is %s\n", path, found);
      memset((char*)&entry, '\0', sizeof(cookies_entry_t));
      strcpy(entry.netloc, netloc);
      strcpy(entry.path, found);
      cookies_load_1(cookies, &entry);
      if(verbose) fprintf(stderr, "cookies_match: out %s\n", cookies->current.out);
      return cookies->current.out;
    } else {
      if(verbose) fprintf(stderr, "cookies_match: failed to match\n");
    }
  }
  return 0;
}

static cookies_entry_t* cookies_parse(cookies_t* cookies, uri_t* url_object, char* cookie)
{
  static cookies_entry_t entry;
  char** splitted;
  int count;
  int i;
  char* name = 0;
  char* value = 0;
  char* path = 0;

  memset((char*)&entry, '\0', sizeof(cookies_entry_t));

  if(strlen(cookie) > COOKIES_LENGTH - 20) {
    fprintf(stderr, "cookies_parse: cookie too long %s, ignore\n", cookie);
    return 0;
  }

  split(cookie, strlen(cookie), &splitted, &count, ';', SPLIT_TRIM);

  if(count >= 1) {
    char* equal = strchr(splitted[0], '=');
    if(equal) {
      name = splitted[0];
      *equal++ = '\0';
      value = equal;
    }
  }
  for(i = 1; i < count; i++) {
    char* p = splitted[i];
    while(*p && isspace((int)*p))
      p++;
    if(!strncasecmp(p, "path", 4)) {
      char* equal = strchr(p, '=');
      if(equal) {
	path = equal + 1;
	/*
	 * Strip leading space and /.
	 */
	while(*path && (isspace((int)*path) || *path == '/'))
	  path++;
	p = path;
	/*
	 * Strip trailing space.
	 */
	while(*p && !isspace((int)*p))
	  p++;
	*p = '\0';
      }
    }
  }

  if(name && value && path) {
    strcpy(entry.in, cookie);
    sprintf(entry.out, "$Version=0; %s=%s; $Path=/%s", name, value, path);
    strcpy(entry.netloc, uri_netloc(url_object));
    strcpy(entry.path, path);
    return &entry;
  } else {
    return 0;
  }
}

static void cookies_load_1(cookies_t* cookies, cookies_entry_t* entry)
{
  static char* query = 0;
  static int query_size = 0;
  char* netloc = entry->netloc;
  int netloc_length = strlen(netloc);
  char* path = entry->path;
  int path_length = strlen(path);
  webbase_t* base = cookies->base;
  MYSQL_RES *res;
  MYSQL_ROW row;
  
  static_alloc(&query, &query_size, 128 + netloc_length + path_length);

  cookies_save_entry(cookies);

  /*
   * Obsoletes the 'no cookies for this netloc' info, regardless.
   */
  cookies->netloc_without_cookies[0] = '\0';

  sprintf(query, "select * from cookies where netloc = '%s' and path = '%s'", netloc, path);
  if(verbose) fprintf(stderr, "cookies_load: query = %s\n", query);
  smysql_query(&base->mysql, query);
  res = smysql_store_result(&base->mysql);
  switch(mysql_num_rows(res)) {
  case 1:
    row = mysql_fetch_row(res);
    cookies_decode(cookies, res, row);
    cookies->current.info |= COOKIES_INFO_IN_BASE;
    if(verbose) fprintf(stderr, "cookies_load_1: found in base\n");
    break;
  case 0:
    /*
     * If the cookie is not in base, just update
     * the cache and consider it changed so that 
     * it will be saved by save_entry.
     */
    cookies->current = *entry;
    cookies->current.info |= COOKIES_INFO_CHANGED;
    if(verbose) fprintf(stderr, "cookies_load_1: set in cache\n");
    break;
  default:
    if(verbose) fprintf(stderr, "cookies_load: unexpected number of rows query = %s, rows = %ld\n", query, (long)mysql_num_rows(res));
    exit(1);
    break;
  }
  mysql_free_result(res);
}

static void cookies_save_entry(cookies_t* cookies)
{
  static char* query = 0;
  static int query_size = 0;
  cookies_entry_t* entry = &cookies->current;
  webbase_t* base = cookies->base;

  static_alloc(&query, &query_size, 128 + strlen(entry->netloc) + strlen(entry->path) + strlen(entry->in) + strlen(entry->out));

  query[0] = '\0';
  /*
   * First save the current modified entry, if it is in use
   */
  if((entry->info & COOKIES_INFO_CHANGED) &&
     (entry->info & COOKIES_INFO_IN_BASE)) {
    sprintf(query, "update cookies set cookie_in = '%s', cookie_out = '%s' where netloc = '%s' and path = '%s'",
	    entry->in,
	    entry->out,
	    entry->netloc,
	    entry->path);
  } else if((entry->info & COOKIES_INFO_CHANGED) &&
	    !(entry->info & COOKIES_INFO_IN_BASE)) {
    sprintf(query, "insert into cookies values ('%s', '%s', '%s', '%s')",
	    entry->netloc,
	    entry->path,
	    entry->in,
	    entry->out);
  } else {
  }

  if(query[0] != '\0') {
    if(verbose) fprintf(stderr, "cookies_save_entry: query = %s\n", query);
    smysql_query(&base->mysql, query);
  } else {
    if(verbose) fprintf(stderr, "cookies_save_entry: nothing to save\n");
  }
}

static void cookies_decode(cookies_t* cookies, MYSQL_RES *res, MYSQL_ROW row)
{
  cookies_entry_t* entry = &cookies->current;
  memset(entry, '\0', sizeof(cookies_entry_t));

  strncpy(entry->netloc, row[0], WEBBASE_URL_LENGTH);
  strncpy(entry->path, row[1], WEBBASE_URL_LENGTH);
  strncpy(entry->in, row[2], COOKIES_LENGTH);
  strncpy(entry->out, row[3], COOKIES_LENGTH);
}

void cookies_free(cookies_t* params)
{
  cookies_save_entry(params);
  hash_free(params->options);
  free(params);
}

static void hnode_free(hnode_t *node, void *context)
{
  free(node->data);
  free(node);
}

static cookies_t* params_alloc()
{
  cookies_t* params = (cookies_t*)smalloc(sizeof(cookies_t));
  memset((char*)params, '\0', sizeof(cookies_t));
  params->options = hash_create(33, 0, 0);
  hash_set_allocator(params->options, 0, hnode_free, 0);

  return params;
}
