/************************************************************************/
/* File		webpublishsynch.cpp					*/
/*									*/
/* Purpose	This C++ program file contains the private WebPublish	*/
/*		member functions that process the --synchronize command	*/
/*		line option. The --synchronize command will remove any	*/
/*		files and directories on the server that are no longer	*/
/*		a part of the local copy of a website. The		*/
/*		--synchronize command can operate on all or part of a	*/
/*		website.						*/
/*									*/
/* Author	This C++ program file was written by Charles Henry	*/
/*		Schoonover for Padre Software. You can contact Charles	*/
/*		Henry Schoonover at charles@padresoftware.com.		*/
/*									*/
/* Owner	The contents of this C++ program file were written for	*/
/*		Padre Software. You can contact Padre Software at	*/
/*		webmaster@padresoftware.com.				*/
/*									*/
/* Version	00.00.00 (Prototype)					*/
/*									*/
/* Date		Sunday, July 7, 2002.					*/
/*									*/
/* Copyright	(C) 2002 by Padre Software Incorporated.		*/
/*		All rights are reserved.				*/
/*									*/
/*		Padre Software has released the source code in this	*/
/*		file to the public domain under the terms of the GNU	*/
/*		General Public License. (See the file COPYING).		*/
/*									*/
/*		This program is free software; you can redistribute it	*/
/*		and/or modify it under the terms of the GNU General	*/
/*		Public License as published by the Free Software	*/
/*		Foundation; either version 2 of the License, or (at	*/
/*		your option) any later version.				*/
/************************************************************************/

#include <string.h>
#include "webpublish.h"

/************************************************************************/
/* Function	condition is_synchronize_path_excluded(			*/
/*		   const String& path)					*/
/*									*/
/* Purpose	This function can be used to see if a given path is	*/
/*		listed in the database as one of the paths that is to	*/
/*		be excluded from being synchronized.			*/
/*									*/
/* Input	This function expects the variable 'path' to contain	*/
/*		the complete path that is to be searched for in the	*/
/*		database.						*/
/*									*/
/* Output	If the given path is listed in the database as one of	*/
/*		the paths that is to be excluded from being		*/
/*		synchronized then this function will return true.	*/
/*		Otherwise, this function will return false.		*/
/************************************************************************/

condition WebPublish::is_synchronize_path_excluded(const String& path)
   {
      condition		result;
      String		key;
      DataRecord	record;

      key		= itsaccount;
      key		+= path;
      itsdatabase.Format_Record(WebPublishSynch, record);
      record.Set_Data(0, key);
      itsdatabase.Get_Record(WebPublishSynch, record);
      record.Get_Key(key);
      if (key.Length() != 0)
         {
	    result	= true;
	 }
      else
         {
	    result	= false;
	 }
      return(result);
   }

/************************************************************************/
/* Function	status synchronize_directory(DataRecord& account,	*/
/*		   const String& directory, const int depth)		*/
/*									*/
/* Purpose	This function is responsible for synchronizing a local	*/
/*		copy of a directory with an account's server. When this	*/
/*		function needs to synchronize a subdirectory, this	*/
/*		function will call itself.				*/
/*									*/
/* Input	This function expects the variable 'account' to contain	*/
/*		a formatted account record containing the account's	*/
/*		information. The variable 'directory' must contain the	*/
/*		local directory path that is to be synchronized with	*/
/*		the account's server. Finally, the variable 'depth'	*/
/*		must contain the number of directory levels to		*/
/*		traverse (or -1 for all subdirectories).		*/
/*									*/
/* Output	If this function is able to synchronize a local		*/
/*		directory with an account's server then this function	*/
/*		will return OK. If this function is not able to		*/
/*		synchronize a local directory then this function will	*/
/*		return ERROR. All errors by this function are reported	*/
/*		to stderr.						*/
/************************************************************************/

status WebPublish::synchronize_directory(DataRecord& account,
   const String& directory, const int depth)
   {
      status		result		= OK;
      String		workdir		= directory;
      String*		list		= (String*)0;
      register int	index;
      register int	index2;
      condition		removeit;
      String		servpath;
      String		localpath;
      String		website;
      String		filename;
      String		offset;
      int		items;

      if (itsverboseflag == true)
         {
	    std::cout << "Synchronizing directory ";
	    if (directory.Length() == 0)
	       {
	          std::cout << "/";
	       }
	    else
	       {
	          std::cout << directory.Data();
	       }
	    std::cout << std::endl;
	    fflush(stdout);
	 }
      if (itsftp.Get_Directory(workdir, items, list) == ERROR)
         {
	    /* Could not get dir from server.				*/

	    std::cout << "Could not get dir from server.\n";
	    result	= ERROR;
	 }
      else
         {
	    account.Get_Data(1, website);
	    if (website.Data()[website.Length() - 1] != '/')
	       {
	          website	+= "/";
	       }
	    account.Get_Data(5, offset);
	    if (offset.Data()[offset.Length() - 1] != '/')
	       {
	          offset	+= "/";
	       }
	 }
      for (index = 0; index < items && result == OK; index++)
         {
	    index2	= list[index].Length();
	    while (list[index].Data()[--index2] != ' ' && index2 > 0);
	    filename	= list[index].Data() + index2 + 1;
            servpath	= workdir;
	    servpath	+= filename;
	    localpath	= website;
	    if (offset.Length() != 0)
	       {
	          if (offset.Length() == 1)
		     {
	                localpath	+= servpath.Data() + offset.Length() - 1;
		     }
		  else
		     {
		        localpath	+= servpath.Data() + offset.Length();
		     }
	       }
	    else
	       {
	          localpath	+= servpath;
	       }
	    if (is_synchronize_path_excluded(servpath) == true)
	       {
	          if (itsverboseflag == true)
		     {
		        std::cout << "Excluding : " << servpath.Data() << std::endl;
			fflush(stdout);
		     }
	       }
	    else if (list[index].Data()[0] == 'D' ||
	       list[index].Data()[0] == 'd' ||
	       strstr(list[index].Data(), "<DIR>") != 0)
	       {
	          if (itsdir.Does_Directory_Exist(localpath) == false)
		     {
		        if (itspromptflag == true)
			   {
			      std::cout << "Remove server directory "
			         << servpath.Data() << "? (Y/N/A) ";
			      switch(response())
			         {
				    case 'Y':
				       removeit	= true;
				       break;
				    case 'N':
				       if (itsverboseflag == true)
				          {
				             std::cout << "Ignoring : " <<
				                servpath.Data() << std::endl;
					     fflush(stdout);
					  }
				       removeit	= false;
				       break;
				    case 'A':
				       removeit	= false;
				       itspath	= servpath;
				       result	= add_synch();
				 }
			   }
			else
			   {
			      removeit	= true;
			   }
			if (removeit == true)
			   {
		              if (itsverboseflag == true)
			         {
			            std::cout << "Removing server directory " <<
			               servpath.Data() << std::endl;
				    fflush(stdout);
			         }
			      if (itsftp.Purge_Directory(servpath) == ERROR)
			         {
			            std::cout << "Could not remove server directory.\n";
			            result	= ERROR;
			            break;
			         }
			   }
		     }
	          else if (depth == -1)
		     {
			servpath	+= "/";
			synchronize_directory(account, servpath, -1);
		     }
		  else if (depth > 1)
		     {
			servpath	+= "/";
		        synchronize_directory(account, servpath, depth - 1);
		     }
	       }
	    else
	       {
	          if (itsfile.Does_File_Exist(localpath) == false)
		     {
		        if (itspromptflag == true)
			   {
			      std::cout << "Remove server file "
			         << servpath.Data() << "? (Y/N/A) ";
			      switch(response())
			         {
				    case 'Y':
				       removeit	= true;
				       break;
				    case 'N':
				       if (itsverboseflag == true)
				          {
				             std::cout << "Ignoring : " <<
				                servpath.Data() << std::endl;
					     fflush(stdout);
					  }
				       removeit	= false;
				       break;
				    case 'A':
				       removeit	= false;
				       itspath	= servpath;
				       result	= add_synch();
				 }
			   }
			else
			   {
			      removeit	= true;
			   }
			if (removeit == true)
			   {
		              if (itsverboseflag == true)
			         {
			            std::cout << "Removing server file " <<
			               servpath.Data() << std::endl;
				    fflush(stdout);
			         }
			      if (itsftp.Change_Work_Directory(workdir) == ERROR)
			         {
				    std::cout << "Could not change work dir.\n";
				    result	= ERROR;
				    break;
				 }
			      if (itsftp.Remove_File(filename) == ERROR)
			         {
			            std::cout << "Could not remove server file.\n";
			            result	= ERROR;
			            break;
			         }
			   }
		     }
	       }
	 }
      if (list != (String*)0)
         {
	    delete [] list;
	 }
      return(result);
   }

/************************************************************************/
/* Function	status synchronize(void)				*/
/*									*/
/*									*/
/* Input	This function expects the private WebPublish member	*/
/*		variables to contain the relevant information.		*/
/*									*/
/* Output	If this function is able to synchronize all or part of	*/
/*		an account's website with the copy of the website on	*/
/*		the account's remote server then this function will	*/
/*		return OK. If this function is not able to synchronize	*/
/*		the account's website then this function will return	*/
/*		ERROR. All errors by this function are reported to	*/
/*		stderr.							*/
/************************************************************************/

status WebPublish::synchronize(void)
   {
      status		result		= OK;
      DataRecord	record;
      String		account;
      String		server;
      String		user;
      String		password;
      String		systype;
      String		directory;
      String		tempdir;

      itsdatabase.Format_Record(WebPublishAccounts, record);
      record.Set_Data(0, itsaccount);

      /* Make sure that the required information was specified.		*/

      if (itsaccount.Length() == 0)
         {
	    /* Missing required information.				*/

	    itserrorinfo	= "Attempted to synchronize an account";
	    itserror		= WebPublishRequire;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Get the account's database record.				*/

      else if (itsdatabase.Get_Record(WebPublishAccounts, record)
         == ERROR)
         {
	    /* Error searching for record.				*/

	    itserrorinfo	= "Attempted to synchronize account";
	    itserrorinfo	+= itsaccount;
	    itserror		= WebPublishBadGet;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
	    /* Make sure that the record was in the database.		*/

	    record.Get_Key(account);
	    if (account.Length() == 0)
	       {
	          /* Account does not exist.				*/

		  itserrorinfo	= "Attempted to synchronize account";
		  itserrorinfo	+= itsaccount;
		  result	= ERROR;
		  Report_Error();
	       }
	    else
	       {
	          /* Establish the FTP connection.			*/

		  record.Get_Data(2, server);
		  record.Get_Data(3, user);
		  record.Get_Data(4, password);
		  if (itsverboseflag == true)
		     {
		        std::cout << "Establishing a connection to "
			   << server.Data() << std::endl;
			fflush(stdout);
		     }
		  if (itsftp.Connect_To_Host(server) == ERROR)
		     {
		        /* Could not establish FTP connection.		*/

			itserrorinfo	= "Attempted to synchronize "
					  "account";
			itserrorinfo	+= itsaccount;
			itserror	= WebPublishBadHost;
			result		= ERROR;
			Report_Error();
		     }
		  else if (itsftp.Logon_To_Host(user, password) == ERROR)
		     {
		        /* Could not logon to host.			*/

			itserrorinfo	= "Attempted to synchronize "
					  "account";
			itserrorinfo	+= itsaccount;
			itserror	= WebPublishBadLogon;
			result		= ERROR;
			Report_Error();
		     }
		  else
		     {
		        if (itsverboseflag == true)
		           {
			      std::cout << "Successfully logged into " <<
			         server.Data() << ".\n";
			      itsftp.Get_System_Type(systype);
			      if (systype.Length() != 0)
			         {
				    std::cout << "System type = " <<
				       systype.Data() << std::endl;
				 }
			      std::cout << "Synchronizing the account "
			         << account.Data() << ".\n";
			      fflush(stdout);
		           }
			record.Get_Data(5, directory);
			if (directory.Length() != 0)
			   {
			      if (directory.Data()[directory.Length() - 1] != '/')
			         {
				    directory	+= "/";
				 }
			   }
		        if (itsdirectory.Length() != 0)
		           {
			      directory	+= itsdirectory;
			      directory	+= "/";
			   }
		        result	= synchronize_directory(record, directory, itsdepth);
			if (itsftp.Close_Connection_To_Host() == ERROR)
			   {
			      /* Failed to close FTP connection.	*/

			      itserrorinfo	= "Attempted to "
						  "synchronize account";
			      itserrorinfo	+= itsaccount;
			      itserror		= WebPublishFTPClose;
			      result		= ERROR;
			      Report_Error();
			   }
			else if (itsverboseflag == true)
			   {
			      std::cout << "Connection to " << server.Data()
			         << " has been terminated.\n\n";
			      fflush(stdout);
			   }
		     }

	       }
	 }
      return(result);
   }
